use schemars::JsonSchema;
use serde::{Deserialize, Serialize};
use serde_json::Value;
use std::collections::HashMap;

use crate::color::Color;

#[derive(Serialize, Deserialize, Debug, JsonSchema)]
#[serde(rename_all = "camelCase")]
pub struct Material {
    pub shininess: f64,

    /// Diffuse property. Both color and map are supported.
    /// Color and map replace each other.
    pub diffuse: Option<ColorMap>,

    /// Specular map. Both color and map are supported. The color (white if
    /// undefined) will be used as a factor to the map, if any.
    pub specular: Option<ColorMap>,

    /// Emission color. No map supported
    pub emission: Option<ColorMap>,

    /// Normal map.
    pub normal: Option<BasicMap>,

    /// Roughness map.
    pub roughness: Option<BasicMap>,

    /// Alpha value or map. Value and map replace each other.
    /// Values in the range 0.0 to 1.0. 0.0 is transparent, 1.0 is opaque.
    /// Map is grayscale. Black is transparent, white is opaque.
    pub alpha: Option<ValueMap>,

    /// Diffuse delta map. The delta map modifies the diffuse
    /// color/map and typically has it's own mapping.
    ///
    /// Format: 24 Bit (RGB)
    ///
    /// The handling (for each channel/color) is:
    /// - linear mapping of [0, 255] to [0, 2]
    /// - multiplication with the diffuse/color value
    ///
    /// 0 sets the original value to 0.
    /// 127/128 keeps the original value.
    /// 255 doubles the original value.
    pub diffuse_delta: Option<TextureMap>,

    /// Roughness delta map. The delta map modifies the roughness
    /// value/map and typically has it's own mapping.
    ///
    /// Format: 8 Bit (Grayscale)
    ///
    /// The handling is:
    /// - linear mapping of [0, 255] to [0, 2]
    /// - multiplication with the roughness value
    ///
    /// 0 sets the original value to 0.
    /// 127/128 keeps the original value.
    /// 255 doubles the original value.
    pub roughness_delta: Option<TextureMap>,

    /// Default mapping, may be overridden by specific texture maps.
    pub mapping: Option<TextureMapping>,

    /// Taxonomy information according to docs/ig_Taxonomy
    pub taxonomy: Option<HashMap<String, Value>>,

    /// Optional geometry-related parameters, to be resolved at client-side.

    /// "DoNotRescale":
    /// Marks the material as not re-scaleable. Re-scaling can be applied
    /// for psychological reasons, for instance.
    /// Target type: Boolean
    ///
    /// "Overlay":
    /// For alpha-map materials, tells the renderer that the associated geometry
    /// should be an overlay to other geometries at the same location.
    /// Target type: Boolean
    ///
    /// IGXC Compatibility: Metamaterial parameters but without redundancy.
    pub parameters: Option<HashMap<String, Value>>,
}

#[derive(Serialize, Deserialize, Debug, JsonSchema)]
#[serde(rename_all = "camelCase")]
/// Texture map with additional scalar value. The relationship between both
/// will be defined in the application context.
pub struct ValueMap {
    #[serde(flatten)]
    pub basic: BasicMap,

    /// A scalar value. If the map is supported or not, and it's relation-
    /// ship to a map entry, is defined in the outer context.
    pub value: Option<f64>,
}

#[derive(Serialize, Deserialize, Debug, JsonSchema)]
#[serde(rename_all = "camelCase")]
/// Texture map with additional color value. The relationship between both
/// will be defined in the application context.
pub struct ColorMap {
    #[serde(flatten)]
    pub basic: BasicMap,

    /// A color value.
    pub color: Color,
}

#[derive(Serialize, Deserialize, Debug, JsonSchema)]
#[serde(rename_all = "camelCase")]
pub struct BasicMap {
    pub map: Option<TextureMap>,
}

#[derive(Serialize, Deserialize, Debug, JsonSchema)]
#[serde(rename_all = "camelCase")]
pub struct TextureMap {
    /// The mandatory format of the map.
    pub map_format: TextureMapFormat,

    /// Absolute or relative path to the texture image.
    pub map: String,

    /// An optional UV mapping.	  
    pub mapping: Option<TextureMapping>,
}

#[derive(Serialize, Deserialize, Debug, JsonSchema)]
pub enum TextureMapFormat {
    JPEG,
    PNG,
}

#[derive(Serialize, Deserialize, Debug, JsonSchema)]
#[serde(rename_all = "camelCase")]
/// A Transformation of the UV Set for a Material  
/// Ordering R///S///T///V  
/// Also see ThreeJS issue #15831
///
/// This order minimizes shearing and improves content creation.
///
/// Combination with a GeometryMapping looks like this:
/// GS///GT///GR///MR///MS///MT///V
///
/// GR .. Matrix of GeometryMapping.Rotation
/// GS .. Matrix of GeometryMapping.Scale///
/// GT .. Matrix of GeometryMapping.Translation///
///
/// MR .. Matrix of TextureMapping.Rotation
/// MS .. Matrix of TextureMapping.Scale///
/// MT .. Matrix of TextureMapping.Translation///
///
/// V .. (UV)-Vector to be transformed
///
/// Rotation is clockwise in degrees
pub struct TextureMapping {
    pub translation_s: f64,

    pub translation_t: f64,

    pub rotation: f64,

    #[serde(default = "default_scale")]
    /// default value = 1
    pub scale_s: f64,

    #[serde(default = "default_scale")]
    /// default value = 1
    pub scale_t: f64,
}

fn default_scale() -> f64 {
    1.0
}
