use schemars::JsonSchema;
use serde::{Deserialize, Serialize};
use serde_json::Value;
use std::collections::HashMap;

#[derive(Serialize, Deserialize, Debug, JsonSchema)]
#[serde(rename_all = "camelCase")]
/// ig Geometry representation, mesh based.
pub struct Geometry {
    /// The mandatory format of the optional preview mesh.
    pub preview_format: Option<GeometryPreviewFormat>,

    /// An optional mesh to be displayed until the main mesh is loaded.
    pub preview: Option<String>,

    /// The mandatory format of the optional preview mesh.

    /// Version: OC 1.1	  
    pub collider_format: Option<GeomtryColliderFormat>,

    /// An optional mesh to be used for collision detection.
    ///
    /// Version: OC 1.1
    pub collider: Option<String>,

    /// The mandatory format of the optional main mesh.
    pub mesh_format: Option<GeometryMeshFormat>,

    /// An optional mesh to be displayed as main mesh. Note, in some case a
    /// geometry does not provide it, e.g. for super deformations.
    pub mesh: Option<String>,

    /// The mandatory format of the optional normal map.
    pub normal_map_format: Option<GeometryNormalMapFormat>,

    /// An optional normal map to be applied to the main mesh.
    pub normal_map: Option<String>,

    /// The mandatory format of the optional deformation.
    /// Allowed formats are "FFD" and "JSON".
    pub deformation_format: Option<GeometryDeformationFormat>,

    /// Deformation description.
    pub deformation: Option<String>,

    #[serde(rename = "deformationXDataFormat")]
    /// The mandatory format of the optional shape-key file, or other extra deformation data.
    pub deformation_xdata_format: Option<GeometryDeformationXDataFormat>,

    #[serde(rename = "deformationXData")]
    /// Shape-key file or other extra data, belonging to a deformation.
    pub deformation_xdata: Option<String>,

    /// Optional geometry-related parameters, to be resolved at client-side.
    ///
    /// "SuperDeformation":
    /// Marks the geometry as a super deformation. The assigned deformation
    /// will be applied to the local geometry (if any) as well as recursively
    /// to all child geometries.
    /// Target type: Boolean
    ///
    /// "NormalMapTiling":
    /// The Geometry Normal Map should be repeated instead of clamped outside 0..1 UV range
    /// Some 3D libraries call this "wrapping mode" if true set to "repeat", default is "clamp"
    /// Target type: Boolean
    /// Version: OC 1.1
    ///
    /// "NormalMapStrength":
    /// Changes the influence of the Geometry Normal Map in light calculation, default: 1,
    /// by multiplying the strength of the geometry normal map with this factor.
    /// Target type: number
    /// Version: OC 1.1
    ///
    /// "RenderOrder"
    /// FRender order for 3D libraries that render transparent objects in a separate, second pass, back to front.
    /// Negative values are rendered earlier in the process, positive values later, 0 is default and keeps library ordering.
    /// Raytracers and renderers with advanced transparency handling must ignore this!
    /// This is NOT intended for effects different from a natural/physical correct scene, i.e. it can not be used for
    /// artistic effects!
    /// Target type: number
    /// Version: OC 1.1
    pub parameters: Option<HashMap<String, Value>>,
}

#[derive(Serialize, Deserialize, Debug, JsonSchema)]
pub enum GeometryPreviewFormat {
    OpenCTM,
    OBJ,
    FBX,
}

#[derive(Serialize, Deserialize, Debug, JsonSchema)]
pub enum GeomtryColliderFormat {
    OpenCTM,
    OBJ,
    FBX,
}

#[derive(Serialize, Deserialize, Debug, JsonSchema)]
pub enum GeometryMeshFormat {
    OpenCTM,
    OBJ,
    FBX,
    DWF,
}

#[derive(Serialize, Deserialize, Debug, JsonSchema)]
pub enum GeometryNormalMapFormat {
    JPEG,
    PNG,
}

#[derive(Serialize, Deserialize, Debug, JsonSchema)]
pub enum GeometryDeformationFormat {
    FFD,
    JSON,
}

#[derive(Serialize, Deserialize, Debug, JsonSchema)]
pub enum GeometryDeformationXDataFormat {
    #[serde(rename = "glTF")]
    GLTF,
    GLB,
}
