use std::collections::HashMap;

use schemars::JsonSchema;
use serde::{Deserialize, Serialize};

use crate::{
    preview::Preview, status::Status, AttachPoint, Commercial, Component, TopView, Transform,
};

#[derive(Serialize, Deserialize, Debug, JsonSchema)]
pub enum ProductKind {
    /// regular product
    Product,
    /// a commercial container for multiple products
    Set,
    Placeholder,
}

#[derive(Serialize, Deserialize, Debug, JsonSchema)]
#[serde(rename_all = "camelCase")]
/// An initial or incremental product representation.
pub struct Product {
    pub kind: Option<ProductKind>,
    /// Optional state information for a product. Even if the state object
    /// exists, not all attributes may be set.
    /// The state information always entirely updates an existing state
    /// information.
    pub state: Option<ProductState>,
    /// Optional world transform of the product representation.
    /// If the transform is not null, it entirely replaces the existing transform.
    pub transform: Option<Transform>,
    /// The mandatory attribute maps material categories to materials.
    /// Both, material categories and materials should be compatible with a
    /// three-level technical namespace.
    /// Implicit material categories consist of prefix '@' and a material,
    /// and should be included too, even if this is kind of redundant.
    ///
    /// IGXC Compatibility: In IGXC this attribute was named Categories.
    pub material_categories: Option<HashMap<String, String>>,
    /// Product categories for the client-side implementation of planning
    /// behavior.
    /// If Categories is not null, it entirely replaces the existing categories.
    ///
    /// IGXC Compatibility: In IGXC, categories were considered from the
    /// structure's root entry. Now they can be updated, even if the
    /// structure does not change.
    pub categories: Option<Vec<String>>,
    /// The tree structure that describes the product in 3D.
    /// If the structure is not null, it entirely replaces the existing
    /// structure.
    pub structure: Option<Vec<Component>>,
    /// An optional product representation from top-view perspective.
    /// Version: OC 1.2
    ///
    pub top_view: Option<TopView>,
    /// An optional product preview image.
    /// Version: OC 1.2
    pub preview: Option<Preview>,
    /// Attachment points for the client-side creation of neighbor and
    /// parent-child relationships.
    /// If Points is not null, it entirely replaces the existing points.
    ///
    /// IGXC Compatibility: In IGXC, points were considered from the
    /// structure's root entry. Now they can be updated, even if the
    /// structure does not change.
    pub points: Option<Vec<AttachPoint>>,
    /// Commercial data related to this product
    pub commercial: Option<Commercial>,
    /// Contains product related status information.
    pub status: Status,
}

#[derive(Serialize, Deserialize, Debug, JsonSchema)]
#[serde(rename_all = "camelCase")]
/// State information of a product.
///
/// IGXC Compatibility: The PBR hash has been removed for now.
pub struct ProductState {
    /// Optional commercial instance id. Will be sent from client to server
    /// (and retour) to identify the object in the configurator/basket.
    ///
    /// Note. comId should match to Commercial.Id if Commercial exists.
    pub com_id: Option<String>,

    /// Optional graphical instance it. Will typically be sent from client
    /// (where this id is created and managed) to the server and returned
    /// from it, for clear assignment of product representation.
    pub gfx_id: Option<String>,

    /// The geometric hash may be created from a configurator service. It
    /// will be sent to the client, which may return it during upcoming
    /// update requests. The service may then detect a non-geometric update
    /// of the product representation and send a corresponding update.
    pub geometric_hash: Option<String>,

    /// The visual hash may be created from a configurator service. It
    /// will be sent to the client, which may return it during upcoming
    /// update requests. The service may then detect a non-visual update
    /// of the product representation and send a corresponding update.
    pub visual_hash: Option<String>,
}
