use schemars::JsonSchema;
use script::Script;
use serde::{Deserialize, Serialize};
use status::Status;
use std::collections::HashMap;

mod attach_point;
mod camera;
pub mod cns;
mod color;
mod commercial;
mod component;
mod geometry_mapping;
pub mod ig;
mod image_format;
mod material;
mod material_index;
mod preview;
mod product;
mod quaternion;
mod script;
mod status;
mod top_view;
mod transform;
mod vector;

pub use attach_point::*;
pub use camera::*;
pub use color::*;
pub use commercial::*;
pub use component::*;
pub use geometry_mapping::*;
pub use image_format::*;
pub use material::*;
pub use product::*;
pub use quaternion::*;
pub use script::*;
pub use top_view::*;
pub use transform::*;
pub use vector::*;

#[derive(Serialize, Deserialize, Debug, JsonSchema)]
#[serde(rename_all = "camelCase")]
/// The Scene is the central part of an OC data set. Each OC data set
/// contains exactly one.
///
/// A Scene may contain initial product representations and/or updates.
///
/// A Scene may contain products that should be inserted instantly, as well
/// as add-ons to be inserted later on and shown as placeholders before.
pub struct Scene {
    /// The mandatory attribute format must be structured as follows:
    ///
    /// "OpenConfiguration_<Major/>.<Minor/> [PRE]"
    ///
    /// Legal combinations of <Major/>.<Minor/> are: 1.0, 1.1 and 1.2
    ///
    /// The optional postfix " PRE" marks a preliminary format.
    pub format: String,
    /// This optional attribute contains informal information about the
    /// creator of this OC data set.
    pub creator: Option<String>,
    /// This attribute contains general status information about the Scene contents.
    pub status: Status,
    /// The representations of the scene. Legal values are:
    ///
    /// "Standard" - Real-time mesh data.
    ///
    /// "CAD_BIM" - CAD/BIM volume bodies.
    ///
    /// "Photo" - High-resolution mesh data.
    ///
    /// This just lists possible usages and can be used for fast filtering,
    /// it does not influence the actual data. A "Photo" renderer would still use
    /// meshes attached as geometry.mesh
    pub representations: Option<Vec<Representation>>,
    /// This optional attribute may contain the id of a server-side session.
    pub configuration_id: Option<String>,
    /// This optional attribute may contain the link of the original configuration.
    pub configuration_link: Option<String>,
    /// Optional, embedded catalog.
    pub catalog: Option<Vec<CatalogEntry>>,
    /// The optional attribute provides a partial URI to be added to all
    /// relative asset uris. A valid base path should normally start with
    /// https:// or file://, and end with either a slash or the path separator character of the operating system.
    pub base_path: Option<String>,
    /// The mandatory attribute provides unique content hashes for assets
    /// that are directly referenced in the OC data set. The specific hash
    /// algorithm is unspecified. It may be an MD5 hash of the binary
    /// content for instance. But low-res assets may use the same hash as
    /// the originals, they are derived from. So, the only operation that is
    /// legal for hash, is to compare them with an optionally existing one.
    ///
    /// Any asset should have an entry here as downloaders may iterate thru
    /// this record rather than exploring the other data entities.
    pub hashes: HashMap<String, Option<String>>,
    /// Redirections maybe needed to convert absolute asset urls into file
    /// names, especially when compiling a self-contained OC zip.
    /// The attribute is optional.
    ///
    /// Version: OC 1.3
    pub redirections: Option<HashMap<String, String>>,
    /// The optional attribute contains all client-side JavaScript packages.
    pub scripts: Option<Vec<Script>>,
    /// The optional attribute provides an embedded geometry index.
    ///
    /// IGXC Compatibility: now it's embedded, self-contained geometries
    /// rather than just geometry names. Thus, separate assignments of
    /// geometry normal maps, deformations, etc. are removed. Also, the
    /// embedded geometry definition is optional. There can be external
    /// geometry definitions, too.
    pub geometries: Option<HashMap<String, Geometry>>,
    /// The optional attribute provides links to external geometry indices.
    /// The key should be a two-level technical namespace to speedup the
    /// geometry lookup.
    /// The value must be an absolute or relative URI.
    /// The content of the value, must be de-serialized as GeometryIndex.
    ///
    /// IGXC Compatibility: In IGXC, this concept did not exist.
    pub geometry_indexes: Option<HashMap<String, String>>,
    /// The optional attribute provides an embedded geometry index.
    ///
    /// IGXC Compatibility: The embedded material definition is optional.
    /// There can be external material definitions, too.
    pub materials: Option<HashMap<String, Material>>,
    /// The optional attribute provides links to external geometry indices.
    /// The key should be a two-level technical namespace to speedup the
    /// geometry lookup.
    /// The value must be an absolute or relative URI.
    /// The content of the value, must be de-serialized as MaterialIndex.
    ///
    /// IGXC Compatibility: In IGXC, this concept did not exist.
    pub material_indexes: Option<HashMap<String, String>>,
    /// An optional camera setup to restore the camera in another viewer.
    pub camera: Option<Camera>,
    /// The mandatory attribute contains the products to be inserted into/
    /// updated in the client world.
    ///
    /// IGXC Compatibility: in IGXC, there is only one product.
    pub products: Vec<Product>,
    /// Commercial products without an own visual representation.
    ///
    /// Version: OC 1.3
    pub com_products: Option<Vec<Commercial>>,
}

#[derive(Serialize, Deserialize, Debug, JsonSchema)]
pub enum Representation {
    Standard,
    #[serde(rename = "CAD_BIM")]
    CadBim,
    Photo,
}

#[derive(Serialize, Deserialize, Debug, JsonSchema)]
#[serde(rename_all = "camelCase")]
pub struct CatalogEntry {
    pub entry_type: CatalogEntryType,
    /// Id of the entry. Needed if catalog processing is done server side.
    pub entry_id: String,
    /// Localized entry text. Key is ISO 639-1 language code.
    pub entry_text: HashMap<String, Vec<String>>,
    /// For type Article, the native id of the article to be created.
    pub article_id: String,
    /// For type Folder, the contained entries: folders and/or articles
    pub entries: Vec<CatalogEntry>,
}

#[derive(Serialize, Deserialize, Debug, JsonSchema)]
pub enum CatalogEntryType {
    Folder,
    Article,
}

#[derive(Serialize, Deserialize, Debug, JsonSchema)]
#[serde(rename_all = "camelCase")]
/// An abstract representation of a 3D Geometry.
pub struct Geometry {
    /// Optional representation in the ig format.
    pub ig: Option<ig::Geometry>,

    /// Optional representation in the CADENAS format.
    pub cns: Option<cns::Geometry>,
}
