use std::collections::HashMap;

use schemars::JsonSchema;
use serde::{Deserialize, Serialize};

use crate::{GeometryMapping, Transform};

#[derive(Serialize, Deserialize, Debug, JsonSchema)]
#[serde(rename_all = "camelCase")]
/// A 3D component that is used to model products. A product corresponds to
/// a tree of Objects. Inside the tree, transforms are always inherited from
/// parent to child and concatenated. Material Categories are inherited from
/// parent to child, as long as the child does not have one.
///
/// IGXC Compatibility: Points and Categories are moved to the Product.
/// Evaluator and Evaluator Parameters have been removed.
pub struct Component {
    /// The Path describes the tree structure. The root path "." is
    /// mandatory. First-level children may be "o1", "o2", "e1", and so on.
    /// Again, "o1.o1" is a child of "o1". And so on.
    /// The sequence of objects must be compatible to the corresponding tree
    /// defined by Path of these objects. So any parent must occur in the
    /// array before it's children, and so on.
    pub path: String,

    #[serde(default = "default_visible")]
    /// The attribute controls the visibility of the object. If it is
    /// invisible, all children are invisible too (recursively). If the
    /// attribute is unspecified in a JSON file, it is considered to be true!
    ///
    /// default value = true
    pub visible: bool,

    #[serde(default)]
    /// The attribute controls if the object is selectable or--if not--what
    /// should happen then. Allowed values are:
    /// "Self", "Parent" (default) and "None" (see above for description).
    ///
    /// default value = "Parent"
    pub selectable: Option<ComponentSelectable>,

    /// The optional attribute specifies a fully scoped geometry.
    pub geometry: Option<String>,

    /// The optional attribute specifies additional parameters to be applied
    /// to the geometry.
    pub geometry_parameters: Option<HashMap<String, String>>,

    /// The optional attribute defines a deformation value in the range of
    /// 0 to 1, to be applied to the geometry. 0 means no deformation, 1
    /// means full deformation.
    /// This setting has a higher priority, compared with the Deformation
    /// specified as Geometry Parameter in ig.Geometry.
    pub deformation: Option<f64>,

    /// The optional attribute defines a layer tag for this object.
    pub layer: Option<String>,

    /// The optional attribute sets a material category for the object and
    /// all children (recursively) that do not set one themselves.
    pub material_category: Option<String>,

    /// The optional attribute sets a mapping to be applied to the non-object
    /// bound UV set - typically, the UV set used for the material.
    pub mapping: Option<GeometryMapping>,

    /// The optional attribute specifies an offset to the parent. The parent's
    /// transform will always be concatenated with the child's one.
    /// The transform of root objects (Path: ".") will be ignored if existing!
    pub transform: Option<Transform>,

    /// Optional attribute that assigns an interactor script to the object.
    /// Fully scoped name of an interactor class. The class must be provided
    /// by the scripts in the scene.
    pub interactor: Option<String>,

    /// Optional attribute that provides further parameters for the created
    /// interactor.
    pub interactor_parameters: Option<HashMap<String, String>>,

    /// One or more symbolic id to identify a component at run-time. Unlike Path, a
    /// Tag is more robust and not biunique.
    /// Version: OC 1.1
    pub tags: Option<Vec<String>>,

    /// The optional attribute defines property ids that are linked to this
    /// object (and it's children). This information may be used for context-
    /// sensitive configuration.
    pub options: Option<Vec<String>>,
}

#[derive(Serialize, Deserialize, Debug, JsonSchema)]

pub enum ComponentSelectable {
    #[serde(rename = "Self")]
    Component,
    Parent,
    None,
}

impl Default for ComponentSelectable {
    fn default() -> Self {
        Self::Parent
    }
}

fn default_visible() -> bool {
    true
}
