use std::collections::HashMap;

use schemars::JsonSchema;
use serde::{Deserialize, Serialize};

#[derive(Serialize, Deserialize, Debug, JsonSchema)]
#[serde(rename_all = "camelCase")]
pub struct Commercial {
    /// The instance id of the product in the configurator/basket.
    pub id: String,

    /// The instance id of an associated master product, e.g. a set.
    pub master_id: Option<String>,

    /// The (basic) article code.
    pub article: Option<String>,

    /// Additional article code that describes the specific variant.
    pub variant: Option<String>,

    /// The native id.
    ///
    /// In case of IDM, this is <series/>|<product/>.
    ///
    /// In case of XcalibuR, this is <model_source_id/>@@<product_source_id/>
    pub native_id: Option<String>,

    /// Short (one-line), localized article text.
    pub short_text: Option<HashMap<String, String>>,

    /// Long, localized article text. Key is ISO 639-1 language code.
    pub long_text: Option<HashMap<String, Vec<String>>>,

    /// Long, localized article text that describes the variant. Used as an addition
    /// to the long text, not as replacement. Key is ISO 639-1 language code.
    pub variant_text: Option<HashMap<String, Vec<String>>>,

    ///  The price unit, for all prices of this article. Possible values:
    ///  "EUR_Ct" - Euro Cent
    ///  "USD_Ct" - USD Cent
    ///  "CHF_Rp" - CHF Rappen
    pub price_unit: Option<String>,

    ///  The article's sales price, according to the price unit.  
    pub sales_price: Option<f64>,

    ///  The article's purchase price, according to the price unit.  
    pub purchase_price: Option<f64>,

    ///
    /// The properties of the article.
    ///
    pub properties: Vec<Property>,
}

#[derive(Serialize, Deserialize, Debug, JsonSchema)]
#[serde(rename_all = "camelCase")]
pub struct Property {
    /// The unique id of the property.
    pub property_id: String,

    // TODO: remove the option from the values once the icom server no longer adds
    // "*": null
    /// Localized property text. Key is ISO 639-1 language code.
    pub property_text: HashMap<String, Option<String>>,

    /// The optional id of the property's current value.
    pub value_id: Option<String>,

    // TODO: remove the option from the values once the icom server no longer adds
    // "*": null
    /// The localized text of the value. Needed if valueId cannot be
    /// resolved in Values.
    pub value_text: Option<HashMap<String, Option<String>>>,

    /// The position of the property in the product's property editor.
    pub position: i32,

    /// The values of the property.
    pub values: Option<Vec<Value>>,
}

#[derive(Serialize, Deserialize, Debug, JsonSchema)]
#[serde(rename_all = "camelCase")]
/// Value of properties
pub struct Value {
    /// The unique id of the property value.
    pub value_id: String,

    /// Localized value text. Key is ISO 639-1 language code.
    pub value_text: HashMap<String, String>,

    /// The position of the value in the property's selection list.
    pub position: i32,

    /// The sub values of the value.
    pub values: Vec<Value>,
}
