use schemars::JsonSchema;
use serde::{Deserialize, Serialize};

use crate::{Quaternion, Vector3};

#[derive(Serialize, Deserialize, Debug, JsonSchema)]
/// The type of the point
/// "Origin" - the origin of a product, matches to "Child"
/// "Child" - a (virtual) child can be placed, matches to "Origin"
/// "Left" - L-R neighborship, matches to "Right"
/// "Right" - L-R neighborship, matches to "Left"
/// "Top" - vertical neighborship, matches to "Bottom"
/// "Bottom" - vertical neighborship, matches to "Top"
/// "Front" - front/back neighborship, matches to "Back"
/// "Back" - front/back neighborship, machtes to "Front"
/// "Alignment" - auxiliary point to support alignments
/// "Maximum" - marks the maximum of the bbox
/// "Minimum" - marks the minimum of the bbox
pub enum AttachPointType {
    Origin,
    Child,
    Left,
    Right,
    Top,
    Bottom,
    Front,
    Back,
    Alignment,
    Maximum,
    Minimum,
}

#[derive(Serialize, Deserialize, Debug, JsonSchema)]
pub enum AttachPointRepresentation {
    Standard,
    Ghost,
    Ghost2,
}

#[derive(Serialize, Deserialize, Debug, JsonSchema)]
#[serde(rename_all = "camelCase")]
/// An attachment point for planning support and placement of children.
/// All attributes are mandatory.
///
/// IGXC Compatibility: the Rotation is changed to Quaternion, in IGXC it
/// was Euler-based.
pub struct AttachPoint {
    #[serde(rename = "type")]

    /// The type of the point
    pub kind: AttachPointType,

    /// Own key of the point. Should match the counterpart's foreign key.
    pub key: String,

    /// Allowed keys of counterpart's points. May contain /// and ? characters.
    pub foreign: String,

    /// The position of the point, relative to the Product.
    pub point: Vector3,

    /// The orientation of the point, relative to the Product.
    pub rotation: Quaternion,

    /// Attribute controls if the point is enabled.
    /// If disabled, it should be invisible too.
    pub enabled: bool,

    /// The visual representation of the attachment point.
    pub representation: AttachPointRepresentation,
}
