use hmac::digest::{
    Update, BlockInput, FixedOutput, Reset,
    generic_array::ArrayLength,
};
use crate::error::Error;

/// Calculate the TOTP code for the current system time.
#[cfg(any(feature = "std", feature = "libc"))]
pub fn totp_now<D>(key: &[u8]) -> Result<u64, Error>
    where D: Update + BlockInput + FixedOutput + Reset + Default + Clone,
          D::BlockSize: ArrayLength<u8>,
{
    totp::<D>(&TotpOptions::with_key_now(key))
}

/// Calculate the TOTP code for the given options.
pub fn totp<D>(options: &TotpOptions) -> Result<u64, Error>
    where D: Update + BlockInput + FixedOutput + Reset + Default + Clone,
          D::BlockSize: ArrayLength<u8>,
{
    if options.step == 0 {
        return Err(Error::InvalidStep);
    }

    if options.timestamp < options.epoch {
        return Err(Error::InvalidTimeOrEpoch);
    }

    let counter = (options.timestamp - options.epoch) / options.step;
    crate::hotp::<D>(options.key, counter, options.digits)
}

/// Options for generating TOTP codes.
pub struct TotpOptions<'a> {
    /// The secret, shared key to use to generate the code.
    pub key: &'a [u8],
    /// The number of digits that should be in the code.
    ///
    /// Defaults to `6`.
    pub digits: u32,
    /// The number of seconds in a generation window.
    ///
    /// Defaults to `30`.
    pub step: u64,
    /// The seconds from the Unix epoch to use as `T0`.
    ///
    /// Defaults to `0`.
    pub epoch: u64,
    /// The timestamp to generate the TOTP code at.
    ///
    /// Defaults to the current system time if `std` or `libc` are enabled.
    pub timestamp: u64,
}

impl<'a> TotpOptions<'a> {
    /// Create a `TotpOptions` with the given key at the current system time.
    ///
    /// `digits`, `step`, and `epoch` are all initialised to their default values.
    #[cfg(any(feature = "std", feature = "libc"))]
    pub fn with_key_now(key: &'a [u8]) -> Self {
        Self {
            key,
            digits: 6,
            step: 30,
            epoch: 0,
            timestamp: now().unwrap_or_default(),
        }
    }
}

#[cfg(all(feature = "std", not(feature = "libc")))]
fn now() -> Option<u64> {
    use std::time::{SystemTime, UNIX_EPOCH};

    SystemTime::now().duration_since(UNIX_EPOCH)
        .ok()
        .map(|t| t.as_secs())
}

#[cfg(feature = "libc")]
fn now() -> Option<u64> {
    let time = unsafe {
        libc::time(core::ptr::null_mut())
    };
    if time < 0 {
        return None;
    }

    Some(time as u64)
}
