use sha1::Sha1;
use crate::totp::TotpOptions;
use sha2::{Sha256, Sha512};

#[test]
fn hotp() {
    const SECRET: &[u8] = b"12345678901234567890";
    const DIGITS: u32 = 6;
    const EXPECTED: &[u64] = &[
        755224,
        287082,
        359152,
        969429,
        338314,
        254676,
        287922,
        162583,
        399871,
        520489,
    ];

    for (i, &otp) in EXPECTED.iter().enumerate() {
        assert_eq!(
            crate::hotp::<Sha1>(SECRET, i as u64, DIGITS).unwrap(),
            otp,
        );
    }
}

#[test]
fn totp() {
    const SECRETS: [&[u8]; 3] = [
        b"12345678901234567890",
        b"12345678901234567890123456789012",
        b"1234567890123456789012345678901234567890123456789012345678901234",
    ];
    const DIGITS: u32 = 8;
    const EXPECTED: &[(u64, u64)] = &[
        (59, 94287082),
        (59, 46119246),
        (59, 90693936),
        (1111111109, 07081804),
        (1111111109, 68084774),
        (1111111109, 25091201),
        (1111111111, 14050471),
        (1111111111, 67062674),
        (1111111111, 99943326),
        (1234567890, 89005924),
        (1234567890, 91819424),
        (1234567890, 93441116),
        (2000000000, 69279037),
        (2000000000, 90698825),
        (2000000000, 38618901),
        (20000000000, 65353130),
        (20000000000, 77737706),
        (20000000000, 47863826),
    ];

    for (i, &(timestamp, otp)) in EXPECTED.iter().enumerate() {
        let options = TotpOptions {
            key: SECRETS[i % 3],
            digits: DIGITS,
            epoch: 0,
            step: 30,
            timestamp,
        };

        let result = match i % 3 {
            0 => crate::totp::<Sha1>(&options),
            1 => crate::totp::<Sha256>(&options),
            2 => crate::totp::<Sha512>(&options),
            _ => unreachable!(),
        };

        assert_eq!(
            result.unwrap(),
            otp,
        );
    }
}
