use hmac::{
    Hmac, Mac, NewMac,
    digest::{
        Update, BlockInput, FixedOutput, Reset,
        generic_array::ArrayLength,
    },
};
use crate::error::Error;

/// Calculate the HOTP code for the given key, counter, and digits.
///
/// Note that according to RFC 4226, the digest algorithm MUST be SHA1.
/// However, this implementation will use any digest algorithm the caller
/// sees fit to use.
pub fn hotp<D>(key: &[u8], counter: u64, digits: u32) -> Result<u64, Error>
where D: Update + BlockInput + FixedOutput + Reset + Default + Clone,
      D::BlockSize: ArrayLength<u8>,
{
    if digits < 6 {
        return Err(Error::InvalidDigitsSize);
    }

    let data = counter.to_be_bytes();

    let mut hmac = Hmac::<D>::new_from_slice(key).map_err(Error::InvalidKeySize)?;
    hmac.update(&data);

    let result = hmac.finalize().into_bytes();

    let offset = usize::from(result[result.len() - 1] & 0xF);
    let binary = (u64::from(result[offset] & 0x7F) << 24)
        | (u64::from(result[offset + 1]) << 16)
        | (u64::from(result[offset + 2]) << 8)
        | u64::from(result[offset + 3]);

    Ok(binary % 10u64.pow(digits))
}
