// SPDX-FileCopyrightText: 2022 Declan Rixon <declan.fraser.rixon@gmail.com>
//
// SPDX-License-Identifier: GPL-3.0-only

#[test]
fn full_page() {
    use crate::*;
    use indoc::indoc;

    struct HeaderLink(Template);

    impl HeaderLink {
        pub fn template() -> Self {
            Self(
                TagPair::new("li", true)
                    .content(
                        TagPair::new("a", true)
                            .value("href", Placeholder("HREF"))
                            .content(Placeholder("TEXT")),
                    )
                    .into(),
            )
        }

        pub fn new<T, U>(&self, text: T, href: U) -> Template
        where
            Template: From<T> + From<U>,
        {
            self.0.clone().sub("HREF", href).sub("TEXT", text).into()
        }
    }

    struct Card(Template);

    impl Card {
        pub fn template() -> Self {
            Self(
                TagPair::new("div", false)
                    .value("class", "card")
                    .content(
                        TagPair::new("div", false)
                            .value("class", "container")
                            .content(TagPair::new("h1", true).content(Placeholder("TITLE")))
                            .content(TagPair::new("p", false).content(Placeholder("CONTENT"))),
                    )
                    .into(),
            )
        }

        pub fn new<T, U>(&self, title: T, content: U) -> Template
        where
            Template: From<T> + From<U>,
        {
            self.0
                .clone()
                .sub("TITLE", title)
                .sub("CONTENT", content)
                .into()
        }
    }

    let head = TagPair::new("head", false)
        .content(TagPair::new("title", true).content("Twisted Speedcubing"))
        .content(
            Tag::new("link")
                .value("rel", "stylesheet")
                .value("href", "style.css"),
        );

    let hl_builder = HeaderLink::template();
    let header = TagPair::new("header", false)
        .content(
            TagPair::new("a", false)
                .value("href", "/")
                .content(Tag::new("img").value("src", "logo.jpg")),
        )
        .content(
            TagPair::new("nav", false).content(
                TagPair::new("ul", false)
                    .content(hl_builder.new("Home", "/"))
                    .content(hl_builder.new("Devilog", "/devilog.html"))
                    .content(hl_builder.new("Solve DB", "/solve_db.html")),
            ),
        );

    let card_builder = Card::template();
    let main = TagPair::new("main", false)
        .content(card_builder.new("Test Heading", "This is a test paragraph."))
        .content(card_builder.new("Another Heading", "This is another test paragraph."))
        .content(card_builder.new("A Third Heading", "This is a third testing paragraph."));

    let body = TagPair::new("body", false).content(header).content(main);

    let document = Template::new()
        .add(DOCTYPE)
        .line()
        .concat(TagPair::new("html", false).content(head).content(body));

    let bytes = document.render().unwrap();

    let correct = indoc! {br#"
        <!DOCTYPE html>
        <html>
            <head>
                <title>Twisted Speedcubing</title>
                <link rel="stylesheet" href="style.css"/>
            </head>
            <body>
                <header>
                    <a href="/">
                        <img src="logo.jpg"/>
                    </a>
                    <nav>
                        <ul>
                            <li><a href="/">Home</a></li>
                            <li><a href="/devilog.html">Devilog</a></li>
                            <li><a href="/solve_db.html">Solve DB</a></li>
                        </ul>
                    </nav>
                </header>
                <main>
                    <div class="card">
                        <div class="container">
                            <h1>Test Heading</h1>
                            <p>
                                This is a test paragraph.
                            </p>
                        </div>
                    </div>
                    <div class="card">
                        <div class="container">
                            <h1>Another Heading</h1>
                            <p>
                                This is another test paragraph.
                            </p>
                        </div>
                    </div>
                    <div class="card">
                        <div class="container">
                            <h1>A Third Heading</h1>
                            <p>
                                This is a third testing paragraph.
                            </p>
                        </div>
                    </div>
                </main>
            </body>
        </html>"#};

    assert_eq!(bytes, correct);
}

#[test]
fn test_loop() {
    use crate::*;
    use indoc::indoc;
    use std::collections::HashMap;

    let img = Tag::new("img")
        .value("src", Placeholder("Image Source"))
        .value("height", "200")
        .value("width", "500");

    let heading = TagPair::new("h1", true).content(Placeholder("Heading"));

    let container = TagPair::new("div", false)
        .value("class", "container")
        .content(heading)
        .content(img);

    let card: Template = TagPair::new("div", false)
        .value("class", "card")
        .content(container)
        .into();

    let containing_div: Template = (1..10)
        .fold(TagPair::new("div", false), |t, i| {
            let mut map = HashMap::new();
            map.insert("Image Source", format!("img_{}.png", i).into());
            map.insert("Heading", format!("Image No: {}", i).into());
            t.content(card.clone().sub_many(&mut map))
        })
        .into();

    let bytes = containing_div.render().unwrap();

    let correct = indoc! {br#"
    <div>
        <div class="card">
            <div class="container">
                <h1>Image No: 1</h1>
                <img src="img_1.png" height="200" width="500"/>
            </div>
        </div>
        <div class="card">
            <div class="container">
                <h1>Image No: 2</h1>
                <img src="img_2.png" height="200" width="500"/>
            </div>
        </div>
        <div class="card">
            <div class="container">
                <h1>Image No: 3</h1>
                <img src="img_3.png" height="200" width="500"/>
            </div>
        </div>
        <div class="card">
            <div class="container">
                <h1>Image No: 4</h1>
                <img src="img_4.png" height="200" width="500"/>
            </div>
        </div>
        <div class="card">
            <div class="container">
                <h1>Image No: 5</h1>
                <img src="img_5.png" height="200" width="500"/>
            </div>
        </div>
        <div class="card">
            <div class="container">
                <h1>Image No: 6</h1>
                <img src="img_6.png" height="200" width="500"/>
            </div>
        </div>
        <div class="card">
            <div class="container">
                <h1>Image No: 7</h1>
                <img src="img_7.png" height="200" width="500"/>
            </div>
        </div>
        <div class="card">
            <div class="container">
                <h1>Image No: 8</h1>
                <img src="img_8.png" height="200" width="500"/>
            </div>
        </div>
        <div class="card">
            <div class="container">
                <h1>Image No: 9</h1>
                <img src="img_9.png" height="200" width="500"/>
            </div>
        </div>
    </div>"#};

    assert_eq!(bytes, correct);
}
