// SPDX-FileCopyrightText: 2022 Declan Rixon <declan.fraser.rixon@gmail.com>
//
// SPDX-License-Identifier: GPL-3.0-only

use crate::*;
use std::collections::HashMap;
use std::fmt::{self, Debug, Display, Formatter};

#[derive(Clone)]
pub struct Template(Vec<Data>);

impl Template {
    pub fn new() -> Self {
        Self(vec![])
    }

    pub fn sub_one<T>(self, placeholder: ID, replace_with: T) -> Self
    where
        Template: From<T>,
    {
        use Data::*;
        let mut sub = Some(replace_with);
        self.into_iter()
            .fold(Template::new(), |template, component| match component {
                Placeholder { indent, id } if id == placeholder => {
                    if let Some(sub) = sub.take() {
                        template.concat(if let Some(depth) = indent {
                            Template::from(sub).indented(depth)
                        } else {
                            sub.into()
                        })
                    } else {
                        template.add(Placeholder { indent, id })
                    }
                }
                other => template.add(other),
            })
    }
    pub fn sub_many(self, substitutions: &mut HashMap<ID, Template>) -> Self {
        use Data::*;
        self.into_iter()
            .fold(Template::new(), |template, component| match component {
                Placeholder { indent, id } => template.concat(
                    substitutions
                        .remove(id)
                        .map(|template| {
                            if let Some(depth) = indent {
                                template.indented(depth)
                            } else {
                                template
                            }
                        })
                        .unwrap_or_else(|| Placeholder { indent, id }.into()),
                ),
                other => template.add(other),
            })
    }
    pub fn sub<T>(self, placeholder: ID, replace_with: T) -> Substitution
    where
        Template: From<T>,
    {
        let mut map = HashMap::new();
        map.insert(placeholder, replace_with.into());
        Substitution(self, map)
    }
    pub fn iter(&self) -> std::slice::Iter<'_, Data> {
        self.0.iter()
    }
    pub fn render(self) -> Result<Vec<u8>, Template> {
        use Data::*;
        let mut template: Template = self
            .into_iter()
            .skip_while(|c| matches!(c, LineStart))
            .map(|component| match component {
                LineStart => Rendered(b"\n".to_vec()),
                c => c,
            })
            .collect();
        if template.0.len() == 1 {
            match template.0.pop() {
                Some(Rendered(mut bytes)) => {
                    if let Some(b'\n') = bytes.last() {
                        bytes.pop();
                    }
                    Ok(bytes)
                }
                Some(x) => Err(Template(vec![x])),
                _ => Err(Template::new()),
            }
        } else {
            Err(template)
        }
    }
    pub fn indented(self, depth: usize) -> Self {
        use Data::*;
        self.into_iter().fold(Template::new(), |t, c| match c {
            LineStart => t.add(c).add(INDENTATION.repeat(depth)),
            Placeholder {
                indent: Some(depth),
                id,
            } => t.add(Placeholder {
                indent: Some(depth + 1),
                id,
            }),
            _ => t.add(c),
        })
    }

    pub fn line(self) -> Self {
        self.add(Data::LineStart)
    }

    pub fn add<T>(mut self, component: T) -> Self
    where
        T: Into<Data>,
    {
        use Data::*;
        match (self.0.last_mut(), component.into()) {
            (Some(LineStart), LineStart) => { /* Do nothing */ }
            (Some(LineStart), Placeholder { indent: None, id }) => self.0.push(Placeholder {
                indent: Some(0),
                id,
            }),
            (Some(Rendered(bytes)), Rendered(mut other_bytes)) => {
                bytes.append(&mut other_bytes);
            }
            (_, x) => {
                self.0.push(x);
            }
        }
        self
    }

    pub fn concat<T>(self, template: T) -> Self
    where
        T: Into<Template>,
    {
        template.into().into_iter().fold(self, |t, c| t.add(c))
    }
}

impl<T> From<T> for Template
where
    Data: From<T>,
{
    fn from(t: T) -> Template {
        Template(vec![t.into()])
    }
}

impl FromIterator<Data> for Template {
    fn from_iter<I: IntoIterator<Item = Data>>(iter: I) -> Self {
        iter.into_iter()
            .fold(Template::new(), |template, component| {
                template.add(component)
            })
    }
}

impl IntoIterator for Template {
    type Item = Data;
    type IntoIter = std::vec::IntoIter<Self::Item>;

    fn into_iter(self) -> Self::IntoIter {
        self.0.into_iter()
    }
}

impl Display for Template {
    fn fmt(&self, f: &mut Formatter<'_>) -> fmt::Result {
        self.iter()
            .map(|component| write!(f, "{}", component))
            .collect()
    }
}

impl Debug for Template {
    fn fmt(&self, f: &mut Formatter<'_>) -> fmt::Result {
        use Data::*;
        f.write_str("template contained the following unfilled placeholders: ")?;
        f.write_str(
            &self
                .iter()
                .filter_map(|component| match component {
                    Placeholder { id, .. } => Some(String::from(*id)),
                    _ => None,
                })
                .collect::<Vec<_>>()
                .join(", "),
        )
    }
}

#[derive(Clone)]
pub struct Substitution(Template, HashMap<ID, Template>);

impl Substitution {
    pub fn sub<T>(mut self, placeholder: ID, replace_with: T) -> Substitution
    where
        Template: From<T>,
    {
        self.1.insert(placeholder, replace_with.into());
        self
    }
}

impl From<Substitution> for Template {
    fn from(Substitution(template, mut substitutions): Substitution) -> Template {
        template.sub_many(&mut substitutions)
    }
}
