// SPDX-FileCopyrightText: 2022 Declan Rixon <declan.fraser.rixon@gmail.com>
//
// SPDX-License-Identifier: GPL-3.0-only

use crate::*;

#[derive(Clone)]
pub struct TagPair {
    name: Vec<u8>,
    inline: bool,
    attributes: Option<Vec<Template>>,
    content: Template,
}

impl TagPair {
    pub fn new<T>(name: T, inline: bool) -> Self
    where
        T: Into<Vec<u8>>,
    {
        Self {
            name: name.into(),
            attributes: None,
            inline,
            content: Template::new(),
        }
    }

    pub fn attribute(mut self, attribute: Attribute) -> Self {
        match &mut self.attributes {
            Some(attributes) => attributes.push(attribute.into()),
            None => self.attributes = Some(vec![attribute.into()]),
        }
        self
    }

    pub fn bool<T>(self, name: T) -> Self
    where
        T: Into<Data>,
    {
        self.attribute(Attribute::new(name))
    }

    pub fn value<T, U>(self, name: T, value: U) -> Self
    where
        T: Into<Data>,
        U: Into<Data>,
    {
        self.attribute(Attribute::new(name).value(value))
    }

    pub fn content<T>(mut self, content: T) -> Self
    where
        T: Into<Template>,
    {
        self.content = if self.inline {
            self.content.concat(content.into())
        } else {
            self.content
                .line()
                .concat(content.into())
        };
        self
    }
}

impl From<TagPair> for Template {
    fn from(paired_tags: TagPair) -> Template {
        let open_tag = paired_tags
            .attributes
            .into_iter()
            .flatten()
            .fold(
                Template::new().add("<").add(paired_tags.name.clone()),
                |t, a| t.add(" ").concat(a),
            )
            .add(">");
        let close_tag = Template::new().add("</").add(paired_tags.name).add(">");
        if paired_tags.inline {
            open_tag.concat(paired_tags.content).concat(close_tag)
        } else {
            Template::new()
                .concat(open_tag)
                .concat(paired_tags.content.indented(1))
                .line()
                .concat(close_tag)
        }
    }
}

#[cfg(test)]
mod tests {
    #[test]
    fn paired_tags() {
        use crate::*;
        use indoc::indoc;

        let img = Tag::new("img")
            .value("src", "test.img")
            .value("height", "200")
            .value("width", "500");

        let heading = TagPair::new("h1", true).content("Test Heading");

        let container = TagPair::new("div", false)
            .value("class", "container")
            .content(heading)
            .content(img);

        let card: Template = TagPair::new("div", false)
            .value("class", "card")
            .content(container)
            .into();

        let bytes = card.render().unwrap();

        let correct = indoc! {br#"
            <div class="card">
                <div class="container">
                    <h1>Test Heading</h1>
                    <img src="test.img" height="200" width="500"/>
                </div>
            </div>"#};

        assert_eq!(bytes, correct);
    }

    #[test]
    fn paired_tags_with_placeholder() {
        use crate::*;
        use indoc::indoc;

        let img = Tag::new("img")
            .value("src", "test.img")
            .value("height", "200")
            .value("width", "500");

        let heading = TagPair::new("h1", true).content("Test Heading");

        let container: Template = TagPair::new("div", false)
            .value("class", "container")
            .content(heading)
            .content(img)
            .into();

        let card: Template = TagPair::new("div", false)
            .value("class", "card")
            .content(Placeholder("CONTAINER"))
            .into();

        let bytes = card.sub_one("CONTAINER", container).render().unwrap();

        let correct = indoc! {br#"
            <div class="card">
                <div class="container">
                    <h1>Test Heading</h1>
                    <img src="test.img" height="200" width="500"/>
                </div>
            </div>"#};

        assert_eq!(bytes, correct);
    }
}
