// SPDX-FileCopyrightText: 2022 Declan Rixon <declan.fraser.rixon@gmail.com>
//
// SPDX-License-Identifier: GPL-3.0-only

use crate::*;

#[derive(Clone)]
pub struct Tag {
    name: Data,
    attributes: Option<Vec<Template>>,
}

impl Tag {
    pub fn new<T>(name: T) -> Self
    where
        T: Into<Data>,
    {
        Self {
            name: name.into(),
            attributes: None,
        }
    }

    pub fn attribute(mut self, attribute: Attribute) -> Self {
        match &mut self.attributes {
            Some(attributes) => attributes.push(attribute.into()),
            None => self.attributes = Some(vec![attribute.into()]),
        }
        self
    }

    pub fn bool<T>(self, name: T) -> Self
    where
        T: Into<Data>,
    {
        self.attribute(Attribute::new(name))
    }

    pub fn value<T, U>(self, name: T, value: U) -> Self
    where
        T: Into<Data>,
        U: Into<Data>,
    {
        self.attribute(Attribute::new(name).value(value))
    }
}

impl From<Tag> for Template {
    fn from(tag: Tag) -> Template {
        let template = Template::new().add("<").add(tag.name);
        if let Some(attributes) = tag.attributes {
            attributes
                .into_iter()
                .fold(template, |t, a| t.add(" ").concat(a))
        } else {
            template
        }
        .add("/>")
    }
}

#[cfg(test)]
mod tests {
    use crate::*;
    #[test]
    fn tag_with_placeholder() {
        let tag: Template = Tag::new("img")
            .value("src", Placeholder("IMG_SRC"))
            .value("height", "200")
            .value("width", "500")
            .into();

        assert_eq!(
            r#"<img src="{ IMG_SRC }" height="200" width="500"/>"#,
            format!("{}", tag)
        );

        let bytes = tag.sub_one("IMG_SRC", "test.img").render().unwrap();

        assert_eq!(bytes, br#"<img src="test.img" height="200" width="500"/>"#);
    }
}
