// Copyright 2020 Johannes Hayeß
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied
// See the License for the specific language governing permissions and
// limitations under the License.

//! This is an intermediate crate that exposes the C API of `libolm` to Rust. If you want to start building things with `libolm` from Rust, check out [`olm-rs`](https://crates.io/crates/olm-rs).
//!
//! ## Supported Platforms
//!
//! - Android
//! - Linux
//! - macOS
//! - Windows
//! - FreeBSD
//! - WebAssembly
//!
//! ## Building
//!
//! This library can either be built by statically or dynamically linking against `libolm`:
//!
//! ### Static
//!
//! This is the default and requires no further action. `libolm` is built locally and then linked against statically.
//!
//! #### Build dependencies
//!
//! - `libstdc++`/`libc++`
//! - cmake
//! - GNU make or a compatible variant (WebAssembly only)
//! - Emscripten (WebAssembly only)
//!
//! ### Dynamic
//!
//! For linking against `libolm` dynamically, first make sure that you have the library in your link path.
//! Then build this library with the `OLM_LINK_VARIANT` environment variable set to `dylib`.
//!
//! For example, building your project using `olm-sys` as a dependency would look like this:
//!
//! ```bash
//! $ OLM_LINK_VARIANT=dylib cargo build
//! ```
//!
//! ### Cross compiling for Android
//!
//! To enable cross compilation for Android set the environment variable
//! `ANDROID_NDK` to the location of your NDK installation, for example:
//!
//! ```bash
//! $ ANRDOID_NDK=/home/user/Android/Sdk/ndk/22.0.7026061/
//! ```
//!
//! The linker needs to be set to an target specific one as well, for example for
//! `aarch64-linux-android` set this into your cargo config:
//!
//! ```toml
//! [target.aarch64-linux-android]
//! ar = "/home/user/Android/Sdk/ndk/22.0.7026061/toolchains/llvm/prebuilt/linux-x86_64/bin/ar"
//! linker = "/home/user/Android/Sdk/ndk/22.0.7026061/toolchains/llvm/prebuilt/linux-x86_64/bin/aarch64-linux-android30-clang"
//! ```
//!
//! After both of these are set, compilation should work as usual using cargo:
//!
//! ```bash
//! $ ANDROID_NDK=~/Android/Sdk/ndk/22.0.7026061 cargo build --target aarch64-linux-android
//! ```

#![allow(non_upper_case_globals)]
#![allow(non_camel_case_types)]
#![allow(non_snake_case)]

/* automatically generated by rust-bindgen 0.57.0 */

pub const _STDINT_H: u32 = 1;
pub const _FEATURES_H: u32 = 1;
pub const _DEFAULT_SOURCE: u32 = 1;
pub const __GLIBC_USE_ISOC2X: u32 = 0;
pub const __USE_ISOC11: u32 = 1;
pub const __USE_ISOC99: u32 = 1;
pub const __USE_ISOC95: u32 = 1;
pub const __USE_POSIX_IMPLICITLY: u32 = 1;
pub const _POSIX_SOURCE: u32 = 1;
pub const _POSIX_C_SOURCE: u32 = 200809;
pub const __USE_POSIX: u32 = 1;
pub const __USE_POSIX2: u32 = 1;
pub const __USE_POSIX199309: u32 = 1;
pub const __USE_POSIX199506: u32 = 1;
pub const __USE_XOPEN2K: u32 = 1;
pub const __USE_XOPEN2K8: u32 = 1;
pub const _ATFILE_SOURCE: u32 = 1;
pub const __USE_MISC: u32 = 1;
pub const __USE_ATFILE: u32 = 1;
pub const __USE_FORTIFY_LEVEL: u32 = 0;
pub const __GLIBC_USE_DEPRECATED_GETS: u32 = 0;
pub const __GLIBC_USE_DEPRECATED_SCANF: u32 = 0;
pub const _STDC_PREDEF_H: u32 = 1;
pub const __STDC_IEC_559__: u32 = 1;
pub const __STDC_IEC_559_COMPLEX__: u32 = 1;
pub const __STDC_ISO_10646__: u32 = 201706;
pub const __GNU_LIBRARY__: u32 = 6;
pub const __GLIBC__: u32 = 2;
pub const __GLIBC_MINOR__: u32 = 31;
pub const _SYS_CDEFS_H: u32 = 1;
pub const __glibc_c99_flexarr_available: u32 = 1;
pub const __WORDSIZE: u32 = 64;
pub const __WORDSIZE_TIME64_COMPAT32: u32 = 1;
pub const __SYSCALL_WORDSIZE: u32 = 64;
pub const __LONG_DOUBLE_USES_FLOAT128: u32 = 0;
pub const __HAVE_GENERIC_SELECTION: u32 = 1;
pub const __GLIBC_USE_LIB_EXT2: u32 = 0;
pub const __GLIBC_USE_IEC_60559_BFP_EXT: u32 = 0;
pub const __GLIBC_USE_IEC_60559_BFP_EXT_C2X: u32 = 0;
pub const __GLIBC_USE_IEC_60559_FUNCS_EXT: u32 = 0;
pub const __GLIBC_USE_IEC_60559_FUNCS_EXT_C2X: u32 = 0;
pub const __GLIBC_USE_IEC_60559_TYPES_EXT: u32 = 0;
pub const _BITS_TYPES_H: u32 = 1;
pub const __TIMESIZE: u32 = 64;
pub const _BITS_TYPESIZES_H: u32 = 1;
pub const __OFF_T_MATCHES_OFF64_T: u32 = 1;
pub const __INO_T_MATCHES_INO64_T: u32 = 1;
pub const __RLIM_T_MATCHES_RLIM64_T: u32 = 1;
pub const __STATFS_MATCHES_STATFS64: u32 = 1;
pub const __FD_SETSIZE: u32 = 1024;
pub const _BITS_TIME64_H: u32 = 1;
pub const _BITS_WCHAR_H: u32 = 1;
pub const _BITS_STDINT_INTN_H: u32 = 1;
pub const _BITS_STDINT_UINTN_H: u32 = 1;
pub const INT8_MIN: i32 = -128;
pub const INT16_MIN: i32 = -32768;
pub const INT32_MIN: i32 = -2147483648;
pub const INT8_MAX: u32 = 127;
pub const INT16_MAX: u32 = 32767;
pub const INT32_MAX: u32 = 2147483647;
pub const UINT8_MAX: u32 = 255;
pub const UINT16_MAX: u32 = 65535;
pub const UINT32_MAX: u32 = 4294967295;
pub const INT_LEAST8_MIN: i32 = -128;
pub const INT_LEAST16_MIN: i32 = -32768;
pub const INT_LEAST32_MIN: i32 = -2147483648;
pub const INT_LEAST8_MAX: u32 = 127;
pub const INT_LEAST16_MAX: u32 = 32767;
pub const INT_LEAST32_MAX: u32 = 2147483647;
pub const UINT_LEAST8_MAX: u32 = 255;
pub const UINT_LEAST16_MAX: u32 = 65535;
pub const UINT_LEAST32_MAX: u32 = 4294967295;
pub const INT_FAST8_MIN: i32 = -128;
pub const INT_FAST16_MIN: i64 = -9223372036854775808;
pub const INT_FAST32_MIN: i64 = -9223372036854775808;
pub const INT_FAST8_MAX: u32 = 127;
pub const INT_FAST16_MAX: u64 = 9223372036854775807;
pub const INT_FAST32_MAX: u64 = 9223372036854775807;
pub const UINT_FAST8_MAX: u32 = 255;
pub const UINT_FAST16_MAX: i32 = -1;
pub const UINT_FAST32_MAX: i32 = -1;
pub const INTPTR_MIN: i64 = -9223372036854775808;
pub const INTPTR_MAX: u64 = 9223372036854775807;
pub const UINTPTR_MAX: i32 = -1;
pub const PTRDIFF_MIN: i64 = -9223372036854775808;
pub const PTRDIFF_MAX: u64 = 9223372036854775807;
pub const SIG_ATOMIC_MIN: i32 = -2147483648;
pub const SIG_ATOMIC_MAX: u32 = 2147483647;
pub const SIZE_MAX: i32 = -1;
pub const WINT_MIN: u32 = 0;
pub const WINT_MAX: u32 = 4294967295;
pub type wchar_t = ::std::os::raw::c_int;
#[repr(C)]
#[repr(align(16))]
#[derive(Debug, Copy, Clone)]
pub struct max_align_t {
    pub __clang_max_align_nonce1: ::std::os::raw::c_longlong,
    pub __bindgen_padding_0: u64,
    pub __clang_max_align_nonce2: u128,
}
#[test]
fn bindgen_test_layout_max_align_t() {
    assert_eq!(
        ::std::mem::size_of::<max_align_t>(),
        32usize,
        concat!("Size of: ", stringify!(max_align_t))
    );
    assert_eq!(
        ::std::mem::align_of::<max_align_t>(),
        16usize,
        concat!("Alignment of ", stringify!(max_align_t))
    );
    assert_eq!(
        unsafe {
            &(*(::std::ptr::null::<max_align_t>())).__clang_max_align_nonce1 as *const _ as usize
        },
        0usize,
        concat!(
            "Offset of field: ",
            stringify!(max_align_t),
            "::",
            stringify!(__clang_max_align_nonce1)
        )
    );
    assert_eq!(
        unsafe {
            &(*(::std::ptr::null::<max_align_t>())).__clang_max_align_nonce2 as *const _ as usize
        },
        16usize,
        concat!(
            "Offset of field: ",
            stringify!(max_align_t),
            "::",
            stringify!(__clang_max_align_nonce2)
        )
    );
}
pub type __u_char = ::std::os::raw::c_uchar;
pub type __u_short = ::std::os::raw::c_ushort;
pub type __u_int = ::std::os::raw::c_uint;
pub type __u_long = ::std::os::raw::c_ulong;
pub type __int8_t = ::std::os::raw::c_schar;
pub type __uint8_t = ::std::os::raw::c_uchar;
pub type __int16_t = ::std::os::raw::c_short;
pub type __uint16_t = ::std::os::raw::c_ushort;
pub type __int32_t = ::std::os::raw::c_int;
pub type __uint32_t = ::std::os::raw::c_uint;
pub type __int64_t = ::std::os::raw::c_long;
pub type __uint64_t = ::std::os::raw::c_ulong;
pub type __int_least8_t = __int8_t;
pub type __uint_least8_t = __uint8_t;
pub type __int_least16_t = __int16_t;
pub type __uint_least16_t = __uint16_t;
pub type __int_least32_t = __int32_t;
pub type __uint_least32_t = __uint32_t;
pub type __int_least64_t = __int64_t;
pub type __uint_least64_t = __uint64_t;
pub type __quad_t = ::std::os::raw::c_long;
pub type __u_quad_t = ::std::os::raw::c_ulong;
pub type __intmax_t = ::std::os::raw::c_long;
pub type __uintmax_t = ::std::os::raw::c_ulong;
pub type __dev_t = ::std::os::raw::c_ulong;
pub type __uid_t = ::std::os::raw::c_uint;
pub type __gid_t = ::std::os::raw::c_uint;
pub type __ino_t = ::std::os::raw::c_ulong;
pub type __ino64_t = ::std::os::raw::c_ulong;
pub type __mode_t = ::std::os::raw::c_uint;
pub type __nlink_t = ::std::os::raw::c_ulong;
pub type __off_t = ::std::os::raw::c_long;
pub type __off64_t = ::std::os::raw::c_long;
pub type __pid_t = ::std::os::raw::c_int;
#[repr(C)]
#[derive(Debug, Copy, Clone)]
pub struct __fsid_t {
    pub __val: [::std::os::raw::c_int; 2usize],
}
#[test]
fn bindgen_test_layout___fsid_t() {
    assert_eq!(
        ::std::mem::size_of::<__fsid_t>(),
        8usize,
        concat!("Size of: ", stringify!(__fsid_t))
    );
    assert_eq!(
        ::std::mem::align_of::<__fsid_t>(),
        4usize,
        concat!("Alignment of ", stringify!(__fsid_t))
    );
    assert_eq!(
        unsafe { &(*(::std::ptr::null::<__fsid_t>())).__val as *const _ as usize },
        0usize,
        concat!(
            "Offset of field: ",
            stringify!(__fsid_t),
            "::",
            stringify!(__val)
        )
    );
}
pub type __clock_t = ::std::os::raw::c_long;
pub type __rlim_t = ::std::os::raw::c_ulong;
pub type __rlim64_t = ::std::os::raw::c_ulong;
pub type __id_t = ::std::os::raw::c_uint;
pub type __time_t = ::std::os::raw::c_long;
pub type __useconds_t = ::std::os::raw::c_uint;
pub type __suseconds_t = ::std::os::raw::c_long;
pub type __daddr_t = ::std::os::raw::c_int;
pub type __key_t = ::std::os::raw::c_int;
pub type __clockid_t = ::std::os::raw::c_int;
pub type __timer_t = *mut ::std::os::raw::c_void;
pub type __blksize_t = ::std::os::raw::c_long;
pub type __blkcnt_t = ::std::os::raw::c_long;
pub type __blkcnt64_t = ::std::os::raw::c_long;
pub type __fsblkcnt_t = ::std::os::raw::c_ulong;
pub type __fsblkcnt64_t = ::std::os::raw::c_ulong;
pub type __fsfilcnt_t = ::std::os::raw::c_ulong;
pub type __fsfilcnt64_t = ::std::os::raw::c_ulong;
pub type __fsword_t = ::std::os::raw::c_long;
pub type __ssize_t = ::std::os::raw::c_long;
pub type __syscall_slong_t = ::std::os::raw::c_long;
pub type __syscall_ulong_t = ::std::os::raw::c_ulong;
pub type __loff_t = __off64_t;
pub type __caddr_t = *mut ::std::os::raw::c_char;
pub type __intptr_t = ::std::os::raw::c_long;
pub type __socklen_t = ::std::os::raw::c_uint;
pub type __sig_atomic_t = ::std::os::raw::c_int;
pub type int_least8_t = __int_least8_t;
pub type int_least16_t = __int_least16_t;
pub type int_least32_t = __int_least32_t;
pub type int_least64_t = __int_least64_t;
pub type uint_least8_t = __uint_least8_t;
pub type uint_least16_t = __uint_least16_t;
pub type uint_least32_t = __uint_least32_t;
pub type uint_least64_t = __uint_least64_t;
pub type int_fast8_t = ::std::os::raw::c_schar;
pub type int_fast16_t = ::std::os::raw::c_long;
pub type int_fast32_t = ::std::os::raw::c_long;
pub type int_fast64_t = ::std::os::raw::c_long;
pub type uint_fast8_t = ::std::os::raw::c_uchar;
pub type uint_fast16_t = ::std::os::raw::c_ulong;
pub type uint_fast32_t = ::std::os::raw::c_ulong;
pub type uint_fast64_t = ::std::os::raw::c_ulong;
pub type intmax_t = __intmax_t;
pub type uintmax_t = __uintmax_t;
#[doc = "< There wasn't an error"]
pub const OlmErrorCode_OLM_SUCCESS: OlmErrorCode = 0;
#[doc = "< Not enough entropy was supplied"]
pub const OlmErrorCode_OLM_NOT_ENOUGH_RANDOM: OlmErrorCode = 1;
#[doc = "< Supplied output buffer is too small"]
pub const OlmErrorCode_OLM_OUTPUT_BUFFER_TOO_SMALL: OlmErrorCode = 2;
#[doc = "< The message version is unsupported"]
pub const OlmErrorCode_OLM_BAD_MESSAGE_VERSION: OlmErrorCode = 3;
#[doc = "< The message couldn't be decoded"]
pub const OlmErrorCode_OLM_BAD_MESSAGE_FORMAT: OlmErrorCode = 4;
#[doc = "< The message couldn't be decrypted"]
pub const OlmErrorCode_OLM_BAD_MESSAGE_MAC: OlmErrorCode = 5;
#[doc = "< The message references an unknown key id"]
pub const OlmErrorCode_OLM_BAD_MESSAGE_KEY_ID: OlmErrorCode = 6;
#[doc = "< The input base64 was invalid"]
pub const OlmErrorCode_OLM_INVALID_BASE64: OlmErrorCode = 7;
#[doc = "< The supplied account key is invalid"]
pub const OlmErrorCode_OLM_BAD_ACCOUNT_KEY: OlmErrorCode = 8;
#[doc = "< The pickled object is too new"]
pub const OlmErrorCode_OLM_UNKNOWN_PICKLE_VERSION: OlmErrorCode = 9;
#[doc = "< The pickled object couldn't be decoded"]
pub const OlmErrorCode_OLM_CORRUPTED_PICKLE: OlmErrorCode = 10;
#[doc = "< Attempt to initialise an inbound group"]
#[doc = "session from an invalid session key"]
pub const OlmErrorCode_OLM_BAD_SESSION_KEY: OlmErrorCode = 11;
#[doc = "< Attempt to decode a message whose"]
#[doc = " index is earlier than our earliest"]
#[doc = " known session key."]
pub const OlmErrorCode_OLM_UNKNOWN_MESSAGE_INDEX: OlmErrorCode = 12;
#[doc = " Attempt to unpickle an account which uses pickle version 1 (which did"]
#[doc = " not save enough space for the Ed25519 key; the key should be considered"]
#[doc = " compromised. We don't let the user reload the account."]
pub const OlmErrorCode_OLM_BAD_LEGACY_ACCOUNT_PICKLE: OlmErrorCode = 13;
#[doc = " Received message had a bad signature"]
pub const OlmErrorCode_OLM_BAD_SIGNATURE: OlmErrorCode = 14;
#[doc = " Received message had a bad signature"]
pub const OlmErrorCode_OLM_INPUT_BUFFER_TOO_SMALL: OlmErrorCode = 15;
#[doc = " SAS doesn't have their key set."]
pub const OlmErrorCode_OLM_SAS_THEIR_KEY_NOT_SET: OlmErrorCode = 16;
#[doc = " The pickled object was successfully decoded, but the unpickling still failed"]
#[doc = " because it had some extraneous junk data at the end."]
pub const OlmErrorCode_OLM_PICKLE_EXTRA_DATA: OlmErrorCode = 17;
pub type OlmErrorCode = ::std::os::raw::c_uint;
extern "C" {
    #[doc = " get a string representation of the given error code."]
    pub fn _olm_error_to_string(error: OlmErrorCode) -> *const ::std::os::raw::c_char;
}
#[repr(C)]
#[derive(Debug, Copy, Clone)]
pub struct OlmInboundGroupSession {
    _unused: [u8; 0],
}
extern "C" {
    #[doc = " get the size of an inbound group session, in bytes."]
    pub fn olm_inbound_group_session_size() -> usize;
}
extern "C" {
    #[doc = " Initialise an inbound group session object using the supplied memory"]
    #[doc = " The supplied memory should be at least olm_inbound_group_session_size()"]
    #[doc = " bytes."]
    pub fn olm_inbound_group_session(
        memory: *mut ::std::os::raw::c_void,
    ) -> *mut OlmInboundGroupSession;
}
extern "C" {
    #[doc = " A null terminated string describing the most recent error to happen to a"]
    #[doc = " group session"]
    pub fn olm_inbound_group_session_last_error(
        session: *const OlmInboundGroupSession,
    ) -> *const ::std::os::raw::c_char;
}
extern "C" {
    #[doc = " An error code describing the most recent error to happen to a group"]
    #[doc = " session"]
    pub fn olm_inbound_group_session_last_error_code(
        session: *const OlmInboundGroupSession,
    ) -> OlmErrorCode;
}
extern "C" {
    #[doc = " Clears the memory used to back this group session"]
    pub fn olm_clear_inbound_group_session(session: *mut OlmInboundGroupSession) -> usize;
}
extern "C" {
    #[doc = " Returns the number of bytes needed to store an inbound group session"]
    pub fn olm_pickle_inbound_group_session_length(session: *const OlmInboundGroupSession)
        -> usize;
}
extern "C" {
    #[doc = " Stores a group session as a base64 string. Encrypts the session using the"]
    #[doc = " supplied key. Returns the length of the session on success."]
    #[doc = ""]
    #[doc = " Returns olm_error() on failure. If the pickle output buffer"]
    #[doc = " is smaller than olm_pickle_inbound_group_session_length() then"]
    #[doc = " olm_inbound_group_session_last_error() will be \"OUTPUT_BUFFER_TOO_SMALL\""]
    pub fn olm_pickle_inbound_group_session(
        session: *mut OlmInboundGroupSession,
        key: *const ::std::os::raw::c_void,
        key_length: usize,
        pickled: *mut ::std::os::raw::c_void,
        pickled_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " Loads a group session from a pickled base64 string. Decrypts the session"]
    #[doc = " using the supplied key."]
    #[doc = ""]
    #[doc = " Returns olm_error() on failure. If the key doesn't match the one used to"]
    #[doc = " encrypt the account then olm_inbound_group_session_last_error() will be"]
    #[doc = " \"BAD_ACCOUNT_KEY\". If the base64 couldn't be decoded then"]
    #[doc = " olm_inbound_group_session_last_error() will be \"INVALID_BASE64\". The input"]
    #[doc = " pickled buffer is destroyed"]
    pub fn olm_unpickle_inbound_group_session(
        session: *mut OlmInboundGroupSession,
        key: *const ::std::os::raw::c_void,
        key_length: usize,
        pickled: *mut ::std::os::raw::c_void,
        pickled_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " Start a new inbound group session, from a key exported from"]
    #[doc = " olm_outbound_group_session_key"]
    #[doc = ""]
    #[doc = " Returns olm_error() on failure. On failure last_error will be set with an"]
    #[doc = " error code. The last_error will be:"]
    #[doc = ""]
    #[doc = "  * OLM_INVALID_BASE64  if the session_key is not valid base64"]
    #[doc = "  * OLM_BAD_SESSION_KEY if the session_key is invalid"]
    pub fn olm_init_inbound_group_session(
        session: *mut OlmInboundGroupSession,
        session_key: *const u8,
        session_key_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " Import an inbound group session, from a previous export."]
    #[doc = ""]
    #[doc = " Returns olm_error() on failure. On failure last_error will be set with an"]
    #[doc = " error code. The last_error will be:"]
    #[doc = ""]
    #[doc = "  * OLM_INVALID_BASE64  if the session_key is not valid base64"]
    #[doc = "  * OLM_BAD_SESSION_KEY if the session_key is invalid"]
    pub fn olm_import_inbound_group_session(
        session: *mut OlmInboundGroupSession,
        session_key: *const u8,
        session_key_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " Get an upper bound on the number of bytes of plain-text the decrypt method"]
    #[doc = " will write for a given input message length. The actual size could be"]
    #[doc = " different due to padding."]
    #[doc = ""]
    #[doc = " The input message buffer is destroyed."]
    #[doc = ""]
    #[doc = " Returns olm_error() on failure."]
    pub fn olm_group_decrypt_max_plaintext_length(
        session: *mut OlmInboundGroupSession,
        message: *mut u8,
        message_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " Decrypt a message."]
    #[doc = ""]
    #[doc = " The input message buffer is destroyed."]
    #[doc = ""]
    #[doc = " Returns the length of the decrypted plain-text, or olm_error() on failure."]
    #[doc = ""]
    #[doc = " On failure last_error will be set with an error code. The last_error will"]
    #[doc = " be:"]
    #[doc = "   * OLM_OUTPUT_BUFFER_TOO_SMALL if the plain-text buffer is too small"]
    #[doc = "   * OLM_INVALID_BASE64 if the message is not valid base-64"]
    #[doc = "   * OLM_BAD_MESSAGE_VERSION if the message was encrypted with an unsupported"]
    #[doc = "     version of the protocol"]
    #[doc = "   * OLM_BAD_MESSAGE_FORMAT if the message headers could not be decoded"]
    #[doc = "   * OLM_BAD_MESSAGE_MAC    if the message could not be verified"]
    #[doc = "   * OLM_UNKNOWN_MESSAGE_INDEX  if we do not have a session key corresponding to the"]
    #[doc = "     message's index (ie, it was sent before the session key was shared with"]
    #[doc = "     us)"]
    pub fn olm_group_decrypt(
        session: *mut OlmInboundGroupSession,
        message: *mut u8,
        message_length: usize,
        plaintext: *mut u8,
        max_plaintext_length: usize,
        message_index: *mut u32,
    ) -> usize;
}
extern "C" {
    #[doc = " Get the number of bytes returned by olm_inbound_group_session_id()"]
    pub fn olm_inbound_group_session_id_length(session: *const OlmInboundGroupSession) -> usize;
}
extern "C" {
    #[doc = " Get a base64-encoded identifier for this session."]
    #[doc = ""]
    #[doc = " Returns the length of the session id on success or olm_error() on"]
    #[doc = " failure. On failure last_error will be set with an error code. The"]
    #[doc = " last_error will be OUTPUT_BUFFER_TOO_SMALL if the id buffer was too"]
    #[doc = " small."]
    pub fn olm_inbound_group_session_id(
        session: *mut OlmInboundGroupSession,
        id: *mut u8,
        id_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " Get the first message index we know how to decrypt."]
    pub fn olm_inbound_group_session_first_known_index(
        session: *const OlmInboundGroupSession,
    ) -> u32;
}
extern "C" {
    #[doc = " Check if the session has been verified as a valid session."]
    #[doc = ""]
    #[doc = " (A session is verified either because the original session share was signed,"]
    #[doc = " or because we have subsequently successfully decrypted a message.)"]
    #[doc = ""]
    #[doc = " This is mainly intended for the unit tests, currently."]
    pub fn olm_inbound_group_session_is_verified(
        session: *const OlmInboundGroupSession,
    ) -> ::std::os::raw::c_int;
}
extern "C" {
    #[doc = " Get the number of bytes returned by olm_export_inbound_group_session()"]
    pub fn olm_export_inbound_group_session_length(session: *const OlmInboundGroupSession)
        -> usize;
}
extern "C" {
    #[doc = " Export the base64-encoded ratchet key for this session, at the given index,"]
    #[doc = " in a format which can be used by olm_import_inbound_group_session"]
    #[doc = ""]
    #[doc = " Returns the length of the ratchet key on success or olm_error() on"]
    #[doc = " failure. On failure last_error will be set with an error code. The"]
    #[doc = " last_error will be:"]
    #[doc = "   * OUTPUT_BUFFER_TOO_SMALL if the buffer was too small"]
    #[doc = "   * OLM_UNKNOWN_MESSAGE_INDEX  if we do not have a session key corresponding to the"]
    #[doc = "     given index (ie, it was sent before the session key was shared with"]
    #[doc = "     us)"]
    pub fn olm_export_inbound_group_session(
        session: *mut OlmInboundGroupSession,
        key: *mut u8,
        key_length: usize,
        message_index: u32,
    ) -> usize;
}
#[repr(C)]
#[derive(Debug, Copy, Clone)]
pub struct OlmOutboundGroupSession {
    _unused: [u8; 0],
}
extern "C" {
    #[doc = " get the size of an outbound group session, in bytes."]
    pub fn olm_outbound_group_session_size() -> usize;
}
extern "C" {
    #[doc = " Initialise an outbound group session object using the supplied memory"]
    #[doc = " The supplied memory should be at least olm_outbound_group_session_size()"]
    #[doc = " bytes."]
    pub fn olm_outbound_group_session(
        memory: *mut ::std::os::raw::c_void,
    ) -> *mut OlmOutboundGroupSession;
}
extern "C" {
    #[doc = " A null terminated string describing the most recent error to happen to a"]
    #[doc = " group session"]
    pub fn olm_outbound_group_session_last_error(
        session: *const OlmOutboundGroupSession,
    ) -> *const ::std::os::raw::c_char;
}
extern "C" {
    #[doc = " An error code describing the most recent error to happen to a group"]
    #[doc = " session"]
    pub fn olm_outbound_group_session_last_error_code(
        session: *const OlmOutboundGroupSession,
    ) -> OlmErrorCode;
}
extern "C" {
    #[doc = " Clears the memory used to back this group session"]
    pub fn olm_clear_outbound_group_session(session: *mut OlmOutboundGroupSession) -> usize;
}
extern "C" {
    #[doc = " Returns the number of bytes needed to store an outbound group session"]
    pub fn olm_pickle_outbound_group_session_length(
        session: *const OlmOutboundGroupSession,
    ) -> usize;
}
extern "C" {
    #[doc = " Stores a group session as a base64 string. Encrypts the session using the"]
    #[doc = " supplied key. Returns the length of the session on success."]
    #[doc = ""]
    #[doc = " Returns olm_error() on failure. If the pickle output buffer"]
    #[doc = " is smaller than olm_pickle_outbound_group_session_length() then"]
    #[doc = " olm_outbound_group_session_last_error() will be \"OUTPUT_BUFFER_TOO_SMALL\""]
    pub fn olm_pickle_outbound_group_session(
        session: *mut OlmOutboundGroupSession,
        key: *const ::std::os::raw::c_void,
        key_length: usize,
        pickled: *mut ::std::os::raw::c_void,
        pickled_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " Loads a group session from a pickled base64 string. Decrypts the session"]
    #[doc = " using the supplied key."]
    #[doc = ""]
    #[doc = " Returns olm_error() on failure. If the key doesn't match the one used to"]
    #[doc = " encrypt the account then olm_outbound_group_session_last_error() will be"]
    #[doc = " \"BAD_ACCOUNT_KEY\". If the base64 couldn't be decoded then"]
    #[doc = " olm_outbound_group_session_last_error() will be \"INVALID_BASE64\". The input"]
    #[doc = " pickled buffer is destroyed"]
    pub fn olm_unpickle_outbound_group_session(
        session: *mut OlmOutboundGroupSession,
        key: *const ::std::os::raw::c_void,
        key_length: usize,
        pickled: *mut ::std::os::raw::c_void,
        pickled_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " The number of random bytes needed to create an outbound group session"]
    pub fn olm_init_outbound_group_session_random_length(
        session: *const OlmOutboundGroupSession,
    ) -> usize;
}
extern "C" {
    #[doc = " Start a new outbound group session. Returns olm_error() on failure. On"]
    #[doc = " failure last_error will be set with an error code. The last_error will be"]
    #[doc = " NOT_ENOUGH_RANDOM if the number of random bytes was too small."]
    pub fn olm_init_outbound_group_session(
        session: *mut OlmOutboundGroupSession,
        random: *mut u8,
        random_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " The number of bytes that will be created by encrypting a message"]
    pub fn olm_group_encrypt_message_length(
        session: *mut OlmOutboundGroupSession,
        plaintext_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " Encrypt some plain-text. Returns the length of the encrypted message or"]
    #[doc = " olm_error() on failure. On failure last_error will be set with an"]
    #[doc = " error code. The last_error will be OUTPUT_BUFFER_TOO_SMALL if the output"]
    #[doc = " buffer is too small."]
    pub fn olm_group_encrypt(
        session: *mut OlmOutboundGroupSession,
        plaintext: *const u8,
        plaintext_length: usize,
        message: *mut u8,
        message_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " Get the number of bytes returned by olm_outbound_group_session_id()"]
    pub fn olm_outbound_group_session_id_length(session: *const OlmOutboundGroupSession) -> usize;
}
extern "C" {
    #[doc = " Get a base64-encoded identifier for this session."]
    #[doc = ""]
    #[doc = " Returns the length of the session id on success or olm_error() on"]
    #[doc = " failure. On failure last_error will be set with an error code. The"]
    #[doc = " last_error will be OUTPUT_BUFFER_TOO_SMALL if the id buffer was too"]
    #[doc = " small."]
    pub fn olm_outbound_group_session_id(
        session: *mut OlmOutboundGroupSession,
        id: *mut u8,
        id_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " Get the current message index for this session."]
    #[doc = ""]
    #[doc = " Each message is sent with an increasing index; this returns the index for"]
    #[doc = " the next message."]
    pub fn olm_outbound_group_session_message_index(session: *mut OlmOutboundGroupSession) -> u32;
}
extern "C" {
    #[doc = " Get the number of bytes returned by olm_outbound_group_session_key()"]
    pub fn olm_outbound_group_session_key_length(session: *const OlmOutboundGroupSession) -> usize;
}
extern "C" {
    #[doc = " Get the base64-encoded current ratchet key for this session."]
    #[doc = ""]
    #[doc = " Each message is sent with a different ratchet key. This function returns the"]
    #[doc = " ratchet key that will be used for the next message."]
    #[doc = ""]
    #[doc = " Returns the length of the ratchet key on success or olm_error() on"]
    #[doc = " failure. On failure last_error will be set with an error code. The"]
    #[doc = " last_error will be OUTPUT_BUFFER_TOO_SMALL if the buffer was too small."]
    pub fn olm_outbound_group_session_key(
        session: *mut OlmOutboundGroupSession,
        key: *mut u8,
        key_length: usize,
    ) -> usize;
}
pub const OLM_MESSAGE_TYPE_PRE_KEY: usize = 0;
pub const OLM_MESSAGE_TYPE_MESSAGE: usize = 1;
#[repr(C)]
#[derive(Debug, Copy, Clone)]
pub struct OlmAccount {
    _unused: [u8; 0],
}
#[repr(C)]
#[derive(Debug, Copy, Clone)]
pub struct OlmSession {
    _unused: [u8; 0],
}
#[repr(C)]
#[derive(Debug, Copy, Clone)]
pub struct OlmUtility {
    _unused: [u8; 0],
}
extern "C" {
    #[doc = " Get the version number of the library."]
    #[doc = " Arguments will be updated if non-null."]
    pub fn olm_get_library_version(major: *mut u8, minor: *mut u8, patch: *mut u8);
}
extern "C" {
    #[doc = " The size of an account object in bytes"]
    pub fn olm_account_size() -> usize;
}
extern "C" {
    #[doc = " The size of a session object in bytes"]
    pub fn olm_session_size() -> usize;
}
extern "C" {
    #[doc = " The size of a utility object in bytes"]
    pub fn olm_utility_size() -> usize;
}
extern "C" {
    #[doc = " Initialise an account object using the supplied memory"]
    #[doc = "  The supplied memory must be at least olm_account_size() bytes"]
    pub fn olm_account(memory: *mut ::std::os::raw::c_void) -> *mut OlmAccount;
}
extern "C" {
    #[doc = " Initialise a session object using the supplied memory"]
    #[doc = "  The supplied memory must be at least olm_session_size() bytes"]
    pub fn olm_session(memory: *mut ::std::os::raw::c_void) -> *mut OlmSession;
}
extern "C" {
    #[doc = " Initialise a utility object using the supplied memory"]
    #[doc = "  The supplied memory must be at least olm_utility_size() bytes"]
    pub fn olm_utility(memory: *mut ::std::os::raw::c_void) -> *mut OlmUtility;
}
extern "C" {
    #[doc = " The value that olm will return from a function if there was an error"]
    pub fn olm_error() -> usize;
}
extern "C" {
    #[doc = " A null terminated string describing the most recent error to happen to an"]
    #[doc = " account"]
    pub fn olm_account_last_error(account: *const OlmAccount) -> *const ::std::os::raw::c_char;
}
extern "C" {
    #[doc = " An error code describing the most recent error to happen to an account"]
    pub fn olm_account_last_error_code(account: *const OlmAccount) -> OlmErrorCode;
}
extern "C" {
    #[doc = " A null terminated string describing the most recent error to happen to a"]
    #[doc = " session"]
    pub fn olm_session_last_error(session: *const OlmSession) -> *const ::std::os::raw::c_char;
}
extern "C" {
    #[doc = " An error code describing the most recent error to happen to a session"]
    pub fn olm_session_last_error_code(session: *const OlmSession) -> OlmErrorCode;
}
extern "C" {
    #[doc = " A null terminated string describing the most recent error to happen to a"]
    #[doc = " utility"]
    pub fn olm_utility_last_error(utility: *const OlmUtility) -> *const ::std::os::raw::c_char;
}
extern "C" {
    #[doc = " An error code describing the most recent error to happen to a utility"]
    pub fn olm_utility_last_error_code(utility: *const OlmUtility) -> OlmErrorCode;
}
extern "C" {
    #[doc = " Clears the memory used to back this account"]
    pub fn olm_clear_account(account: *mut OlmAccount) -> usize;
}
extern "C" {
    #[doc = " Clears the memory used to back this session"]
    pub fn olm_clear_session(session: *mut OlmSession) -> usize;
}
extern "C" {
    #[doc = " Clears the memory used to back this utility"]
    pub fn olm_clear_utility(utility: *mut OlmUtility) -> usize;
}
extern "C" {
    #[doc = " Returns the number of bytes needed to store an account"]
    pub fn olm_pickle_account_length(account: *const OlmAccount) -> usize;
}
extern "C" {
    #[doc = " Returns the number of bytes needed to store a session"]
    pub fn olm_pickle_session_length(session: *const OlmSession) -> usize;
}
extern "C" {
    #[doc = " Stores an account as a base64 string. Encrypts the account using the"]
    #[doc = " supplied key. Returns the length of the pickled account on success."]
    #[doc = " Returns olm_error() on failure. If the pickle output buffer"]
    #[doc = " is smaller than olm_pickle_account_length() then"]
    #[doc = " olm_account_last_error() will be \"OUTPUT_BUFFER_TOO_SMALL\""]
    pub fn olm_pickle_account(
        account: *mut OlmAccount,
        key: *const ::std::os::raw::c_void,
        key_length: usize,
        pickled: *mut ::std::os::raw::c_void,
        pickled_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " Stores a session as a base64 string. Encrypts the session using the"]
    #[doc = " supplied key. Returns the length of the pickled session on success."]
    #[doc = " Returns olm_error() on failure. If the pickle output buffer"]
    #[doc = " is smaller than olm_pickle_session_length() then"]
    #[doc = " olm_session_last_error() will be \"OUTPUT_BUFFER_TOO_SMALL\""]
    pub fn olm_pickle_session(
        session: *mut OlmSession,
        key: *const ::std::os::raw::c_void,
        key_length: usize,
        pickled: *mut ::std::os::raw::c_void,
        pickled_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " Loads an account from a pickled base64 string. Decrypts the account using"]
    #[doc = " the supplied key. Returns olm_error() on failure. If the key doesn't"]
    #[doc = " match the one used to encrypt the account then olm_account_last_error()"]
    #[doc = " will be \"BAD_ACCOUNT_KEY\". If the base64 couldn't be decoded then"]
    #[doc = " olm_account_last_error() will be \"INVALID_BASE64\". The input pickled"]
    #[doc = " buffer is destroyed"]
    pub fn olm_unpickle_account(
        account: *mut OlmAccount,
        key: *const ::std::os::raw::c_void,
        key_length: usize,
        pickled: *mut ::std::os::raw::c_void,
        pickled_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " Loads a session from a pickled base64 string. Decrypts the session using"]
    #[doc = " the supplied key. Returns olm_error() on failure. If the key doesn't"]
    #[doc = " match the one used to encrypt the account then olm_session_last_error()"]
    #[doc = " will be \"BAD_ACCOUNT_KEY\". If the base64 couldn't be decoded then"]
    #[doc = " olm_session_last_error() will be \"INVALID_BASE64\". The input pickled"]
    #[doc = " buffer is destroyed"]
    pub fn olm_unpickle_session(
        session: *mut OlmSession,
        key: *const ::std::os::raw::c_void,
        key_length: usize,
        pickled: *mut ::std::os::raw::c_void,
        pickled_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " The number of random bytes needed to create an account."]
    pub fn olm_create_account_random_length(account: *const OlmAccount) -> usize;
}
extern "C" {
    #[doc = " Creates a new account. Returns olm_error() on failure. If there weren't"]
    #[doc = " enough random bytes then olm_account_last_error() will be"]
    #[doc = " \"NOT_ENOUGH_RANDOM\""]
    pub fn olm_create_account(
        account: *mut OlmAccount,
        random: *mut ::std::os::raw::c_void,
        random_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " The size of the output buffer needed to hold the identity keys"]
    pub fn olm_account_identity_keys_length(account: *const OlmAccount) -> usize;
}
extern "C" {
    #[doc = " Writes the public parts of the identity keys for the account into the"]
    #[doc = " identity_keys output buffer. Returns olm_error() on failure. If the"]
    #[doc = " identity_keys buffer was too small then olm_account_last_error() will be"]
    #[doc = " \"OUTPUT_BUFFER_TOO_SMALL\"."]
    pub fn olm_account_identity_keys(
        account: *mut OlmAccount,
        identity_keys: *mut ::std::os::raw::c_void,
        identity_key_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " The length of an ed25519 signature encoded as base64."]
    pub fn olm_account_signature_length(account: *const OlmAccount) -> usize;
}
extern "C" {
    #[doc = " Signs a message with the ed25519 key for this account. Returns olm_error()"]
    #[doc = " on failure. If the signature buffer was too small then"]
    #[doc = " olm_account_last_error() will be \"OUTPUT_BUFFER_TOO_SMALL\""]
    pub fn olm_account_sign(
        account: *mut OlmAccount,
        message: *const ::std::os::raw::c_void,
        message_length: usize,
        signature: *mut ::std::os::raw::c_void,
        signature_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " The size of the output buffer needed to hold the one time keys"]
    pub fn olm_account_one_time_keys_length(account: *const OlmAccount) -> usize;
}
extern "C" {
    #[doc = " Writes the public parts of the unpublished one time keys for the account"]
    #[doc = " into the one_time_keys output buffer."]
    #[doc = " <p>"]
    #[doc = " The returned data is a JSON-formatted object with the single property"]
    #[doc = " <tt>curve25519</tt>, which is itself an object mapping key id to"]
    #[doc = " base64-encoded Curve25519 key. For example:"]
    #[doc = " <pre>"]
    #[doc = " {"]
    #[doc = "     curve25519: {"]
    #[doc = "         \"AAAAAA\": \"wo76WcYtb0Vk/pBOdmduiGJ0wIEjW4IBMbbQn7aSnTo\","]
    #[doc = "         \"AAAAAB\": \"LRvjo46L1X2vx69sS9QNFD29HWulxrmW11Up5AfAjgU\""]
    #[doc = "     }"]
    #[doc = " }"]
    #[doc = " </pre>"]
    #[doc = " Returns olm_error() on failure."]
    #[doc = " <p>"]
    #[doc = " If the one_time_keys buffer was too small then olm_account_last_error()"]
    #[doc = " will be \"OUTPUT_BUFFER_TOO_SMALL\"."]
    pub fn olm_account_one_time_keys(
        account: *mut OlmAccount,
        one_time_keys: *mut ::std::os::raw::c_void,
        one_time_keys_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " Marks the current set of one time keys as being published."]
    pub fn olm_account_mark_keys_as_published(account: *mut OlmAccount) -> usize;
}
extern "C" {
    #[doc = " The largest number of one time keys this account can store."]
    pub fn olm_account_max_number_of_one_time_keys(account: *const OlmAccount) -> usize;
}
extern "C" {
    #[doc = " The number of random bytes needed to generate a given number of new one"]
    #[doc = " time keys."]
    pub fn olm_account_generate_one_time_keys_random_length(
        account: *const OlmAccount,
        number_of_keys: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " Generates a number of new one time keys. If the total number of keys stored"]
    #[doc = " by this account exceeds max_number_of_one_time_keys() then the old keys are"]
    #[doc = " discarded. Returns olm_error() on error. If the number of random bytes is"]
    #[doc = " too small then olm_account_last_error() will be \"NOT_ENOUGH_RANDOM\"."]
    pub fn olm_account_generate_one_time_keys(
        account: *mut OlmAccount,
        number_of_keys: usize,
        random: *mut ::std::os::raw::c_void,
        random_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " The number of random bytes needed to generate a fallback key."]
    pub fn olm_account_generate_fallback_key_random_length(account: *const OlmAccount) -> usize;
}
extern "C" {
    #[doc = " Generates a new fallback key. Only one previous fallback key is"]
    #[doc = " stored. Returns olm_error() on error. If the number of random bytes is too"]
    #[doc = " small then olm_account_last_error() will be \"NOT_ENOUGH_RANDOM\"."]
    pub fn olm_account_generate_fallback_key(
        account: *mut OlmAccount,
        random: *mut ::std::os::raw::c_void,
        random_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " The number of bytes needed to hold the fallback key as returned by"]
    #[doc = " olm_account_fallback_key."]
    pub fn olm_account_fallback_key_length(account: *const OlmAccount) -> usize;
}
extern "C" {
    pub fn olm_account_fallback_key(
        account: *mut OlmAccount,
        fallback_key: *mut ::std::os::raw::c_void,
        fallback_key_size: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " The number of random bytes needed to create an outbound session"]
    pub fn olm_create_outbound_session_random_length(session: *const OlmSession) -> usize;
}
extern "C" {
    #[doc = " Creates a new out-bound session for sending messages to a given identity_key"]
    #[doc = " and one_time_key. Returns olm_error() on failure. If the keys couldn't be"]
    #[doc = " decoded as base64 then olm_session_last_error() will be \"INVALID_BASE64\""]
    #[doc = " If there weren't enough random bytes then olm_session_last_error() will"]
    #[doc = " be \"NOT_ENOUGH_RANDOM\"."]
    pub fn olm_create_outbound_session(
        session: *mut OlmSession,
        account: *const OlmAccount,
        their_identity_key: *const ::std::os::raw::c_void,
        their_identity_key_length: usize,
        their_one_time_key: *const ::std::os::raw::c_void,
        their_one_time_key_length: usize,
        random: *mut ::std::os::raw::c_void,
        random_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " Create a new in-bound session for sending/receiving messages from an"]
    #[doc = " incoming PRE_KEY message. Returns olm_error() on failure. If the base64"]
    #[doc = " couldn't be decoded then olm_session_last_error will be \"INVALID_BASE64\"."]
    #[doc = " If the message was for an unsupported protocol version then"]
    #[doc = " olm_session_last_error() will be \"BAD_MESSAGE_VERSION\". If the message"]
    #[doc = " couldn't be decoded then olm_session_last_error() will be"]
    #[doc = " \"BAD_MESSAGE_FORMAT\". If the message refers to an unknown one time"]
    #[doc = " key then olm_session_last_error() will be \"BAD_MESSAGE_KEY_ID\"."]
    pub fn olm_create_inbound_session(
        session: *mut OlmSession,
        account: *mut OlmAccount,
        one_time_key_message: *mut ::std::os::raw::c_void,
        message_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " Same as olm_create_inbound_session, but ensures that the identity key"]
    #[doc = " in the pre-key message matches the expected identity key, supplied via the"]
    #[doc = " `their_identity_key` parameter. Fails early if there is no match."]
    pub fn olm_create_inbound_session_from(
        session: *mut OlmSession,
        account: *mut OlmAccount,
        their_identity_key: *const ::std::os::raw::c_void,
        their_identity_key_length: usize,
        one_time_key_message: *mut ::std::os::raw::c_void,
        message_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " The length of the buffer needed to return the id for this session."]
    pub fn olm_session_id_length(session: *const OlmSession) -> usize;
}
extern "C" {
    #[doc = " An identifier for this session. Will be the same for both ends of the"]
    #[doc = " conversation. If the id buffer is too small then olm_session_last_error()"]
    #[doc = " will be \"OUTPUT_BUFFER_TOO_SMALL\"."]
    pub fn olm_session_id(
        session: *mut OlmSession,
        id: *mut ::std::os::raw::c_void,
        id_length: usize,
    ) -> usize;
}
extern "C" {
    pub fn olm_session_has_received_message(session: *const OlmSession) -> ::std::os::raw::c_int;
}
extern "C" {
    #[doc = " Write a null-terminated string describing the internal state of an olm"]
    #[doc = " session to the buffer provided for debugging and logging purposes."]
    pub fn olm_session_describe(
        session: *mut OlmSession,
        buf: *mut ::std::os::raw::c_char,
        buflen: usize,
    );
}
extern "C" {
    #[doc = " Checks if the PRE_KEY message is for this in-bound session. This can happen"]
    #[doc = " if multiple messages are sent to this account before this account sends a"]
    #[doc = " message in reply. The one_time_key_message buffer is destroyed. Returns 1 if"]
    #[doc = " the session matches. Returns 0 if the session does not match. Returns"]
    #[doc = " olm_error() on failure. If the base64 couldn't be decoded then"]
    #[doc = " olm_session_last_error will be \"INVALID_BASE64\".  If the message was for an"]
    #[doc = " unsupported protocol version then olm_session_last_error() will be"]
    #[doc = " \"BAD_MESSAGE_VERSION\". If the message couldn't be decoded then then"]
    #[doc = " olm_session_last_error() will be \"BAD_MESSAGE_FORMAT\"."]
    pub fn olm_matches_inbound_session(
        session: *mut OlmSession,
        one_time_key_message: *mut ::std::os::raw::c_void,
        message_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " Checks if the PRE_KEY message is for this in-bound session. This can happen"]
    #[doc = " if multiple messages are sent to this account before this account sends a"]
    #[doc = " message in reply. The one_time_key_message buffer is destroyed. Returns 1 if"]
    #[doc = " the session matches. Returns 0 if the session does not match. Returns"]
    #[doc = " olm_error() on failure. If the base64 couldn't be decoded then"]
    #[doc = " olm_session_last_error will be \"INVALID_BASE64\".  If the message was for an"]
    #[doc = " unsupported protocol version then olm_session_last_error() will be"]
    #[doc = " \"BAD_MESSAGE_VERSION\". If the message couldn't be decoded then then"]
    #[doc = " olm_session_last_error() will be \"BAD_MESSAGE_FORMAT\"."]
    pub fn olm_matches_inbound_session_from(
        session: *mut OlmSession,
        their_identity_key: *const ::std::os::raw::c_void,
        their_identity_key_length: usize,
        one_time_key_message: *mut ::std::os::raw::c_void,
        message_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " Removes the one time keys that the session used from the account. Returns"]
    #[doc = " olm_error() on failure. If the account doesn't have any matching one time"]
    #[doc = " keys then olm_account_last_error() will be \"BAD_MESSAGE_KEY_ID\"."]
    pub fn olm_remove_one_time_keys(account: *mut OlmAccount, session: *mut OlmSession) -> usize;
}
extern "C" {
    #[doc = " The type of the next message that olm_encrypt() will return. Returns"]
    #[doc = " OLM_MESSAGE_TYPE_PRE_KEY if the message will be a PRE_KEY message."]
    #[doc = " Returns OLM_MESSAGE_TYPE_MESSAGE if the message will be a normal message."]
    #[doc = " Returns olm_error on failure."]
    pub fn olm_encrypt_message_type(session: *const OlmSession) -> usize;
}
extern "C" {
    #[doc = " The number of random bytes needed to encrypt the next message."]
    pub fn olm_encrypt_random_length(session: *const OlmSession) -> usize;
}
extern "C" {
    #[doc = " The size of the next message in bytes for the given number of plain-text"]
    #[doc = " bytes."]
    pub fn olm_encrypt_message_length(session: *const OlmSession, plaintext_length: usize)
        -> usize;
}
extern "C" {
    #[doc = " Encrypts a message using the session. Returns the length of the message in"]
    #[doc = " bytes on success. Writes the message as base64 into the message buffer."]
    #[doc = " Returns olm_error() on failure. If the message buffer is too small then"]
    #[doc = " olm_session_last_error() will be \"OUTPUT_BUFFER_TOO_SMALL\". If there"]
    #[doc = " weren't enough random bytes then olm_session_last_error() will be"]
    #[doc = " \"NOT_ENOUGH_RANDOM\"."]
    pub fn olm_encrypt(
        session: *mut OlmSession,
        plaintext: *const ::std::os::raw::c_void,
        plaintext_length: usize,
        random: *mut ::std::os::raw::c_void,
        random_length: usize,
        message: *mut ::std::os::raw::c_void,
        message_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " The maximum number of bytes of plain-text a given message could decode to."]
    #[doc = " The actual size could be different due to padding. The input message buffer"]
    #[doc = " is destroyed. Returns olm_error() on failure. If the message base64"]
    #[doc = " couldn't be decoded then olm_session_last_error() will be"]
    #[doc = " \"INVALID_BASE64\". If the message is for an unsupported version of the"]
    #[doc = " protocol then olm_session_last_error() will be \"BAD_MESSAGE_VERSION\"."]
    #[doc = " If the message couldn't be decoded then olm_session_last_error() will be"]
    #[doc = " \"BAD_MESSAGE_FORMAT\"."]
    pub fn olm_decrypt_max_plaintext_length(
        session: *mut OlmSession,
        message_type: usize,
        message: *mut ::std::os::raw::c_void,
        message_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " Decrypts a message using the session. The input message buffer is destroyed."]
    #[doc = " Returns the length of the plain-text on success. Returns olm_error() on"]
    #[doc = " failure. If the plain-text buffer is smaller than"]
    #[doc = " olm_decrypt_max_plaintext_length() then olm_session_last_error()"]
    #[doc = " will be \"OUTPUT_BUFFER_TOO_SMALL\". If the base64 couldn't be decoded then"]
    #[doc = " olm_session_last_error() will be \"INVALID_BASE64\". If the message is for"]
    #[doc = " an unsupported version of the protocol then olm_session_last_error() will"]
    #[doc = " be \"BAD_MESSAGE_VERSION\". If the message couldn't be decoded then"]
    #[doc = " olm_session_last_error() will be BAD_MESSAGE_FORMAT\"."]
    #[doc = " If the MAC on the message was invalid then olm_session_last_error() will"]
    #[doc = " be \"BAD_MESSAGE_MAC\"."]
    pub fn olm_decrypt(
        session: *mut OlmSession,
        message_type: usize,
        message: *mut ::std::os::raw::c_void,
        message_length: usize,
        plaintext: *mut ::std::os::raw::c_void,
        max_plaintext_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " The length of the buffer needed to hold the SHA-256 hash."]
    pub fn olm_sha256_length(utility: *const OlmUtility) -> usize;
}
extern "C" {
    #[doc = " Calculates the SHA-256 hash of the input and encodes it as base64. If the"]
    #[doc = " output buffer is smaller than olm_sha256_length() then"]
    #[doc = " olm_utility_last_error() will be \"OUTPUT_BUFFER_TOO_SMALL\"."]
    pub fn olm_sha256(
        utility: *mut OlmUtility,
        input: *const ::std::os::raw::c_void,
        input_length: usize,
        output: *mut ::std::os::raw::c_void,
        output_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " Verify an ed25519 signature. If the key was too small then"]
    #[doc = " olm_utility_last_error() will be \"INVALID_BASE64\". If the signature was invalid"]
    #[doc = " then olm_utility_last_error() will be \"BAD_MESSAGE_MAC\"."]
    pub fn olm_ed25519_verify(
        utility: *mut OlmUtility,
        key: *const ::std::os::raw::c_void,
        key_length: usize,
        message: *const ::std::os::raw::c_void,
        message_length: usize,
        signature: *mut ::std::os::raw::c_void,
        signature_length: usize,
    ) -> usize;
}
#[repr(C)]
#[derive(Debug, Copy, Clone)]
pub struct OlmSAS {
    _unused: [u8; 0],
}
extern "C" {
    #[doc = " A null terminated string describing the most recent error to happen to an"]
    #[doc = " SAS object."]
    pub fn olm_sas_last_error(sas: *const OlmSAS) -> *const ::std::os::raw::c_char;
}
extern "C" {
    #[doc = " An error code describing the most recent error to happen to an SAS"]
    #[doc = " object."]
    pub fn olm_sas_last_error_code(sas: *const OlmSAS) -> OlmErrorCode;
}
extern "C" {
    #[doc = " The size of an SAS object in bytes."]
    pub fn olm_sas_size() -> usize;
}
extern "C" {
    #[doc = " Initialize an SAS object using the supplied memory."]
    #[doc = " The supplied memory must be at least `olm_sas_size()` bytes."]
    pub fn olm_sas(memory: *mut ::std::os::raw::c_void) -> *mut OlmSAS;
}
extern "C" {
    #[doc = " Clears the memory used to back an SAS object."]
    pub fn olm_clear_sas(sas: *mut OlmSAS) -> usize;
}
extern "C" {
    #[doc = " The number of random bytes needed to create an SAS object."]
    pub fn olm_create_sas_random_length(sas: *const OlmSAS) -> usize;
}
extern "C" {
    #[doc = " Creates a new SAS object."]
    #[doc = ""]
    #[doc = " @param[in] sas the SAS object to create, initialized by `olm_sas()`."]
    #[doc = " @param[in] random array of random bytes.  The contents of the buffer may be"]
    #[doc = "     overwritten."]
    #[doc = " @param[in] random_length the number of random bytes provided.  Must be at"]
    #[doc = "    least `olm_create_sas_random_length()`."]
    #[doc = ""]
    #[doc = " @return `olm_error()` on failure.  If there weren't enough random bytes then"]
    #[doc = " `olm_sas_last_error()` will be `NOT_ENOUGH_RANDOM`."]
    pub fn olm_create_sas(
        sas: *mut OlmSAS,
        random: *mut ::std::os::raw::c_void,
        random_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " The size of a public key in bytes."]
    pub fn olm_sas_pubkey_length(sas: *const OlmSAS) -> usize;
}
extern "C" {
    #[doc = " Get the public key for the SAS object."]
    #[doc = ""]
    #[doc = " @param[in] sas the SAS object."]
    #[doc = " @param[out] pubkey buffer to store the public key."]
    #[doc = " @param[in] pubkey_length the size of the `pubkey` buffer.  Must be at least"]
    #[doc = "   `olm_sas_pubkey_length()`."]
    #[doc = ""]
    #[doc = " @return `olm_error()` on failure.  If the `pubkey` buffer is too small, then"]
    #[doc = " `olm_sas_last_error()` will be `OUTPUT_BUFFER_TOO_SMALL`."]
    pub fn olm_sas_get_pubkey(
        sas: *mut OlmSAS,
        pubkey: *mut ::std::os::raw::c_void,
        pubkey_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " Sets the public key of other user."]
    #[doc = ""]
    #[doc = " @param[in] sas the SAS object."]
    #[doc = " @param[in] their_key the other user's public key.  The contents of the"]
    #[doc = "     buffer will be overwritten."]
    #[doc = " @param[in] their_key_length the size of the `their_key` buffer."]
    #[doc = ""]
    #[doc = " @return `olm_error()` on failure.  If the `their_key` buffer is too small,"]
    #[doc = " then `olm_sas_last_error()` will be `INPUT_BUFFER_TOO_SMALL`."]
    pub fn olm_sas_set_their_key(
        sas: *mut OlmSAS,
        their_key: *mut ::std::os::raw::c_void,
        their_key_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " Checks if their key was set."]
    #[doc = ""]
    #[doc = " @param[in] sas the SAS object."]
    #[doc = ""]
    pub fn olm_sas_is_their_key_set(sas: *const OlmSAS) -> ::std::os::raw::c_int;
}
extern "C" {
    #[doc = " Generate bytes to use for the short authentication string."]
    #[doc = ""]
    #[doc = " @param[in] sas the SAS object."]
    #[doc = " @param[in] info extra information to mix in when generating the bytes, as"]
    #[doc = "     per the Matrix spec."]
    #[doc = " @param[in] info_length the length of the `info` parameter."]
    #[doc = " @param[out] output the output buffer."]
    #[doc = " @param[in] output_length the size of the output buffer.  For hex-based SAS"]
    #[doc = "     as in the Matrix spec, this will be 5."]
    #[doc = ""]
    #[doc = " @return `olm_error()` on failure. If their key wasn't set then"]
    #[doc = " `olm_sas_last_error()` will be `SAS_THEIR_KEY_NOT_SET`."]
    pub fn olm_sas_generate_bytes(
        sas: *mut OlmSAS,
        info: *const ::std::os::raw::c_void,
        info_length: usize,
        output: *mut ::std::os::raw::c_void,
        output_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " The size of the message authentication code generated by"]
    #[doc = " olm_sas_calculate_mac()`."]
    pub fn olm_sas_mac_length(sas: *const OlmSAS) -> usize;
}
extern "C" {
    #[doc = " Generate a message authentication code (MAC) based on the shared secret."]
    #[doc = ""]
    #[doc = " @param[in] sas the SAS object."]
    #[doc = " @param[in] input the message to produce the authentication code for."]
    #[doc = " @param[in] input_length the length of the message."]
    #[doc = " @param[in] info extra information to mix in when generating the MAC, as per"]
    #[doc = "     the Matrix spec."]
    #[doc = " @param[in] info_length the length of the `info` parameter."]
    #[doc = " @param[out] mac the buffer in which to store the MAC."]
    #[doc = " @param[in] mac_length the size of the `mac` buffer.  Must be at least"]
    #[doc = " `olm_sas_mac_length()`"]
    #[doc = ""]
    #[doc = " @return `olm_error()` on failure.  If the `mac` buffer is too small, then"]
    #[doc = " `olm_sas_last_error()` will be `OUTPUT_BUFFER_TOO_SMALL`."]
    pub fn olm_sas_calculate_mac(
        sas: *mut OlmSAS,
        input: *const ::std::os::raw::c_void,
        input_length: usize,
        info: *const ::std::os::raw::c_void,
        info_length: usize,
        mac: *mut ::std::os::raw::c_void,
        mac_length: usize,
    ) -> usize;
}
extern "C" {
    pub fn olm_sas_calculate_mac_fixed_base64(
        sas: *mut OlmSAS,
        input: *const ::std::os::raw::c_void,
        input_length: usize,
        info: *const ::std::os::raw::c_void,
        info_length: usize,
        mac: *mut ::std::os::raw::c_void,
        mac_length: usize,
    ) -> usize;
}
extern "C" {
    pub fn olm_sas_calculate_mac_long_kdf(
        sas: *mut OlmSAS,
        input: *const ::std::os::raw::c_void,
        input_length: usize,
        info: *const ::std::os::raw::c_void,
        info_length: usize,
        mac: *mut ::std::os::raw::c_void,
        mac_length: usize,
    ) -> usize;
}
#[repr(C)]
#[derive(Debug, Copy, Clone)]
pub struct OlmPkEncryption {
    _unused: [u8; 0],
}
extern "C" {
    pub fn olm_pk_encryption_size() -> usize;
}
extern "C" {
    #[doc = " Initialise an encryption object using the supplied memory"]
    #[doc = "  The supplied memory must be at least olm_pk_encryption_size() bytes"]
    pub fn olm_pk_encryption(memory: *mut ::std::os::raw::c_void) -> *mut OlmPkEncryption;
}
extern "C" {
    #[doc = " A null terminated string describing the most recent error to happen to an"]
    #[doc = " encryption object"]
    pub fn olm_pk_encryption_last_error(
        encryption: *const OlmPkEncryption,
    ) -> *const ::std::os::raw::c_char;
}
extern "C" {
    #[doc = " An error code describing the most recent error to happen to an encryption"]
    #[doc = " object"]
    pub fn olm_pk_encryption_last_error_code(encryption: *const OlmPkEncryption) -> OlmErrorCode;
}
extern "C" {
    #[doc = " Clears the memory used to back this encryption object"]
    pub fn olm_clear_pk_encryption(encryption: *mut OlmPkEncryption) -> usize;
}
extern "C" {
    #[doc = " Set the recipient's public key for encrypting to"]
    pub fn olm_pk_encryption_set_recipient_key(
        encryption: *mut OlmPkEncryption,
        public_key: *const ::std::os::raw::c_void,
        public_key_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " Get the length of the ciphertext that will correspond to a plaintext of the"]
    #[doc = " given length."]
    pub fn olm_pk_ciphertext_length(
        encryption: *const OlmPkEncryption,
        plaintext_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " Get the length of the message authentication code."]
    pub fn olm_pk_mac_length(encryption: *const OlmPkEncryption) -> usize;
}
extern "C" {
    #[doc = " Get the length of a public or ephemeral key"]
    pub fn olm_pk_key_length() -> usize;
}
extern "C" {
    #[doc = " The number of random bytes needed to encrypt a message."]
    pub fn olm_pk_encrypt_random_length(encryption: *const OlmPkEncryption) -> usize;
}
extern "C" {
    #[doc = " Encrypt a plaintext for the recipient set using"]
    #[doc = " olm_pk_encryption_set_recipient_key. Writes to the ciphertext, mac, and"]
    #[doc = " ephemeral_key buffers, whose values should be sent to the recipient. mac is"]
    #[doc = " a Message Authentication Code to ensure that the data is received and"]
    #[doc = " decrypted properly. ephemeral_key is the public part of the ephemeral key"]
    #[doc = " used (together with the recipient's key) to generate a symmetric encryption"]
    #[doc = " key. Returns olm_error() on failure. If the ciphertext, mac, or"]
    #[doc = " ephemeral_key buffers were too small then olm_pk_encryption_last_error()"]
    #[doc = " will be \"OUTPUT_BUFFER_TOO_SMALL\". If there weren't enough random bytes then"]
    #[doc = " olm_pk_encryption_last_error() will be \"OLM_INPUT_BUFFER_TOO_SMALL\"."]
    pub fn olm_pk_encrypt(
        encryption: *mut OlmPkEncryption,
        plaintext: *const ::std::os::raw::c_void,
        plaintext_length: usize,
        ciphertext: *mut ::std::os::raw::c_void,
        ciphertext_length: usize,
        mac: *mut ::std::os::raw::c_void,
        mac_length: usize,
        ephemeral_key: *mut ::std::os::raw::c_void,
        ephemeral_key_size: usize,
        random: *const ::std::os::raw::c_void,
        random_length: usize,
    ) -> usize;
}
#[repr(C)]
#[derive(Debug, Copy, Clone)]
pub struct OlmPkDecryption {
    _unused: [u8; 0],
}
extern "C" {
    pub fn olm_pk_decryption_size() -> usize;
}
extern "C" {
    #[doc = " Initialise a decryption object using the supplied memory"]
    #[doc = "  The supplied memory must be at least olm_pk_decryption_size() bytes"]
    pub fn olm_pk_decryption(memory: *mut ::std::os::raw::c_void) -> *mut OlmPkDecryption;
}
extern "C" {
    #[doc = " A null terminated string describing the most recent error to happen to a"]
    #[doc = " decription object"]
    pub fn olm_pk_decryption_last_error(
        decryption: *const OlmPkDecryption,
    ) -> *const ::std::os::raw::c_char;
}
extern "C" {
    #[doc = " An error code describing the most recent error to happen to a decription"]
    #[doc = " object"]
    pub fn olm_pk_decryption_last_error_code(decryption: *const OlmPkDecryption) -> OlmErrorCode;
}
extern "C" {
    #[doc = " Clears the memory used to back this decryption object"]
    pub fn olm_clear_pk_decryption(decryption: *mut OlmPkDecryption) -> usize;
}
extern "C" {
    #[doc = " Get the number of bytes required to store an olm private key"]
    pub fn olm_pk_private_key_length() -> usize;
}
extern "C" {
    #[doc = " DEPRECATED: Use olm_pk_private_key_length()"]
    pub fn olm_pk_generate_key_random_length() -> usize;
}
extern "C" {
    #[doc = " Initialise the key from the private part of a key as returned by"]
    #[doc = " olm_pk_get_private_key(). The associated public key will be written to the"]
    #[doc = " pubkey buffer. Returns olm_error() on failure. If the pubkey buffer is too"]
    #[doc = " small then olm_pk_decryption_last_error() will be \"OUTPUT_BUFFER_TOO_SMALL\"."]
    #[doc = " If the private key was not long enough then olm_pk_decryption_last_error()"]
    #[doc = " will be \"OLM_INPUT_BUFFER_TOO_SMALL\"."]
    #[doc = ""]
    #[doc = " Note that the pubkey is a base64 encoded string, but the private key is"]
    #[doc = " an unencoded byte array"]
    pub fn olm_pk_key_from_private(
        decryption: *mut OlmPkDecryption,
        pubkey: *mut ::std::os::raw::c_void,
        pubkey_length: usize,
        privkey: *const ::std::os::raw::c_void,
        privkey_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " DEPRECATED: Use olm_pk_key_from_private"]
    pub fn olm_pk_generate_key(
        decryption: *mut OlmPkDecryption,
        pubkey: *mut ::std::os::raw::c_void,
        pubkey_length: usize,
        privkey: *const ::std::os::raw::c_void,
        privkey_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " Returns the number of bytes needed to store a decryption object."]
    pub fn olm_pickle_pk_decryption_length(decryption: *const OlmPkDecryption) -> usize;
}
extern "C" {
    #[doc = " Stores decryption object as a base64 string. Encrypts the object using the"]
    #[doc = " supplied key. Returns the length of the pickled object on success."]
    #[doc = " Returns olm_error() on failure. If the pickle output buffer"]
    #[doc = " is smaller than olm_pickle_pk_decryption_length() then"]
    #[doc = " olm_pk_decryption_last_error() will be \"OUTPUT_BUFFER_TOO_SMALL\""]
    pub fn olm_pickle_pk_decryption(
        decryption: *mut OlmPkDecryption,
        key: *const ::std::os::raw::c_void,
        key_length: usize,
        pickled: *mut ::std::os::raw::c_void,
        pickled_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " Loads a decryption object from a pickled base64 string. The associated"]
    #[doc = " public key will be written to the pubkey buffer. Decrypts the object using"]
    #[doc = " the supplied key. Returns olm_error() on failure. If the key doesn't"]
    #[doc = " match the one used to encrypt the account then olm_pk_decryption_last_error()"]
    #[doc = " will be \"BAD_ACCOUNT_KEY\". If the base64 couldn't be decoded then"]
    #[doc = " olm_pk_decryption_last_error() will be \"INVALID_BASE64\". The input pickled"]
    #[doc = " buffer is destroyed"]
    pub fn olm_unpickle_pk_decryption(
        decryption: *mut OlmPkDecryption,
        key: *const ::std::os::raw::c_void,
        key_length: usize,
        pickled: *mut ::std::os::raw::c_void,
        pickled_length: usize,
        pubkey: *mut ::std::os::raw::c_void,
        pubkey_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " Get the length of the plaintext that will correspond to a ciphertext of the"]
    #[doc = " given length."]
    pub fn olm_pk_max_plaintext_length(
        decryption: *const OlmPkDecryption,
        ciphertext_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " Decrypt a ciphertext. The input ciphertext buffer is destroyed. See the"]
    #[doc = " olm_pk_encrypt function for descriptions of the ephemeral_key and mac"]
    #[doc = " arguments. Returns the length of the plaintext on success. Returns"]
    #[doc = " olm_error() on failure. If the plaintext buffer is too small then"]
    #[doc = " olm_pk_encryption_last_error() will be \"OUTPUT_BUFFER_TOO_SMALL\"."]
    pub fn olm_pk_decrypt(
        decryption: *mut OlmPkDecryption,
        ephemeral_key: *const ::std::os::raw::c_void,
        ephemeral_key_length: usize,
        mac: *const ::std::os::raw::c_void,
        mac_length: usize,
        ciphertext: *mut ::std::os::raw::c_void,
        ciphertext_length: usize,
        plaintext: *mut ::std::os::raw::c_void,
        max_plaintext_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " Get the private key for an OlmDecryption object as an unencoded byte array"]
    #[doc = " private_key must be a pointer to a buffer of at least"]
    #[doc = " olm_pk_private_key_length() bytes and this length must be passed in"]
    #[doc = " private_key_length. If the given buffer is too small, returns olm_error()"]
    #[doc = " and olm_pk_encryption_last_error() will be \"OUTPUT_BUFFER_TOO_SMALL\"."]
    #[doc = " Returns the number of bytes written."]
    pub fn olm_pk_get_private_key(
        decryption: *mut OlmPkDecryption,
        private_key: *mut ::std::os::raw::c_void,
        private_key_length: usize,
    ) -> usize;
}
#[repr(C)]
#[derive(Debug, Copy, Clone)]
pub struct OlmPkSigning {
    _unused: [u8; 0],
}
extern "C" {
    pub fn olm_pk_signing_size() -> usize;
}
extern "C" {
    #[doc = " Initialise a signing object using the supplied memory"]
    #[doc = "  The supplied memory must be at least olm_pk_signing_size() bytes"]
    pub fn olm_pk_signing(memory: *mut ::std::os::raw::c_void) -> *mut OlmPkSigning;
}
extern "C" {
    #[doc = " A null terminated string describing the most recent error to happen to a"]
    #[doc = " signing object"]
    pub fn olm_pk_signing_last_error(sign: *const OlmPkSigning) -> *const ::std::os::raw::c_char;
}
extern "C" {
    #[doc = " A null terminated string describing the most recent error to happen to a"]
    #[doc = " signing object"]
    pub fn olm_pk_signing_last_error_code(sign: *const OlmPkSigning) -> OlmErrorCode;
}
extern "C" {
    #[doc = " Clears the memory used to back this signing object"]
    pub fn olm_clear_pk_signing(sign: *mut OlmPkSigning) -> usize;
}
extern "C" {
    #[doc = " Initialise the signing object with a public/private keypair from a seed. The"]
    #[doc = " associated public key will be written to the pubkey buffer. Returns"]
    #[doc = " olm_error() on failure. If the public key buffer is too small then"]
    #[doc = " olm_pk_signing_last_error() will be \"OUTPUT_BUFFER_TOO_SMALL\".  If the seed"]
    #[doc = " buffer is too small then olm_pk_signing_last_error() will be"]
    #[doc = " \"INPUT_BUFFER_TOO_SMALL\"."]
    pub fn olm_pk_signing_key_from_seed(
        sign: *mut OlmPkSigning,
        pubkey: *mut ::std::os::raw::c_void,
        pubkey_length: usize,
        seed: *const ::std::os::raw::c_void,
        seed_length: usize,
    ) -> usize;
}
extern "C" {
    #[doc = " The size required for the seed for initialising a signing object."]
    pub fn olm_pk_signing_seed_length() -> usize;
}
extern "C" {
    #[doc = " The size of the public key of a signing object."]
    pub fn olm_pk_signing_public_key_length() -> usize;
}
extern "C" {
    #[doc = " The size of a signature created by a signing object."]
    pub fn olm_pk_signature_length() -> usize;
}
extern "C" {
    #[doc = " Sign a message. The signature will be written to the signature"]
    #[doc = " buffer. Returns olm_error() on failure. If the signature buffer is too"]
    #[doc = " small, olm_pk_signing_last_error() will be \"OUTPUT_BUFFER_TOO_SMALL\"."]
    pub fn olm_pk_sign(
        sign: *mut OlmPkSigning,
        message: *const u8,
        message_length: usize,
        signature: *mut u8,
        signature_length: usize,
    ) -> usize;
}
