use bevy::prelude::*;
use serde::{Deserialize, Serialize};
use okizeme_utils::*;
use okizeme_types::PlayerId;
use okizeme_defense::{
  Hurtbox,
  BlockState,
};
use crate::{
  Collision
};

/// Box generated by attacks in game
#[cfg_attr(feature = "debug", derive(bevy_inspector_egui::Inspectable))]
#[derive(Component,Debug, Clone, Copy, Default)]
pub struct Hitbox {
  /// Attack level, effects hit/block stun
  attack_level: u8,
  /// Base damage of the hitbox
  damage: u8,
  /// Proration when this hitbox connects first in a combo
  proration: f32,
  /// Force to be applied when a player is hit by this
  force: Vec2,
  /// Has the hitbox connected
  hit_state: HitState,
  /// Can the hitbox be blocked in the air
  air_blockable: bool,
  /// The block property of the hitbox
  property: AttackProperty,
  /// How many frames will this hitbox stay out
  duration: u8,
  /// Does this hitbox cause damage when blocked
  chip: bool,
  /// Is the hitbox currently active
  active: bool,
  /// Is the hitbox attached to the player that generated it
  projectile: bool,
}

impl Hitbox {
  /// if possible, lower the hitboxes duration by 1 frame
  pub fn tick(&mut self) {
    self.duration = countdown(self.duration);
  }

  /// return if the hitbox should be removed
  pub fn is_finished(&self) -> bool {
    return self.duration == 0;
  }

  pub fn generate_collision(&self, hurtbox: &Hurtbox) -> Collision {
    return Collision::new(self.clone(), self.is_blocked(hurtbox));
  }

  /// Returns if a Hitbox is blocked by a Hurtbox it overlaps
  pub fn is_blocked(&self, hurtbox: &Hurtbox) -> bool {
    use AttackProperty::*;
    use BlockState::*;
    match hurtbox.block_state {
      Stand {barrier:_, instant:_} => {
        match self.property {
          Low => return false,
          _ => return true
        }
      },
      Crouch {barrier:_, instant:_} => {
        match self.property {
          High => return false,
          _ => return true
        }
      },
      Air {barrier, instant:_} => {
        if self.air_blockable {
          return true
        } else {
          if barrier {
            return true
          } else {
            return false
          }
        }
      },
      None => return false,
    }
  }
}


#[cfg_attr(feature = "debug", derive(bevy_inspector_egui::Inspectable))]
#[derive(Deserialize, Serialize, Debug, Clone, Copy, Component)]
pub enum AttackProperty {
  Mid,
  Low,
  High,
}

impl Default for AttackProperty {
  fn default() -> Self {
    AttackProperty::Mid
  }
}

#[cfg_attr(feature = "debug", derive(bevy_inspector_egui::Inspectable))]
#[derive(Debug, Clone, Copy)]
pub enum HitState {
  None,
  Hit,
  Blocked
}

impl Default for HitState {
  fn default() -> Self {
      HitState::None
  }
}

#[derive(Component)]
pub struct Damage(u8);

impl Damage {
  fn apply_damage(&self) -> u8 {
    return self.0;
  }
}

#[derive(Component)]
pub enum AttackLevel {
  L0,
  L1,
  L2,
  L3,
  L4
}

#[derive(Component)]
pub enum Knockdown {
  None,
  Stand,
  Soft,
  Mid,
  Hard
}

pub struct StunValues {
  hitstop: u8,
  standing_hitstun: u8,
  crouching_hitstun: u8,
  aerial_hitstun: u8,
  blockstun: u8
}

impl StunValues {
  fn new(hitstop: u8, standing_hitstun: u8, crouching_hitstun: u8, aerial_hitstun: u8, blockstun: u8) -> Self {
    StunValues {
      hitstop,
      standing_hitstun,
      crouching_hitstun,
      aerial_hitstun,
      blockstun
    }
  }

  pub fn from_attack_level(attack_level: &AttackLevel) -> Self {
    use AttackLevel::*;
    match attack_level {
      L0 => StunValues::new(11, 12, 13, 14, 9),
      L1 => StunValues::new(12, 14, 15, 16, 11),
      L2 => StunValues::new(13, 16, 17, 18, 13),
      L3 => StunValues::new(14, 19, 20, 21, 16),
      L4 => StunValues::new(15, 21, 22, 23, 18)
    }
  }
}
#[derive(Bundle)]
pub struct HitboxBundle {
  #[bundle]
  sprite_bundle: SpriteBundle,
  player_id: PlayerId,
  attack_property: AttackProperty,
  attack_level: AttackLevel,
  damage: Damage,
  knockdown: Knockdown
}
