use anyhow::Result;

use crate::Client;

pub struct Projects {
    client: Client,
}

impl Projects {
    #[doc(hidden)]
    pub fn new(client: Client) -> Self {
        Projects { client }
    }

    /**
     * List organization projects.
     *
     * This function performs a `GET` to the `/orgs/{org}/projects` endpoint.
     *
     * Lists the projects in an organization. Returns a `404 Not Found` status if projects are disabled in the organization. If you do not have sufficient privileges to perform this action, a `401 Unauthorized` or `410 Gone` status is returned.
     *
     * FROM: <https://docs.github.com/rest/reference/projects#list-organization-projects>
     *
     * **Parameters:**
     *
     * * `org: &str`
     * * `state: crate::types::IssuesListState` -- Indicates the state of the projects to return. Can be either `open`, `closed`, or `all`.
     * * `per_page: i64` -- Results per page (max 100).
     * * `page: i64` -- Page number of the results to fetch.
     */
    pub async fn list_for_org(
        &self,
        org: &str,
        state: crate::types::IssuesListState,
        per_page: i64,
        page: i64,
    ) -> Result<Vec<crate::types::Project>> {
        let url = format!(
            "/orgs/{}/projects?page={}&per_page={}&state={}",
            crate::progenitor_support::encode_path(&org.to_string()),
            format!("{}", page),
            format!("{}", per_page),
            state,
        );

        self.client.get(&url).await
    }

    /**
     * List organization projects.
     *
     * This function performs a `GET` to the `/orgs/{org}/projects` endpoint.
     *
     * As opposed to `list_for_org`, this function returns all the pages of the request at once.
     *
     * Lists the projects in an organization. Returns a `404 Not Found` status if projects are disabled in the organization. If you do not have sufficient privileges to perform this action, a `401 Unauthorized` or `410 Gone` status is returned.
     *
     * FROM: <https://docs.github.com/rest/reference/projects#list-organization-projects>
     */
    pub async fn list_all_for_org(
        &self,
        org: &str,
        state: crate::types::IssuesListState,
    ) -> Result<Vec<crate::types::Project>> {
        let url = format!(
            "/orgs/{}/projects?state={}",
            crate::progenitor_support::encode_path(&org.to_string()),
            state,
        );

        self.client.get_all_pages(&url).await
    }

    /**
     * Create an organization project.
     *
     * This function performs a `POST` to the `/orgs/{org}/projects` endpoint.
     *
     * Creates an organization project board. Returns a `404 Not Found` status if projects are disabled in the organization. If you do not have sufficient privileges to perform this action, a `401 Unauthorized` or `410 Gone` status is returned.
     *
     * FROM: <https://docs.github.com/rest/reference/projects#create-an-organization-project>
     *
     * **Parameters:**
     *
     * * `org: &str`
     */
    pub async fn create_for_org(
        &self,
        org: &str,
        body: &crate::types::ProjectsCreateOrgRequest,
    ) -> Result<crate::types::Project> {
        let url = format!(
            "/orgs/{}/projects",
            crate::progenitor_support::encode_path(&org.to_string()),
        );

        self.client
            .post(
                &url,
                Some(reqwest::Body::from(serde_json::to_vec(body).unwrap())),
            )
            .await
    }

    /**
     * Get a project card.
     *
     * This function performs a `GET` to the `/projects/columns/cards/{card_id}` endpoint.
     *
     *
     *
     * FROM: <https://docs.github.com/rest/reference/projects#get-a-project-card>
     *
     * **Parameters:**
     *
     * * `card_id: i64` -- card_id parameter.
     */
    pub async fn get_card(&self, card_id: i64) -> Result<crate::types::ProjectCard> {
        let url = format!(
            "/projects/columns/cards/{}",
            crate::progenitor_support::encode_path(&card_id.to_string()),
        );

        self.client.get(&url).await
    }

    /**
     * Delete a project card.
     *
     * This function performs a `DELETE` to the `/projects/columns/cards/{card_id}` endpoint.
     *
     *
     *
     * FROM: <https://docs.github.com/rest/reference/projects#delete-a-project-card>
     *
     * **Parameters:**
     *
     * * `card_id: i64` -- card_id parameter.
     */
    pub async fn delete_card(&self, card_id: i64) -> Result<()> {
        let url = format!(
            "/projects/columns/cards/{}",
            crate::progenitor_support::encode_path(&card_id.to_string()),
        );

        self.client.delete(&url, None).await
    }

    /**
     * Update an existing project card.
     *
     * This function performs a `PATCH` to the `/projects/columns/cards/{card_id}` endpoint.
     *
     *
     *
     * FROM: <https://docs.github.com/rest/reference/projects#update-a-project-card>
     *
     * **Parameters:**
     *
     * * `card_id: i64` -- card_id parameter.
     */
    pub async fn update_card(
        &self,
        card_id: i64,
        body: &crate::types::ProjectsUpdateCardRequest,
    ) -> Result<crate::types::ProjectCard> {
        let url = format!(
            "/projects/columns/cards/{}",
            crate::progenitor_support::encode_path(&card_id.to_string()),
        );

        self.client
            .patch(
                &url,
                Some(reqwest::Body::from(serde_json::to_vec(body).unwrap())),
            )
            .await
    }

    /**
     * Move a project card.
     *
     * This function performs a `POST` to the `/projects/columns/cards/{card_id}/moves` endpoint.
     *
     *
     *
     * FROM: <https://docs.github.com/rest/reference/projects#move-a-project-card>
     *
     * **Parameters:**
     *
     * * `card_id: i64` -- card_id parameter.
     */
    pub async fn move_card(
        &self,
        card_id: i64,
        body: &crate::types::ProjectsMoveCardRequest,
    ) -> Result<crate::types::Data> {
        let url = format!(
            "/projects/columns/cards/{}/moves",
            crate::progenitor_support::encode_path(&card_id.to_string()),
        );

        self.client
            .post(
                &url,
                Some(reqwest::Body::from(serde_json::to_vec(body).unwrap())),
            )
            .await
    }

    /**
     * Get a project column.
     *
     * This function performs a `GET` to the `/projects/columns/{column_id}` endpoint.
     *
     *
     *
     * FROM: <https://docs.github.com/rest/reference/projects#get-a-project-column>
     *
     * **Parameters:**
     *
     * * `column_id: i64` -- column_id parameter.
     */
    pub async fn get_column(&self, column_id: i64) -> Result<crate::types::ProjectColumn> {
        let url = format!(
            "/projects/columns/{}",
            crate::progenitor_support::encode_path(&column_id.to_string()),
        );

        self.client.get(&url).await
    }

    /**
     * Delete a project column.
     *
     * This function performs a `DELETE` to the `/projects/columns/{column_id}` endpoint.
     *
     *
     *
     * FROM: <https://docs.github.com/rest/reference/projects#delete-a-project-column>
     *
     * **Parameters:**
     *
     * * `column_id: i64` -- column_id parameter.
     */
    pub async fn delete_column(&self, column_id: i64) -> Result<()> {
        let url = format!(
            "/projects/columns/{}",
            crate::progenitor_support::encode_path(&column_id.to_string()),
        );

        self.client.delete(&url, None).await
    }

    /**
     * Update an existing project column.
     *
     * This function performs a `PATCH` to the `/projects/columns/{column_id}` endpoint.
     *
     *
     *
     * FROM: <https://docs.github.com/rest/reference/projects#update-a-project-column>
     *
     * **Parameters:**
     *
     * * `column_id: i64` -- column_id parameter.
     */
    pub async fn update_column(
        &self,
        column_id: i64,
        body: &crate::types::ProjectsUpdateColumnRequest,
    ) -> Result<crate::types::ProjectColumn> {
        let url = format!(
            "/projects/columns/{}",
            crate::progenitor_support::encode_path(&column_id.to_string()),
        );

        self.client
            .patch(
                &url,
                Some(reqwest::Body::from(serde_json::to_vec(body).unwrap())),
            )
            .await
    }

    /**
     * List project cards.
     *
     * This function performs a `GET` to the `/projects/columns/{column_id}/cards` endpoint.
     *
     *
     *
     * FROM: <https://docs.github.com/rest/reference/projects#list-project-cards>
     *
     * **Parameters:**
     *
     * * `column_id: i64` -- column_id parameter.
     * * `archived_state: crate::types::ArchivedState` -- Filters the project cards that are returned by the card's state. Can be one of `all`,`archived`, or `not_archived`.
     * * `per_page: i64` -- Results per page (max 100).
     * * `page: i64` -- Page number of the results to fetch.
     */
    pub async fn list_cards(
        &self,
        column_id: i64,
        archived_state: crate::types::ArchivedState,
        per_page: i64,
        page: i64,
    ) -> Result<Vec<crate::types::ProjectCard>> {
        let url = format!(
            "/projects/columns/{}/cards?archived_state={}&page={}&per_page={}",
            crate::progenitor_support::encode_path(&column_id.to_string()),
            archived_state,
            format!("{}", page),
            format!("{}", per_page),
        );

        self.client.get(&url).await
    }

    /**
     * List project cards.
     *
     * This function performs a `GET` to the `/projects/columns/{column_id}/cards` endpoint.
     *
     * As opposed to `list_cards`, this function returns all the pages of the request at once.
     *
     *
     *
     * FROM: <https://docs.github.com/rest/reference/projects#list-project-cards>
     */
    pub async fn list_all_cards(
        &self,
        column_id: i64,
        archived_state: crate::types::ArchivedState,
    ) -> Result<Vec<crate::types::ProjectCard>> {
        let url = format!(
            "/projects/columns/{}/cards?archived_state={}",
            crate::progenitor_support::encode_path(&column_id.to_string()),
            archived_state,
        );

        self.client.get_all_pages(&url).await
    }

    /**
     * Create a project card.
     *
     * This function performs a `POST` to the `/projects/columns/{column_id}/cards` endpoint.
     *
     *
     *
     * FROM: <https://docs.github.com/rest/reference/projects#create-a-project-card>
     *
     * **Parameters:**
     *
     * * `column_id: i64` -- column_id parameter.
     */
    pub async fn create_card(
        &self,
        column_id: i64,
        body: &crate::types::ProjectsCreateCardRequest,
    ) -> Result<crate::types::ProjectCard> {
        let url = format!(
            "/projects/columns/{}/cards",
            crate::progenitor_support::encode_path(&column_id.to_string()),
        );

        self.client
            .post(
                &url,
                Some(reqwest::Body::from(serde_json::to_vec(body).unwrap())),
            )
            .await
    }

    /**
     * Move a project column.
     *
     * This function performs a `POST` to the `/projects/columns/{column_id}/moves` endpoint.
     *
     *
     *
     * FROM: <https://docs.github.com/rest/reference/projects#move-a-project-column>
     *
     * **Parameters:**
     *
     * * `column_id: i64` -- column_id parameter.
     */
    pub async fn move_column(
        &self,
        column_id: i64,
        body: &crate::types::ProjectsMoveColumnRequest,
    ) -> Result<crate::types::Data> {
        let url = format!(
            "/projects/columns/{}/moves",
            crate::progenitor_support::encode_path(&column_id.to_string()),
        );

        self.client
            .post(
                &url,
                Some(reqwest::Body::from(serde_json::to_vec(body).unwrap())),
            )
            .await
    }

    /**
     * Get a project.
     *
     * This function performs a `GET` to the `/projects/{project_id}` endpoint.
     *
     * Gets a project by its `id`. Returns a `404 Not Found` status if projects are disabled. If you do not have sufficient privileges to perform this action, a `401 Unauthorized` or `410 Gone` status is returned.
     *
     * FROM: <https://docs.github.com/rest/reference/projects#get-a-project>
     *
     * **Parameters:**
     *
     * * `project_id: i64`
     */
    pub async fn get(&self, project_id: i64) -> Result<crate::types::Project> {
        let url = format!(
            "/projects/{}",
            crate::progenitor_support::encode_path(&project_id.to_string()),
        );

        self.client.get(&url).await
    }

    /**
     * Delete a project.
     *
     * This function performs a `DELETE` to the `/projects/{project_id}` endpoint.
     *
     * Deletes a project board. Returns a `404 Not Found` status if projects are disabled.
     *
     * FROM: <https://docs.github.com/rest/reference/projects#delete-a-project>
     *
     * **Parameters:**
     *
     * * `project_id: i64`
     */
    pub async fn delete(&self, project_id: i64) -> Result<()> {
        let url = format!(
            "/projects/{}",
            crate::progenitor_support::encode_path(&project_id.to_string()),
        );

        self.client.delete(&url, None).await
    }

    /**
     * Update a project.
     *
     * This function performs a `PATCH` to the `/projects/{project_id}` endpoint.
     *
     * Updates a project board's information. Returns a `404 Not Found` status if projects are disabled. If you do not have sufficient privileges to perform this action, a `401 Unauthorized` or `410 Gone` status is returned.
     *
     * FROM: <https://docs.github.com/rest/reference/projects#update-a-project>
     *
     * **Parameters:**
     *
     * * `project_id: i64`
     */
    pub async fn update(
        &self,
        project_id: i64,
        body: &crate::types::ProjectsUpdateRequest,
    ) -> Result<crate::types::Project> {
        let url = format!(
            "/projects/{}",
            crate::progenitor_support::encode_path(&project_id.to_string()),
        );

        self.client
            .patch(
                &url,
                Some(reqwest::Body::from(serde_json::to_vec(body).unwrap())),
            )
            .await
    }

    /**
     * List project collaborators.
     *
     * This function performs a `GET` to the `/projects/{project_id}/collaborators` endpoint.
     *
     * Lists the collaborators for an organization project. For a project, the list of collaborators includes outside collaborators, organization members that are direct collaborators, organization members with access through team memberships, organization members with access through default organization permissions, and organization owners. You must be an organization owner or a project `admin` to list collaborators.
     *
     * FROM: <https://docs.github.com/rest/reference/projects#list-project-collaborators>
     *
     * **Parameters:**
     *
     * * `project_id: i64`
     * * `affiliation: crate::types::Affiliation` -- Filters the collaborators by their affiliation. Can be one of:  
     *  \* `outside`: Outside collaborators of a project that are not a member of the project's organization.  
     *  \* `direct`: Collaborators with permissions to a project, regardless of organization membership status.  
     *  \* `all`: All collaborators the authenticated user can see.
     * * `per_page: i64` -- Results per page (max 100).
     * * `page: i64` -- Page number of the results to fetch.
     */
    pub async fn list_collaborators(
        &self,
        project_id: i64,
        affiliation: crate::types::Affiliation,
        per_page: i64,
        page: i64,
    ) -> Result<Vec<crate::types::User>> {
        let url = format!(
            "/projects/{}/collaborators?affiliation={}&page={}&per_page={}",
            crate::progenitor_support::encode_path(&project_id.to_string()),
            affiliation,
            format!("{}", page),
            format!("{}", per_page),
        );

        self.client.get(&url).await
    }

    /**
     * List project collaborators.
     *
     * This function performs a `GET` to the `/projects/{project_id}/collaborators` endpoint.
     *
     * As opposed to `list_collaborators`, this function returns all the pages of the request at once.
     *
     * Lists the collaborators for an organization project. For a project, the list of collaborators includes outside collaborators, organization members that are direct collaborators, organization members with access through team memberships, organization members with access through default organization permissions, and organization owners. You must be an organization owner or a project `admin` to list collaborators.
     *
     * FROM: <https://docs.github.com/rest/reference/projects#list-project-collaborators>
     */
    pub async fn list_all_collaborators(
        &self,
        project_id: i64,
        affiliation: crate::types::Affiliation,
    ) -> Result<Vec<crate::types::User>> {
        let url = format!(
            "/projects/{}/collaborators?affiliation={}",
            crate::progenitor_support::encode_path(&project_id.to_string()),
            affiliation,
        );

        self.client.get_all_pages(&url).await
    }

    /**
     * Add project collaborator.
     *
     * This function performs a `PUT` to the `/projects/{project_id}/collaborators/{username}` endpoint.
     *
     * Adds a collaborator to an organization project and sets their permission level. You must be an organization owner or a project `admin` to add a collaborator.
     *
     * FROM: <https://docs.github.com/rest/reference/projects#add-project-collaborator>
     *
     * **Parameters:**
     *
     * * `project_id: i64`
     * * `username: &str`
     */
    pub async fn add_collaborator(
        &self,
        project_id: i64,
        username: &str,
        body: &crate::types::ProjectsAddCollaboratorRequest,
    ) -> Result<()> {
        let url = format!(
            "/projects/{}/collaborators/{}",
            crate::progenitor_support::encode_path(&project_id.to_string()),
            crate::progenitor_support::encode_path(&username.to_string()),
        );

        self.client
            .put(
                &url,
                Some(reqwest::Body::from(serde_json::to_vec(body).unwrap())),
            )
            .await
    }

    /**
     * Remove user as a collaborator.
     *
     * This function performs a `DELETE` to the `/projects/{project_id}/collaborators/{username}` endpoint.
     *
     * Removes a collaborator from an organization project. You must be an organization owner or a project `admin` to remove a collaborator.
     *
     * FROM: <https://docs.github.com/rest/reference/projects#remove-project-collaborator>
     *
     * **Parameters:**
     *
     * * `project_id: i64`
     * * `username: &str`
     */
    pub async fn remove_collaborator(&self, project_id: i64, username: &str) -> Result<()> {
        let url = format!(
            "/projects/{}/collaborators/{}",
            crate::progenitor_support::encode_path(&project_id.to_string()),
            crate::progenitor_support::encode_path(&username.to_string()),
        );

        self.client.delete(&url, None).await
    }

    /**
     * Get project permission for a user.
     *
     * This function performs a `GET` to the `/projects/{project_id}/collaborators/{username}/permission` endpoint.
     *
     * Returns the collaborator's permission level for an organization project. Possible values for the `permission` key: `admin`, `write`, `read`, `none`. You must be an organization owner or a project `admin` to review a user's permission level.
     *
     * FROM: <https://docs.github.com/rest/reference/projects#get-project-permission-for-a-user>
     *
     * **Parameters:**
     *
     * * `project_id: i64`
     * * `username: &str`
     */
    pub async fn get_permission_for_user(
        &self,
        project_id: i64,
        username: &str,
    ) -> Result<crate::types::RepositoryCollaboratorPermission> {
        let url = format!(
            "/projects/{}/collaborators/{}/permission",
            crate::progenitor_support::encode_path(&project_id.to_string()),
            crate::progenitor_support::encode_path(&username.to_string()),
        );

        self.client.get(&url).await
    }

    /**
     * List project columns.
     *
     * This function performs a `GET` to the `/projects/{project_id}/columns` endpoint.
     *
     *
     *
     * FROM: <https://docs.github.com/rest/reference/projects#list-project-columns>
     *
     * **Parameters:**
     *
     * * `project_id: i64`
     * * `per_page: i64` -- Results per page (max 100).
     * * `page: i64` -- Page number of the results to fetch.
     */
    pub async fn list_columns(
        &self,
        project_id: i64,
        per_page: i64,
        page: i64,
    ) -> Result<Vec<crate::types::ProjectColumn>> {
        let url = format!(
            "/projects/{}/columns?page={}&per_page={}",
            crate::progenitor_support::encode_path(&project_id.to_string()),
            format!("{}", page),
            format!("{}", per_page),
        );

        self.client.get(&url).await
    }

    /**
     * List project columns.
     *
     * This function performs a `GET` to the `/projects/{project_id}/columns` endpoint.
     *
     * As opposed to `list_columns`, this function returns all the pages of the request at once.
     *
     *
     *
     * FROM: <https://docs.github.com/rest/reference/projects#list-project-columns>
     */
    pub async fn list_all_columns(
        &self,
        project_id: i64,
    ) -> Result<Vec<crate::types::ProjectColumn>> {
        let url = format!(
            "/projects/{}/columns",
            crate::progenitor_support::encode_path(&project_id.to_string()),
        );

        self.client.get_all_pages(&url).await
    }

    /**
     * Create a project column.
     *
     * This function performs a `POST` to the `/projects/{project_id}/columns` endpoint.
     *
     *
     *
     * FROM: <https://docs.github.com/rest/reference/projects#create-a-project-column>
     *
     * **Parameters:**
     *
     * * `project_id: i64`
     */
    pub async fn create_column(
        &self,
        project_id: i64,
        body: &crate::types::ProjectsUpdateColumnRequest,
    ) -> Result<crate::types::ProjectColumn> {
        let url = format!(
            "/projects/{}/columns",
            crate::progenitor_support::encode_path(&project_id.to_string()),
        );

        self.client
            .post(
                &url,
                Some(reqwest::Body::from(serde_json::to_vec(body).unwrap())),
            )
            .await
    }

    /**
     * List repository projects.
     *
     * This function performs a `GET` to the `/repos/{owner}/{repo}/projects` endpoint.
     *
     * Lists the projects in a repository. Returns a `404 Not Found` status if projects are disabled in the repository. If you do not have sufficient privileges to perform this action, a `401 Unauthorized` or `410 Gone` status is returned.
     *
     * FROM: <https://docs.github.com/rest/reference/projects#list-repository-projects>
     *
     * **Parameters:**
     *
     * * `owner: &str`
     * * `repo: &str`
     * * `state: crate::types::IssuesListState` -- Indicates the state of the projects to return. Can be either `open`, `closed`, or `all`.
     * * `per_page: i64` -- Results per page (max 100).
     * * `page: i64` -- Page number of the results to fetch.
     */
    pub async fn list_for_repo(
        &self,
        owner: &str,
        repo: &str,
        state: crate::types::IssuesListState,
        per_page: i64,
        page: i64,
    ) -> Result<Vec<crate::types::Project>> {
        let url = format!(
            "/repos/{}/{}/projects?page={}&per_page={}&state={}",
            crate::progenitor_support::encode_path(&owner.to_string()),
            crate::progenitor_support::encode_path(&repo.to_string()),
            format!("{}", page),
            format!("{}", per_page),
            state,
        );

        self.client.get(&url).await
    }

    /**
     * List repository projects.
     *
     * This function performs a `GET` to the `/repos/{owner}/{repo}/projects` endpoint.
     *
     * As opposed to `list_for_repo`, this function returns all the pages of the request at once.
     *
     * Lists the projects in a repository. Returns a `404 Not Found` status if projects are disabled in the repository. If you do not have sufficient privileges to perform this action, a `401 Unauthorized` or `410 Gone` status is returned.
     *
     * FROM: <https://docs.github.com/rest/reference/projects#list-repository-projects>
     */
    pub async fn list_all_for_repo(
        &self,
        owner: &str,
        repo: &str,
        state: crate::types::IssuesListState,
    ) -> Result<Vec<crate::types::Project>> {
        let url = format!(
            "/repos/{}/{}/projects?state={}",
            crate::progenitor_support::encode_path(&owner.to_string()),
            crate::progenitor_support::encode_path(&repo.to_string()),
            state,
        );

        self.client.get_all_pages(&url).await
    }

    /**
     * Create a repository project.
     *
     * This function performs a `POST` to the `/repos/{owner}/{repo}/projects` endpoint.
     *
     * Creates a repository project board. Returns a `404 Not Found` status if projects are disabled in the repository. If you do not have sufficient privileges to perform this action, a `401 Unauthorized` or `410 Gone` status is returned.
     *
     * FROM: <https://docs.github.com/rest/reference/projects#create-a-repository-project>
     *
     * **Parameters:**
     *
     * * `owner: &str`
     * * `repo: &str`
     */
    pub async fn create_for_repo(
        &self,
        owner: &str,
        repo: &str,
        body: &crate::types::ProjectsCreateOrgRequest,
    ) -> Result<crate::types::Project> {
        let url = format!(
            "/repos/{}/{}/projects",
            crate::progenitor_support::encode_path(&owner.to_string()),
            crate::progenitor_support::encode_path(&repo.to_string()),
        );

        self.client
            .post(
                &url,
                Some(reqwest::Body::from(serde_json::to_vec(body).unwrap())),
            )
            .await
    }

    /**
     * Create a user project.
     *
     * This function performs a `POST` to the `/user/projects` endpoint.
     *
     *
     *
     * FROM: <https://docs.github.com/rest/reference/projects#create-a-user-project>
     */
    pub async fn create_for_authenticated_user(
        &self,
        body: &crate::types::ProjectsCreateRequest,
    ) -> Result<crate::types::Project> {
        let url = "/user/projects".to_string();
        self.client
            .post(
                &url,
                Some(reqwest::Body::from(serde_json::to_vec(body).unwrap())),
            )
            .await
    }

    /**
     * List user projects.
     *
     * This function performs a `GET` to the `/users/{username}/projects` endpoint.
     *
     *
     *
     * FROM: <https://docs.github.com/rest/reference/projects#list-user-projects>
     *
     * **Parameters:**
     *
     * * `username: &str`
     * * `state: crate::types::IssuesListState` -- Indicates the state of the projects to return. Can be either `open`, `closed`, or `all`.
     * * `per_page: i64` -- Results per page (max 100).
     * * `page: i64` -- Page number of the results to fetch.
     */
    pub async fn list_for_user(
        &self,
        username: &str,
        state: crate::types::IssuesListState,
        per_page: i64,
        page: i64,
    ) -> Result<Vec<crate::types::Project>> {
        let url = format!(
            "/users/{}/projects?page={}&per_page={}&state={}",
            crate::progenitor_support::encode_path(&username.to_string()),
            format!("{}", page),
            format!("{}", per_page),
            state,
        );

        self.client.get(&url).await
    }

    /**
     * List user projects.
     *
     * This function performs a `GET` to the `/users/{username}/projects` endpoint.
     *
     * As opposed to `list_for_user`, this function returns all the pages of the request at once.
     *
     *
     *
     * FROM: <https://docs.github.com/rest/reference/projects#list-user-projects>
     */
    pub async fn list_all_for_user(
        &self,
        username: &str,
        state: crate::types::IssuesListState,
    ) -> Result<Vec<crate::types::Project>> {
        let url = format!(
            "/users/{}/projects?state={}",
            crate::progenitor_support::encode_path(&username.to_string()),
            state,
        );

        self.client.get_all_pages(&url).await
    }
}
