//! OCSP response  
use tracing::{error, trace, warn};

use crate::common::asn1::Bytes;
use crate::common::{
    asn1::{
        asn1_encode_length, CertId, GeneralizedTime, Oid, ASN1_ENUMERATED, ASN1_EXPLICIT_0,
        ASN1_EXPLICIT_1,
    },
    ocsp::OcspExtI,
};
use crate::{common::asn1::asn1_encode_bit_string, oid::OCSP_OID_DOT_LIST};
use crate::{
    common::asn1::{ASN1_EXPLICIT_2, ASN1_OCTET, ASN1_SEQUENCE},
    err::{OcspError, Result},
    oid::OCSP_RESPONSE_BASIC_ID,
};

/// possible revocation reasons, See RFC 5280
#[repr(u8)]
#[derive(Debug, Copy, Clone)]
pub enum CrlReason {
    /// RFC defined revocation reason
    OcspRevokeUnspecified = 0u8,
    /// RFC defined revocation reason
    OcspRevokeKeyCompromise = 1u8,
    /// RFC defined revocation reason
    OcspRevokeCaCompromise = 2u8,
    /// RFC defined revocation reason
    OcspRevokeAffChanged = 3u8,
    /// RFC defined revocation reason
    OcspRevokeSuperseded = 4u8,
    /// RFC defined revocation reason
    OcspRevokeCessOperation = 5u8,
    /// RFC defined revocation reason
    OcspRevokeCertHold = 6u8,
    /// RFC defined revocation reason
    OcspRevokeRemoveFromCrl = 8u8,
    /// RFC defined revocation reason
    OcspRevokePrivWithdrawn = 9u8,
    /// RFC defined revocation reason
    OcspRevokeAaCompromise = 10u8,
}

/// RFC 6960 Revokeinfo
#[derive(Debug, Clone)]
pub struct RevokedInfo {
    /// revocation time
    pub revocation_time: GeneralizedTime,
    /// revocation reason, EXP 0 tagging, ASN.1 ENUMERATED
    pub revocation_reason: Option<CrlReason>,
}

impl RevokedInfo {
    /// create new RevokeInfo
    pub async fn new(gt: GeneralizedTime, reason: Option<CrlReason>) -> Self {
        RevokedInfo {
            revocation_time: gt,
            revocation_reason: reason,
        }
    }

    /// encode to ASN.1 DER
    pub async fn to_der(&self) -> Result<Bytes> {
        trace!(
            "Encoding revokeinfo with reason {:?}",
            self.revocation_reason
        );
        let mut time = self.revocation_time.to_der_utc().await?;
        let mut reason = vec![];
        if let Some(re) = self.revocation_reason {
            reason = vec![ASN1_EXPLICIT_0, 0x03, ASN1_ENUMERATED, 0x01, re as u8];
        }
        time.extend(reason);
        let len = asn1_encode_length(time.len()).await?;
        let mut tag = vec![ASN1_EXPLICIT_1];
        tag.extend(len);
        tag.extend(time);

        trace!("Revokeinfo successfully encoded");
        trace!("Revokeinfo {}", hex::encode(&tag));
        Ok(tag)
    }
}

/// Certificate status enum, value is defined in RFC 6960
#[repr(u8)]
#[derive(Debug, Clone)]
pub enum CertStatusCode {
    /// cert is valid
    Good = 0x80,
    /// cert is revoked
    Revoked = 0xa1,
    /// The "unknown" state indicates that the responder doesn't know about  
    /// the certificate being requested, usually because the request  
    /// indicates an unrecognized issuer that is not served by this responder.
    Unknown = 0x82,
}

/// RFC 6960 cert status
#[derive(Debug, Clone)]
pub struct CertStatus {
    code: CertStatusCode,
    revoke_info: Option<RevokedInfo>,
}

impl CertStatus {
    /// create new status
    pub async fn new(status: CertStatusCode, rev_info: Option<RevokedInfo>) -> Self {
        match status {
            CertStatusCode::Good | CertStatusCode::Unknown => {
                if rev_info.is_some() {
                    warn!("Cert status good or unknown with revoke info, ignored.");
                }
                CertStatus {
                    code: status,
                    revoke_info: None,
                }
            }

            CertStatusCode::Revoked => CertStatus {
                code: status,
                revoke_info: rev_info,
            },
        }
    }

    /// encode to ASN.1 DER
    pub async fn to_der(&self) -> Result<Bytes> {
        trace!("Encoding cert status {:?}", self);
        match self.code {
            CertStatusCode::Good => Ok(vec![CertStatusCode::Good as u8, 0x00]),
            CertStatusCode::Unknown => Ok(vec![CertStatusCode::Unknown as u8, 0x00]),
            CertStatusCode::Revoked => {
                let v;
                match &self.revoke_info {
                    Some(r) => {
                        // revoke_info to_der contains status code
                        v = r.to_der().await?
                    }
                    None => return Err(OcspError::GenRevokeInfoNotFound),
                }
                Ok(v)
            }
        }
    }
}

/// RFC 6960 response of single cert
#[derive(Debug, Clone)]
pub struct OneResp {
    /// certid of a resp
    pub cid: CertId,
    /// cert status
    pub cert_status: CertStatus,
    /// Responses whose thisUpdate time is later than the local system time SHOULD be considered unreliable.
    pub this_update: GeneralizedTime,
    /// Responses whose nextUpdate value is earlier than the local system time value SHOULD be considered unreliable
    pub next_update: Option<GeneralizedTime>,
    /// extension for a resp
    pub one_resp_ext: Option<Vec<OcspExtI>>,
}

impl OneResp {
    /// encode list of resp to ASN.1 DER
    pub async fn list_to_der(list: &[OneResp]) -> Result<Bytes> {
        trace!("Encoding {} OneResp", list.len());

        let mut v = vec![];
        for i in list {
            v.extend(i.to_der().await?);
        }

        let len = asn1_encode_length(v.len()).await?;
        let mut tlv = vec![ASN1_SEQUENCE];
        tlv.extend(len);
        tlv.extend(v);

        trace!("OneResp list successfully encoded");
        Ok(tlv)
    }

    /// encode to ASN.1 DER
    pub async fn to_der(&self) -> Result<Bytes> {
        trace!(
            "Encoding OneResp sn {} with status {:?}",
            hex::encode(&self.cid.serial_num),
            self.cert_status
        );
        trace!("OneResp: {:?}", self);
        let mut certid = self.cid.to_der().await?;
        let status = self.cert_status.to_der().await?;
        let this = self.this_update.to_der_utc().await?;

        certid.extend(status);
        certid.extend(this);

        if let Some(t) = self.next_update {
            trace!("Found OneResp nextUpdate");
            let next = t.to_der_utc().await?;
            let len = asn1_encode_length(next.len()).await?;
            let mut tagging = vec![ASN1_EXPLICIT_0];
            tagging.extend(len);
            tagging.extend(next);
            certid.extend(tagging);
        }

        if let Some(e) = self.one_resp_ext.clone() {
            trace!("Found {} OneResp extensions", e.len());
            // list_to_der comes with explicit tagging
            let list = OcspExtI::list_to_der(&e, ASN1_EXPLICIT_1).await?;
            certid.extend(list);
        }

        let len = asn1_encode_length(certid.len()).await?;
        let mut r = vec![ASN1_SEQUENCE];
        r.extend(len);
        r.extend(certid);

        trace!("OneResp successfully encoded");
        trace!("OneResp {}", hex::encode(&r));
        Ok(r)
    }
}

/// RFC 6960 defined responder types
#[allow(non_camel_case_types)]
#[derive(Debug)]
pub enum ResponderType {
    /// responder by name
    BY_NAME = 0x00,
    /// responder by key hash
    BY_KEY_HASH = 0x01,
}

/// Munzo responder id
#[derive(Debug)]
pub struct ResponderId {
    /// id by name or key hash
    pub id_by: ResponderType,
    /// id
    pub id: Bytes,
}

impl ResponderId {
    /// create new responder id
    pub async fn new_key_hash(key_hash: &[u8]) -> Self {
        ResponderId {
            id_by: ResponderType::BY_KEY_HASH,
            id: key_hash.to_vec(),
        }
    }

    /// encode to ASN.1, by name unimplemented!
    // example by name
    // a1 56
    //  30 54
    //      31 0b 30 09 06 03 55 04 06 13 02 41 55
    //      31 13 30 11 06 03 55 04 08 0c 0a 53 6f 6d 65 2d 53 74 61 74 65
    //      31 21 30 1f 06 03 55 04 0a 0c 18 49 6e 74 65 72 6e 65 74 20 57 69 64 67 69 74 73 20 50 74 79 20 4c 74 64
    //      31 0d 30 0b 06 03 55 04 03 0c 04 4f 43 53 50
    pub async fn to_der(&self) -> Result<Bytes> {
        trace!("Encoding Responder Id by {:?}", self.id_by);
        trace!("Responder Id: {:?}", self);

        let mut v = vec![];
        match self.id_by {
            ResponderType::BY_NAME => {
                // FIXME:
                unimplemented!()
            }
            ResponderType::BY_KEY_HASH => {
                let len = asn1_encode_length(self.id.len()).await?;
                let mut octet = vec![ASN1_OCTET];
                octet.extend(len);
                octet.extend(self.id.clone());
                let len = asn1_encode_length(octet.len()).await?;
                v.push(ASN1_EXPLICIT_2);
                v.extend(len);
                v.extend(octet);
            }
        }

        trace!("Responder id successfully encoded");
        trace!("Responder id {}", hex::encode(&v));
        Ok(v)
    }
}

/// RFC 6960 Response Data
#[derive(Debug)]
pub struct ResponseData {
    // REVIEW:
    // version
    /// responder id  
    /// in case of KeyHash ::= OCTET STRING  
    /// SHA-1 hash of responder's public key (excluding the tag and length fields)
    pub responder_id: ResponderId,
    /// time of creating response
    pub produced_at: GeneralizedTime,
    /// list of responses
    pub responses: Vec<OneResp>,
    /// exp 1
    pub resp_ext: Option<Vec<OcspExtI>>,
}

impl ResponseData {
    /// return new response data
    pub async fn new(
        id: ResponderId,
        produce: GeneralizedTime,
        list: Vec<OneResp>,
        ext: Option<Vec<OcspExtI>>,
    ) -> Self {
        ResponseData {
            responder_id: id,
            produced_at: produce,
            responses: list,
            resp_ext: ext,
        }
    }

    /// encode to ASN.1 DER
    pub async fn to_der(&self) -> Result<Bytes> {
        trace!("Encoding response data: {:?}", self);

        let mut v = vec![];
        v.extend(self.responder_id.to_der().await?);
        v.extend(self.produced_at.to_der_utc().await?);
        v.extend(OneResp::list_to_der(&self.responses).await?);
        if let Some(l) = &self.resp_ext {
            v.extend(OcspExtI::list_to_der(l, ASN1_EXPLICIT_1).await?);
        }

        let len = asn1_encode_length(v.len()).await?;
        let mut tlv = vec![ASN1_SEQUENCE];
        tlv.extend(len);
        tlv.extend(v);

        trace!("Response data successfully encoded");
        trace!("Response data {}", hex::encode(&tlv));
        Ok(tlv)
    }
}

/// RFC 6960 Basic Response
#[derive(Debug)]
pub struct BasicResponse {
    /// response data
    pub tbs_resp_data: ResponseData,
    /// signature algorithm oid
    pub signature_algo: Oid,
    ///  The value for signature SHALL be computed on the hash of the DER encoding of ResponseData
    pub signature: Bytes,
    /// The responder MAY include certificates in  
    /// the certs field of BasicOCSPResponse that help the OCSP client verify  
    /// the responder's signature.  
    /// If no certificates are included, then certs SHOULD be absent
    pub certs: Option<Bytes>,
}

impl BasicResponse {
    /// return new response data
    pub async fn new(data: ResponseData, algo: Oid, sign: Bytes, cert: Option<Bytes>) -> Self {
        BasicResponse {
            tbs_resp_data: data,
            signature_algo: algo,
            signature: sign,
            certs: cert,
        }
    }

    /// encode to ASN.1 DER
    pub async fn to_der(&self) -> Result<Bytes> {
        trace!("Encoding basic response: {:?}", self);

        let mut v = vec![];

        v.extend(self.tbs_resp_data.to_der().await?);
        v.extend(self.signature_algo.to_der_with_null().await?);
        let mut pad = vec![0x00u8];
        pad.extend(&self.signature);
        v.extend(asn1_encode_bit_string(&pad).await?);
        // FIXME:
        if let Some(_c) = &self.certs {
            unimplemented!()
        }

        let len = asn1_encode_length(v.len()).await?;
        let mut tlv = vec![ASN1_SEQUENCE];
        tlv.extend(len);
        tlv.extend(v);

        trace!("Basic response successfully encoded");
        trace!("Basic response {}", hex::encode(&tlv));
        Ok(tlv)
    }
}

/// RFC 6960 Response Bytes
/// The value for responseBytes consists of an OBJECT IDENTIFIER and a  
/// response syntax identified by that OID encoded as an OCTET STRING  
/// only basic response is implemented
#[derive(Debug)]
pub struct ResponseBytes {
    /// For a basic OCSP responder, responseType will be id-pkix-ocsp-basic
    pub response_type: Oid,
    /// basic response
    pub response_data: BasicResponse,
}

impl ResponseBytes {
    /// return new ResponseBytes
    /// currently only support basic response
    pub async fn new_basic(oid: Oid, data: BasicResponse) -> Result<Self> {
        if oid.index != OCSP_RESPONSE_BASIC_ID {
            let dot_name = OCSP_OID_DOT_LIST[oid.index];
            error!("Response type {} is not supported", dot_name);
            return Err(OcspError::OcspUnsupportedResponseType);
        }

        Ok(ResponseBytes {
            response_type: oid,
            response_data: data,
        })
    }

    /// encode to ASN.1 DER  
    /// - with explicit 0 tagging  
    /// - with octet header for basic response
    pub async fn to_der(&self) -> Result<Bytes> {
        trace!("Encoding response bytes {:?}", self);

        let mut v = vec![];
        v.extend(self.response_type.to_der_raw().await?);

        // octet header for basic response
        let basic = self.response_data.to_der().await?;
        let mut octet = vec![ASN1_OCTET];
        let len = asn1_encode_length(basic.len()).await?;
        octet.extend(len);
        octet.extend(basic);
        v.extend(octet);

        // response sequence header
        let len = asn1_encode_length(v.len()).await?;
        let mut tlv = vec![ASN1_SEQUENCE];
        tlv.extend(len);
        tlv.extend(v);

        // response byte exp 0 tagging
        let len = asn1_encode_length(tlv.len()).await?;
        let mut tagging = vec![ASN1_EXPLICIT_0];
        tagging.extend(len);
        tagging.extend(tlv);

        trace!("Response bytes successfully encoded");
        trace!("Response bytes {}", hex::encode(&tagging));
        Ok(tagging)
    }
}

/// ocsp request status enum
#[repr(u8)]
#[derive(Debug, Clone, Copy)]
pub enum OcspRespStatus {
    /// See RFC 6960
    Successful = 0u8,
    /// See RFC 6960
    MalformedReq = 1u8,
    /// See RFC 6960
    InternalError = 2u8,
    /// See RFC 6960
    TryLater = 3u8,
    /// See RFC 6960
    SigRequired = 5u8,
    /// See RFC 6960
    Unauthorized = 6u8,
}

/// RFC6960 Ocsp Response
#[derive(Debug)]
pub struct OcspResponse {
    /// response status
    pub resp_status: OcspRespStatus,
    /// If the value of responseStatus is one of the error conditions,  
    /// the responseBytes field is not set
    pub resp_bytes: Option<ResponseBytes>,
}

impl OcspResponse {
    /// crate new response with successful status
    pub async fn new_success(data: ResponseBytes) -> Self {
        OcspResponse {
            resp_status: OcspRespStatus::Successful,
            resp_bytes: Some(data),
        }
    }

    /// create new response with non-success status
    pub async fn new_non_success(status: OcspRespStatus) -> Result<Self> {
        match status as u8 {
            0 => Err(OcspError::OcspRespStatusError(
                "creating a success response with non_success function",
            )),
            1 | 2 | 3 | 5 | 6 => Ok(OcspResponse {
                resp_status: status,
                resp_bytes: None,
            }),
            _ => Err(OcspError::OcspRespUndefinedStatus(status as u8)),
        }
    }

    /// encode to ASN.1 DER
    pub async fn to_der(&self) -> Result<Bytes> {
        trace!("Ocsp Response: {:?}", self);

        let mut v = vec![ASN1_ENUMERATED, 0x01];
        v.push(self.resp_status as u8);

        if let Some(r) = &self.resp_bytes {
            v.extend(r.to_der().await?);
        }

        let len = asn1_encode_length(v.len()).await?;
        let mut tlv = vec![ASN1_SEQUENCE];
        tlv.extend(len);
        tlv.extend(v);

        trace!("Ocsp response successfully encoded");
        trace!("Ocsp response {}", hex::encode(&tlv));
        Ok(tlv)
    }
}

#[cfg(test)]
mod test {
    use crate::oid::{ALGO_SHA1_DOT, ALGO_SHA1_WITH_RSA_ENCRYPTION_DOT, OCSP_RESPONSE_BASIC_DOT};
    use tracing_subscriber::{layer::SubscriberExt, EnvFilter, Registry};

    use super::*;

    /// ocsp response to ASN.1 DER
    #[tokio::test]
    async fn ocsp_resp_success_to_der() {
        let key = [
            0x36, 0x6f, 0x35, 0xfb, 0xef, 0x16, 0xc6, 0xba, 0x8a, 0x31, 0x83, 0x42, 0x6d, 0x97,
            0xba, 0x89, 0x4d, 0x55, 0x6e, 0x91,
        ];
        let id = ResponderId::new_key_hash(&key).await;
        let produce = GeneralizedTime::new(2021, 1, 12, 3, 26, 43).await.unwrap();
        let oid = Oid::new_from_dot(ALGO_SHA1_DOT).await.unwrap();
        let name = vec![
            0x69, 0x4d, 0x18, 0xa9, 0xbe, 0x42, 0xf7, 0x80, 0x26, 0x14, 0xd4, 0x84, 0x4f, 0x23,
            0x60, 0x14, 0x78, 0xb7, 0x88, 0x20,
        ];
        let key = vec![
            0x39, 0x7b, 0xe0, 0x02, 0xa2, 0xf5, 0x71, 0xfd, 0x80, 0xdc, 0xeb, 0x52, 0xa1, 0x7a,
            0x7f, 0x8b, 0x63, 0x2b, 0xe7, 0x55,
        ];
        let sn = vec![0x41, 0x30, 0x09, 0x83, 0x33, 0x1f, 0x9d, 0x4f];
        let certid = CertId::new(oid.clone(), &name, &key, &sn).await;
        let good = CertStatus::new(CertStatusCode::Good, None).await;
        let gt = GeneralizedTime::new(2021, 1, 12, 3, 26, 43).await.unwrap();

        let one = OneResp {
            cid: certid.clone(),
            cert_status: good,
            this_update: gt.clone(),
            next_update: None,
            one_resp_ext: None,
        };

        let sn2 = vec![0x63, 0x78, 0xe5, 0x1d, 0x44, 0x8f, 0xf4, 0x6d];
        let certid2 = CertId::new(oid, &name, &key, &sn2).await;
        let rev_t = GeneralizedTime::new(2020, 11, 30, 1, 48, 25).await.unwrap();
        let rev_info = RevokedInfo::new(rev_t, Some(CrlReason::OcspRevokeUnspecified)).await;
        let revoke = CertStatus::new(CertStatusCode::Revoked, Some(rev_info)).await;
        let two = OneResp {
            cid: certid2,
            cert_status: revoke,
            this_update: gt,
            next_update: None,
            one_resp_ext: None,
        };

        let list = [one, two].to_vec();
        let data = ResponseData::new(id, produce, list, None).await;

        let oid = Oid::new_from_dot(ALGO_SHA1_WITH_RSA_ENCRYPTION_DOT)
            .await
            .unwrap();
        let sign = vec![
            0x1e, 0x02, 0x2d, 0x5b, 0xa2, 0x5a, 0xa6, 0xee, 0x97, 0xc5, 0xd9, 0x10, 0xc6, 0x1e,
            0xbe, 0xb7, 0x3d, 0xb7, 0x5a, 0x76, 0x7d, 0xeb, 0x43, 0xaf, 0x88, 0xc2, 0xa5, 0x63,
            0x77, 0xd9, 0xe5, 0xae, 0xaa, 0x54, 0x84, 0x30, 0x08, 0x7b, 0x54, 0x29, 0xd9, 0xb9,
            0x0b, 0x30, 0x56, 0x9f, 0x94, 0x44, 0x67, 0x6a, 0xd3, 0xa9, 0x88, 0x5f, 0xb6, 0xd2,
            0x9c, 0xd4, 0x64, 0x89, 0xea, 0x1a, 0x82, 0xc3, 0x69, 0x79, 0x0d, 0x2a, 0x49, 0x43,
            0xf4, 0xca, 0x93, 0xc9, 0x77, 0x06, 0xc9, 0x29, 0x70, 0x7f, 0xb6, 0xe5, 0xb4, 0x9d,
            0x43, 0x3b, 0x84, 0x00, 0x3b, 0xd9, 0xaa, 0x24, 0xa3, 0x95, 0x27, 0x8a, 0xb6, 0x3e,
            0x7a, 0x26, 0x22, 0xd2, 0xec, 0x7d, 0x35, 0x79, 0x45, 0x3e, 0x79, 0x60, 0xbb, 0xcf,
            0xca, 0x6d, 0x0d, 0x3d, 0xb0, 0xfe, 0x46, 0x0f, 0x7c, 0x2b, 0xba, 0xf7, 0x2e, 0x8c,
            0x6f, 0xb8, 0x5c, 0x7c, 0x65, 0x37, 0xea, 0x0c, 0xb3, 0xc3, 0x68, 0x11, 0xa8, 0x95,
            0x0f, 0x73, 0x96, 0x98, 0x75, 0x98, 0xa5, 0xb3, 0xc8, 0x9f, 0xc1, 0x46, 0x6c, 0xb1,
            0x7c, 0x55, 0x95, 0x89, 0xd8, 0x5d, 0x8a, 0xf9, 0x54, 0xd6, 0x07, 0xc4, 0x3f, 0xf7,
            0x08, 0xdd, 0xf5, 0xd6, 0x67, 0x2f, 0xaa, 0x14, 0xfb, 0xc7, 0x17, 0xb5, 0x53, 0x52,
            0xc2, 0x11, 0x04, 0x50, 0xf2, 0x20, 0xe8, 0xa0, 0xbe, 0x9e, 0x6a, 0x86, 0x64, 0xa6,
            0xac, 0xdc, 0xa6, 0x3a, 0xe3, 0xa7, 0x06, 0xb7, 0x2c, 0xc1, 0x9d, 0xa2, 0x27, 0xce,
            0x5e, 0x1a, 0x8f, 0x69, 0xad, 0xce, 0x38, 0xf4, 0x5f, 0x8d, 0xd3, 0x87, 0x48, 0x85,
            0x89, 0x8d, 0x7c, 0xeb, 0xd6, 0x05, 0x7f, 0xd8, 0xe5, 0xf3, 0x27, 0x69, 0x41, 0x98,
            0xed, 0xd9, 0x0f, 0xe6, 0xe8, 0x21, 0x61, 0x3b, 0xe7, 0x1e, 0x3b, 0xa2, 0x4f, 0x4d,
            0xb8, 0x5f, 0x10, 0xa7,
        ];

        let basic = BasicResponse::new(data, oid, sign, None).await;
        let resp_type = Oid::new_from_dot(OCSP_RESPONSE_BASIC_DOT).await.unwrap();
        let bytes = ResponseBytes::new_basic(resp_type, basic).await.unwrap();
        let ocsp = OcspResponse::new_success(bytes).await;
        let v = ocsp.to_der().await.unwrap();

        let c = vec![
            0x30, 0x82, 0x02, 0x27, 0x0a, 0x01, 0x00, 0xa0, 0x82, 0x02, 0x20, 0x30, 0x82, 0x02,
            0x1c, 0x06, 0x09, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x01, 0x01, 0x04, 0x82,
            0x02, 0x0d, 0x30, 0x82, 0x02, 0x09, 0x30, 0x81, 0xf2, 0xa2, 0x16, 0x04, 0x14, 0x36,
            0x6f, 0x35, 0xfb, 0xef, 0x16, 0xc6, 0xba, 0x8a, 0x31, 0x83, 0x42, 0x6d, 0x97, 0xba,
            0x89, 0x4d, 0x55, 0x6e, 0x91, 0x18, 0x0f, 0x32, 0x30, 0x32, 0x31, 0x30, 0x31, 0x31,
            0x32, 0x30, 0x33, 0x32, 0x36, 0x34, 0x33, 0x5a, 0x30, 0x81, 0xc6, 0x30, 0x56, 0x30,
            0x41, 0x30, 0x09, 0x06, 0x05, 0x2b, 0x0e, 0x03, 0x02, 0x1a, 0x05, 0x00, 0x04, 0x14,
            0x69, 0x4d, 0x18, 0xa9, 0xbe, 0x42, 0xf7, 0x80, 0x26, 0x14, 0xd4, 0x84, 0x4f, 0x23,
            0x60, 0x14, 0x78, 0xb7, 0x88, 0x20, 0x04, 0x14, 0x39, 0x7b, 0xe0, 0x02, 0xa2, 0xf5,
            0x71, 0xfd, 0x80, 0xdc, 0xeb, 0x52, 0xa1, 0x7a, 0x7f, 0x8b, 0x63, 0x2b, 0xe7, 0x55,
            0x02, 0x08, 0x41, 0x30, 0x09, 0x83, 0x33, 0x1f, 0x9d, 0x4f, 0x80, 0x00, 0x18, 0x0f,
            0x32, 0x30, 0x32, 0x31, 0x30, 0x31, 0x31, 0x32, 0x30, 0x33, 0x32, 0x36, 0x34, 0x33,
            0x5a, 0x30, 0x6c, 0x30, 0x41, 0x30, 0x09, 0x06, 0x05, 0x2b, 0x0e, 0x03, 0x02, 0x1a,
            0x05, 0x00, 0x04, 0x14, 0x69, 0x4d, 0x18, 0xa9, 0xbe, 0x42, 0xf7, 0x80, 0x26, 0x14,
            0xd4, 0x84, 0x4f, 0x23, 0x60, 0x14, 0x78, 0xb7, 0x88, 0x20, 0x04, 0x14, 0x39, 0x7b,
            0xe0, 0x02, 0xa2, 0xf5, 0x71, 0xfd, 0x80, 0xdc, 0xeb, 0x52, 0xa1, 0x7a, 0x7f, 0x8b,
            0x63, 0x2b, 0xe7, 0x55, 0x02, 0x08, 0x63, 0x78, 0xe5, 0x1d, 0x44, 0x8f, 0xf4, 0x6d,
            0xa1, 0x16, 0x18, 0x0f, 0x32, 0x30, 0x32, 0x30, 0x31, 0x31, 0x33, 0x30, 0x30, 0x31,
            0x34, 0x38, 0x32, 0x35, 0x5a, 0xa0, 0x03, 0x0a, 0x01, 0x00, 0x18, 0x0f, 0x32, 0x30,
            0x32, 0x31, 0x30, 0x31, 0x31, 0x32, 0x30, 0x33, 0x32, 0x36, 0x34, 0x33, 0x5a, 0x30,
            0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x05, 0x05, 0x00,
            0x03, 0x82, 0x01, 0x01, 0x00, 0x1e, 0x02, 0x2d, 0x5b, 0xa2, 0x5a, 0xa6, 0xee, 0x97,
            0xc5, 0xd9, 0x10, 0xc6, 0x1e, 0xbe, 0xb7, 0x3d, 0xb7, 0x5a, 0x76, 0x7d, 0xeb, 0x43,
            0xaf, 0x88, 0xc2, 0xa5, 0x63, 0x77, 0xd9, 0xe5, 0xae, 0xaa, 0x54, 0x84, 0x30, 0x08,
            0x7b, 0x54, 0x29, 0xd9, 0xb9, 0x0b, 0x30, 0x56, 0x9f, 0x94, 0x44, 0x67, 0x6a, 0xd3,
            0xa9, 0x88, 0x5f, 0xb6, 0xd2, 0x9c, 0xd4, 0x64, 0x89, 0xea, 0x1a, 0x82, 0xc3, 0x69,
            0x79, 0x0d, 0x2a, 0x49, 0x43, 0xf4, 0xca, 0x93, 0xc9, 0x77, 0x06, 0xc9, 0x29, 0x70,
            0x7f, 0xb6, 0xe5, 0xb4, 0x9d, 0x43, 0x3b, 0x84, 0x00, 0x3b, 0xd9, 0xaa, 0x24, 0xa3,
            0x95, 0x27, 0x8a, 0xb6, 0x3e, 0x7a, 0x26, 0x22, 0xd2, 0xec, 0x7d, 0x35, 0x79, 0x45,
            0x3e, 0x79, 0x60, 0xbb, 0xcf, 0xca, 0x6d, 0x0d, 0x3d, 0xb0, 0xfe, 0x46, 0x0f, 0x7c,
            0x2b, 0xba, 0xf7, 0x2e, 0x8c, 0x6f, 0xb8, 0x5c, 0x7c, 0x65, 0x37, 0xea, 0x0c, 0xb3,
            0xc3, 0x68, 0x11, 0xa8, 0x95, 0x0f, 0x73, 0x96, 0x98, 0x75, 0x98, 0xa5, 0xb3, 0xc8,
            0x9f, 0xc1, 0x46, 0x6c, 0xb1, 0x7c, 0x55, 0x95, 0x89, 0xd8, 0x5d, 0x8a, 0xf9, 0x54,
            0xd6, 0x07, 0xc4, 0x3f, 0xf7, 0x08, 0xdd, 0xf5, 0xd6, 0x67, 0x2f, 0xaa, 0x14, 0xfb,
            0xc7, 0x17, 0xb5, 0x53, 0x52, 0xc2, 0x11, 0x04, 0x50, 0xf2, 0x20, 0xe8, 0xa0, 0xbe,
            0x9e, 0x6a, 0x86, 0x64, 0xa6, 0xac, 0xdc, 0xa6, 0x3a, 0xe3, 0xa7, 0x06, 0xb7, 0x2c,
            0xc1, 0x9d, 0xa2, 0x27, 0xce, 0x5e, 0x1a, 0x8f, 0x69, 0xad, 0xce, 0x38, 0xf4, 0x5f,
            0x8d, 0xd3, 0x87, 0x48, 0x85, 0x89, 0x8d, 0x7c, 0xeb, 0xd6, 0x05, 0x7f, 0xd8, 0xe5,
            0xf3, 0x27, 0x69, 0x41, 0x98, 0xed, 0xd9, 0x0f, 0xe6, 0xe8, 0x21, 0x61, 0x3b, 0xe7,
            0x1e, 0x3b, 0xa2, 0x4f, 0x4d, 0xb8, 0x5f, 0x10, 0xa7,
        ];

        assert_eq!(c, v);
    }

    /// basic resp to ASN.1 DER
    #[tokio::test]
    async fn basic_resp_to_der() {
        let key = [
            0x36, 0x6f, 0x35, 0xfb, 0xef, 0x16, 0xc6, 0xba, 0x8a, 0x31, 0x83, 0x42, 0x6d, 0x97,
            0xba, 0x89, 0x4d, 0x55, 0x6e, 0x91,
        ];
        let id = ResponderId::new_key_hash(&key).await;
        let produce = GeneralizedTime::new(2021, 1, 12, 3, 26, 43).await.unwrap();
        let oid = Oid::new_from_dot(ALGO_SHA1_DOT).await.unwrap();
        let name = vec![
            0x69, 0x4d, 0x18, 0xa9, 0xbe, 0x42, 0xf7, 0x80, 0x26, 0x14, 0xd4, 0x84, 0x4f, 0x23,
            0x60, 0x14, 0x78, 0xb7, 0x88, 0x20,
        ];
        let key = vec![
            0x39, 0x7b, 0xe0, 0x02, 0xa2, 0xf5, 0x71, 0xfd, 0x80, 0xdc, 0xeb, 0x52, 0xa1, 0x7a,
            0x7f, 0x8b, 0x63, 0x2b, 0xe7, 0x55,
        ];
        let sn = vec![0x41, 0x30, 0x09, 0x83, 0x33, 0x1f, 0x9d, 0x4f];
        let certid = CertId::new(oid.clone(), &name, &key, &sn).await;
        let good = CertStatus::new(CertStatusCode::Good, None).await;
        let gt = GeneralizedTime::new(2021, 1, 12, 3, 26, 43).await.unwrap();

        let one = OneResp {
            cid: certid.clone(),
            cert_status: good,
            this_update: gt.clone(),
            next_update: None,
            one_resp_ext: None,
        };

        let sn2 = vec![0x63, 0x78, 0xe5, 0x1d, 0x44, 0x8f, 0xf4, 0x6d];
        let certid2 = CertId::new(oid, &name, &key, &sn2).await;
        let rev_t = GeneralizedTime::new(2020, 11, 30, 1, 48, 25).await.unwrap();
        let rev_info = RevokedInfo::new(rev_t, Some(CrlReason::OcspRevokeUnspecified)).await;
        let revoke = CertStatus::new(CertStatusCode::Revoked, Some(rev_info)).await;
        let two = OneResp {
            cid: certid2,
            cert_status: revoke,
            this_update: gt,
            next_update: None,
            one_resp_ext: None,
        };

        let list = [one, two].to_vec();
        let data = ResponseData::new(id, produce, list, None).await;

        let oid = Oid::new_from_dot(ALGO_SHA1_WITH_RSA_ENCRYPTION_DOT)
            .await
            .unwrap();
        let sign = vec![
            0x1e, 0x02, 0x2d, 0x5b, 0xa2, 0x5a, 0xa6, 0xee, 0x97, 0xc5, 0xd9, 0x10, 0xc6, 0x1e,
            0xbe, 0xb7, 0x3d, 0xb7, 0x5a, 0x76, 0x7d, 0xeb, 0x43, 0xaf, 0x88, 0xc2, 0xa5, 0x63,
            0x77, 0xd9, 0xe5, 0xae, 0xaa, 0x54, 0x84, 0x30, 0x08, 0x7b, 0x54, 0x29, 0xd9, 0xb9,
            0x0b, 0x30, 0x56, 0x9f, 0x94, 0x44, 0x67, 0x6a, 0xd3, 0xa9, 0x88, 0x5f, 0xb6, 0xd2,
            0x9c, 0xd4, 0x64, 0x89, 0xea, 0x1a, 0x82, 0xc3, 0x69, 0x79, 0x0d, 0x2a, 0x49, 0x43,
            0xf4, 0xca, 0x93, 0xc9, 0x77, 0x06, 0xc9, 0x29, 0x70, 0x7f, 0xb6, 0xe5, 0xb4, 0x9d,
            0x43, 0x3b, 0x84, 0x00, 0x3b, 0xd9, 0xaa, 0x24, 0xa3, 0x95, 0x27, 0x8a, 0xb6, 0x3e,
            0x7a, 0x26, 0x22, 0xd2, 0xec, 0x7d, 0x35, 0x79, 0x45, 0x3e, 0x79, 0x60, 0xbb, 0xcf,
            0xca, 0x6d, 0x0d, 0x3d, 0xb0, 0xfe, 0x46, 0x0f, 0x7c, 0x2b, 0xba, 0xf7, 0x2e, 0x8c,
            0x6f, 0xb8, 0x5c, 0x7c, 0x65, 0x37, 0xea, 0x0c, 0xb3, 0xc3, 0x68, 0x11, 0xa8, 0x95,
            0x0f, 0x73, 0x96, 0x98, 0x75, 0x98, 0xa5, 0xb3, 0xc8, 0x9f, 0xc1, 0x46, 0x6c, 0xb1,
            0x7c, 0x55, 0x95, 0x89, 0xd8, 0x5d, 0x8a, 0xf9, 0x54, 0xd6, 0x07, 0xc4, 0x3f, 0xf7,
            0x08, 0xdd, 0xf5, 0xd6, 0x67, 0x2f, 0xaa, 0x14, 0xfb, 0xc7, 0x17, 0xb5, 0x53, 0x52,
            0xc2, 0x11, 0x04, 0x50, 0xf2, 0x20, 0xe8, 0xa0, 0xbe, 0x9e, 0x6a, 0x86, 0x64, 0xa6,
            0xac, 0xdc, 0xa6, 0x3a, 0xe3, 0xa7, 0x06, 0xb7, 0x2c, 0xc1, 0x9d, 0xa2, 0x27, 0xce,
            0x5e, 0x1a, 0x8f, 0x69, 0xad, 0xce, 0x38, 0xf4, 0x5f, 0x8d, 0xd3, 0x87, 0x48, 0x85,
            0x89, 0x8d, 0x7c, 0xeb, 0xd6, 0x05, 0x7f, 0xd8, 0xe5, 0xf3, 0x27, 0x69, 0x41, 0x98,
            0xed, 0xd9, 0x0f, 0xe6, 0xe8, 0x21, 0x61, 0x3b, 0xe7, 0x1e, 0x3b, 0xa2, 0x4f, 0x4d,
            0xb8, 0x5f, 0x10, 0xa7,
        ];

        let basic = BasicResponse::new(data, oid, sign, None).await;
        let v = basic.to_der().await.unwrap();

        let c = vec![
            0x30, 0x82, 0x02, 0x09, 0x30, 0x81, 0xf2, 0xa2, 0x16, 0x04, 0x14, 0x36, 0x6f, 0x35,
            0xfb, 0xef, 0x16, 0xc6, 0xba, 0x8a, 0x31, 0x83, 0x42, 0x6d, 0x97, 0xba, 0x89, 0x4d,
            0x55, 0x6e, 0x91, 0x18, 0x0f, 0x32, 0x30, 0x32, 0x31, 0x30, 0x31, 0x31, 0x32, 0x30,
            0x33, 0x32, 0x36, 0x34, 0x33, 0x5a, 0x30, 0x81, 0xc6, 0x30, 0x56, 0x30, 0x41, 0x30,
            0x09, 0x06, 0x05, 0x2b, 0x0e, 0x03, 0x02, 0x1a, 0x05, 0x00, 0x04, 0x14, 0x69, 0x4d,
            0x18, 0xa9, 0xbe, 0x42, 0xf7, 0x80, 0x26, 0x14, 0xd4, 0x84, 0x4f, 0x23, 0x60, 0x14,
            0x78, 0xb7, 0x88, 0x20, 0x04, 0x14, 0x39, 0x7b, 0xe0, 0x02, 0xa2, 0xf5, 0x71, 0xfd,
            0x80, 0xdc, 0xeb, 0x52, 0xa1, 0x7a, 0x7f, 0x8b, 0x63, 0x2b, 0xe7, 0x55, 0x02, 0x08,
            0x41, 0x30, 0x09, 0x83, 0x33, 0x1f, 0x9d, 0x4f, 0x80, 0x00, 0x18, 0x0f, 0x32, 0x30,
            0x32, 0x31, 0x30, 0x31, 0x31, 0x32, 0x30, 0x33, 0x32, 0x36, 0x34, 0x33, 0x5a, 0x30,
            0x6c, 0x30, 0x41, 0x30, 0x09, 0x06, 0x05, 0x2b, 0x0e, 0x03, 0x02, 0x1a, 0x05, 0x00,
            0x04, 0x14, 0x69, 0x4d, 0x18, 0xa9, 0xbe, 0x42, 0xf7, 0x80, 0x26, 0x14, 0xd4, 0x84,
            0x4f, 0x23, 0x60, 0x14, 0x78, 0xb7, 0x88, 0x20, 0x04, 0x14, 0x39, 0x7b, 0xe0, 0x02,
            0xa2, 0xf5, 0x71, 0xfd, 0x80, 0xdc, 0xeb, 0x52, 0xa1, 0x7a, 0x7f, 0x8b, 0x63, 0x2b,
            0xe7, 0x55, 0x02, 0x08, 0x63, 0x78, 0xe5, 0x1d, 0x44, 0x8f, 0xf4, 0x6d, 0xa1, 0x16,
            0x18, 0x0f, 0x32, 0x30, 0x32, 0x30, 0x31, 0x31, 0x33, 0x30, 0x30, 0x31, 0x34, 0x38,
            0x32, 0x35, 0x5a, 0xa0, 0x03, 0x0a, 0x01, 0x00, 0x18, 0x0f, 0x32, 0x30, 0x32, 0x31,
            0x30, 0x31, 0x31, 0x32, 0x30, 0x33, 0x32, 0x36, 0x34, 0x33, 0x5a, 0x30, 0x0d, 0x06,
            0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x03, 0x82,
            0x01, 0x01, 0x00, 0x1e, 0x02, 0x2d, 0x5b, 0xa2, 0x5a, 0xa6, 0xee, 0x97, 0xc5, 0xd9,
            0x10, 0xc6, 0x1e, 0xbe, 0xb7, 0x3d, 0xb7, 0x5a, 0x76, 0x7d, 0xeb, 0x43, 0xaf, 0x88,
            0xc2, 0xa5, 0x63, 0x77, 0xd9, 0xe5, 0xae, 0xaa, 0x54, 0x84, 0x30, 0x08, 0x7b, 0x54,
            0x29, 0xd9, 0xb9, 0x0b, 0x30, 0x56, 0x9f, 0x94, 0x44, 0x67, 0x6a, 0xd3, 0xa9, 0x88,
            0x5f, 0xb6, 0xd2, 0x9c, 0xd4, 0x64, 0x89, 0xea, 0x1a, 0x82, 0xc3, 0x69, 0x79, 0x0d,
            0x2a, 0x49, 0x43, 0xf4, 0xca, 0x93, 0xc9, 0x77, 0x06, 0xc9, 0x29, 0x70, 0x7f, 0xb6,
            0xe5, 0xb4, 0x9d, 0x43, 0x3b, 0x84, 0x00, 0x3b, 0xd9, 0xaa, 0x24, 0xa3, 0x95, 0x27,
            0x8a, 0xb6, 0x3e, 0x7a, 0x26, 0x22, 0xd2, 0xec, 0x7d, 0x35, 0x79, 0x45, 0x3e, 0x79,
            0x60, 0xbb, 0xcf, 0xca, 0x6d, 0x0d, 0x3d, 0xb0, 0xfe, 0x46, 0x0f, 0x7c, 0x2b, 0xba,
            0xf7, 0x2e, 0x8c, 0x6f, 0xb8, 0x5c, 0x7c, 0x65, 0x37, 0xea, 0x0c, 0xb3, 0xc3, 0x68,
            0x11, 0xa8, 0x95, 0x0f, 0x73, 0x96, 0x98, 0x75, 0x98, 0xa5, 0xb3, 0xc8, 0x9f, 0xc1,
            0x46, 0x6c, 0xb1, 0x7c, 0x55, 0x95, 0x89, 0xd8, 0x5d, 0x8a, 0xf9, 0x54, 0xd6, 0x07,
            0xc4, 0x3f, 0xf7, 0x08, 0xdd, 0xf5, 0xd6, 0x67, 0x2f, 0xaa, 0x14, 0xfb, 0xc7, 0x17,
            0xb5, 0x53, 0x52, 0xc2, 0x11, 0x04, 0x50, 0xf2, 0x20, 0xe8, 0xa0, 0xbe, 0x9e, 0x6a,
            0x86, 0x64, 0xa6, 0xac, 0xdc, 0xa6, 0x3a, 0xe3, 0xa7, 0x06, 0xb7, 0x2c, 0xc1, 0x9d,
            0xa2, 0x27, 0xce, 0x5e, 0x1a, 0x8f, 0x69, 0xad, 0xce, 0x38, 0xf4, 0x5f, 0x8d, 0xd3,
            0x87, 0x48, 0x85, 0x89, 0x8d, 0x7c, 0xeb, 0xd6, 0x05, 0x7f, 0xd8, 0xe5, 0xf3, 0x27,
            0x69, 0x41, 0x98, 0xed, 0xd9, 0x0f, 0xe6, 0xe8, 0x21, 0x61, 0x3b, 0xe7, 0x1e, 0x3b,
            0xa2, 0x4f, 0x4d, 0xb8, 0x5f, 0x10, 0xa7,
        ];

        assert_eq!(c, v);
    }

    /// response data to ASN.1 DER
    #[tokio::test]
    async fn response_data_to_der() {
        let key = [
            0x36, 0x6f, 0x35, 0xfb, 0xef, 0x16, 0xc6, 0xba, 0x8a, 0x31, 0x83, 0x42, 0x6d, 0x97,
            0xba, 0x89, 0x4d, 0x55, 0x6e, 0x91,
        ];
        let id = ResponderId::new_key_hash(&key).await;
        let produce = GeneralizedTime::new(2021, 1, 12, 3, 26, 43).await.unwrap();
        let oid = Oid::new_from_dot(ALGO_SHA1_DOT).await.unwrap();
        let name = vec![
            0x69, 0x4d, 0x18, 0xa9, 0xbe, 0x42, 0xf7, 0x80, 0x26, 0x14, 0xd4, 0x84, 0x4f, 0x23,
            0x60, 0x14, 0x78, 0xb7, 0x88, 0x20,
        ];
        let key = vec![
            0x39, 0x7b, 0xe0, 0x02, 0xa2, 0xf5, 0x71, 0xfd, 0x80, 0xdc, 0xeb, 0x52, 0xa1, 0x7a,
            0x7f, 0x8b, 0x63, 0x2b, 0xe7, 0x55,
        ];
        let sn = vec![0x41, 0x30, 0x09, 0x83, 0x33, 0x1f, 0x9d, 0x4f];
        let certid = CertId::new(oid.clone(), &name, &key, &sn).await;
        let good = CertStatus::new(CertStatusCode::Good, None).await;
        let gt = GeneralizedTime::new(2021, 1, 12, 3, 26, 43).await.unwrap();

        let one = OneResp {
            cid: certid.clone(),
            cert_status: good,
            this_update: gt.clone(),
            next_update: None,
            one_resp_ext: None,
        };

        let sn2 = vec![0x63, 0x78, 0xe5, 0x1d, 0x44, 0x8f, 0xf4, 0x6d];
        let certid2 = CertId::new(oid, &name, &key, &sn2).await;
        let rev_t = GeneralizedTime::new(2020, 11, 30, 1, 48, 25).await.unwrap();
        let rev_info = RevokedInfo::new(rev_t, Some(CrlReason::OcspRevokeUnspecified)).await;
        let revoke = CertStatus::new(CertStatusCode::Revoked, Some(rev_info)).await;
        let two = OneResp {
            cid: certid2,
            cert_status: revoke,
            this_update: gt,
            next_update: None,
            one_resp_ext: None,
        };

        let list = [one, two].to_vec();
        let data = ResponseData::new(id, produce, list, None).await;
        let v = data.to_der().await.unwrap();

        let c = vec![
            0x30, 0x81, 0xf2, 0xa2, 0x16, 0x04, 0x14, 0x36, 0x6f, 0x35, 0xfb, 0xef, 0x16, 0xc6,
            0xba, 0x8a, 0x31, 0x83, 0x42, 0x6d, 0x97, 0xba, 0x89, 0x4d, 0x55, 0x6e, 0x91, 0x18,
            0x0f, 0x32, 0x30, 0x32, 0x31, 0x30, 0x31, 0x31, 0x32, 0x30, 0x33, 0x32, 0x36, 0x34,
            0x33, 0x5a, 0x30, 0x81, 0xc6, 0x30, 0x56, 0x30, 0x41, 0x30, 0x09, 0x06, 0x05, 0x2b,
            0x0e, 0x03, 0x02, 0x1a, 0x05, 0x00, 0x04, 0x14, 0x69, 0x4d, 0x18, 0xa9, 0xbe, 0x42,
            0xf7, 0x80, 0x26, 0x14, 0xd4, 0x84, 0x4f, 0x23, 0x60, 0x14, 0x78, 0xb7, 0x88, 0x20,
            0x04, 0x14, 0x39, 0x7b, 0xe0, 0x02, 0xa2, 0xf5, 0x71, 0xfd, 0x80, 0xdc, 0xeb, 0x52,
            0xa1, 0x7a, 0x7f, 0x8b, 0x63, 0x2b, 0xe7, 0x55, 0x02, 0x08, 0x41, 0x30, 0x09, 0x83,
            0x33, 0x1f, 0x9d, 0x4f, 0x80, 0x00, 0x18, 0x0f, 0x32, 0x30, 0x32, 0x31, 0x30, 0x31,
            0x31, 0x32, 0x30, 0x33, 0x32, 0x36, 0x34, 0x33, 0x5a, 0x30, 0x6c, 0x30, 0x41, 0x30,
            0x09, 0x06, 0x05, 0x2b, 0x0e, 0x03, 0x02, 0x1a, 0x05, 0x00, 0x04, 0x14, 0x69, 0x4d,
            0x18, 0xa9, 0xbe, 0x42, 0xf7, 0x80, 0x26, 0x14, 0xd4, 0x84, 0x4f, 0x23, 0x60, 0x14,
            0x78, 0xb7, 0x88, 0x20, 0x04, 0x14, 0x39, 0x7b, 0xe0, 0x02, 0xa2, 0xf5, 0x71, 0xfd,
            0x80, 0xdc, 0xeb, 0x52, 0xa1, 0x7a, 0x7f, 0x8b, 0x63, 0x2b, 0xe7, 0x55, 0x02, 0x08,
            0x63, 0x78, 0xe5, 0x1d, 0x44, 0x8f, 0xf4, 0x6d, 0xa1, 0x16, 0x18, 0x0f, 0x32, 0x30,
            0x32, 0x30, 0x31, 0x31, 0x33, 0x30, 0x30, 0x31, 0x34, 0x38, 0x32, 0x35, 0x5a, 0xa0,
            0x03, 0x0a, 0x01, 0x00, 0x18, 0x0f, 0x32, 0x30, 0x32, 0x31, 0x30, 0x31, 0x31, 0x32,
            0x30, 0x33, 0x32, 0x36, 0x34, 0x33, 0x5a,
        ];

        assert_eq!(c, v);
    }

    /// responder by key hash
    #[tokio::test]
    async fn responder_by_key_to_der() {
        let key = [
            0x36, 0x6f, 0x35, 0xfb, 0xef, 0x16, 0xc6, 0xba, 0x8a, 0x31, 0x83, 0x42, 0x6d, 0x97,
            0xba, 0x89, 0x4d, 0x55, 0x6e, 0x91,
        ];
        let id = ResponderId::new_key_hash(&key).await;
        let v = id.to_der().await.unwrap();
        let c = vec![
            0xa2, 0x16, 0x04, 0x14, 0x36, 0x6f, 0x35, 0xfb, 0xef, 0x16, 0xc6, 0xba, 0x8a, 0x31,
            0x83, 0x42, 0x6d, 0x97, 0xba, 0x89, 0x4d, 0x55, 0x6e, 0x91,
        ];

        assert_eq!(c, v);
    }

    /// two resp to ASN.1 DER
    #[tokio::test]
    async fn two_resp_to_der() {
        let oid = Oid::new_from_dot(ALGO_SHA1_DOT).await.unwrap();
        let name = vec![
            0x69, 0x4d, 0x18, 0xa9, 0xbe, 0x42, 0xf7, 0x80, 0x26, 0x14, 0xd4, 0x84, 0x4f, 0x23,
            0x60, 0x14, 0x78, 0xb7, 0x88, 0x20,
        ];
        let key = vec![
            0x39, 0x7b, 0xe0, 0x02, 0xa2, 0xf5, 0x71, 0xfd, 0x80, 0xdc, 0xeb, 0x52, 0xa1, 0x7a,
            0x7f, 0x8b, 0x63, 0x2b, 0xe7, 0x55,
        ];
        let sn = vec![0x41, 0x30, 0x09, 0x83, 0x33, 0x1f, 0x9d, 0x4f];
        let certid = CertId::new(oid.clone(), &name, &key, &sn).await;
        let good = CertStatus::new(CertStatusCode::Good, None).await;
        let gt = GeneralizedTime::new(2021, 1, 12, 3, 26, 43).await.unwrap();

        let one = OneResp {
            cid: certid.clone(),
            cert_status: good,
            this_update: gt.clone(),
            next_update: None,
            one_resp_ext: None,
        };

        let sn2 = vec![0x63, 0x78, 0xe5, 0x1d, 0x44, 0x8f, 0xf4, 0x6d];
        let certid2 = CertId::new(oid, &name, &key, &sn2).await;
        let rev_t = GeneralizedTime::new(2020, 11, 30, 1, 48, 25).await.unwrap();
        let rev_info = RevokedInfo::new(rev_t, Some(CrlReason::OcspRevokeUnspecified)).await;
        let revoke = CertStatus::new(CertStatusCode::Revoked, Some(rev_info)).await;
        let two = OneResp {
            cid: certid2,
            cert_status: revoke,
            this_update: gt,
            next_update: None,
            one_resp_ext: None,
        };

        let resp = [one, two];
        let v = OneResp::list_to_der(&resp).await.unwrap();

        let c = vec![
            0x30, 0x81, 0xc6, 0x30, 0x56, 0x30, 0x41, 0x30, 0x09, 0x06, 0x05, 0x2b, 0x0e, 0x03,
            0x02, 0x1a, 0x05, 0x00, 0x04, 0x14, 0x69, 0x4d, 0x18, 0xa9, 0xbe, 0x42, 0xf7, 0x80,
            0x26, 0x14, 0xd4, 0x84, 0x4f, 0x23, 0x60, 0x14, 0x78, 0xb7, 0x88, 0x20, 0x04, 0x14,
            0x39, 0x7b, 0xe0, 0x02, 0xa2, 0xf5, 0x71, 0xfd, 0x80, 0xdc, 0xeb, 0x52, 0xa1, 0x7a,
            0x7f, 0x8b, 0x63, 0x2b, 0xe7, 0x55, 0x02, 0x08, 0x41, 0x30, 0x09, 0x83, 0x33, 0x1f,
            0x9d, 0x4f, 0x80, 0x00, 0x18, 0x0f, 0x32, 0x30, 0x32, 0x31, 0x30, 0x31, 0x31, 0x32,
            0x30, 0x33, 0x32, 0x36, 0x34, 0x33, 0x5a, 0x30, 0x6c, 0x30, 0x41, 0x30, 0x09, 0x06,
            0x05, 0x2b, 0x0e, 0x03, 0x02, 0x1a, 0x05, 0x00, 0x04, 0x14, 0x69, 0x4d, 0x18, 0xa9,
            0xbe, 0x42, 0xf7, 0x80, 0x26, 0x14, 0xd4, 0x84, 0x4f, 0x23, 0x60, 0x14, 0x78, 0xb7,
            0x88, 0x20, 0x04, 0x14, 0x39, 0x7b, 0xe0, 0x02, 0xa2, 0xf5, 0x71, 0xfd, 0x80, 0xdc,
            0xeb, 0x52, 0xa1, 0x7a, 0x7f, 0x8b, 0x63, 0x2b, 0xe7, 0x55, 0x02, 0x08, 0x63, 0x78,
            0xe5, 0x1d, 0x44, 0x8f, 0xf4, 0x6d, 0xa1, 0x16, 0x18, 0x0f, 0x32, 0x30, 0x32, 0x30,
            0x31, 0x31, 0x33, 0x30, 0x30, 0x31, 0x34, 0x38, 0x32, 0x35, 0x5a, 0xa0, 0x03, 0x0a,
            0x01, 0x00, 0x18, 0x0f, 0x32, 0x30, 0x32, 0x31, 0x30, 0x31, 0x31, 0x32, 0x30, 0x33,
            0x32, 0x36, 0x34, 0x33, 0x5a,
        ];

        assert_eq!(c, v);
    }

    /// good one resp with next update
    #[tokio::test]
    async fn one_resp_good_next_update_to_der() {
        let oid = Oid::new_from_dot(ALGO_SHA1_DOT).await.unwrap();
        let name = vec![
            0x69, 0x4d, 0x18, 0xa9, 0xbe, 0x42, 0xf7, 0x80, 0x26, 0x14, 0xd4, 0x84, 0x4f, 0x23,
            0x60, 0x14, 0x78, 0xb7, 0x88, 0x20,
        ];
        let key = vec![
            0x39, 0x7b, 0xe0, 0x02, 0xa2, 0xf5, 0x71, 0xfd, 0x80, 0xdc, 0xeb, 0x52, 0xa1, 0x7a,
            0x7f, 0x8b, 0x63, 0x2b, 0xe7, 0x55,
        ];
        let sn = vec![0x41, 0x30, 0x09, 0x83, 0x33, 0x1f, 0x9d, 0x4f];
        let certid = CertId::new(oid, &name, &key, &sn).await;
        let good = CertStatus::new(CertStatusCode::Good, None).await;
        let gt = GeneralizedTime::new(2021, 1, 13, 3, 9, 25).await.unwrap();

        let one = OneResp {
            cid: certid,
            cert_status: good,
            this_update: gt.clone(),
            next_update: Some(gt),
            one_resp_ext: None,
        };

        let v = one.to_der().await.unwrap();

        let c = vec![
            0x30, 0x69, 0x30, 0x41, 0x30, 0x09, 0x06, 0x05, 0x2b, 0x0e, 0x03, 0x02, 0x1a, 0x05,
            0x00, 0x04, 0x14, 0x69, 0x4d, 0x18, 0xa9, 0xbe, 0x42, 0xf7, 0x80, 0x26, 0x14, 0xd4,
            0x84, 0x4f, 0x23, 0x60, 0x14, 0x78, 0xb7, 0x88, 0x20, 0x04, 0x14, 0x39, 0x7b, 0xe0,
            0x02, 0xa2, 0xf5, 0x71, 0xfd, 0x80, 0xdc, 0xeb, 0x52, 0xa1, 0x7a, 0x7f, 0x8b, 0x63,
            0x2b, 0xe7, 0x55, 0x02, 0x08, 0x41, 0x30, 0x09, 0x83, 0x33, 0x1f, 0x9d, 0x4f, 0x80,
            0x00, 0x18, 0x0f, 0x32, 0x30, 0x32, 0x31, 0x30, 0x31, 0x31, 0x33, 0x30, 0x33, 0x30,
            0x39, 0x32, 0x35, 0x5a, 0xa0, 0x11, 0x18, 0x0f, 0x32, 0x30, 0x32, 0x31, 0x30, 0x31,
            0x31, 0x33, 0x30, 0x33, 0x30, 0x39, 0x32, 0x35, 0x5a,
        ];

        assert_eq!(c, v);
    }

    // test encode one resp to ASN.1 DER
    #[tokio::test]
    async fn one_resp_good_to_der() {
        let oid = Oid::new_from_dot(ALGO_SHA1_DOT).await.unwrap();
        let name = vec![
            0x69, 0x4d, 0x18, 0xa9, 0xbe, 0x42, 0xf7, 0x80, 0x26, 0x14, 0xd4, 0x84, 0x4f, 0x23,
            0x60, 0x14, 0x78, 0xb7, 0x88, 0x20,
        ];
        let key = vec![
            0x39, 0x7b, 0xe0, 0x02, 0xa2, 0xf5, 0x71, 0xfd, 0x80, 0xdc, 0xeb, 0x52, 0xa1, 0x7a,
            0x7f, 0x8b, 0x63, 0x2b, 0xe7, 0x55,
        ];
        let sn = vec![0x41, 0x30, 0x09, 0x83, 0x33, 0x1f, 0x9d, 0x4f];
        let certid = CertId::new(oid, &name, &key, &sn).await;
        let good = CertStatus::new(CertStatusCode::Good, None).await;
        let gt = GeneralizedTime::new(2021, 1, 13, 3, 9, 25).await.unwrap();

        let one = OneResp {
            cid: certid,
            cert_status: good,
            this_update: gt.clone(),
            next_update: None,
            one_resp_ext: None,
        };

        let v = one.to_der().await.unwrap();

        let c = vec![
            0x30, 0x56, 0x30, 0x41, 0x30, 0x09, 0x06, 0x05, 0x2b, 0x0e, 0x03, 0x02, 0x1a, 0x05,
            0x00, 0x04, 0x14, 0x69, 0x4d, 0x18, 0xa9, 0xbe, 0x42, 0xf7, 0x80, 0x26, 0x14, 0xd4,
            0x84, 0x4f, 0x23, 0x60, 0x14, 0x78, 0xb7, 0x88, 0x20, 0x04, 0x14, 0x39, 0x7b, 0xe0,
            0x02, 0xa2, 0xf5, 0x71, 0xfd, 0x80, 0xdc, 0xeb, 0x52, 0xa1, 0x7a, 0x7f, 0x8b, 0x63,
            0x2b, 0xe7, 0x55, 0x02, 0x08, 0x41, 0x30, 0x09, 0x83, 0x33, 0x1f, 0x9d, 0x4f, 0x80,
            0x00, 0x18, 0x0f, 0x32, 0x30, 0x32, 0x31, 0x30, 0x31, 0x31, 0x33, 0x30, 0x33, 0x30,
            0x39, 0x32, 0x35, 0x5a,
        ];

        assert_eq!(c, v);
    }

    // test good & unknown status don't deal with revoke info
    #[tokio::test]
    async fn cert_good_with_rev_info() {
        let rev_info = RevokedInfo::new(
            GeneralizedTime::new(2021, 1, 1, 1, 1, 1).await.unwrap(),
            Some(CrlReason::OcspRevokeUnspecified),
        )
        .await;

        let good_rev = CertStatus::new(CertStatusCode::Good, Some(rev_info)).await;
        assert!(good_rev.revoke_info.is_none());

        let rev_info = RevokedInfo::new(
            GeneralizedTime::new(2021, 1, 1, 1, 1, 1).await.unwrap(),
            Some(CrlReason::OcspRevokeUnspecified),
        )
        .await;

        let unknown_rev = CertStatus::new(CertStatusCode::Unknown, Some(rev_info)).await;
        assert!(unknown_rev.revoke_info.is_none());
    }

    // test unknown cert status
    #[tokio::test]
    async fn cert_unknown() {
        let unknown = CertStatus::new(CertStatusCode::Unknown, None).await;
        let v = unknown.to_der().await.unwrap();
        assert_eq!(vec![0x82, 0x00], v);
    }

    // test revoke cert status
    #[tokio::test]
    async fn cert_revoke() {
        let rev_info = RevokedInfo::new(
            GeneralizedTime::new(2021, 1, 1, 1, 1, 1).await.unwrap(),
            Some(CrlReason::OcspRevokeUnspecified),
        )
        .await;
        let revoke = CertStatus::new(CertStatusCode::Revoked, Some(rev_info)).await;
        let v = revoke.to_der().await.unwrap();

        assert_eq!(
            vec![
                0xa1, 0x16, 0x18, 0x0f, 0x32, 0x30, 0x32, 0x31, 0x30, 0x31, 0x30, 0x31, 0x30, 0x31,
                0x30, 0x31, 0x30, 0x31, 0x5a, 0xa0, 0x03, 0x0a, 0x01, 0x00
            ],
            v
        );
    }

    // return good cert status
    #[tokio::test]
    async fn cert_good() {
        let good = CertStatus::new(CertStatusCode::Good, None).await;
        let v = good.to_der().await.unwrap();
        assert_eq!(vec![0x80, 0x00], v);
    }

    // test revoke info to der without reason
    #[tokio::test]
    async fn revoke_info_to_der_no_reason() {
        let ri = RevokedInfo {
            revocation_time: GeneralizedTime::new(2021, 1, 12, 8, 32, 56).await.unwrap(),
            revocation_reason: None,
        };

        let v = ri.to_der().await.unwrap();
        assert_eq!(
            vec![
                0xa1, 0x11, 0x18, 0x0f, 0x32, 0x30, 0x32, 0x31, 0x30, 0x31, 0x31, 0x32, 0x30, 0x38,
                0x33, 0x32, 0x35, 0x36, 0x5a
            ],
            v
        );
    }

    // test revoke info to der with reason
    #[tokio::test]
    async fn revoke_info_to_der() {
        let ri = RevokedInfo {
            revocation_time: GeneralizedTime::new(2020, 11, 30, 1, 48, 25).await.unwrap(),
            revocation_reason: Some(CrlReason::OcspRevokeUnspecified),
        };

        let v = ri.to_der().await.unwrap();
        assert_eq!(
            vec![
                0xa1, 0x16, 0x18, 0x0f, 0x32, 0x30, 0x32, 0x30, 0x31, 0x31, 0x33, 0x30, 0x30, 0x31,
                0x34, 0x38, 0x32, 0x35, 0x5a, 0xa0, 0x03, 0x0a, 0x01, 0x00
            ],
            v
        );
    }

    // init log
    #[allow(dead_code)]
    fn init() {
        let log_level = "debug";
        let env_layer = EnvFilter::try_from_default_env().unwrap_or(EnvFilter::new(log_level));
        let fmt_layer = tracing_subscriber::fmt::layer().with_target(true).pretty();
        let reg = Registry::default().with(env_layer).with(fmt_layer);
        tracing_log::LogTracer::init().unwrap();
        tracing::subscriber::set_global_default(reg).unwrap();
    }
}
