//!
//! # 6. Versions module
//! Type: Configuration Module
//! This is the required base module of OCPI.
//! This module is the starting point for any OCPI connection.
//! Via this module, clients can learn which versions of OCPI a server supports,
//! and which modules it supports for each of the versions.
//!

use crate::{types, Result};

pub trait VersionsModule {
    /// # 6.1. Version information endpoint
    /// This endpoint lists all the available OCPI versions and the corresponding
    /// URLs to where version specific details such as the
    /// supported endpoints can be found.
    /// Endpoint structure definition:
    ///
    /// No structure defined. This is open for every party to define themselves.
    /// Examples:
    /// https://www.server.com/ocpi/cpo/versions
    /// https://www.server.com/ocpi/emsp/versions
    /// https://ocpi.server.com/versions
    /// The exact URL to the implemented version endpoint should be given (offline)
    /// to parties that want to communicate with your OCPI implementation.
    /// Both, CPOs and eMSPs MUST implement such a version endpoint.
    fn versions_get(&self) -> Result<Vec<types::Version>> {
        Ok(Vec::new())
    }

    /// 6.2. Version details endpoint
    ///
    /// Via the version details, the parties can exchange which modules are implemented
    /// for a specific version of OCPI, which interface role is implemented,
    /// and what the endpoint URL is for this interface.
    /// Parties that are both CPO and eMSP (or a Hub) can implement one
    /// version endpoint that covers both roles.
    /// With the information that is available in the version details,
    /// parties don’t need to implement a separate endpoint per role (CPO or eMSP) anymore.
    /// In practice this means that when a company is both a CPO and an eMSP and it connects
    /// to another party that implements both interfaces, only one OCPI connection is needed.
    ///
    /// __NOTE__
    /// OCPI 2.2 introduces the role field in the version details.
    /// Older versions of OCPI do not support this.
    ///
    /// Endpoint structure definition:
    /// No structure defined. This is open for every party to define themselves.
    /// __Examples__:
    /// `https://www.server.com/ocpi/cpo/2.2`
    /// `https://www.server.com/ocpi/emsp/2.2`
    /// `https://ocpi.server.com/2.2/details`
    ///
    /// This endpoint lists the supported endpoints and their URLs for a specific OCPI version.
    /// To notify the other party that the list of endpoints of your current version has changed,
    /// you can send a PUT request to the corresponding credentials endpoint
    /// (see the credentials chapter).
    ///
    /// Both the CPO and the eMSP MUST implement this endpoint.
    fn versions_get_details(
        &self,
        _version_number: types::VersionNumber,
    ) -> Result<Option<types::VersionDetails>> {
        Ok(None)
    }
}

impl<DB> VersionsModule for crate::Cpo<DB>
where
    DB: crate::Store,
{
    fn versions_get(&self) -> Result<Vec<types::Version>> {
        let url = self
            .base_url
            .join("versions")
            .expect("versions")
            .join("2.2")
            .expect("URL 2.2");

        Ok(vec![types::Version {
            version: types::VersionNumber::V2_2,
            url,
        }])
    }

    fn versions_get_details(
        &self,
        version_number: types::VersionNumber,
    ) -> Result<Option<types::VersionDetails>> {
        match version_number {
            types::VersionNumber::V2_2 => Ok(Some(types::VersionDetails {
                version: types::VersionNumber::V2_2,
                endpoints: vec![types::Endpoint {
                    identifier: types::ModuleId::Credentials,
                    role: types::InterfaceRole::Receiver,
                    url: self
                        .base_url
                        .join("2.2")
                        .expect("Parsing url `2.2`")
                        .join("credentials")
                        .expect("Parsing URL `credentials`"),
                }],
            })),

            version => Ok(Some(types::VersionDetails {
                version,
                endpoints: vec![],
            })),
        }
    }
}

#[cfg(test)]
mod tests {

    use super::*;
    use crate::types;

    #[test]
    fn test_get_versions() {
        let cpo = crate::cpo();

        let res = cpo.versions_get().expect("getting versions");
        assert_eq!(res.len(), 1);
        assert_eq!(res[0].version, types::VersionNumber::V2_2);
    }

    #[test]
    fn test_get_version_details() {
        let cpo = crate::cpo();

        let res = cpo
            .versions_get_details(types::VersionNumber::V2_2)
            .expect("getting versions");

        assert!(res.is_some());
    }
}
