pub mod common;
pub mod credentials_module;
pub mod types;
pub mod versions_module;

mod client;
mod context;
mod error;
mod party;
mod party_store;

pub use {
    client::Client,
    common::*,
    context::Context,
    credentials_module::CredentialsModule,
    error::{ClientError, Error, HubError, ServerError},
    party::Party,
    party_store::PartyStore,
    versions_module::VersionsModule,
};

use {std::borrow::Cow, url::Url};

/// The Result type used by all OCPI functions.
/// Can be converted in a Response
pub type Result<T> = std::result::Result<T, Error>;

/// An Ocpi Response structure.
#[derive(serde::Serialize)]
pub struct Response<T>
where
    T: serde::Serialize,
{
    pub code: u32,
    #[serde(skip_serializing_if = "Option::is_none")]
    pub data: Option<T>,

    #[serde(rename = "status_message", skip_serializing_if = "Option::is_none")]
    pub message: Option<Cow<'static, str>>,

    pub timestamp: types::DateTime,
}

impl<T> From<Result<T>> for Response<T>
where
    T: serde::Serialize,
{
    fn from(res: Result<T>) -> Self {
        match res {
            Ok(data) => Response {
                code: 1000,
                data: Some(data),
                message: Some(Cow::Borrowed("Success")),
                timestamp: types::DateTime::now(),
            },

            Err(err) => Response {
                code: err.code(),
                data: None,
                message: Some(Cow::Owned(err.to_string())),
                timestamp: types::DateTime::now(),
            },
        }
    }
}

/// Cpo implements the CPO role of the OCPI Protocol.
///
/// Every module supplies an implementation of it self
/// on this type.
#[derive(Clone)]
pub struct Cpo<DB>
where
    DB: Store,
{
    base_url: Url,
    db: DB,
    client: Client,
}

impl<DB> Cpo<DB>
where
    DB: Store,
{
    /// Creates a new Cpo instance.
    /// the base_url must be the url to the base ocpi endpoint.
    /// __NOT__ the versions module.
    /// This base url will be used to add on each module url.
    /// so `versions` will be appended and the versions module
    /// should be served from that path.
    pub fn new(base_url: Url, db: DB, client: Client) -> Self {
        Self {
            base_url,
            db,
            client,
        }
    }
}

pub trait Store
where
    Self: Clone,
{
}

#[cfg(test)]
pub(crate) fn cpo() -> Cpo<()> {
    impl Store for () {}

    let base_url = "http://localhost".parse().expect("Parsing localhost url");
    Cpo {
        base_url,
        db: (),
        client: Client::default(),
    }
}
