use std::{fmt, str::FromStr};

#[derive(Clone, Copy, Debug, PartialEq, Eq)]
pub enum ModuleId {
    Cdrs,
    ChargingProfiles,
    Commands,
    Credentials,
    HubClientInfo,
    Locations,
    Sessions,
    Tariffs,
    Tokens,
}

impl fmt::Display for ModuleId {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            Self::Cdrs => f.write_str("cdrs"),
            Self::ChargingProfiles => f.write_str("chargingprofiles"),
            Self::Commands => f.write_str("commands"),
            Self::Credentials => f.write_str("credentials"),
            Self::HubClientInfo => f.write_str("hubclientinfo"),
            Self::Locations => f.write_str("locations"),
            Self::Sessions => f.write_str("sessions"),
            Self::Tariffs => f.write_str("tariffs"),
            Self::Tokens => f.write_str("tokens"),
        }
    }
}

impl FromStr for ModuleId {
    type Err = String;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        match s {
            "cdrs" => Ok(Self::Cdrs),
            "chargingprofiles" => Ok(Self::ChargingProfiles),
            "commands" => Ok(Self::Commands),
            "credentials" => Ok(Self::Credentials),
            "hubclientinfo" => Ok(Self::HubClientInfo),
            "locations" => Ok(Self::Locations),
            "sessions" => Ok(Self::Sessions),
            "tariffs" => Ok(Self::Tariffs),
            "tokens" => Ok(Self::Tokens),
            s => Err(format!("Invalid ModuleId `{}`", s)),
        }
    }
}

impl serde::Serialize for ModuleId {
    fn serialize<S>(&self, s: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        match self {
            Self::Cdrs => s.serialize_str("cdrs"),
            Self::ChargingProfiles => s.serialize_str("chargingprofiles"),
            Self::Commands => s.serialize_str("commands"),
            Self::Credentials => s.serialize_str("credentials"),
            Self::HubClientInfo => s.serialize_str("hubclientinfo"),
            Self::Locations => s.serialize_str("locations"),
            Self::Sessions => s.serialize_str("sessions"),
            Self::Tariffs => s.serialize_str("tariffs"),
            Self::Tokens => s.serialize_str("tokens"),
        }
    }
}

impl<'de> serde::Deserialize<'de> for ModuleId {
    fn deserialize<D>(deserializer: D) -> Result<ModuleId, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        deserializer.deserialize_str(crate::common::FromStrVisitor::expecting("a valid ModuleId"))
    }
}

#[cfg(test)]
mod tests {
    use super::ModuleId;

    #[test]
    fn test_serialize() {
        assert_eq!(
            serde_json::to_string(&ModuleId::Cdrs).expect("Serializing"),
            r#""cdrs""#
        );
        assert_eq!(
            serde_json::to_string(&ModuleId::ChargingProfiles).expect("Serializing"),
            r#""chargingprofiles""#
        );
        assert_eq!(
            serde_json::to_string(&ModuleId::Commands).expect("Serializing"),
            r#""commands""#
        );
        assert_eq!(
            serde_json::to_string(&ModuleId::Credentials).expect("Serializing"),
            r#""credentials""#
        );
        assert_eq!(
            serde_json::to_string(&ModuleId::HubClientInfo).expect("Serializing"),
            r#""hubclientinfo""#
        );
        assert_eq!(
            serde_json::to_string(&ModuleId::Locations).expect("Serializing"),
            r#""locations""#
        );
        assert_eq!(
            serde_json::to_string(&ModuleId::Sessions).expect("Serializing"),
            r#""sessions""#
        );
        assert_eq!(
            serde_json::to_string(&ModuleId::Tariffs).expect("Serializing"),
            r#""tariffs""#
        );
        assert_eq!(
            serde_json::to_string(&ModuleId::Tokens).expect("Serializing"),
            r#""tokens""#
        );
    }

    #[test]
    fn test_deserialize() {
        assert_eq!(
            ModuleId::Cdrs,
            serde_json::from_str(r#""cdrs""#).expect("Deserializing"),
        );
        assert_eq!(
            ModuleId::ChargingProfiles,
            serde_json::from_str(r#""chargingprofiles""#).expect("Deserializing"),
        );

        assert_eq!(
            ModuleId::Commands,
            serde_json::from_str(r#""commands""#).expect("Deserializing"),
        );
        assert_eq!(
            ModuleId::Credentials,
            serde_json::from_str(r#""credentials""#).expect("Deserializing"),
        );
        assert_eq!(
            ModuleId::HubClientInfo,
            serde_json::from_str(r#""hubclientinfo""#).expect("Deserializing"),
        );
        assert_eq!(
            ModuleId::Locations,
            serde_json::from_str(r#""locations""#).expect("Deserializing"),
        );
        assert_eq!(
            ModuleId::Sessions,
            serde_json::from_str(r#""sessions""#).expect("Deserializing"),
        );
        assert_eq!(
            ModuleId::Tariffs,
            serde_json::from_str(r#""tariffs""#).expect("Deserializing"),
        );
        assert_eq!(
            ModuleId::Tokens,
            serde_json::from_str(r#""tokens""#).expect("Deserializing"),
        );
    }
}
