use url::Url;

#[derive(Clone, Debug, PartialEq, Eq, serde::Serialize, serde::Deserialize)]
pub struct Credential {
    /// Case Sensitive, ASCII only. The credentials token for the other party to
    /// authenticate in your system. Not encoded in Base64 or any other
    /// encoding.
    pub token: super::CredentialsToken,

    /// The URL to your API versions endpoint.
    pub url: Url,

    /// List of the roles this party provides.
    pub roles: Vec<super::CredentialsRole>,
}

#[cfg(test)]
mod tests {

    use super::*;
    use crate::types;

    #[test]
    fn test_deserialize_minimal_example() {
        let s = r#"
{
"token": "ebf3b399-779f-4497-9b9d-ac6ad3cc44d2",
"url": "https://example.com/ocpi/versions/",
"roles": [{
"role": "CPO",
"party_id": "EXA",
"country_code": "NL",
"business_details": {
"name": "Example Operator"
}
}]
}
"#;

        let res = serde_json::from_str::<Credential>(s).expect("Deserializing");

        assert_eq!(res.token, "ebf3b399-779f-4497-9b9d-ac6ad3cc44d2");
        assert_eq!(res.url.as_str(), "https://example.com/ocpi/versions/");
        assert_eq!(res.roles[0].role, types::Role::Cpo);
        assert_eq!(res.roles[0].party_id, "EXA");
        assert_eq!(res.roles[0].country_code, "NL");
        assert_eq!(res.roles[0].business_details.name, "Example Operator");
    }

    #[test]
    fn test_deserialize_cpo_emsp_example() {
        let s = r#"
{
"token": "9e80a9c4-28be-11e9-b210-d663bd873d93",
"url": "https://ocpi.example.com/versions/",
"roles": [{
"role": "CPO",
"party_id": "EXA",
"country_code": "NL",
"business_details": {
"name": "Example Operator"
}
}, {
"role": "EMSP",
"party_id": "EXA",
"country_code": "NL",
"business_details": {
"name": "Example Provider"
}
}]
}
"#;

        let res = serde_json::from_str::<Credential>(s).expect("Deserializing");

        assert_eq!(res.token, "9e80a9c4-28be-11e9-b210-d663bd873d93");
        assert_eq!(res.url.as_str(), "https://ocpi.example.com/versions/");

        assert_eq!(res.roles[0].role, types::Role::Cpo);
        assert_eq!(res.roles[0].party_id, "EXA");
        assert_eq!(res.roles[0].country_code, "NL");
        assert_eq!(res.roles[0].business_details.name, "Example Operator");

        assert_eq!(res.roles[1].role, types::Role::Emsp);
        assert_eq!(res.roles[1].party_id, "EXA");
        assert_eq!(res.roles[1].country_code, "NL");
        assert_eq!(res.roles[1].business_details.name, "Example Provider");
    }

    #[test]
    fn test_deserialize_full_cpo_example() {
        let s = r#"
{
"token": "9e80ae10-28be-11e9-b210-d663bd873d93",
"url": "https://example.com/ocpi/versions/",
"roles": [{
"role": "CPO",
"party_id": "EXA",
"country_code": "NL",
"business_details": {
"name": "Example Operator",
"logo": {
"url": "https://example.com/img/logo.jpg",
"thumbnail": "https://example.com/img/logo_thumb.jpg",
"category": "OPERATOR",
"type": "jpeg",
"width": 512,
"height": 512
},
"website": "http://example.com"
}
}]
}

"#;

        let res = serde_json::from_str::<Credential>(s).expect("Deserializing");

        assert_eq!(res.token, "9e80ae10-28be-11e9-b210-d663bd873d93");
        assert_eq!(res.url.as_str(), "https://example.com/ocpi/versions/");

        assert_eq!(res.roles[0].role, types::Role::Cpo);
        assert_eq!(res.roles[0].party_id, "EXA");
        assert_eq!(res.roles[0].country_code, "NL");
        assert_eq!(res.roles[0].business_details.name, "Example Operator");
        assert!(res.roles[0].business_details.logo.is_some());
        let logo = res.roles[0].business_details.logo.as_ref().unwrap();

        assert_eq!(logo.url.as_str(), "https://example.com/img/logo.jpg");
        assert_eq!(
            logo.thumbnail.as_ref().unwrap().as_str(),
            "https://example.com/img/logo_thumb.jpg"
        );
        assert_eq!(logo.category, types::ImageCategory::Operator);
        assert_eq!(logo.r#type.as_str(), "jpeg");
        assert_eq!(logo.width, Some(512));
        assert_eq!(logo.height, Some(512));
    }
}
