use std::{fmt, str::FromStr};

#[derive(Clone, Copy, Debug, PartialEq, Eq)]
pub enum VersionNumber {
    V2_0,
    V2_1,
    V2_1_1,
    V2_2,
}

impl fmt::Display for VersionNumber {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            Self::V2_0 => f.write_str("2.0"),
            Self::V2_1 => f.write_str("2.1"),
            Self::V2_1_1 => f.write_str("2.1.1"),
            Self::V2_2 => f.write_str("2.2"),
        }
    }
}

impl FromStr for VersionNumber {
    type Err = String;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        match s {
            "2.0" => Ok(Self::V2_0),
            "2.1" => Ok(Self::V2_1),
            "2.1.1" => Ok(Self::V2_1_1),
            "2.2" => Ok(Self::V2_2),

            s => Err(format!("Invalid version `{}`", s)),
        }
    }
}

impl serde::Serialize for VersionNumber {
    fn serialize<S>(&self, s: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        match self {
            Self::V2_0 => s.serialize_str("2.0"),
            Self::V2_1 => s.serialize_str("2.1"),
            Self::V2_1_1 => s.serialize_str("2.1.1"),
            Self::V2_2 => s.serialize_str("2.2"),
        }
    }
}

impl<'de> serde::Deserialize<'de> for VersionNumber {
    fn deserialize<D>(deserializer: D) -> Result<VersionNumber, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        deserializer.deserialize_str(crate::common::FromStrVisitor::expecting(
            "a valid OCPI version",
        ))
    }
}

#[cfg(test)]
mod tests {

    use super::*;
    #[test]
    fn test_serialize_version_number() {
        assert_eq!(
            serde_json::to_string(&VersionNumber::V2_0).expect("Serializing"),
            r#""2.0""#
        );
        assert_eq!(
            serde_json::to_string(&VersionNumber::V2_1).expect("Serializing"),
            r#""2.1""#
        );
        assert_eq!(
            serde_json::to_string(&VersionNumber::V2_1_1).expect("Serializing"),
            r#""2.1.1""#
        );
        assert_eq!(
            serde_json::to_string(&VersionNumber::V2_2).expect("Serializing"),
            r#""2.2""#
        );
    }

    #[test]
    fn test_deserialize_version_number() {
        assert_eq!(
            VersionNumber::V2_0,
            serde_json::from_str(r#""2.0""#).expect("Deserializing"),
        );
        assert_eq!(
            VersionNumber::V2_1,
            serde_json::from_str(r#""2.1""#).expect("Deserializing"),
        );

        assert_eq!(
            VersionNumber::V2_1_1,
            serde_json::from_str(r#""2.1.1""#).expect("Deserializing"),
        );
        assert_eq!(
            VersionNumber::V2_2,
            serde_json::from_str(r#""2.2""#).expect("Deserializing"),
        );
    }
}
