use crate::{error::ServerError, types, Result};
use reqwest::Method;

/// Implements the different OCPI calls as a client.
pub struct Client {
    http: reqwest::Client,
}

impl Clone for Client {
    fn clone(&self) -> Self {
        Self {
            http: self.http.clone(),
        }
    }
}

impl Default for Client {
    fn default() -> Self {
        Self {
            http: reqwest::Client::new(),
        }
    }
}

impl Client {
    pub fn new(http: reqwest::Client) -> Self {
        Self { http }
    }

    /// Given a version number and the URL of the
    /// root OCPI versions endpoint and the client.
    /// The function first calls the versions URL to confirm
    /// that the required version exists.
    /// And then retrieves the available endpoints for that version.
    pub async fn get_endpoints_for_version(
        &self,
        token: &str,
        versions_url: types::Url,
        desired_version: types::VersionNumber,
    ) -> Result<types::VersionDetails> {
        let versions = self
            .http
            .request(Method::GET, versions_url.clone())
            .set_ocpi_token(token)
            .send()
            .await
            .map_err(|err| {
                ServerError::unusable_api(format!("Error calling api `{}`: {}", versions_url, err))
            })?
            .json::<Vec<types::Version>>()
            .await
            .map_err(|err| {
                ServerError::unusable_api(format!(
                    "Error parsing result from `{}` as json: {}",
                    versions_url, err
                ))
            })?;

        // Try to find the matching version.
        let version = versions
            .into_iter()
            .find(|v| v.version == desired_version)
            .ok_or(ServerError::IncompatibleEndpoints)?;

        let version_details = self
            .http
            .request(Method::GET, version.url.clone())
            .set_ocpi_token(token)
            .send()
            .await
            .map_err(|err| {
                ServerError::unusable_api(format!("Error calling api `{}`: {}", version.url, err))
            })?
            .json::<types::VersionDetails>()
            .await
            .map_err(|err| {
                ServerError::unusable_api(format!(
                    "Error parsing result from `{}` as json: {}",
                    version.url, err
                ))
            })?;

        Ok(version_details)
    }
}

trait SetOcpiToken {
    fn set_ocpi_token(self, raw_token: &str) -> Self;
}

impl SetOcpiToken for reqwest::RequestBuilder {
    fn set_ocpi_token(self, raw_token: &str) -> Self {
        let b64 = base64::encode(raw_token);
        self.header("Authorization", format!("Token {}", b64))
    }
}

#[cfg(test)]
mod tests {

    use serde_json::json;
    use wiremock::{matchers, Mock, MockServer, ResponseTemplate};

    use super::*;
    #[tokio::test]
    #[rustfmt::skip::macros(json)]
    async fn test_endpoints_for_version() {
        let cli = Client::default();
        let mock = MockServer::start().await;

        Mock::given(matchers::method("GET"))
            .and(matchers::path("/versions"))
            .respond_with(ResponseTemplate::new(200).set_body_json(json!(
		[
                    {
			"version": "2.1.1",
			"url": "http://www.server.com/ocpi/2.1.1/"
                    },
                    {
			"version": "2.2",
			"url": format!("{}/2.2", mock.uri())
                    }
		]
            )))
            .mount(&mock)
            .await;

        Mock::given(matchers::method("GET"))
            .and(matchers::path("/2.2"))
            .respond_with(ResponseTemplate::new(200).set_body_json(json!({
		"version": "2.2",
		"endpoints": [
		    {
			"identifier": "credentials",
			"role": "SENDER",
			"url": format!("{}/2.2/credentials", mock.uri())
		    }
		]
	    })))
            .mount(&mock)
            .await;

        let versions_url = format!("{}/versions", mock.uri())
            .parse::<types::Url>()
            .expect("Versions url");

        let details = cli
            .get_endpoints_for_version("imatoken", versions_url.clone(), types::VersionNumber::V2_2)
            .await
            .expect(&format!("Making request to {}", versions_url));

        assert_eq!(details.version, types::VersionNumber::V2_2);
        assert_eq!(details.endpoints.len(), 1);
        assert_eq!(
            details.endpoints[0].identifier,
            types::ModuleId::Credentials
        );
        assert_eq!(details.endpoints[0].role, types::InterfaceRole::Sender);
    }
}
