use super::{Endpoint, VersionNumber};

#[derive(Clone, Debug, PartialEq, Eq, serde::Serialize, serde::Deserialize)]
pub struct VersionDetails {
    pub version: VersionNumber,
    pub endpoints: Vec<Endpoint>,
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::types::{InterfaceRole, ModuleId};

    #[test]
    fn deserialize_example() {
        let example = r#"
{
"version": "2.2",
"endpoints": [
{
"identifier": "credentials",
"role": "SENDER",
"url": "https://example.com/ocpi/2.2/credentials/"
},
{
"identifier": "locations",
"role": "SENDER",
"url": "https://example.com/ocpi/cpo/2.2/locations/"
}
]
}
"#;

        let des = serde_json::from_str::<VersionDetails>(example).expect("Deserializing");
        assert_eq!(des.version, VersionNumber::V2_2);

        assert_eq!(des.endpoints[0].identifier, ModuleId::Credentials);
        assert_eq!(des.endpoints[0].role, InterfaceRole::Sender);
        assert_eq!(
            des.endpoints[0].url.to_string(),
            "https://example.com/ocpi/2.2/credentials/"
        );

        assert_eq!(des.endpoints[1].identifier, ModuleId::Locations);
        assert_eq!(des.endpoints[1].role, InterfaceRole::Sender);
        assert_eq!(
            des.endpoints[1].url.to_string(),
            "https://example.com/ocpi/cpo/2.2/locations/"
        );
    }
}
