use std::{fmt, str::FromStr};

#[derive(Clone, Copy, Debug, PartialEq, Eq)]
pub enum InterfaceRole {
    Sender,
    Receiver,
}

impl fmt::Display for InterfaceRole {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            Self::Sender => f.write_str("SENDER"),
            Self::Receiver => f.write_str("RECEIVER"),
        }
    }
}

impl FromStr for InterfaceRole {
    type Err = String;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        match s {
            "SENDER" => Ok(Self::Sender),
            "RECEIVER" => Ok(Self::Receiver),
            s => Err(format!(
                "Invalid InterfaceRole `{}`. Must be one of `RECEIVER`, `SENDER`",
                s
            )),
        }
    }
}

impl serde::Serialize for InterfaceRole {
    fn serialize<S>(&self, s: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        match self {
            Self::Sender => s.serialize_str("SENDER"),
            Self::Receiver => s.serialize_str("RECEIVER"),
        }
    }
}

impl<'de> serde::Deserialize<'de> for InterfaceRole {
    fn deserialize<D>(deserializer: D) -> Result<InterfaceRole, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        deserializer.deserialize_str(crate::common::FromStrVisitor::expecting(
            "a valid Interface Role, one of [`RECEIVER`, `SENDER`]",
        ))
    }
}

#[cfg(test)]
mod tests {
    use super::InterfaceRole;

    #[test]
    fn test_serialize() {
        assert_eq!(
            serde_json::to_string(&InterfaceRole::Sender).expect("Serializing"),
            r#""SENDER""#
        );
        assert_eq!(
            serde_json::to_string(&InterfaceRole::Receiver).expect("Serializing"),
            r#""RECEIVER""#
        );
    }

    #[test]
    fn test_deserialize() {
        assert_eq!(
            InterfaceRole::Sender,
            serde_json::from_str(r#""SENDER""#).expect("Deserializing"),
        );
        assert_eq!(
            InterfaceRole::Receiver,
            serde_json::from_str(r#""RECEIVER""#).expect("Deserializing"),
        );
    }
}
