mod test_store;

use {
    ocpi::{
        types::{self, CiString, CredentialsToken, CsString},
        CredentialsModule,
    },
    serde_json::json,
    wiremock::{matchers, Mock, MockServer, ResponseTemplate},
};

use std::str::FromStr;

use ocpi::Party;
pub use test_store::TestStore;
use wiremock::http::HeaderValue;

pub fn create_cpo_and_store() -> (ocpi::Cpo<TestStore>, TestStore) {
    let store = TestStore::default();
    (
        ocpi::Cpo::new(
            "http://localhost:8000".parse().expect("parsing unused url"),
            store.clone(),
            ocpi::Client::default(),
        ),
        store,
    )
}

/// Tests registration by using a mock server as EMP initiator
/// and a test store.
#[tokio::test]
#[rustfmt::skip::macros(json)]
async fn test_registration() {
    let mock = MockServer::start().await;
    // The token we give CPO and expect it to use back.
    let token_for_cpo = "<i_really_am_a_dummy_token>"
        .parse::<CredentialsToken>()
        .expect("Parsing dummy token");

    let b64_token = base64::encode(token_for_cpo.as_str());

    let mock_url = mock.uri().parse::<types::Url>().expect("Mock URI as URL");

    Mock::given(matchers::method("GET"))
        .and(matchers::path("/versions"))
        .and(matchers::header(
            "Authorization",
            HeaderValue::from_str(&format!("Token {}", b64_token)).expect("HeaderValue"),
        ))
        .respond_with(ResponseTemplate::new(200).set_body_json(json!(
            [
                {
		    "version": "2.1.1",
		    "url": "http://www.server.com/ocpi/2.1.1/"
                },
                {
		    "version": "2.2",
		    "url": format!("{}/2.2", mock.uri())
                }
            ]
        )))
        .mount(&mock)
        .await;

    Mock::given(matchers::method("GET"))
        .and(matchers::path("/2.2"))
        .and(matchers::header(
            "Authorization",
            HeaderValue::from_str(&format!("Token {}", b64_token)).expect("HeaderValue"),
        ))
        .respond_with(ResponseTemplate::new(200).set_body_json(json!({
            "version": "2.2",
            "endpoints": [
		{
		    "identifier": "credentials",
		    "role": "SENDER",
		    "url": format!("{}/2.2/credentials", mock.uri())
		}
            ]
        })))
        .mount(&mock)
        .await;

    let (cpo, store) = create_cpo_and_store();
    let temp_token = store.create_temp_token(); // token to use in registration
    let emsp_name: CsString<100> = "TestingEMSPParty".parse().expect("emsp_name");
    let emsp_party_id: CiString<3> = "WUT".parse().expect("emsp_party_id");
    let emsp_country_code: CiString<2> = "se".parse().expect("emsp_country_code");

    let roles = vec![ocpi::types::CredentialsRole {
        role: ocpi::types::Role::Emsp,
        business_details: ocpi::types::BusinessDetails {
            name: emsp_name.clone(),
            website: None,
            logo: None,
        },
        party_id: emsp_party_id.clone(),
        country_code: emsp_country_code.clone(),
    }];

    let cpo_credentials = cpo
        .credentials_post(
            ocpi::Context::new(temp_token),
            ocpi::types::Credential {
                token: token_for_cpo.clone(),
                url: mock_url.join("/versions").expect("Versions url"),
                roles,
            },
        )
        .await
        .expect("Posting credentials");

    let emsp_party = store
        .by_token(&token_for_cpo)
        .expect("Expected a party to be created");

    assert_ne!(cpo_credentials.token, token_for_cpo);
    assert_eq!(cpo_credentials.token, emsp_party.token_they_use());

    assert_eq!(emsp_party.name, emsp_name);
    assert_eq!(emsp_party.roles.len(), 1);
    assert_eq!(emsp_party.roles[0].business_details.name, emsp_name);
    assert_eq!(emsp_party.roles[0].party_id, emsp_party_id);
    assert_eq!(emsp_party.roles[0].country_code, emsp_country_code);
}
