use super::VersionNumber;
use url::Url;

#[derive(Clone, Debug, PartialEq, Eq, serde::Serialize, serde::Deserialize)]
pub struct Version {
    pub version: VersionNumber,
    pub url: Url,
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_deserialize_information_example() {
        let example = r#"
[
{
"version": "2.1.1",
"url": "https://www.server.com/ocpi/2.1.1/"
},
{
"version": "2.2",
"url": "https://www.server.com/ocpi/2.2/"
}
]
"#;

        let des = serde_json::from_str::<Vec<Version>>(example).expect("Deserializing");

        assert_eq!(des[0].version, VersionNumber::V2_1_1);
        assert_eq!(des[0].url.to_string(), "https://www.server.com/ocpi/2.1.1/");

        assert_eq!(des[1].version, VersionNumber::V2_2);
        assert_eq!(des[1].url.to_string(), "https://www.server.com/ocpi/2.2/");
    }

    #[test]
    fn test_serialize_information() {
        let v = Version {
            version: VersionNumber::V2_0,
            url: "https://www.server.com/ocpi/2.2/"
                .parse::<Url>()
                .expect("URL"),
        };

        assert_eq!(
            r#"{"version":"2.0","url":"https://www.server.com/ocpi/2.2/"}"#,
            serde_json::to_string(&v).expect("Serializing")
        );
    }
}
