use std::{fmt, str::FromStr};

#[derive(Clone, Copy, Debug, PartialEq, Eq, Hash)]
pub enum ImageCategory {
    /// Photo of the physical device that contains one or more EVSEs.
    Charger,

    /// Location entrance photo. Should show the car entrance to the location from street side.
    Entrance,

    /// Location overview photo.
    Location,

    /// Logo of an associated roaming network to be displayed with the EVSE
    /// for example in lists, maps and detailed information views.
    Network,

    /// Logo of the charge point operator, for example a municipality,
    /// to be displayed in the EVSEs detailed information view or in lists and maps,
    /// if no network logo is present.
    Operator,

    /// Other
    Other,

    /// Logo of the charge point owner, for example a local store, to be displayed in the
    /// EVSEs detailed information view.
    Owner,
}

impl FromStr for ImageCategory {
    type Err = String;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        match s {
            "CHARGER" => Ok(Self::Charger),
            "ENTRANCE" => Ok(Self::Entrance),
            "LOCATION" => Ok(Self::Location),
            "NETWORK" => Ok(Self::Network),
            "OPERATOR" => Ok(Self::Operator),
            "OTHER" => Ok(Self::Other),
            "OWNER" => Ok(Self::Owner),
            s => Err(format!("Invalid ImageCategory `{}`", s)),
        }
    }
}

impl fmt::Display for ImageCategory {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        let s = match self {
            Self::Charger => "CHARGER",
            Self::Entrance => "ENTRANCE",
            Self::Location => "LOCATION",
            Self::Network => "NETWORK",
            Self::Operator => "OPERATOR",
            Self::Other => "OTHER",
            Self::Owner => "OWNER",
        };

        f.write_str(s)
    }
}

impl serde::Serialize for ImageCategory {
    fn serialize<S>(&self, ser: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        let s = match self {
            Self::Charger => "CHARGER",
            Self::Entrance => "ENTRANCE",
            Self::Location => "LOCATION",
            Self::Network => "NETWORK",
            Self::Operator => "OPERATOR",
            Self::Other => "OTHER",
            Self::Owner => "OWNER",
        };

        ser.serialize_str(s)
    }
}

impl<'de> serde::Deserialize<'de> for ImageCategory {
    fn deserialize<D>(deserializer: D) -> Result<ImageCategory, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        deserializer.deserialize_str(crate::common::FromStrVisitor::expecting(
            "a valid ImageCategory",
        ))
    }
}
