//! # 7. Credentials module
//! Module Identifier: `credentials`
//! Type: Configuration Module
//! The credentials module is used to exchange the credentials token that has to
//! be used by parties for authorization of requests.
//! Every OCPI request is required to contain a credentials token in the
//! HTTP Authorization header.
use crate::{types, Context, Cpo, Party, PartyStore, Result};
use async_trait::async_trait;

#[async_trait]
pub trait CredentialsModule {
    type PartyModel: Party;
    type TemporaryModel;

    /// # 7.2.1. GET Method
    /// Retrieves the credentials object to access the server’s platform.
    /// The request body is empty, the response contains the credentials
    /// object to access the server’s platform.
    /// This credentials object also contains extra information about the server
    /// such as its business details.
    async fn credentials_get(&self, ctx: Context<Self::PartyModel>) -> Result<types::Credential>;

    /// # 7.2.2. POST Method
    /// Provides the server with credentials to access the client’s system.
    /// This credentials object also contains extra information about the
    /// client such as its business details.
    /// A POST initiates the registration process for this endpoint’s version.
    /// The server must also fetch the client’s endpoints for this version.
    async fn credentials_post(
        &self,
        ctx: Context<Self::TemporaryModel>,
        param: types::Credential,
    ) -> Result<types::Credential>;

    /// # 7.2.3. PUT Method
    /// Provides the server with updated credentials to access the client’s system.
    /// This credentials object also contains extra information about the
    /// client such as its business details.
    /// A PUT will switch to the version that contains this credentials endpoint if
    /// it’s different from the current version.
    /// The server must fetch the client’s endpoints again, even if the version has not changed.
    /// If successful, the server must generate a new credentials token for the client and
    /// respond with the client’s updated credentials to access the server’s system.
    /// The credentials object in the response also contains extra information about the server
    /// such as its business details.
    /// This method MUST return a
    /// `HTTP status code 405: method not allowed`
    /// if the client has not been registered yet.
    async fn credentials_put(
        &self,
        ctx: Context<Self::PartyModel>,
        param: types::Credential,
    ) -> Result<types::Credential>;

    /// # 7.2.4. DELETE Method
    /// Informs the server that its credentials to access the client’s system are now
    /// invalid and can no longer be used.
    /// Both parties must end any automated communication.
    /// This is the unregistration process.
    /// This method __MUST__ return a HTTP status code
    /// 405: method not allowed if the client has not been registered before.
    async fn credentials_delete(&self, ctx: Context<Self::PartyModel>) -> Result<()>;
}

#[async_trait]
impl<Db, M, Tm> CredentialsModule for Cpo<Db>
where
    M: Party + Send + 'static,
    Tm: Send + 'static,
    Db: PartyStore<PartyModel = M, TemporaryModel = Tm> + Sync,
{
    type PartyModel = M;
    type TemporaryModel = Tm;

    async fn credentials_get(&self, ctx: Context<Self::PartyModel>) -> Result<types::Credential> {
        let roles = self.db.get_our_roles().await?;

        Ok(types::Credential {
            token: ctx.as_ref().token_they_use(),
            url: self.base_url.clone(),
            roles,
        })
    }

    /// Provides credentials for the Server to access the Clients system.
    /// This is done using the Temporary model.
    /// If this the authentication of the provided Credential is succesful
    ///
    /// The provided Credentials will be used to create a new Model.
    /// The Credentials for this new Model is what is returned.
    async fn credentials_post(
        &self,
        ctx: Context<Self::TemporaryModel>,
        param: types::Credential,
    ) -> Result<types::Credential> {
        let version_details = self
            .client
            .get_endpoints_for_version(
                param.token.as_str(),
                param.url.clone(),
                types::VersionNumber::V2_2,
            )
            .await?;

        let party = self
            .db
            .save_new_party(ctx.into_inner(), param, version_details)
            .await?;

        let roles = self.db.get_our_roles().await?;

        Ok(types::Credential {
            token: party.token_they_use(),
            url: self.base_url.clone(),
            roles,
        })
    }

    async fn credentials_put(
        &self,
        ctx: Context<Self::PartyModel>,
        param: types::Credential,
    ) -> Result<types::Credential> {
        let version_details = self
            .client
            .get_endpoints_for_version(
                param.token.as_str(),
                param.url.clone(),
                types::VersionNumber::V2_2,
            )
            .await?;

        let party = self
            .db
            .update_party(ctx.into_inner(), param, version_details)
            .await?;

        let roles = self.db.get_our_roles().await?;

        Ok(types::Credential {
            token: party.token_they_use(),
            url: self.base_url.clone(),
            // @TODO: Add a Store component that can retreive our own roles.
            // For now, dummy hardcode this stuff.
            roles,
        })
    }

    async fn credentials_delete(&self, ctx: Context<Self::PartyModel>) -> Result<()> {
        let id = ctx.as_ref().id();
        self.db.delete_party(id).await?;
        Ok(())
    }
}
