//! Occlum SGX enclaves management.

#![allow(non_upper_case_globals)]
#![allow(non_camel_case_types)]
#![allow(non_snake_case)]
#![allow(deref_nullptr)]
#![allow(dead_code)]

mod sys;
use sys::*;

use std::ffi::CString;
use std::fmt;
use std::os::raw::{c_char, c_int};
use std::os::unix::io::AsRawFd;
use std::ptr;
use std::rc::Rc;

/// Occlum-PAL error codes.
#[derive(Debug, Clone, Copy, Eq, PartialEq)]
pub enum Error {
    PalApiError,
    InitError,
    CreateError,
    ExecError,
    ArgumentsError,
    CStringError,
}

impl fmt::Display for Error {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Error::PalApiError => write!(f, "PAL API version mismatch"),
            Error::InitError => write!(f, "Initialization error"),
            Error::CreateError => write!(f, "Process creation error"),
            Error::ExecError => write!(f, "Process execution error"),
            Error::ArgumentsError => write!(f, "Arguments list error"),
            Error::CStringError => write!(f, "String contains a bare \\0 character"),
        }
    }
}

impl std::error::Error for Error {}

/// Log level.
#[derive(Debug, Copy, Clone, PartialEq, Eq)]
pub enum LogLevel {
    Off,
    Error,
    Warn,
    Info,
    Trace,
}

impl LogLevel {
    fn as_bytes(&self) -> &'static [u8] {
        match *self {
            LogLevel::Off => b"off\0",
            LogLevel::Error => b"error\0",
            LogLevel::Warn => b"warn\0",
            LogLevel::Info => b"info\0",
            LogLevel::Trace => b"trace\0",
        }
    }
}

/// Enclave configuration.
#[derive(Debug, Clone, PartialEq, Eq)]
pub struct Config {
    instance_dir: CString,
    log_level: LogLevel,
}

impl Config {
    /// Create a new enclave configuration.
    /// `instance_dir` specifies the path of an Occlum instance directory, which is usually created with the
    /// `occlum new` command. The default value is "."; that is, the current working directory
    /// is the Occlum instance directory.
    pub fn new(instance_dir: impl ToString) -> Result<Self, Error> {
        Ok(Self {
            instance_dir: CString::new(instance_dir.to_string().as_bytes())
                .map_err(|_| Error::CStringError)?,
            log_level: LogLevel::Off,
        })
    }
}

impl Default for Config {
    fn default() -> Self {
        Config {
            instance_dir: CString::new(b".".to_vec()).unwrap(),
            log_level: LogLevel::Off,
        }
    }
}

impl Config {
    fn to_api(&self) -> occlum_pal_attr {
        occlum_pal_attr {
            instance_dir: self.instance_dir.as_ptr() as *const c_char,
            log_level: self.log_level.as_bytes().as_ptr() as *const c_char,
        }
    }
}

#[derive(Debug, Clone, PartialEq, Eq)]
pub struct Stdio {
    stdin: c_int,
    stdout: c_int,
    stderr: c_int,
}

impl Default for Stdio {
    fn default() -> Self {
        Self {
            stdin: 0,
            stdout: 1,
            stderr: 2,
        }
    }
}

impl Stdio {
    /// Redirect the process standard descriptors to existing descriptors.
    pub fn new(stdin: impl AsRawFd, stdout: impl AsRawFd, stderr: impl AsRawFd) -> Self {
        Self {
            stdin: stdin.as_raw_fd() as _,
            stdout: stdout.as_raw_fd() as _,
            stderr: stderr.as_raw_fd() as _,
        }
    }

    fn to_api(&self) -> occlum_stdio_fds {
        occlum_stdio_fds {
            stdin_fd: self.stdin,
            stdout_fd: self.stdout,
            stderr_fd: self.stderr,
        }
    }
}

/// A process running in an enclave.
#[derive(Debug, PartialEq, Eq)]
pub struct ProcessBuilder {
    path: CString,
    argv: Vec<CString>,
    argv_api: Option<Vec<*const c_char>>,
    env: Vec<CString>,
    env_api: Option<Vec<*const c_char>>,
    pid: c_int,
    stdio: Option<Stdio>,
}

/// A process to be run in an enclave.
#[derive(Debug)]
pub struct Process {
    builder: ProcessBuilder,
    enclave: Rc<Enclave>,
    pid: c_int,
}

impl Process {
    /// Return the process identifier.
    pub fn process_id(&self) -> ProcessId {
        ProcessId { pid: self.pid }
    }

    /// Execute the process inside the Occlum enclave
    pub fn exec(&self) -> Result<ExitCode, Error> {
        let mut exit_code: c_int = -1;
        let mut exec_args = occlum_pal_exec_args {
            pid: self.pid,
            exit_value: &mut exit_code,
        };
        let exec_result = unsafe { occlum_pal_exec(&mut exec_args) };
        if exec_result == 0 {
            Ok(exit_code)
        } else {
            Err(Error::CreateError)
        }
    }
}

/// A process identifier, that can be used to send signals to the process.
#[derive(Debug, Clone, PartialEq, Eq, Hash)]
pub struct ProcessId {
    pid: c_int,
}

impl ProcessId {
    /// Kill the process.
    pub fn kill(self) -> Result<(), Error> {
        let result = unsafe { occlum_pal_kill(self.pid, 9) };
        if result == 0 {
            Ok(())
        } else {
            Err(Error::ExecError)
        }
    }

    /// Send SIGTERM to the process.
    pub fn terminate(self) -> Result<(), Error> {
        let result = unsafe { occlum_pal_kill(self.pid, 16) };
        if result == 0 {
            Ok(())
        } else {
            Err(Error::ExecError)
        }
    }
}

pub type ExitCode = c_int;

impl ProcessBuilder {
    /// Create a new process.
    /// `path` is the path of the executable to run.
    /// `argv` is the list of arguments to pass to the executable. If not `None`, the first argument must be the application name, as seen by the application itself.
    /// `env` is the list of environment variables to pass to the application.
    /// `stdio` is the standard I/O descriptors to use for the process.
    pub fn new(
        path: impl ToString,
        argv: Option<&[String]>,
        env: Option<&[String]>,
        stdio: Option<Stdio>,
    ) -> Result<Self, Error> where {
        let path = CString::new(path.to_string().as_bytes()).map_err(|_| Error::CStringError)?;
        let argv = match argv {
            None => vec![path.clone()],
            Some(argv) => {
                let mut y = vec![];
                for x in argv {
                    y.push(CString::new(x.to_string().as_bytes()).map_err(|_| Error::CStringError)?)
                }
                y
            }
        };
        if argv.is_empty() {
            return Err(Error::ArgumentsError);
        }

        let env = env.unwrap_or(&[]);
        let env = {
            let mut y = vec![];
            for x in env {
                y.push(CString::new(x.to_string().as_bytes()).map_err(|_| Error::CStringError)?)
            }
            y
        };

        Ok(Self {
            path,
            argv,
            argv_api: None,
            env,
            env_api: None,
            pid: -1,
            stdio,
        })
    }

    /// Create a new process to be run in the enclave.
    pub fn build(mut self, enclave: &Rc<Enclave>) -> Result<Process, Error> {
        let mut argv_api: Vec<*const c_char> = vec![];
        for x in &self.argv {
            argv_api.push(x.as_ptr());
        }
        argv_api.push(ptr::null());

        let mut env_api: Vec<*const c_char> = vec![];
        for x in &self.env {
            env_api.push(x.as_ptr());
        }
        env_api.push(ptr::null());

        self.argv_api = Some(argv_api);
        self.env_api = Some(env_api);

        let stdio = Stdio::default().to_api();

        let mut process_api = occlum_pal_create_process_args {
            path: self.path.as_ptr(),
            argv: self.argv_api.as_mut().unwrap().as_mut_ptr(),
            env: self.env_api.as_mut().unwrap().as_mut_ptr(),
            stdio: &stdio,
            pid: &mut self.pid,
        };

        if unsafe { occlum_pal_create_process(&mut process_api) } != 0 {
            return Err(Error::CreateError);
        }
        let pid = unsafe { *process_api.pid };
        Ok(Process {
            builder: self,
            enclave: enclave.clone(),
            pid,
        })
    }
}

/// An Occlum SGX enclave.
#[derive(Debug)]
pub struct Enclave {
    config: Config,
}

impl Enclave {
    /// Create a new SGX enclave with the given configuration.
    pub fn new(config: Config) -> Result<Rc<Self>, Error> {
        if (unsafe { occlum_pal_get_version() } <= 0) {}

        let config_api = config.to_api();
        if unsafe { occlum_pal_init(&config_api) } != 0 {
            return Err(Error::InitError);
        }
        Ok(Rc::new(Enclave { config }))
    }

    fn destroy(&mut self) {
        unsafe {
            occlum_pal_destroy();
        }
    }
}

impl Drop for Enclave {
    fn drop(&mut self) {
        self.destroy();
    }
}
