//! Occlum SGX enclaves management.

#![allow(non_upper_case_globals)]
#![allow(non_camel_case_types)]
#![allow(non_snake_case)]
#![allow(deref_nullptr)]

use std::fmt;
use std::os::raw::{c_char, c_int};
use std::os::unix::io::AsRawFd;
use std::ptr;

include!(concat!(env!("OUT_DIR"), "/bindings.rs"));

/// Occlum-PAL error codes.
#[derive(Debug, Clone, Copy, Eq, PartialEq)]
pub enum Error {
    InitError,
    CreateError,
    ExecError,
    ArgumentsError,
}

impl fmt::Display for Error {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Error::InitError => write!(f, "Initialization error"),
            Error::CreateError => write!(f, "Process creation error"),
            Error::ExecError => write!(f, "Process execution error"),
            Error::ArgumentsError => write!(f, "Arguments list error"),
        }
    }
}

impl std::error::Error for Error {}

/// Log level.
#[derive(Debug, Copy, Clone, PartialEq, Eq)]
pub enum LogLevel {
    Off,
    Error,
    Warn,
    Info,
    Trace,
}

impl LogLevel {
    fn as_str(&self) -> &'static str {
        match *self {
            LogLevel::Off => "off",
            LogLevel::Error => "error",
            LogLevel::Warn => "warn",
            LogLevel::Info => "info",
            LogLevel::Trace => "trace",
        }
    }
}

/// Enclave configuration.
#[derive(Debug, Clone, PartialEq, Eq)]
pub struct Config {
    pub instance_dir: String,
    pub log_level: LogLevel,
}

impl Config {
    /// Create a new enclave configuration.
    /// `instance_dir` specifies the path of an Occlum instance directory, which is usually created with the
    /// `occlum new` command. The default value is "."; that is, the current working directory
    /// is the Occlum instance directory.
    pub fn new(instance_dir: impl ToString) -> Self {
        Self {
            instance_dir: instance_dir.to_string(),
            log_level: LogLevel::Info,
        }
    }
}

impl Default for Config {
    fn default() -> Self {
        Config {
            instance_dir: ".".to_string(),
            log_level: LogLevel::Off,
        }
    }
}

impl Config {
    fn to_api(&self) -> occlum_pal_attr {
        occlum_pal_attr {
            instance_dir: self.instance_dir.as_ptr() as *const c_char,
            log_level: self.log_level.as_str().as_ptr() as *const c_char,
        }
    }
}

#[derive(Debug, Clone, PartialEq, Eq)]
pub struct Stdio {
    stdin_api: c_int,
    stdout_api: c_int,
    stderr_api: c_int,
}

impl Default for Stdio {
    fn default() -> Self {
        Self {
            stdin_api: 0,
            stdout_api: 1,
            stderr_api: 2,
        }
    }
}

impl Stdio {
    /// Redirect the process standard descriptors to existing descriptors.
    pub fn new(stdin: impl AsRawFd, stdout: impl AsRawFd, stderr: impl AsRawFd) -> Self {
        Self {
            stdin_api: stdin.as_raw_fd() as _,
            stdout_api: stdout.as_raw_fd() as _,
            stderr_api: stderr.as_raw_fd() as _,
        }
    }
}

/// A process running in an enclave.
#[derive(Debug, PartialEq, Eq)]
pub struct ProcessBuilder {
    path: String,
    argv: Vec<String>,
    argv_api: Vec<*const c_char>,
    env: Option<Vec<String>>,
    env_api: Option<Vec<*const c_char>>,
    stdio: Option<Stdio>,
    pid_api: c_int,
}

/// A process to be run in an enclave.
#[derive(Debug, PartialEq, Eq, Hash)]
pub struct Process {
    pid_api: c_int,
}

impl Process {
    /// Return the process identifier.
    pub fn process_id(&self) -> ProcessId {
        ProcessId {
            pid_api: self.pid_api,
        }
    }

    /// Execute the process inside the Occlum enclave
    pub fn exec(&self) -> Result<ExitCode, Error> {
        let mut exit_code: c_int = -1;
        let mut exec_args = occlum_pal_exec_args {
            pid: self.pid_api,
            exit_value: &mut exit_code,
        };
        let exec_result = unsafe { occlum_pal_exec(&mut exec_args) };
        if exec_result == 0 {
            Ok(exit_code)
        } else {
            Err(Error::CreateError)
        }
    }
}

/// A process identifier, that can be used to send signals to the process.
#[derive(Debug, Clone, PartialEq, Eq, Hash)]
pub struct ProcessId {
    pid_api: c_int,
}

impl ProcessId {
    /// Kill the process.
    pub fn kill(self) -> Result<(), Error> {
        let result = unsafe { occlum_pal_kill(self.pid_api, 9) };
        if result == 0 {
            Ok(())
        } else {
            Err(Error::ExecError)
        }
    }

    /// Send SIGTERM to the process.
    pub fn terminate(self) -> Result<(), Error> {
        let result = unsafe { occlum_pal_kill(self.pid_api, 16) };
        if result == 0 {
            Ok(())
        } else {
            Err(Error::ExecError)
        }
    }
}

pub type ExitCode = c_int;

impl ProcessBuilder {
    /// Create a new process.
    /// `path` is the path of the executable to run.
    /// `argv` is the list of arguments to pass to the executable. If not `None`, the first argument must be the application name, as seen by the application itself.
    /// `env` is the list of environment variables to pass to the application.
    /// `stdio` is the standard I/O descriptors to use for the process.
    pub fn new<T>(
        path: impl ToString,
        argv: Option<&[impl ToString]>,
        env: Option<&[impl ToString]>,
        stdio: Option<Stdio>,
    ) -> Result<Self, Error> where {
        let path = path.to_string();
        let argv = match argv {
            None => vec![path.clone()],
            Some(argv) => argv.iter().map(|s| s.to_string()).collect(),
        };
        if argv.is_empty() {
            return Err(Error::ArgumentsError);
        }
        let argv_api = argv.iter().map(|s| s.as_ptr() as *const c_char).collect();
        let env: Option<Vec<_>> = env.map(|env| env.iter().map(|s| s.to_string()).collect());
        let env_api = env.as_ref().map(|env| env.iter().map(|s| s.as_ptr() as *const c_char).collect());
        Ok(Self {
            path,
            argv,
            argv_api,
            env,
            env_api,
            stdio,
            pid_api: -1,
        })
    }

    fn to_api(&mut self) -> occlum_pal_create_process_args {
        occlum_pal_create_process_args {
            path: self.path.as_ptr() as *const c_char,
            argv: self.argv_api.as_ptr() as *mut _,
            env: match &self.env_api {
                None => ptr::null_mut::<*const c_char>(),
                Some(env_api) => env_api.as_ptr() as *mut _,
            },
            stdio: ptr::null(),
            pid: &mut self.pid_api,
        }
    }

    /// Create a new process to be run in the enclave.
    pub fn build(self, mut process: ProcessBuilder) -> Result<Process, Error> {
        let mut process_api = process.to_api();
        if unsafe { occlum_pal_create_process(&mut process_api) } != 0 {
            return Err(Error::CreateError);
        }
        Ok(Process {
            pid_api: self.pid_api,
        })
    }
}

/// An Occlum SGX enclave.
#[derive(Debug)]
pub struct Enclave;

impl Enclave {
    /// Create a new SGX enclave with the given configuration.
    pub fn new(config: &Config) -> Result<Self, Error> {
        let config_api = config.to_api();
        if unsafe { occlum_pal_init(&config_api) } != 0 {
            return Err(Error::InitError);
        }
        Ok(Enclave)
    }

    /// Create a new process to be run in the enclave.
    pub fn create_process(&mut self, mut process: ProcessBuilder) -> Result<(), Error> {
        let mut process_api = process.to_api();
        if unsafe { occlum_pal_create_process(&mut process_api) } != 0 {
            return Err(Error::CreateError);
        }
        Ok(())
    }
}

impl Drop for Enclave {
    fn drop(&mut self) {
        unsafe {
            occlum_pal_destroy();
        }
    }
}
