use clap::{ArgEnum, Parser};
use serde_yaml::{Mapping, Value};
use std::process::exit;

#[derive(ArgEnum, Clone, Copy, Debug)]
enum OutputType {
    Toml,
    Yaml,
    Json,
}

#[derive(Parser, Debug)]
#[clap(version = env ! ("CARGO_PKG_VERSION"))]
struct Args {
    #[clap(short, long, help = "Build an array")]
    array: bool,

    #[clap(arg_enum, short, long, default_value_t = OutputType::Json, help = "Toast")]
    output_type: OutputType,

    values: Vec<String>,
}

fn to_value(input: &String) -> Value {
    serde_yaml::from_str(input.as_str()).unwrap_or(Value::Null)
}

fn to_sequence(input: Vec<String>) -> Value {
    input.iter().map(to_value).collect()
}

fn to_mapping(input: Vec<String>) -> Value {
    let mut mapping = Mapping::new();

    for item in input {
        let mut split = item.split("=");
        let key = if let Some(key) = split.next() {
            key
        } else {
            break;
        };

        let value = split
            .fold(String::new(), |a, b| a + "=" + b)
            .chars()
            .skip(1)
            .collect::<String>();

        let value = if item.contains("=") && value.is_empty() {
            Value::String(String::new())
        } else {
            serde_yaml::from_str(value.as_str()).unwrap_or(Value::Null)
        };

        mapping.insert(Value::String(key.to_string()), value);
    }

    Value::Mapping(mapping)
}

impl OutputType {
    pub fn stringify_value(&self, value: Value) -> Result<String, Box<dyn std::error::Error>> {
        Ok(match self {
            OutputType::Toml => toml::to_string(&value)?,
            OutputType::Yaml => serde_yaml::to_string(&value)?,
            OutputType::Json => serde_json::to_string(&value)?,
        })
    }
}

fn array_implied(values: &Vec<String>) -> bool {
    for value in values {
        if value.contains("=") {
            return false;
        }
    }

    true
}

fn main() {
    let args = <Args as Parser>::parse();

    let value = if args.array || array_implied(&args.values) {
        to_sequence(args.values)
    } else {
        to_mapping(args.values)
    };

    match args.output_type.stringify_value(value) {
        Ok(value) => println!("{}", value),
        Err(error) => {
            eprintln!("Could not generate output: {}", error);

            exit(1);
        }
    }
}
