use crate::window::{Application, Frame};

#[cfg(windows)]
pub fn run_app(application: Box<dyn Application>) {
    // Get window title
    let title = application.title();
    // Get attributes
    let attributes = application.attributes();
    // startup process
    application.startup();

    // Create a window
    use windows_sys::{Win32::Foundation::*, Win32::Graphics::Gdi::ValidateRect, Win32::System::LibraryLoader::GetModuleHandleA, Win32::UI::WindowsAndMessaging::*};
    extern "system" fn wndproc(window: HWND, message: u32, wparam: WPARAM, lparam: LPARAM) -> LRESULT {
        unsafe {
            match message as u32 {
                WM_PAINT => {
                    println!("WM_PAINT");
                    ValidateRect(window, std::ptr::null());
                    0
                }
                WM_DESTROY => {
                    println!("WM_DESTROY");
                    PostQuitMessage(0);
                    0
                }
                _ => DefWindowProcA(window, message, wparam, lparam),
            }
        }
    }

    unsafe {
        let instance = GetModuleHandleA(std::ptr::null_mut());
        debug_assert!(instance != 0);

        let wc = WNDCLASSA {
            hCursor: LoadCursorW(0, IDC_ARROW),
            hInstance: instance,
            lpszClassName: "window".as_ptr(),
            style: CS_HREDRAW | CS_VREDRAW,
            lpfnWndProc: Some(wndproc),
            cbClsExtra: 0,
            cbWndExtra: 0,
            hIcon: 0,
            hbrBackground: 0,
            lpszMenuName: std::ptr::null_mut(),
        };

        let atom = RegisterClassA(&wc);
        debug_assert!(atom != 0);

        let win = CreateWindowExA(0, "window".as_ptr(), title.as_ptr(), attributes.style, attributes.x, attributes.y, attributes.width, attributes.height, 0, 0, instance, std::ptr::null_mut());
        let frame = Frame::new(win);
        application.ui(frame);
        let mut message = std::mem::zeroed();

        while GetMessageA(&mut message, 0, 0, 0) != 0 {
            DispatchMessageA(&message);
        }

        application.exit();
    }

}

#[cfg(unix)]
pub fn run_app(application: Box<dyn Application>) {
    // Get window title
    let title = application.title();
    // Get attributes
    let attributes = application.attributes();
    // startup process
    application.startup();

    // Create a window
    use gtk::prelude::*;
    gtk::init().expect("Failed to initialize GTK.");

    let window = gtk::Window::new(gtk::WindowType::Toplevel);

    window.set_title(title.as_str());

    window.connect_delete_event(|_, _| {
        gtk::main_quit();
        Inhibit(false)
    });
    let frame = Frame::new(window.clone());
    application.ui(frame);

    window.show_all();

    gtk::main();
    application.exit();
}