//! Provides traits and tools for displaying windows.

pub mod demowindow;

#[cfg(windows)]
pub struct Frame {
    /// Window body
    handle: windows_sys::Win32::Foundation::HWND
}

#[cfg(windows)]
impl Frame {
    /// Create a new Frame
    pub fn new(window: windows_sys::Win32::Foundation::HWND) -> Self {
        Self {
            handle: window
        }
    }

    /// Display the window
    pub fn show(&self) {
        use windows_sys::Win32::UI::WindowsAndMessaging::ShowWindow;
        use windows_sys::Win32::UI::WindowsAndMessaging::SW_SHOWNORMAL;
        unsafe { ShowWindow(self.handle,SW_SHOWNORMAL); }
    }

    /// Maximize the window
    pub fn show_maximized(&self) {
        use windows_sys::Win32::UI::WindowsAndMessaging::ShowWindow;
        use windows_sys::Win32::UI::WindowsAndMessaging::SW_SHOWMAXIMIZED;
        unsafe { ShowWindow(self.handle,SW_SHOWMAXIMIZED); }
    }

    /// Minimize the window
    pub fn show_minimized(&self) {
        use windows_sys::Win32::UI::WindowsAndMessaging::ShowWindow;
        use windows_sys::Win32::UI::WindowsAndMessaging::SW_SHOWMINIMIZED;
        unsafe { ShowWindow(self.handle,SW_SHOWMINIMIZED); }
    }

    /// Set window title
    pub fn set_title(&self,title: String) {
        use windows_sys::Win32::UI::WindowsAndMessaging::SetWindowTextA;
        unsafe {
            SetWindowTextA(self.handle,title.as_ptr() as _);
        }
    }

    /// Set window position
    pub fn set_position(&self,x: i32,y: i32) {
        use windows_sys::Win32::UI::WindowsAndMessaging::SetWindowPos;
        unsafe {
            SetWindowPos(self.handle, 0, x, y, x, y, 0);
        }
    }
}

#[cfg(windows)]
pub struct WindowAttributes {
    pub style: windows_sys::Win32::UI::WindowsAndMessaging::WINDOW_STYLE,
    pub width: i32,
    pub height: i32,
    pub x: i32,
    pub y: i32
}

#[cfg(windows)]
impl WindowAttributes {
    pub fn new(style: windows_sys::Win32::UI::WindowsAndMessaging::WINDOW_STYLE,width: i32,height: i32,x: i32,y: i32) -> Self {
        Self {
            style,
            width,
            height,
            x,
            y
        }
    }
}

#[cfg(windows)]
pub trait Application {
    /// Returns the name of the window.
    fn title(&self) -> String;
    /// Returns WindowAttributes.
    fn attributes(&self) -> WindowAttributes;
    /// Processing at application startup (optional)
    fn startup(&self) {

    }
    /// Writing UI processes and events
    fn ui(&self,frame: Frame);
    /// Processing at application termination (optional)
    fn exit(&self) {

    }
}



#[cfg(unix)]
pub struct Frame {
    /// Window body
    handle: gtk::Window
}

#[cfg(unix)]
impl Frame {
    /// Create a new Frame
    pub fn new(window: gtk::Window) -> Self {
        Self {
            handle: window
        }
    }

    /// Display the window
    pub fn show(&self) {
        use gtk::prelude::*;
        self.handle.show();
    }

    /// Maximize the window
    pub fn show_maximized(&self) {
        use gtk::prelude::*;
        self.handle.maximize();
    }

    /// Minimize the window
    pub fn show_minimized(&self) {
        use gtk::prelude::*;
        self.handle.unmaximize();
    }
}

#[cfg(unix)]
pub struct WindowAttributes {
    pub style: gtk::WindowType,
    pub width: i32,
    pub height: i32,
    pub x: i32,
    pub y: i32
}

#[cfg(unix)]
impl WindowAttributes {
    pub fn new(style: gtk::WindowType,width: i32,height: i32,x: i32,y: i32) -> Self {
        Self {
            style,
            width,
            height,
            x,
            y
        }
    }
}

#[cfg(unix)]
pub trait Application {
    /// Returns the name of the window.
    fn title(&self) -> String;
    /// Returns WindowAttributes.
    fn attributes(&self) -> WindowAttributes;
    /// Processing at application startup (optional)
    fn startup(&self) {

    }
    /// Writing UI processes and events
    fn ui(&self,frame: Frame);
    /// Processing at application termination (optional)
    fn exit(&self) {

    }
}