use crate::manifest::WindowManifest;
#[cfg(unix)]
use gtk::gdk::WindowTypeHint;
#[cfg(unix)]
use gtk::prelude::*;
#[cfg(unix)]
use gtk::*;
use windows_sys::{
    Win32::Foundation::*, Win32::Graphics::Gdi::ValidateRect,
    Win32::System::LibraryLoader::GetModuleHandleA, Win32::UI::WindowsAndMessaging::*,
};

#[cfg(windows)]
#[derive(Clone)]
pub struct WindowAttributes {
    pub style: WINDOW_STYLE,
    pub title: String,
    pub width: i32,
    pub height: i32,
    pub x: i32,
    pub y: i32,
}

#[cfg(unix)]
#[derive(Clone)]
pub struct WindowAttributes {
    pub style: WindowType,
    pub title: String,
    pub width: i32,
    pub height: i32,
    pub x: i32,
    pub y: i32,
}

#[cfg(windows)]
impl WindowAttributes {
    pub fn new(
        style: WINDOW_STYLE,
        title: String,
        width: i32,
        height: i32,
        x: i32,
        y: i32,
    ) -> Self {
        Self {
            style,
            title,
            width,
            height,
            x,
            y,
        }
    }

    pub fn get_style(self) -> WINDOW_STYLE {
        self.style
    }

    pub fn get_title(self) -> String {
        self.title
    }

    pub fn get_width(self) -> i32 {
        self.width
    }

    pub fn get_height(self) -> i32 {
        self.height
    }

    pub fn get_x(self) -> i32 {
        self.x
    }

    pub fn get_y(self) -> i32 {
        self.y
    }
}

#[cfg(unix)]
impl WindowAttributes {
    pub fn new(
        class: WindowClass,
        style: WindowType,
        title: String,
        width: i32,
        height: i32,
        x: i32,
        y: i32,
    ) -> Self {
        Self {
            class,
            style,
            title,
            width,
            height,
            x,
            y,
        }
    }

    pub fn get_class(self) -> WindowClass {
        self.class
    }

    pub fn get_style(self) -> WindowType {
        self.style
    }

    pub fn get_title(self) -> String {
        self.title
    }

    pub fn get_width(self) -> i32 {
        self.width
    }

    pub fn get_height(self) -> i32 {
        self.height
    }

    pub fn get_x(self) -> i32 {
        self.x
    }

    pub fn get_y(self) -> i32 {
        self.y
    }
}

#[cfg(windows)]
pub trait Window {
    fn attributes(&self) -> WindowAttributes;
    fn manifest(&self) -> WindowManifest;
    fn startup(&self) {
        println!("Startup Phase")
    }
    fn ui(&self) {
        println!("UI Phase")
    }
    fn run(&self, attributes: WindowAttributes, manifest: WindowManifest) {
        println!("Run Phase");
        let window: HWND;
        extern "system" fn wndproc(
            window: HWND,
            message: u32,
            wparam: WPARAM,
            lparam: LPARAM,
        ) -> LRESULT {
            unsafe {
                match message as u32 {
                    WM_PAINT => {
                        println!("WM_PAINT");
                        ValidateRect(window, std::ptr::null());
                        0
                    }
                    WM_DESTROY => {
                        println!("WM_DESTROY");
                        PostQuitMessage(0);
                        0
                    }
                    _ => DefWindowProcA(window, message, wparam, lparam),
                }
            }
        }
        unsafe {
            let instance = GetModuleHandleA(std::ptr::null_mut());
            debug_assert!(instance != 0);
            let wc = WNDCLASSA {
                hCursor: LoadCursorW(0, IDC_ARROW),
                hInstance: instance,
                lpszClassName: manifest.class,
                style: CS_HREDRAW | CS_VREDRAW,
                lpfnWndProc: Some(wndproc),
                cbClsExtra: 0,
                cbWndExtra: 0,
                hIcon: 0,
                hbrBackground: 0,
                lpszMenuName: std::ptr::null_mut(),
            };

            let atom = RegisterClassA(&wc);
            debug_assert!(atom != 0);

            let win = CreateWindowExA(
                0,
                manifest.class,
                attributes.title.as_ptr() as _,
                WS_VISIBLE | attributes.style,
                CW_USEDEFAULT,
                CW_USEDEFAULT,
                attributes.width,
                attributes.height,
                0,
                0,
                instance,
                std::ptr::null_mut(),
            );
            SendMessageA(
                win,
                WM_SETICON,
                WPARAM::default(),
                LPARAM::from(manifest.icon),
            );

            let mut message = std::mem::zeroed();

            while GetMessageA(&mut message, 0, 0, 0) != 0 {
                DispatchMessageA(&mut message);
            }
        }
    }
    fn exit(&self) {
        println!("Exit Phase")
    }
}

#[cfg(unix)]
pub trait Window {
    fn attributes(&self) -> WindowAttributes;
    fn startup(&self) {
        println!("Startup Phase")
    }
    fn ui(&self) {
        println!("UI Phase")
    }
    fn run(&self, attributes: WindowAttributes) {
        use gtk::prelude::*;
        let application = gtk::Application::new(
            Some(&*attributes.clone().class.get_application_id()),
            Default::default(),
        );
        let window = gtk::ApplicationWindow::new(&application);

        window.set_title(&*attributes.clone().title);
        window.set_border_width(10);
        window.set_position(gtk::WindowPosition::Center);
        window.set_default_size(attributes.clone().width, attributes.clone().height);

        window.show_all();

        application.run();
    }
    fn exit(&self) {
        println!("Exit Phase")
    }
}
