use windows_sys::{
    Win32::Foundation::*, Win32::Graphics::Gdi::ValidateRect,
    Win32::System::LibraryLoader::GetModuleHandleA, Win32::UI::WindowsAndMessaging::*,
};

#[cfg(windows)]
pub struct Window {
    pub class: super::windowclass::WindowClass,
    pub title: String,
    pub width: i32,
    pub height: i32,
    pub x: i32,
    pub y: i32,
}

#[cfg(linux)]
pub struct Window {
    pub class: PSTR,
    pub title: String,
    pub width: i32,
    pub height: i32,
    pub x: i32,
    pub y: i32,
}

#[cfg(windows)]
impl Window {
    ///
    /// Create a new window
    ///
    pub fn new(
        class: super::windowclass::WindowClass,
        title: String,
        width: i32,
        height: i32,
        x: i32,
        y: i32,
    ) -> Self {
        Self {
            class,
            title,
            width,
            height,
            x,
            y,
        }
    }

    ///
    /// Returns the title
    ///
    pub fn get_title(self) -> String {
        self.title
    }

    ///
    /// Returns the width value.
    ///
    pub fn get_width(self) -> i32 {
        self.width
    }

    ///
    /// Returns the height value.
    ///
    pub fn get_height(self) -> i32 {
        self.height
    }

    ///
    /// Returns the x value.
    ///
    pub fn get_x_pos(self) -> i32 {
        self.x
    }

    ///
    /// Returns the y value.
    ///
    pub fn get_y_pos(self) -> i32 {
        self.y
    }

    ///
    /// Display the window
    ///
    /// # Examples
    /// ```
    /// use nxui::{messagebox, window};
    /// use nxui::NativesAndMessaging::*;
    ///
    /// fn main() {
    ///     let win = nxui::window::Window { class: DEFAULT_WINDOW_WINDOW_CLASS, title: "Hello World".to_string(), width: DEFAULT_WINDOW_WIDTH, height: DEFAULT_WINDOW_HEIGHT, x: DEFAULT_WINDOW_X, y: DEFAULT_WINDOW_Y };
    ///     win.show(WINDOWSTYLE_OVERLAPPED);
    /// }
    /// ```
    pub fn show(&self, window_style: WINDOW_STYLE) {
        extern "system" fn wndproc(
            window: HWND,
            message: u32,
            wparam: WPARAM,
            lparam: LPARAM,
        ) -> LRESULT {
            unsafe {
                match message as u32 {
                    WM_PAINT => {
                        println!("WM_PAINT");
                        ValidateRect(window, std::ptr::null());
                        0
                    }
                    WM_DESTROY => {
                        println!("WM_DESTROY");
                        PostQuitMessage(0);
                        0
                    }
                    _ => DefWindowProcA(window, message, wparam, lparam),
                }
            }
        }
        unsafe {
            let instance = GetModuleHandleA(std::ptr::null_mut());
            debug_assert!(instance != 0);
            let class = self.class.clone().get_class();
            let wc = WNDCLASSA {
                hCursor: LoadCursorW(0, IDC_ARROW),
                hInstance: instance,
                lpszClassName: class,
                style: CS_HREDRAW | CS_VREDRAW,
                lpfnWndProc: Some(wndproc),
                cbClsExtra: 0,
                cbWndExtra: 0,
                hIcon: 0,
                hbrBackground: 0,
                lpszMenuName: std::ptr::null_mut(),
            };

            let atom = RegisterClassA(&wc);
            debug_assert!(atom != 0);

            CreateWindowExA(
                0,
                class,
                self.title.as_ptr() as _,
                WS_VISIBLE | window_style,
                CW_USEDEFAULT,
                CW_USEDEFAULT,
                self.width,
                self.height,
                0,
                0,
                instance,
                std::ptr::null_mut(),
            );

            let mut message = std::mem::zeroed();

            while GetMessageA(&mut message, 0, 0, 0) != 0 {
                DispatchMessageA(&mut message);
            }
        }
    }
}

#[cfg(linux)]
impl Window {
    ///
    /// Create a new window
    ///
    pub fn new(
        class: super::windowclass::WindowClass,
        title: String,
        width: i32,
        height: i32,
        x: i32,
        y: i32,
    ) -> Self {
        Self {
            class,
            title,
            width,
            height,
            x,
            y,
        }
    }

    ///
    /// Returns the title
    ///
    pub fn get_title(self) -> String {
        self.title
    }

    ///
    /// Returns the width value.
    ///
    pub fn get_width(self) -> i32 {
        self.width
    }

    ///
    /// Returns the height value.
    ///
    pub fn get_height(self) -> i32 {
        self.height
    }

    ///
    /// Returns the x value.
    ///
    pub fn get_x_pos(self) -> i32 {
        self.x
    }

    ///
    /// Returns the y value.
    ///
    pub fn get_y_pos(self) -> i32 {
        self.y
    }

    ///
    /// Display the window
    ///
    /// # Examples
    /// ```
    /// use nxui::{messagebox, window};
    /// use nxui::NativesAndMessaging::*;
    ///
    /// fn main() {
    ///     let win = nxui::window::Window { class: DEFAULT_WINDOW_WINDOW_CLASS, title: "Hello World".to_string(), width: DEFAULT_WINDOW_WIDTH, height: DEFAULT_WINDOW_HEIGHT, x: DEFAULT_WINDOW_X, y: DEFAULT_WINDOW_Y };
    ///     win.show(WINDOWSTYLE_OVERLAPPED);
    /// }
    /// ```
    pub fn show(&self, window_style: WINDOW_STYLE) {
        use gtk::prelude::*;

        let window = gtk::ApplicationWindow::new(application);

        window.set_title(&*self.title);
        window.set_border_width(10);
        window.set_position(gtk::WindowPosition::Center);
        window.set_default_size(self.width, self.height);

        window.show_all();

        let application = gtk::Application::new(
            Some(self.class.clone().get_application_id()),
            Default::default(),
        );

        application.run();
    }
}
