use crate::messagebox::MessageBox;
use crate::natives_and_messaging::*;
use crate::window::{Attributes, Frame};

#[cfg(windows)]
pub struct Label {
    pub handle: windows_sys::Win32::Foundation::HWND
}

#[cfg(windows)]
impl Label {
    pub fn new(attributes: Attributes,frame: &Frame) -> Self {
        use windows_sys::Win32::UI::WindowsAndMessaging::*;
        unsafe {
            let handle = CreateWindowExA(0, "Static\0".as_ptr(), format!("{}\0",attributes.title).as_ptr(), WS_CHILD, attributes.x, attributes.y, attributes.width, attributes.height, frame.get_window(), 0, 0, std::ptr::null_mut());
            if handle == 0 {
                match MessageBox::new("NXUI Warning", "NXUI Warning - Code 0002\rWidget could not be initialized\rRunning it as is may disrupt the UI\rPress Yes to continue program execution\r \rDetailed information:\rPanicked at nxui::widget::label\rTo resolve this error, contact the application developer or\rPlease report at https://github.com/LatteSyobon/nxui/issues", DS_WARNING, BS_YESNO).show(Frame::default())
                {
                    DR_NO => {
                        panic!("Exit request from user");
                    }
                    _ => {}
                };
            }
            info!("Label has been created. handle: {}",handle);
            Self {
                handle
            }
        }
    }

    pub fn show(&self) {
        use windows_sys::Win32::UI::WindowsAndMessaging::ShowWindow;
        use windows_sys::Win32::UI::WindowsAndMessaging::SW_SHOWNORMAL;
        unsafe {
            ShowWindow(self.handle,SW_SHOWNORMAL);
            info!("handle: {}",self.handle);
        }
    }
}

#[cfg(unix)]
pub struct Label {
    pub handle: gtk::Label
}

#[cfg(unix)]
impl Label {
    pub fn new(attributes: Attributes,frame: &Frame) -> Self {
        use gtk::prelude::*;
        let handle = gtk::Label::new(Some(&*attributes.title));
        handle.set_height_request(attributes.height/10);
        handle.set_width_request(attributes.width/10);
        handle.set_margin_top(attributes.y);
        handle.set_margin_start(attributes.x);
        frame.get_window().add(&handle);
        Self {
            handle
        }
    }

    pub fn show(&self) {
        use gtk::prelude::*;
        self.handle.show();
    }
}