use std::ops::Deref;
use std::sync::{Arc, mpsc, Mutex};
use std::thread;
use std::thread::park_timeout;
use std::time::Duration;
use winapi::shared::minwindef::WPARAM;
use windows_sys::core::PCSTR;
use windows_sys::Win32::Foundation::*;
use windows_sys::Win32::System::LibraryLoader::GetModuleHandleA;
use windows_sys::Win32::UI::WindowsAndMessaging::*;
use crate::natives_and_messaging::AdvancedWindowStyle;
use crate::widget::NWidget;
use super::*;

#[derive(Clone)]
pub struct Window {
    handle: windows_sys::Win32::Foundation::HWND,
    message: WPARAM,

    attributes: Attributes,
    pub wh: Arc<Mutex<WindowHandler>>,
    pub is_handle_created: bool
}

impl Window {
    pub fn new(attributes: Attributes) -> Self {
        Self {
            handle: 0,
            message: 0,

            attributes,
            wh: Arc::new(Mutex::new(WindowHandler::new(0))),
            is_handle_created: false
        }
    }

    pub fn run(self, action: fn(wh: WindowHandler)) {
        let (wtx, wrx) = mpsc::channel();
        let (atx, arx) = mpsc::channel();
        wtx.send(self.clone()).unwrap();
        atx.send(self.clone().wh);
        let build_h = thread::spawn(move || {
            wrx.recv().unwrap().build();
        });
        park_timeout(Duration::new(1,0));
        println!("{}",self.wh.lock().unwrap().handle);
        let action_h = thread::spawn(move || {
            action(*arx.recv().unwrap().lock().unwrap())
        });

        dbg!(build_h.join());
        dbg!(action_h.join());

        println!("Hyper Threading!!");
    }

    fn build(&mut self) {
        unsafe {
            let instance = GetModuleHandleA(std::ptr::null());
            debug_assert!(instance != 0);

            let window_class = b"window\0".as_ptr();

            let wc = WNDCLASSA {
                hCursor: LoadCursorW(0, IDC_ARROW),
                hInstance: instance,
                lpszClassName: window_class,
                style: CS_HREDRAW | CS_VREDRAW,
                lpfnWndProc: Some(Self::wndproc),
                cbClsExtra: 0,
                cbWndExtra: 0,
                hIcon: 0,
                hbrBackground: 5,
                lpszMenuName: std::ptr::null(),
            };

            let atom = RegisterClassA(&wc);
            debug_assert!(atom != 0);

            let handle = CreateWindowExA(0, window_class, format!("{}\0",self.attributes.title).as_ptr(), self.attributes.style, self.attributes.x, self.attributes.y, self.attributes.width, self.attributes.height, 0, 0, instance, self as *mut _ as _);

            let b = CreateWindowExA(
                0,
                "BUTTON\0".as_ptr(),
                format!("{}\0", self.attributes.title).as_ptr(),
                0,
                self.attributes.x,
                self.attributes.y,
                self.attributes.width,
                self.attributes.height,
                0,
                0,
                0,
                std::ptr::null_mut(),
            );
            SetWindowLongA(b,GWL_STYLE,(WS_VISIBLE|WS_CHILD) as i32);
            SetParent(b,handle);
            self.wh.lock().unwrap().handle = handle;
            let mut message = std::mem::zeroed();
            while GetMessageA(&mut message, 0, 0, 0) != 0 {
                DispatchMessageA(&message);
            }
        }
        // let handle = CreateWindowExA(0, b"window\0".as_ptr(), format!("{}\0",self.attributes.title).as_ptr(), WS_OVERLAPPEDWINDOW, self.attributes.x, self.attributes.y, self.attributes.width, self.attributes.height, 0, 0, instance,  self as *mut _ as _);
    }

    extern "system" fn wndproc(window: HWND, message: u32, wparam: WPARAM, lparam: LPARAM) -> LRESULT {
        unsafe {
            match message {
                WM_DESTROY => {
                    PostQuitMessage(0);
                    return 0
                }
                _ => {}
            }

            if message == WM_NCCREATE {
                let cs = lparam as *const CREATESTRUCTA;
                println!("CREATESTRUCT A {:?}",cs);
                let this = (*cs).lpCreateParams as *mut Self;
                println!("This : {:?}",this);
                (*this).handle = window;
                println!("Finish");

                SetWindowLongW(window, GWLP_USERDATA, this as _);
                println!("Set Window Long");
            } else {
                let this = GetWindowLongW(window, GWLP_USERDATA) as *mut Self;
            }

            DefWindowProcA(window, message, wparam, lparam)
        }
    }

}

#[derive(Clone,Copy)]
pub struct WindowHandler {
    handle: windows_sys::Win32::Foundation::HWND
}

impl WindowHandler {
    pub fn new(handle: windows_sys::Win32::Foundation::HWND) -> Self {
        Self {
            handle
        }
    }

    pub fn show(&self) {
        use windows_sys::Win32::UI::WindowsAndMessaging::ShowWindow;
        use windows_sys::Win32::UI::WindowsAndMessaging::SW_SHOWNORMAL;
        unsafe {
            println!("Window Handler handle: {}",self.handle);
            ShowWindow(self.handle, SW_SHOWNORMAL);
            info!("A window has appeared. handle: {}", self.handle);
        }
    }

    pub fn hide(&self) {
        use windows_sys::Win32::UI::WindowsAndMessaging::ShowWindow;
        use windows_sys::Win32::UI::WindowsAndMessaging::SW_HIDE;
        unsafe {
            ShowWindow(self.handle, SW_HIDE);
            info!("The window has been hidden. handle: {}", self.handle);
        }
    }

    pub fn add(&self,widget: impl NWidget + Copy) {
        use windows_sys::Win32::UI::WindowsAndMessaging::SetParent;
        unsafe {
            SetWindowLongA(widget.get_handle(),GWL_STYLE,(WS_VISIBLE|WS_CHILD) as i32);
            SetParent(widget.get_handle(),self.handle);
        }
    }

    pub fn set_advanced_style(&self,style: AdvancedWindowStyle) {
        unsafe {
            SetWindowLongA(self.handle, GWL_EXSTYLE, style as i32);
        }
    }

    pub fn set_style(&self,style: WindowStyle) {
        unsafe {
            SetWindowLongA(self.handle,GWL_STYLE,style as i32);
        }
    }

    /// Returns the window handle.
    pub fn get_window(&self) -> windows_sys::Win32::Foundation::HWND {
        self.handle
    }

    /// Set window title
    pub fn set_title(&self, title: String) {
        use windows_sys::Win32::UI::WindowsAndMessaging::SetWindowTextA;
        unsafe {
            SetWindowTextA(self.handle, format!("{}\0",title).as_ptr() as _);
            info!(
                "The title of the window has been changed to '{}'. handle: {}",
                title, self.handle
            );
        }
    }

    /// Set window position
    pub fn set_position(&self, x: i32, y: i32) {
        use windows_sys::Win32::UI::WindowsAndMessaging::SetWindowPos;
        unsafe {
            SetWindowPos(self.handle, 0, x, y, x, y, 0);
        }
    }

    /// Set window size
    /// **Warning: This method is unstable.**
    pub fn set_size(&self, width: i32, height: i32) {
        use windows_sys::Win32::Foundation::RECT;
        use windows_sys::Win32::UI::WindowsAndMessaging::{SetWindowPos,GetWindowRect,ShowWindow,SW_SHOWDEFAULT};
        unsafe {
            let rect = &mut RECT {
                left: 0,
                top: 0,
                right: 0,
                bottom: 0,
            };
            GetWindowRect(self.handle, rect);

            ShowWindow(self.handle, SW_SHOWDEFAULT);
            SetWindowPos(self.handle, 0, rect.left, rect.top, width, height, 0);
        }
    }

    pub fn set_menu(self, menu: MenuBar) {
        use windows_sys::Win32::UI::WindowsAndMessaging::SetMenu;
        unsafe {
            SetMenu(self.handle, menu.get_menu());
        }
    }

    pub fn set_icon(&self, icon: Icon) {
        use windows_sys::Win32::Foundation::WPARAM;
        use windows_sys::Win32::UI::WindowsAndMessaging::*;
        unsafe {
            SendMessageA(self.handle, WM_SETICON, ICON_BIG as WPARAM, icon.get_icon());
            SendMessageA(
                self.handle,
                WM_SETICON,
                ICON_SMALL as WPARAM,
                icon.get_icon(),
            );
        }
    }

    pub fn set_blur(&self) {
        use windows_sys::Win32::Graphics::Dwm::*;
        use windows_sys::Win32::Foundation::BOOL;
        unsafe {
            let bb = &mut DWM_BLURBEHIND {
                dwFlags: DWM_BB_ENABLE,
                fEnable: BOOL::from(true),
                hRgnBlur: 0,
                fTransitionOnMaximized: 0
            };
            DwmEnableBlurBehindWindow(self.handle,bb);
        }
    }
}