use std::ptr::null_mut;

pub struct DirectoryChooser<'a> {
    pub title: &'a str
}

impl<'a> DirectoryChooser<'a> {
    pub fn new(title: &'a str) -> Self {
        Self {
            title
        }
    }

    pub fn show(&self) -> String {
        unsafe {
            use winapi::shared::winerror::SUCCEEDED;
            use winapi::um::combaseapi::{CoCreateInstance, CoInitializeEx, CoUninitialize, CLSCTX_ALL};
            use winapi::um::objbase::{COINIT_APARTMENTTHREADED, COINIT_DISABLE_OLE1DDE};
            use winapi::um::shobjidl::*;
            use winapi::um::shobjidl_core::{IShellItem, SIGDN_FILESYSPATH};
            use winapi::Interface;
            use winapi::um::winnt::PWSTR;
            use crate::util::pwstr_to_string;

            #[allow(non_upper_case_globals)]
            const CLSID_FileOpenDialog: winapi::shared::guiddef::GUID = winapi::shared::guiddef::GUID {
                Data1: 0xdc1c5a9c,
                Data2: 0xe88a,
                Data3: 0x4dde,
                Data4: [0xa5, 0xa1, 0x60, 0xf8, 0x2a, 0x20, 0xae, 0xf7],
            };
            let mut sel_dir: String = "".to_string();

            let mut hr = CoInitializeEx(
                null_mut(),
                COINIT_APARTMENTTHREADED | COINIT_DISABLE_OLE1DDE,
            );
            if SUCCEEDED(hr) {
                let mut pfd: *mut IFileDialog = std::mem::uninitialized();
                hr = CoCreateInstance(
                    &CLSID_FileOpenDialog,
                    null_mut(),
                    CLSCTX_ALL,
                    &IFileOpenDialog::uuidof(),
                    &mut pfd as *mut *mut IFileDialog as *mut _,
                );
                if SUCCEEDED(hr) {
                    let mut fop: FILEOPENDIALOGOPTIONS = std::mem::zeroed();
                    if SUCCEEDED((*pfd).GetOptions(&mut fop)) {
                        (*pfd).SetOptions(
                            fop | FOS_PICKFOLDERS
                                | FOS_FORCESHOWHIDDEN
                                | FOS_PATHMUSTEXIST
                                | FOS_FORCEFILESYSTEM,
                        );
                    }
                    if SUCCEEDED((*pfd).Show(null_mut())) {
                        let mut psi: *mut IShellItem = std::mem::zeroed();
                        if SUCCEEDED((*pfd).GetResult(&mut psi)) {
                            // Provide a pointer to a buffer so windows can swap it for its own buffer
                            let mut buffer: PWSTR = std::ptr::null_mut();
                            if SUCCEEDED((*psi).GetDisplayName(SIGDN_FILESYSPATH, &mut buffer)) {
                                sel_dir = pwstr_to_string(buffer);
                            }
                            winapi::um::combaseapi::CoTaskMemFree(std::mem::transmute(buffer));
                        }
                        (*psi).Release();
                    }
                    (*pfd).Release();
                }
                CoUninitialize();
            }
            sel_dir
        }
    }
}