//! Displays an OS-specific message dialog.
//! Display OS-specific dialogs in Windows environment, and GTK dialogs in Linux environment.

use crate::window::Frame;

#[cfg(windows)]
pub struct MessageBox {
    /// Message to be displayed
    pub title: String,
    /// Message to be displayed
    pub message: String,
    /// Dialog style
    pub style: windows_sys::Win32::UI::WindowsAndMessaging::MESSAGEBOX_STYLE,
    /// Dialog button style
    pub button_style: windows_sys::Win32::UI::WindowsAndMessaging::MESSAGEBOX_STYLE,
}

#[cfg(unix)]
pub struct MessageBox {
    /// Message to be displayed
    pub title: String,
    /// Message to be displayed
    pub message: String,
    /// Dialog style
    pub style: gtk::MessageType,
    /// Dialog button style
    pub button_style: gtk::ButtonsType,
}

#[cfg(windows)]
impl MessageBox {
    /// Create a new MessageBox.
    pub fn new(
        title: String,
        message: String,
        style: windows_sys::Win32::UI::WindowsAndMessaging::MESSAGEBOX_STYLE,
        button_style: windows_sys::Win32::UI::WindowsAndMessaging::MESSAGEBOX_STYLE,
    ) -> Self {
        Self {
            title,
            message,
            style,
            button_style,
        }
    }

    /// Display the dialog
    /// In Windows environment, returns windows-sys::Win32::UI::WindowsAndMessaging::MESSAGEBOX_RESULT
    /// # Examples
    /// ```
    /// use nxui::messagebox::MessageBox;
    /// use nxui::natives_and_messaging::{DIALOGSTYLE_INFORMATION,BUTTONSTYLE_OK};
    ///
    /// fn main() {
    ///     MessageBox::new("Hello NXUI!".to_string(),"This is a test".to_string(),DIALOGSTYLE_INFORMATION,BUTTONSTYLE_OK).show();
    /// }
    /// ```
    pub fn show(&self,frame: Frame) -> windows_sys::Win32::UI::WindowsAndMessaging::MESSAGEBOX_RESULT {
        unsafe {
            use windows_sys::{
                Win32::Foundation::*, Win32::System::Threading::*,
                Win32::UI::WindowsAndMessaging::*,
            };
            let event = CreateEventW(std::ptr::null(), 1, 0, std::ptr::null());
            SetEvent(event);
            WaitForSingleObject(event, 0);
            CloseHandle(event);
            info!("HANDLE: {}",frame.get_window());

            let result = MessageBoxW(
                frame.get_window(),
                self.message.encode_utf16().collect::<Vec<u16>>().as_mut_ptr(),
                self.title.encode_utf16().collect::<Vec<u16>>().as_mut_ptr(),
                self.style | self.button_style,
            );
            info!("The Result of MessageBox is {}",result);
            result
        }
    }
}

#[cfg(unix)]
impl MessageBox {
    /// Create a new MessageBox.
    pub fn new(
        title: String,
        message: String,
        style: gtk::MessageType,
        button_style: gtk::ButtonsType,
    ) -> Self {
        Self {
            title,
            message,
            style,
            button_style,
        }
    }

    /// Display the dialog
    /// In Windows environment, returns windows-sys::Win32::UI::WindowsAndMessaging::MESSAGEBOX_RESULT
    /// # Examples
    /// ```
    /// use nxui::messagebox::MessageBox;
    /// use nxui::natives_and_messaging::{DIALOGSTYLE_INFORMATION,BUTTONSTYLE_OK};
    ///
    /// fn main() {
    ///     MessageBox::new("Hello NXUI!".to_string(),"This is a test".to_string(),DIALOGSTYLE_INFORMATION,BUTTONSTYLE_OK).show();
    /// }
    /// ```
    pub fn show(&self,frame: Frame) -> gtk::ResponseType {
        use gtk::prelude::*;
        gtk::init().expect("Failed to initialize gtk.");
        let window = gtk::Window::new(gtk::WindowType::Toplevel);

        window.set_title(self.title.as_str());

        let dialog = gtk::MessageDialog::new(
            Some(&window),
            gtk::DialogFlags::empty(),
            self.style,
            self.button_style,
            self.message.as_str(),
        );
        let result = dialog.run();
        unsafe {
            dialog.destroy();
        }
        result
    }
}
