use crate::io::get_app_dirs;
use crate::io::storage::Storage;
use crate::window::{Application, Frame};

#[cfg(windows)]
/// Run nxui::window::Application
pub fn create_new_app(application: Box<dyn Application>) {
    // Get window title
    let app_name = application.app_name();
    // Get attributes
    let attributes = application.attributes();
    let storage = Storage::new(get_app_dirs(app_name));
    // startup process
    if application.is_child_window() == false {
        storage.create_app_dir();
    }
    application.startup(storage);

    // Create a window
    use windows_sys::{Win32::Foundation::*, Win32::UI::WindowsAndMessaging::*,Win32::System::LibraryLoader::GetModuleHandleA};
    extern "system" fn wndproc(window: HWND, message: u32, wparam: WPARAM, lparam: LPARAM) -> LRESULT {
        unsafe {
            match message as u32 {
                WM_DESTROY => {
                    println!("WM_DESTROY");
                    PostQuitMessage(0);
                    0
                }
                _ => DefWindowProcA(window, message, wparam, lparam),
            }
        }
    }

    unsafe {
        let instance = GetModuleHandleA(std::ptr::null_mut());
        debug_assert!(instance != 0);

        let wc = WNDCLASSA {
            hCursor: LoadCursorW(0, IDC_ARROW),
            hInstance: instance,
            lpszClassName: "window".as_ptr(),
            style: CS_HREDRAW | CS_VREDRAW,
            lpfnWndProc: Some(wndproc),
            cbClsExtra: 0,
            cbWndExtra: 0,
            hIcon: 0,
            hbrBackground: 5,
            lpszMenuName: std::ptr::null_mut(),
        };

        let atom = RegisterClassA(&wc);
        debug_assert!(atom != 0);

        let win = CreateWindowExA(application.advanced_options().get_style(), "window".as_ptr(), attributes.title.as_ptr(), attributes.style|WS_VISIBLE, attributes.x, attributes.y, attributes.width, attributes.height, 0, 0, instance, std::ptr::null_mut());

        let frame = Frame::new(win);
        application.ui(frame);

        let mut message = std::mem::zeroed();

        while GetMessageA(&mut message, 0, 0, 0) != 0 {
            DispatchMessageA(&message);
        }
        application.exit();
    }
}

#[cfg(unix)]
/// Run nxui::window::Application
pub fn create_new_app(application: Box<dyn Application>) {
    // Get window title
    let app_name = application.app_name();
    // Get attributes
    let attributes = application.attributes();
    // Get advanced_options
    let advanced_options = application.advanced_options();
    // let storage = Storage::new(get_app_dirs(app_name));
    // startup process
    // application.startup(storage);

    // Create a window
    use gtk::prelude::*;

    let window = gtk::Window::new(gtk::WindowType::Toplevel);

    window.set_title(attributes.title.as_str());
    window.set_default_size(attributes.width,attributes.height);
    window.set_window_position(gtk::WindowPosition::Center);
    window.set_type_hint(advanced_options.get_style());

    window.connect_delete_event(|_, _| {
        gtk::main_quit();
        Inhibit(false)
    });
    let frame = Frame::new(window.clone());
    application.ui(frame);

    window.show_all();

    gtk::main();
    application.exit();
}