//! Displays an OS-specific message dialog.
//! Display OS-specific dialogs in Windows environment, and GTK dialogs in Linux environment.

use crate::window::Frame;

#[cfg(windows)]
pub struct MessageBox {
    /// Message to be displayed
    pub title: String,
    /// Message to be displayed
    pub message: String,
    /// Dialog style
    pub style: windows_sys::Win32::UI::WindowsAndMessaging::MESSAGEBOX_STYLE,
    /// Dialog button style
    pub button_style: windows_sys::Win32::UI::WindowsAndMessaging::MESSAGEBOX_STYLE,
}

#[cfg(unix)]
pub struct MessageBox {
    /// Message to be displayed
    pub title: String,
    /// Message to be displayed
    pub message: String,
    /// Dialog style
    pub style: gtk::MessageType,
    /// Dialog button style
    pub button_style: gtk::ButtonsType,
}

#[cfg(windows)]
impl MessageBox {
    /// Create a new MessageBox.
    pub fn new(
        title: String,
        message: String,
        style: windows_sys::Win32::UI::WindowsAndMessaging::MESSAGEBOX_STYLE,
        button_style: windows_sys::Win32::UI::WindowsAndMessaging::MESSAGEBOX_STYLE,
    ) -> Self {
        Self {
            title,
            message,
            style,
            button_style,
        }
    }

    /// Display the dialog
    /// In Windows environment, returns windows-sys::Win32::UI::WindowsAndMessaging::MESSAGEBOX_RESULT
    /// # Examples
    /// ```
    /// use nxui::messagebox::MessageBox;
    /// use nxui::natives_and_messaging::{DIALOGSTYLE_INFORMATION,BUTTONSTYLE_OK};
    ///
    /// fn main() {
    ///     MessageBox::new("Hello NXUI!".to_string(),"This is a test".to_string(),DIALOGSTYLE_INFORMATION,BUTTONSTYLE_OK).show();
    /// }
    /// ```
    pub fn show(&self,frame: Frame) -> windows_sys::Win32::UI::WindowsAndMessaging::MESSAGEBOX_RESULT {
        unsafe {
            use windows_sys::{
                Win32::Foundation::*, Win32::System::Threading::*,
                Win32::UI::WindowsAndMessaging::*,
            };
            enable_visual_style();
            let event = CreateEventW(std::ptr::null(), 1, 0, std::ptr::null());
            SetEvent(event);
            WaitForSingleObject(event, 0);
            CloseHandle(event);

            let result = MessageBoxW(
                frame.get_window(),
                self.message.encode_utf16().collect::<Vec<u16>>().as_mut_ptr(),
                self.title.encode_utf16().collect::<Vec<u16>>().as_mut_ptr(),
                self.style | self.button_style,
            );
            result
        }
    }
}

#[cfg(unix)]
impl MessageBox {
    /// Create a new MessageBox.
    pub fn new(
        title: String,
        message: String,
        style: gtk::MessageType,
        button_style: gtk::ButtonsType,
    ) -> Self {
        Self {
            title,
            message,
            style,
            button_style,
        }
    }

    /// Display the dialog
    /// In Windows environment, returns windows-sys::Win32::UI::WindowsAndMessaging::MESSAGEBOX_RESULT
    /// # Examples
    /// ```
    /// use nxui::messagebox::MessageBox;
    /// use nxui::natives_and_messaging::{DIALOGSTYLE_INFORMATION,BUTTONSTYLE_OK};
    ///
    /// fn main() {
    ///     MessageBox::new("Hello NXUI!".to_string(),"This is a test".to_string(),DIALOGSTYLE_INFORMATION,BUTTONSTYLE_OK).show();
    /// }
    /// ```
    pub fn show(&self,frame: Frame) -> gtk::ResponseType {
        use gtk::prelude::*;
        gtk::init().expect("Failed to initialize gtk.");
        let window = gtk::Window::new(gtk::WindowType::Toplevel);

        window.set_title(self.title.as_str());

        let dialog = gtk::MessageDialog::new(
            Some(&window),
            gtk::DialogFlags::empty(),
            self.style,
            self.button_style,
            self.message.as_str(),
        );
        let result = dialog.run();
        unsafe {
            dialog.destroy();
        }
        result
    }
}

/// Enabling Visual Style
#[cfg(windows)]
fn enable_visual_style() {
    use winapi::shared::minwindef::{ULONG, DWORD};
    use winapi::shared::basetsd::ULONG_PTR;
    use winapi::um::winbase::{ACTCTXW, CreateActCtxW, ActivateActCtx};
    use winapi::um::sysinfoapi::GetSystemDirectoryW;
    use std::{mem, ptr};
    use windows_sys::Win32::Foundation::MAX_PATH;

    const ACTCTX_FLAG_RESOURCE_NAME_VALID: DWORD = 0x008;
    const ACTCTX_FLAG_SET_PROCESS_DEFAULT: DWORD = 0x010;
    const ACTCTX_FLAG_ASSEMBLY_DIRECTORY_VALID: DWORD = 0x004;

    let mut sys_dir: Vec<u16> = Vec::with_capacity(MAX_PATH as usize);
    unsafe {
        sys_dir.set_len(MAX_PATH as usize);
        GetSystemDirectoryW(sys_dir.as_mut_ptr(), MAX_PATH as u32);
    }

    let mut source = "shell32.dll".encode_utf16().collect::<Vec<u16>>();

    let mut activation_cookie: ULONG_PTR = 0;
    let mut act_ctx = ACTCTXW {
        cbSize: mem::size_of::<ACTCTXW> as ULONG,
        dwFlags: ACTCTX_FLAG_RESOURCE_NAME_VALID | ACTCTX_FLAG_SET_PROCESS_DEFAULT | ACTCTX_FLAG_ASSEMBLY_DIRECTORY_VALID,
        lpSource: source.as_mut_ptr(),
        wProcessorArchitecture: 0,
        wLangId: 0,
        lpAssemblyDirectory: sys_dir.as_mut_ptr(),
        lpResourceName: unsafe { mem::transmute(124usize) },
        lpApplicationName: ptr::null_mut(),
        hModule: ptr::null_mut()
    };

    unsafe {
        let handle = CreateActCtxW(&mut act_ctx);
        ActivateActCtx(handle, &mut activation_cookie);
    }
}
