//! Provides traits and tools for displaying windows.

use crate::io::storage::Storage;
use crate::widget::NWidget;

pub mod demowindow;

#[cfg(windows)]
#[derive(Clone)]
pub struct Frame {
    /// Window body
    handle: windows_sys::Win32::Foundation::HWND
}

#[cfg(windows)]
impl Frame {
    /// Create a new Frame
    pub fn new(window: windows_sys::Win32::Foundation::HWND) -> Self {
        Self {
            handle: window
        }
    }

    /// Add a widget to the window
    pub fn add(&self,widget: &dyn NWidget,frame: Frame) {

    }

    /// Display the window
    pub fn show(&self) {
        use windows_sys::Win32::UI::WindowsAndMessaging::ShowWindow;
        use windows_sys::Win32::UI::WindowsAndMessaging::SW_SHOWNORMAL;
        unsafe { ShowWindow(self.handle,SW_SHOWNORMAL); }
    }

    /// Maximize the window
    pub fn show_maximized(&self) {
        use windows_sys::Win32::UI::WindowsAndMessaging::ShowWindow;
        use windows_sys::Win32::UI::WindowsAndMessaging::SW_MAXIMIZE;
        unsafe { ShowWindow(self.handle,SW_MAXIMIZE); }
    }

    /// Minimize the window
    pub fn show_minimized(&self) {
        use windows_sys::Win32::UI::WindowsAndMessaging::ShowWindow;
        use windows_sys::Win32::UI::WindowsAndMessaging::SW_MINIMIZE;
        unsafe { ShowWindow(self.handle,SW_MINIMIZE); }
    }

    /// Hide the window
    pub fn hide(&self) {
        use windows_sys::Win32::UI::WindowsAndMessaging::ShowWindow;
        use windows_sys::Win32::UI::WindowsAndMessaging::SW_HIDE;
        unsafe { ShowWindow(self.handle,SW_HIDE); }
    }

    /// Returns the window handle.
    pub fn get_window(&self) -> windows_sys::Win32::Foundation::HWND {
        self.handle
    }

    /// Set window title
    pub fn set_title(&self,title: String) {
        use windows_sys::Win32::UI::WindowsAndMessaging::SetWindowTextA;
        unsafe {
            SetWindowTextA(self.handle,title.as_ptr() as _);
        }
    }

    /// Set window position
    pub fn set_position(&self,x: i32,y: i32) {
        use windows_sys::Win32::UI::WindowsAndMessaging::SetWindowPos;
        unsafe {
            SetWindowPos(self.handle, 0, x, y, x, y, 0);
        }
    }

    /// Set window size
    /// **Warning: This method is unstable.**
    pub fn set_size(&self,width: i32,height: i32) {
        use windows_sys::Win32::Foundation::RECT;
        use windows_sys::Win32::UI::WindowsAndMessaging::*;
        unsafe {
            let rect = &mut RECT {
                left: 0,
                top: 0,
                right: 0,
                bottom: 0
            };
            GetWindowRect(self.handle,rect);

            ShowWindow(self.handle,SW_SHOWDEFAULT);
            SetWindowPos(self.handle, 0, rect.left, rect.top, width, height, 0);
        }
    }
}

#[cfg(windows)]
impl Default for Frame {
    fn default() -> Self {
        Self {
            handle: 0
        }
    }
}

#[cfg(windows)]
pub struct Attributes {
    pub style: windows_sys::Win32::UI::WindowsAndMessaging::WINDOW_STYLE,
    pub title: String,
    pub width: i32,
    pub height: i32,
    pub x: i32,
    pub y: i32,
}

#[cfg(windows)]
impl Attributes {
    pub fn new(style: windows_sys::Win32::UI::WindowsAndMessaging::WINDOW_STYLE,title: String,width: i32,height: i32,x: i32,y: i32) -> Self {
        Self {
            style,
            title,
            width,
            height,
            x,
            y
        }
    }
}

#[cfg(windows)]
pub struct AdvancedOptions {
    advanced_style: windows_sys::Win32::UI::WindowsAndMessaging::WINDOW_EX_STYLE
}

#[cfg(windows)]
impl AdvancedOptions {
    pub fn new(advanced_style: windows_sys::Win32::UI::WindowsAndMessaging::WINDOW_EX_STYLE) -> Self {
        Self {
            advanced_style
        }
    }

    pub fn get_style(&self) -> windows_sys::Win32::UI::WindowsAndMessaging::WINDOW_EX_STYLE {
        self.advanced_style
    }
}

#[cfg(windows)]
impl Default for AdvancedOptions {
    fn default() -> Self {
        Self {
            advanced_style: 0
        }
    }
}

#[cfg(windows)]
pub trait Application {
    /// Returns the name of the application.
    fn app_name(&self) -> String;
    /// Returns WindowAttributes.
    fn attributes(&self) -> Attributes;

    fn is_child_window(&self) -> bool;

    fn advanced_options(&self) -> AdvancedOptions {
        AdvancedOptions::default()
    }

    /// Processing at application startup (optional)
    fn startup(&self,_storage: Storage) {
    }

    /// Writing UI processes and events
    fn ui(&self,frame: Frame);
    /// Processing at application termination (optional)
    fn exit(&self) {
    }
}

#[cfg(unix)]
#[derive(Clone)]
pub struct Frame {
    /// Window body
    handle: gtk::Window
}

#[cfg(unix)]
impl Frame {
    /// Create a new Frame
    pub fn new(window: gtk::Window) -> Self {
        Self {
            handle: window
        }
    }

    /// Display the window
    pub fn show(&self) {
        use gtk::prelude::*;
        self.handle.show();
    }

    /// Maximize the window
    pub fn show_maximized(&self) {
        use gtk::prelude::*;
        self.handle.maximize();
    }

    /// Minimize the window
    pub fn show_minimized(&self) {
        use gtk::prelude::*;
        self.handle.unmaximize();
    }

    /// Hide the window
    pub fn hide(&self) {
        use gtk::prelude::*;
        self.handle.hide();
    }

    /// Set window title
    pub fn set_title(&self,title: String) {
        use gtk::prelude::*;
        self.handle.set_title(title.as_str());
    }

    /// Returns the window handle.
    pub fn get_window(&self) -> gtk::Window {
        use std::borrow::Borrow;
        self.handle.borrow().clone()
    }

    /// Set window position
    pub fn set_position(&self,x: i32,y: i32) {
        use gtk::prelude::*;
        self.handle.set_window_position(gtk::WindowPosition::Center);
    }

    /// Set window size
    pub fn set_size(&self,width: i32,height: i32) {
        use gtk::prelude::*;
        self.handle.set_default_size(width,height);
    }
}

#[cfg(unix)]
impl Default for Frame {
    fn default() -> Self {
        Self {
            handle: gtk::Window::default()
        }
    }
}

#[cfg(unix)]
pub struct Attributes {
    pub style: gtk::WindowType,
    pub title: String,
    pub width: i32,
    pub height: i32,
    pub x: i32,
    pub y: i32,
}

#[cfg(unix)]
impl Attributes {
    pub fn new(style: gtk::WindowType,title: String,width: i32,height: i32,x: i32,y: i32) -> Self {
        Self {
            style,
            title,
            width,
            height,
            x,
            y,
        }
    }
}

#[cfg(unix)]
pub struct AdvancedOptions {
    advanced_style: gtk::gdk::WindowTypeHint
}

#[cfg(unix)]
impl AdvancedOptions {
    pub fn new(advanced_style: gtk::gdk::WindowTypeHint) -> Self {
        Self {
            advanced_style
        }
    }

    pub fn get_style(&self) -> gtk::gdk::WindowTypeHint {
        self.advanced_style
    }
}

#[cfg(unix)]
impl Default for AdvancedOptions {
    fn default() -> Self {
        Self {
            advanced_style: gtk::gdk::WindowTypeHint::Normal
        }
    }
}

#[cfg(unix)]
pub trait Application {
    /// Returns the name of the window.
    fn app_name(&self) -> String;
    /// Returns WindowAttributes.
    fn attributes(&self) -> Attributes;

    #[cfg(feature="nxui-awc")]
    fn advanced_options(&self) -> AdvancedOptions {
        AdvancedOptions::default()
    }

    /// Processing at application startup (optional)
    fn startup(&self,storage: Storage) {

    }
    /// Writing UI processes and events
    fn ui(&self,frame: Frame);
    /// Processing at application termination (optional)
    fn exit(&self) {

    }
}