use diesel::prelude::*;
use diesel::SqliteConnection;
use uuid::Uuid;

use crate::models::Block;
use crate::utils::rank::SaveFileRanking;
use std::str::FromStr;

pub struct BlockRanking {
    pub previous_id: Option<String>,
    pub next_id: Option<String>,
}

// todo: return Result
trait BlockCrud {
    fn create_block(&self, ranking: BlockRanking, block_type: String) -> Block;
    fn retrieve_block(&self, block_id: String) -> Block;
    fn retrieve_all_blocks(&self) -> Vec<Block>;
    fn update_block_rank(&self, block_id: String, ranking: BlockRanking) -> Block;
    fn delete_block(&self, block_id: String);
}

impl BlockCrud for SqliteConnection {
    fn create_block(&self, ranking: BlockRanking, block_type: String) -> Block {
        use crate::schema::blocks;
        use crate::schema::blocks::dsl::*;

        let block_id = Uuid::new_v4().as_bytes().to_vec();
        let block_rank = self.generate_ranking(ranking.previous_id, ranking.next_id);

        let new_block = Block {
            id: block_id.clone(),
            rank: block_rank,
            type_: block_type,
        };

        diesel::insert_into(blocks::table)
            .values(&new_block)
            .execute(self)
            .and_then(|_| blocks.filter(id.eq(&block_id)).first(self))
            .expect("Error saving new block")
    }

    fn retrieve_block(&self, block_id: String) -> Block {
        use crate::schema::blocks::dsl::*;

        let block_uuid = Uuid::from_str(&block_id.as_str()).unwrap().as_bytes().to_vec();

        blocks
            .filter(id.eq(block_uuid))
            .first(self)
            .expect(&format!("Could not retrieve block with {}", block_id))
    }

    fn retrieve_all_blocks(&self) -> Vec<Block> {
        use crate::schema::blocks::dsl::*;

        blocks
            .get_results(self)
            .expect("Could not retrieve all blocks from save file")
    }

    fn update_block_rank(&self, block_id: String, ranking: BlockRanking) -> Block {
        use crate::schema::blocks::dsl::*;

        let block_uuid = Uuid::from_str(&block_id.as_str()).unwrap().as_bytes().to_vec();
        let block_rank = self.generate_ranking(ranking.previous_id, ranking.next_id);

        diesel::update(blocks.find(&block_uuid))
            .set(rank.eq(block_rank))
            .execute(self)
            .and_then(|_| blocks.filter(id.eq(&block_uuid)).first(self))
            .expect(&format!("Could not find block {}", block_id))
    }

    fn delete_block(&self, block_id: String) {
        use crate::schema::blocks::dsl::*;

        let block_uuid = Uuid::from_str(&block_id.as_str()).unwrap().as_bytes().to_vec();

        diesel::delete(blocks.find(block_uuid))
            .execute(self)
            .expect(&format!("Could not delete block {}", block_id));
    }
}
