//////////////////// Known small primes ////////////////////
// The prime table here is used for prime buffer initialization, and also used
// for trivial division in integer factorization and moebius mu function

// only list primes to 256 if big-table feature is disabled
/// A static table of small primes
#[cfg(not(feature = "big-table"))]
pub const SMALL_PRIMES: [u8; 54] = [
    2, 3, 5, 7, 11, 13, 17, 19, 23, 29, 31, 37, 41, 43, 47, 53, 59, 61, 67, 71, 73, 79, 83, 89, 97,
    101, 103, 107, 109, 113, 127, 131, 137, 139, 149, 151, 157, 163, 167, 173, 179, 181, 191, 193,
    197, 199, 211, 223, 227, 229, 233, 239, 241, 251,
];
/// Next prime of the last one in SMALL_PRIMES
#[cfg(not(feature = "big-table"))]
pub const SMALL_PRIMES_NEXT: u64 = 257;

// list 1024 primes if big-table feature is enabled
/// A static table of small primes
#[cfg(feature = "big-table")]
pub const SMALL_PRIMES: [u16; 1024] = [
    2, 3, 5, 7, 11, 13, 17, 19, 23, 29, 31, 37, 41, 43, 47, 53, 59, 61, 67, 71, 73, 79, 83, 89, 97,
    101, 103, 107, 109, 113, 127, 131, 137, 139, 149, 151, 157, 163, 167, 173, 179, 181, 191, 193,
    197, 199, 211, 223, 227, 229, 233, 239, 241, 251, 257, 263, 269, 271, 277, 281, 283, 293, 307,
    311, 313, 317, 331, 337, 347, 349, 353, 359, 367, 373, 379, 383, 389, 397, 401, 409, 419, 421,
    431, 433, 439, 443, 449, 457, 461, 463, 467, 479, 487, 491, 499, 503, 509, 521, 523, 541, 547,
    557, 563, 569, 571, 577, 587, 593, 599, 601, 607, 613, 617, 619, 631, 641, 643, 647, 653, 659,
    661, 673, 677, 683, 691, 701, 709, 719, 727, 733, 739, 743, 751, 757, 761, 769, 773, 787, 797,
    809, 811, 821, 823, 827, 829, 839, 853, 857, 859, 863, 877, 881, 883, 887, 907, 911, 919, 929,
    937, 941, 947, 953, 967, 971, 977, 983, 991, 997, 1009, 1013, 1019, 1021, 1031, 1033, 1039,
    1049, 1051, 1061, 1063, 1069, 1087, 1091, 1093, 1097, 1103, 1109, 1117, 1123, 1129, 1151, 1153,
    1163, 1171, 1181, 1187, 1193, 1201, 1213, 1217, 1223, 1229, 1231, 1237, 1249, 1259, 1277, 1279,
    1283, 1289, 1291, 1297, 1301, 1303, 1307, 1319, 1321, 1327, 1361, 1367, 1373, 1381, 1399, 1409,
    1423, 1427, 1429, 1433, 1439, 1447, 1451, 1453, 1459, 1471, 1481, 1483, 1487, 1489, 1493, 1499,
    1511, 1523, 1531, 1543, 1549, 1553, 1559, 1567, 1571, 1579, 1583, 1597, 1601, 1607, 1609, 1613,
    1619, 1621, 1627, 1637, 1657, 1663, 1667, 1669, 1693, 1697, 1699, 1709, 1721, 1723, 1733, 1741,
    1747, 1753, 1759, 1777, 1783, 1787, 1789, 1801, 1811, 1823, 1831, 1847, 1861, 1867, 1871, 1873,
    1877, 1879, 1889, 1901, 1907, 1913, 1931, 1933, 1949, 1951, 1973, 1979, 1987, 1993, 1997, 1999,
    2003, 2011, 2017, 2027, 2029, 2039, 2053, 2063, 2069, 2081, 2083, 2087, 2089, 2099, 2111, 2113,
    2129, 2131, 2137, 2141, 2143, 2153, 2161, 2179, 2203, 2207, 2213, 2221, 2237, 2239, 2243, 2251,
    2267, 2269, 2273, 2281, 2287, 2293, 2297, 2309, 2311, 2333, 2339, 2341, 2347, 2351, 2357, 2371,
    2377, 2381, 2383, 2389, 2393, 2399, 2411, 2417, 2423, 2437, 2441, 2447, 2459, 2467, 2473, 2477,
    2503, 2521, 2531, 2539, 2543, 2549, 2551, 2557, 2579, 2591, 2593, 2609, 2617, 2621, 2633, 2647,
    2657, 2659, 2663, 2671, 2677, 2683, 2687, 2689, 2693, 2699, 2707, 2711, 2713, 2719, 2729, 2731,
    2741, 2749, 2753, 2767, 2777, 2789, 2791, 2797, 2801, 2803, 2819, 2833, 2837, 2843, 2851, 2857,
    2861, 2879, 2887, 2897, 2903, 2909, 2917, 2927, 2939, 2953, 2957, 2963, 2969, 2971, 2999, 3001,
    3011, 3019, 3023, 3037, 3041, 3049, 3061, 3067, 3079, 3083, 3089, 3109, 3119, 3121, 3137, 3163,
    3167, 3169, 3181, 3187, 3191, 3203, 3209, 3217, 3221, 3229, 3251, 3253, 3257, 3259, 3271, 3299,
    3301, 3307, 3313, 3319, 3323, 3329, 3331, 3343, 3347, 3359, 3361, 3371, 3373, 3389, 3391, 3407,
    3413, 3433, 3449, 3457, 3461, 3463, 3467, 3469, 3491, 3499, 3511, 3517, 3527, 3529, 3533, 3539,
    3541, 3547, 3557, 3559, 3571, 3581, 3583, 3593, 3607, 3613, 3617, 3623, 3631, 3637, 3643, 3659,
    3671, 3673, 3677, 3691, 3697, 3701, 3709, 3719, 3727, 3733, 3739, 3761, 3767, 3769, 3779, 3793,
    3797, 3803, 3821, 3823, 3833, 3847, 3851, 3853, 3863, 3877, 3881, 3889, 3907, 3911, 3917, 3919,
    3923, 3929, 3931, 3943, 3947, 3967, 3989, 4001, 4003, 4007, 4013, 4019, 4021, 4027, 4049, 4051,
    4057, 4073, 4079, 4091, 4093, 4099, 4111, 4127, 4129, 4133, 4139, 4153, 4157, 4159, 4177, 4201,
    4211, 4217, 4219, 4229, 4231, 4241, 4243, 4253, 4259, 4261, 4271, 4273, 4283, 4289, 4297, 4327,
    4337, 4339, 4349, 4357, 4363, 4373, 4391, 4397, 4409, 4421, 4423, 4441, 4447, 4451, 4457, 4463,
    4481, 4483, 4493, 4507, 4513, 4517, 4519, 4523, 4547, 4549, 4561, 4567, 4583, 4591, 4597, 4603,
    4621, 4637, 4639, 4643, 4649, 4651, 4657, 4663, 4673, 4679, 4691, 4703, 4721, 4723, 4729, 4733,
    4751, 4759, 4783, 4787, 4789, 4793, 4799, 4801, 4813, 4817, 4831, 4861, 4871, 4877, 4889, 4903,
    4909, 4919, 4931, 4933, 4937, 4943, 4951, 4957, 4967, 4969, 4973, 4987, 4993, 4999, 5003, 5009,
    5011, 5021, 5023, 5039, 5051, 5059, 5077, 5081, 5087, 5099, 5101, 5107, 5113, 5119, 5147, 5153,
    5167, 5171, 5179, 5189, 5197, 5209, 5227, 5231, 5233, 5237, 5261, 5273, 5279, 5281, 5297, 5303,
    5309, 5323, 5333, 5347, 5351, 5381, 5387, 5393, 5399, 5407, 5413, 5417, 5419, 5431, 5437, 5441,
    5443, 5449, 5471, 5477, 5479, 5483, 5501, 5503, 5507, 5519, 5521, 5527, 5531, 5557, 5563, 5569,
    5573, 5581, 5591, 5623, 5639, 5641, 5647, 5651, 5653, 5657, 5659, 5669, 5683, 5689, 5693, 5701,
    5711, 5717, 5737, 5741, 5743, 5749, 5779, 5783, 5791, 5801, 5807, 5813, 5821, 5827, 5839, 5843,
    5849, 5851, 5857, 5861, 5867, 5869, 5879, 5881, 5897, 5903, 5923, 5927, 5939, 5953, 5981, 5987,
    6007, 6011, 6029, 6037, 6043, 6047, 6053, 6067, 6073, 6079, 6089, 6091, 6101, 6113, 6121, 6131,
    6133, 6143, 6151, 6163, 6173, 6197, 6199, 6203, 6211, 6217, 6221, 6229, 6247, 6257, 6263, 6269,
    6271, 6277, 6287, 6299, 6301, 6311, 6317, 6323, 6329, 6337, 6343, 6353, 6359, 6361, 6367, 6373,
    6379, 6389, 6397, 6421, 6427, 6449, 6451, 6469, 6473, 6481, 6491, 6521, 6529, 6547, 6551, 6553,
    6563, 6569, 6571, 6577, 6581, 6599, 6607, 6619, 6637, 6653, 6659, 6661, 6673, 6679, 6689, 6691,
    6701, 6703, 6709, 6719, 6733, 6737, 6761, 6763, 6779, 6781, 6791, 6793, 6803, 6823, 6827, 6829,
    6833, 6841, 6857, 6863, 6869, 6871, 6883, 6899, 6907, 6911, 6917, 6947, 6949, 6959, 6961, 6967,
    6971, 6977, 6983, 6991, 6997, 7001, 7013, 7019, 7027, 7039, 7043, 7057, 7069, 7079, 7103, 7109,
    7121, 7127, 7129, 7151, 7159, 7177, 7187, 7193, 7207, 7211, 7213, 7219, 7229, 7237, 7243, 7247,
    7253, 7283, 7297, 7307, 7309, 7321, 7331, 7333, 7349, 7351, 7369, 7393, 7411, 7417, 7433, 7451,
    7457, 7459, 7477, 7481, 7487, 7489, 7499, 7507, 7517, 7523, 7529, 7537, 7541, 7547, 7549, 7559,
    7561, 7573, 7577, 7583, 7589, 7591, 7603, 7607, 7621, 7639, 7643, 7649, 7669, 7673, 7681, 7687,
    7691, 7699, 7703, 7717, 7723, 7727, 7741, 7753, 7757, 7759, 7789, 7793, 7817, 7823, 7829, 7841,
    7853, 7867, 7873, 7877, 7879, 7883, 7901, 7907, 7919, 7927, 7933, 7937, 7949, 7951, 7963, 7993,
    8009, 8011, 8017, 8039, 8053, 8059, 8069, 8081, 8087, 8089, 8093, 8101, 8111, 8117, 8123, 8147,
    8161, // 8167, 8171, 8179, 8191
];

/// Next prime of the last one in SMALL_PRIMES
#[cfg(feature = "big-table")]
pub const SMALL_PRIMES_NEXT: u64 = 8167;

//////////////////// Pre-computed inversions for primes ////////////////////

/// Precomputed modular inverse for fast divisibility check
///
/// We could have store them in u16 and do double width check, but don't bother.
/// Note that the factor 2 should be eliminated before checking the primes
/// Reference: <https://math.stackexchange.com/questions/1251327/is-there-a-fast-divisibility-check-for-a-fixed-divisor>
#[cfg(feature = "big-table")]
pub const SMALL_PRIMES_INV: [u64; 1024] = [
    0x0000000000000000,
    0xaaaaaaaaaaaaaaab,
    0xcccccccccccccccd,
    0x6db6db6db6db6db7,
    0x2e8ba2e8ba2e8ba3,
    0x4ec4ec4ec4ec4ec5,
    0xf0f0f0f0f0f0f0f1,
    0x86bca1af286bca1b,
    0xd37a6f4de9bd37a7,
    0x34f72c234f72c235,
    0xef7bdef7bdef7bdf,
    0x14c1bacf914c1bad,
    0x8f9c18f9c18f9c19,
    0x82fa0be82fa0be83,
    0x51b3bea3677d46cf,
    0x21cfb2b78c13521d,
    0xcbeea4e1a08ad8f3,
    0x4fbcda3ac10c9715,
    0xf0b7672a07a44c6b,
    0x193d4bb7e327a977,
    0x7e3f1f8fc7e3f1f9,
    0x9b8b577e613716af,
    0xa3784a062b2e43db,
    0xf47e8fd1fa3f47e9,
    0xa3a0fd5c5f02a3a1,
    0x3a4c0a237c32b16d,
    0xdab7ec1dd3431b57,
    0x77a04c8f8d28ac43,
    0xa6c0964fda6c0965,
    0x90fdbc090fdbc091,
    0x7efdfbf7efdfbf7f,
    0x03e88cb3c9484e2b,
    0xe21a291c077975b9,
    0x3aef6ca970586723,
    0xdf5b0f768ce2cabd,
    0x6fe4dfc9bf937f27,
    0x5b4fe5e92c0685b5,
    0x1f693a1c451ab30b,
    0x8d07aa27db35a717,
    0x882383b30d516325,
    0xed6866f8d962ae7b,
    0x3454dca410f8ed9d,
    0x1d7ca632ee936f3f,
    0x70bf015390948f41,
    0xc96bdb9d3d137e0d,
    0x2697cc8aef46c0f7,
    0xc0e8f2a76e68575b,
    0x687763dfdb43bb1f,
    0x1b10ea929ba144cb,
    0x1d10c4c0478bbced,
    0x63fb9aeb1fdcd759,
    0x64afaa4f437b2e0f,
    0xf010fef010fef011,
    0x28cbfbeb9a020a33,
    0xff00ff00ff00ff01,
    0xd624fd1470e99cb7,
    0x8fb3ddbd6205b5c5,
    0xd57da36ca27acdef,
    0xee70c03b25e4463d,
    0xc5b1a6b80749cb29,
    0x47768073c9b97113,
    0x2591e94884ce32ad,
    0xf02806abc74be1fb,
    0x7ec3e8f3a7198487,
    0x58550f8a39409d09,
    0xec9e48ae6f71de15,
    0x2ff3a018bfce8063,
    0x7f9ec3fcf61fe7b1,
    0x89f5abe570e046d3,
    0xda971b23f1545af5,
    0x79d5f00b9a7862a1,
    0x4dba1df32a128a57,
    0x87530217b7747d8f,
    0x30baae53bb5e06dd,
    0xee70206c12e9b5b3,
    0xcdde9462ec9dbe7f,
    0xafb64b05ec41cf4d,
    0x02944ff5aec02945,
    0x2cb033128382df71,
    0x1ccacc0c84b1c2a9,
    0x19a93db575eb3a0b,
    0xcebeef94fa86fe2d,
    0x6faa77fb3f8df54f,
    0x68a58af00975a751,
    0xd56e36d0c3efac07,
    0xd8b44c47a8299b73,
    0x02d9ccaf9ba70e41,
    0x0985e1c023d9e879,
    0x2a343316c494d305,
    0x70cb7916ab67652f,
    0xd398f132fb10fe5b,
    0x6f2a38a6bf54fa1f,
    0x211df689b98f81d7,
    0x0e994983e90f1ec3,
    0xad671e44bed87f3b,
    0xf9623a0516e70fc7,
    0x4b7129be9dece355,
    0x190f3b7473f62c39,
    0x63dacc9aad46f9a3,
    0xc1108fda24e8d035,
    0xb77578472319bd8b,
    0x473d20a1c7ed9da5,
    0xfbe85af0fea2c8fb,
    0x58a1f7e6ce0f4c09,
    0x1a00e58c544986f3,
    0x7194a17f55a10dc1,
    0x7084944785e33763,
    0xba10679bd84886b1,
    0xebe9c6bb31260967,
    0x97a3fe4bd1ff25e9,
    0x6c6388395b84d99f,
    0x8c51da6a1335df6d,
    0x46f3234475d5add9,
    0x905605ca3c619a43,
    0xcee8dff304767747,
    0xff99c27f00663d81,
    0xacca407f671ddc2b,
    0xe71298bac1e12337,
    0xfa1e94309cd09045,
    0xbebccb8e91496b9b,
    0x312fa30cc7d7b8bd,
    0x6160ff9e9f006161,
    0x6b03673b5e28152d,
    0xfe802ffa00bfe803,
    0xe66fe25c9e907c7b,
    0x3f8b236c76528895,
    0xf6f923bf01ce2c0d,
    0x6c3d3d98bed7c42f,
    0x30981efcd4b010e7,
    0x6f691fc81ebbe575,
    0xb10480ddb47b52cb,
    0x74cd59ed64f3f0d7,
    0x0105cb81316d6c0f,
    0x9be64c6d91c1195d,
    0x71b3f945a27b1f49,
    0x77d80d50e508fd01,
    0xa5eb778e133551cd,
    0x18657d3c2d8a3f1b,
    0x2e40e220c34ad735,
    0xa76593c70a714919,
    0x1eef452124eea383,
    0x38206dc242ba771d,
    0x4cd4c35807772287,
    0x83de917d5e69ddf3,
    0x882ef0403b4a6c15,
    0xf8fb6c51c606b677,
    0xb4abaac446d3e1fd,
    0xa9f83bbe484a14e9,
    0x0bebbc0d1ce874d3,
    0xbd418eaf0473189f,
    0x44e3af6f372b7e65,
    0xc87fdace4f9e5d91,
    0xec93479c446bd9bb,
    0xdac4d592e777c647,
    0xa63ea8c8f61f0c23,
    0xe476062ea5cbbb6f,
    0xdf68761c69daac27,
    0xb813d737637aa061,
    0xa3a77aac1fb15099,
    0x17f0c3e0712c5825,
    0xfd912a70ff30637b,
    0xfbb3b5dc01131289,
    0x856d560a0f5acdf7,
    0x96472f314d3f89e3,
    0xa76f5c7ed2253531,
    0x816eae7c7bf69fe7,
    0xb6a2bea4cfb1781f,
    0xa3900c53318e81ed,
    0x60aa7f5d9f148d11,
    0x6be8c0102c7a505d,
    0x8ff3f0ed28728f33,
    0x680e0a87e5ec7155,
    0xbbf70fa49fe829b7,
    0xd69d1e7b6a50ca39,
    0x1a1e0f46b6d26aef,
    0x7429f9a7a8251829,
    0xd9c2219d1b863613,
    0x91406c1820d077ad,
    0x521f4ec02e3d2b97,
    0xbb8283b63dc8eba5,
    0x431eda153229ebbf,
    0xaf0bf78d7e01686b,
    0xa9ced0742c086e8d,
    0xc26458ad9f632df9,
    0xbbff1255dff892af,
    0xcbd49a333f04d8fd,
    0xec84ed6f9cfdeff5,
    0x97980cc40bda9d4b,
    0x777f34d524f5cbd9,
    0x2797051d94cbbb7f,
    0xea769051b4f43b81,
    0xce7910f3034d4323,
    0x92791d1374f5b99b,
    0x89a5645cc68ea1b5,
    0x5f8aacf796c0cf0b,
    0xf2e90a15e33edf99,
    0x8e99e5feb897c451,
    0xaca2eda38fb91695,
    0x5d9b737be5ea8b41,
    0x4aefe1db93fd7cf7,
    0xa0994ef20b3f8805,
    0x103890bda912822f,
    0xb441659d13a9147d,
    0x1e2134440c4c3f21,
    0x263a27727a6883c3,
    0x78e221472ab33855,
    0x95eac88e82e6faff,
    0xf66c258317be8dab,
    0x09ee202c7cb91939,
    0x8d2fca1042a09ea3,
    0x82779c856d8b8bf1,
    0x3879361cba8a223d,
    0xf23f43639c3182a7,
    0xa03868fc474bcd13,
    0x651e78b8c5311a97,
    0x8ffce639c00c6719,
    0xf7b460754b0b61cf,
    0x7b03f3359b8e63b1,
    0xa55c5326041eb667,
    0x647f88ab896a76f5,
    0x8fd971434a55a46d,
    0x9fbf969958046447,
    0x9986feba69be3a81,
    0xa668b3e6d053796f,
    0x97694e6589f4e09b,
    0x37890c00b7721dbd,
    0x5ac094a235f37ea9,
    0x31cff775f2d5d65f,
    0xddad8e6b36505217,
    0x5a27df897062cd03,
    0xe2396fe0fdb5a625,
    0xb352a4957e82317b,
    0xd8ab3f2c60c2ea3f,
    0x6893f702f0452479,
    0x9686fdc182acf7e3,
    0x6854037173dce12f,
    0x7f0ded1685c27331,
    0xeeda72e1fe490b7d,
    0x9e7bfc959a8e6e53,
    0x49b314d6d4753dd7,
    0x2e8f8c5ac4aa1b3b,
    0xb8ef723481163d33,
    0x6a2ec96a594287b7,
    0xdba41c6d13aab8c5,
    0xc2adbe648dc3aaf1,
    0x87a2bade565f91a7,
    0x4d6fe8798c01f5df,
    0x3791310c8c23d98b,
    0xf80e446b01228883,
    0x9aed1436fbf500cf,
    0x7839b54cc8b24115,
    0xc128c646ad0309c1,
    0x14de631624a3c377,
    0x3f7b9fe68b0ecbf9,
    0x284ffd75ec00a285,
    0x37803cb80dea2ddb,
    0x86b63f7c9ac4c6fd,
    0x8b6851d1bd99b9d3,
    0xb62fda77ca343b6d,
    0x1f0dc009e34383c9,
    0x496dc21ddd35b97f,
    0xb0e96ce17090f82b,
    0xaadf05acdd7d024d,
    0xcb138196746eafb5,
    0x347f523736755d61,
    0xd14a48a051f7dd0b,
    0x474d71b1ce914d25,
    0x386063f5e28c1f89,
    0x1db7325e32d04e73,
    0xfef748d3893b880d,
    0x2f3351506e935605,
    0x7a3637fa2376415b,
    0x4ac525d2baa21969,
    0x3a11c16b42cd351f,
    0x6c7abde0049c2a11,
    0x54dad0303e069ac7,
    0xebf1ac9fdfe91433,
    0xfafdda8237cec655,
    0xdce3ff6e71ffb739,
    0xbed5737d6286db1b,
    0xe479e431fe08b4df,
    0x9dd9b0dd7742f897,
    0x8f09d7402c5a5e87,
    0x9216d5c4d958738d,
    0xb3139ba11d34ca63,
    0x47d54f7ed644afaf,
    0x92a81d85cf11a1b1,
    0x754b26533253bdfd,
    0xbbe0efc980bfd467,
    0xc0d8d594f024dca1,
    0x8238d43bcaac1a65,
    0x27779c1fae6175bb,
    0xa746ca9af708b2c9,
    0x93f3cd9f389be823,
    0x5cb4a4c04c489345,
    0xbf6047743e85b6b5,
    0x61c147831563545f,
    0xedb47c0ae62dee9d,
    0x0a3824386673a573,
    0xa4a77d19e575a0eb,
    0xa2bee045e066c279,
    0xc23618de8ab43d05,
    0x266b515216cb9f2f,
    0xe279edd9e9c2e85b,
    0xd0c591c221dc9c53,
    0x06da8ee9c9ee7c21,
    0x9dfebcaf4c27e8c3,
    0x49aeff9f19dd6de5,
    0x86976a57a296e9c7,
    0xa3b9abf4872b84cd,
    0x34fca6483895e6ef,
    0x34b5a333988f873d,
    0xd9dd4f19b5f17be1,
    0xb935b507fd0ce78b,
    0xb450f5540660e797,
    0x63ff82831ffc1419,
    0x8992f718c22a32fb,
    0x5f3253ad0d37e7bf,
    0x007c0ffe0fc007c1,
    0x4d8ebadc0c0640b1,
    0xe2729af831037bdb,
    0xb8f64bf30feebfe9,
    0xda93124b544c0bf5,
    0x9cf7ff0b593c539f,
    0xd6bd8861fa0e07d9,
    0x5cfe75c0bd8ab891,
    0x43e808757c2e862b,
    0x90caa96d595c9d93,
    0x8fd550625d07135f,
    0x76b010a86e209f2d,
    0xecc0426447769b25,
    0xe381339caabe3295,
    0xd1b190a2d0c7673f,
    0xc3bce3cf26b0e7eb,
    0x5f87e76f56c61ce3,
    0xc06c6857a124b353,
    0x38c040fcba630f75,
    0xd078bc4fbd533b21,
    0xde8e15c5dd354f59,
    0xca61d53d7414260f,
    0xb56bf5ba8eae635d,
    0x44a72cb0fb6e3949,
    0x879839a714f45bcd,
    0x02a8994fde5314b7,
    0xb971920cf2b90135,
    0x8a8fd0b7df9a6e8b,
    0xb31f9a84c1c6eaad,
    0x92293b02823c6d83,
    0xeee77ff20fe5ddcf,
    0x0e1ea0f6c496c11d,
    0xfdf2d3d6f88ccb6b,
    0xfa9d74a3457738f9,
    0xefc3ca3db71a5785,
    0x8e2071718d0d6daf,
    0xbc0fdbfeb6cfabfd,
    0x1eeab613e5e5aee9,
    0x2d2388e90e9e929f,
    0x81dbafba588ddb43,
    0x52eebc51c4799791,
    0x1c6bc4693b45a047,
    0x06eee0974498874d,
    0xd85b7377a9953cb9,
    0x4b6df412d4caf56f,
    0x6b8afbbb4a053493,
    0xcc5299c96ac7720b,
    0xadce84b5c710aa99,
    0x9d673f5aa3804225,
    0xe6541268efbce7f7,
    0xfcf41e76cf5be669,
    0x5c3eb5dc31c383cb,
    0x301832d11d8ad6c3,
    0x2e9c0942f1ce450f,
    0x97f3f2be37a39a5d,
    0xe8b7d8a9654187c7,
    0xb5d024d7da5b1b55,
    0xb8ba9d6e7ae3501b,
    0xf50865f71b90f1df,
    0x739c1682847df9e1,
    0xc470a4d842b90ed1,
    0x1fb1be11698cc409,
    0xd8d5512a7cd35d15,
    0xa5496821723e07f9,
    0xbcc8c6d7abaa8167,
    0x52c396c95eb619a1,
    0x6eb7e380878ec74b,
    0x3d5513b504537157,
    0x314391f8862e948f,
    0xdc0b17cfcd81f5dd,
    0x2f6bea3ec89044b3,
    0xce13a05869f1b57f,
    0x7593474e8ace3581,
    0x07fc329295a05e4d,
    0xb05377cba4908d23,
    0xe7b2131a628aa39b,
    0x9031dbed7de01527,
    0x76844b1c670aa9a9,
    0x6a03f4533b08915f,
    0x1dbca579db0a3999,
    0x002ffe800bffa003,
    0x478ab1a3e936139d,
    0x66e722bc4c5cc095,
    0x7a8f63c717278541,
    0xdf6eee24d292bc2f,
    0x9fc20d17237dd569,
    0xcdf9932356bda2ed,
    0x97b5e332e80f68d7,
    0x46eee26fd875e2e5,
    0x3548a8e65157a611,
    0xc288d03be9b71e3b,
    0x8151186db38937ab,
    0x7800b910895a45f1,
    0xaee0b024182eec3d,
    0x96323eda173b5713,
    0x0ed0dbd03ae77c8b,
    0xf73800b7828dc119,
    0x1b61715ec22b7ca5,
    0xa8533a991ead64bf,
    0x7f6c7290e46c2e77,
    0x6325e8d907b01db1,
    0x28909f70152a1067,
    0xea7077af0997a0f5,
    0x7e605cad10c32e6d,
    0x471b33570635b38f,
    0xab559fa997a61bb3,
    0xad4bdae562bddab9,
    0x055e1b2f2ed62f45,
    0x03cd328b1a2dca9b,
    0xd28f4e08733218a9,
    0xb6800b077f186293,
    0x6fbd138c3fd9c207,
    0xb117ccd12ae88a89,
    0x2f1a1a044046bceb,
    0x548aba0b060541e3,
    0xcf4e808cea111b2f,
    0xdbec1b4fa855a475,
    0xe3f794eb600d7821,
    0x34fae0d9a11f7c59,
    0xf006b0ccbbac085d,
    0x3f45076dc3114733,
    0xeef49bfa58a1a1b7,
    0x12c4218bea691fa3,
    0xbc7504e3bd5e64f1,
    0x4ee21c292bb92fad,
    0x34338b7327a4bacf,
    0x3fe5c0833d6fccd1,
    0xb1e70743535203c1,
    0xefbb5dcdfb4e43d3,
    0xca68467ca5394f9f,
    0x8c51c081408b97a1,
    0x3275a899dfa5dd65,
    0x9e674cb62e1b78bb,
    0xa37ff5bb2a998d47,
    0x792a999db131a22b,
    0x1b48841bc30d29b9,
    0xf06721d2011d3471,
    0x93fd2386dff85ebd,
    0x4ce72f54c07ed9b5,
    0xd6d0fd3e71dd827b,
    0x856405fb1eed819d,
    0x8ea8aceb7c443989,
    0x34a13026f62e5873,
    0x1eea0208ec0af4f7,
    0x63679853cea598cb,
    0xc30b3ebd61f2d0ed,
    0x7eb9037bc7f43bc3,
    0xa583e6f6ce016411,
    0xf1938d895f1a74c7,
    0x80cf1491c1e81e33,
    0x3c0f12886ba8f301,
    0x0e4b786e0dfcc5ab,
    0x672684c93f2d41ef,
    0xe00757badb35c51b,
    0xd6d84afe66472edf,
    0xfbbc0eedcbbfb6e1,
    0x250f43aa08a84983,
    0x04400e927b1acaa5,
    0x56572be34b9d3215,
    0x87964ef7781c62bf,
    0x29ed84051c06e9af,
    0xb00acd11ed3f87fd,
    0x06307881744152d9,
    0x7a786459f5c1ccc9,
    0x1308125d74563281,
    0x395310a480b3e34d,
    0x35985baa8b202837,
    0x96304a6e052b3223,
    0xbd8265fc9af8fd45,
    0x1b6d0b383ec58e0b,
    0xc21a7c3b68b28503,
    0x236fa180fbfd6007,
    0xc42accd440ed9595,
    0x7acf7128236ba3f7,
    0xf909367a987b9c79,
    0xb64efb252bfba705,
    0x980d4f5a7e4cd25b,
    0xe1ecc4ef27b0c37d,
    0x9111aebb81d72653,
    0x8951f985cb2c67ed,
    0xc439d4fc54e0b5d7,
    0xe857bf31896d533b,
    0xb614bb4cb5023755,
    0x938a89e5473bf1ff,
    0xeac481aca34de039,
    0x14b961badf4809a7,
    0x76784fecba352435,
    0xefa689bb58aef5e1,
    0xb2b2c4db9c3a8197,
    0x2503bc992279f8cf,
    0xd2ab9aec5ca1541d,
    0x3e78ba1460f99af3,
    0x0a01426572cfcb63,
    0xbea857968f3cbd67,
    0x78db213eefe659e9,
    0x963e8541a74d35f5,
    0x9e22d152776f2e43,
    0x05d10d39d1e1f291,
    0x374468dccaced1dd,
    0x8d145c7d110c5ad5,
    0x3251a39f5acb5737,
    0xa66e50171443506f,
    0x124f69ad91dd4cbd,
    0xec24f8f2a61a2793,
    0xb472148e656b7a51,
    0x0adf9570e1142f07,
    0x89bf33b065119789,
    0x8f0149803cb291eb,
    0x8334b63afd190a31,
    0x920908d50d6aba7d,
    0x57d8b018c5a33d53,
    0xea1773092dc27ee5,
    0xcae5f38b7bf2e00f,
    0x2bd02df34f695349,
    0xddfecd5be62e2eb7,
    0xdbf849ebec96c4a3,
    0xda31d4d0187357c5,
    0xe34e21cc2d5418a7,
    0x68ca5137a9e574ad,
    0x3eaa0d0f804bfd19,
    0x554fb753cc20e9d1,
    0x797afcca1300756b,
    0x8b8d950b52eeea77,
    0xfb6cd166acabc185,
    0x4eb6c5ed9437a7af,
    0xd1eddbd91b790cdb,
    0x93d714ea4d8948e9,
    0x3ca13ed8145188d3,
    0x829086016da89c57,
    0xd7da1f432124a543,
    0x7ead5581632fb07f,
    0x35443837f63ec3bd,
    0x89e2b200e5519461,
    0xe9ae44f0b7289c0b,
    0x94387a277b9fa817,
    0xc84f1a58abfc2c25,
    0x71101d8e3c83377b,
    0xc024abe5c50ba69d,
    0x15de4eb365a65d73,
    0x09ed28a76bcca931,
    0x816bffbf4a00205b,
    0x1f5c71543d558069,
    0xf25c64d0ec53b859,
    0x96c02c2ef1e0ff0f,
    0x19a804816870a333,
    0x6de49add0971c555,
    0x528087e684c71aab,
    0xa94152c269bcdeef,
    0x0379450a3c2b6bdf,
    0xd2cd38bafe5373e1,
    0xc29df2bea71d8bad,
    0xc15862775f302e83,
    0x1016af2fe55ede09,
    0x3d26dbd9d1910715,
    0x621dab2dfaf3dfbf,
    0xb6f1d7ac287338b1,
    0x8d9e9f0c3f9e7fd9,
    0x60a93f8762e914bb,
    0xb14371f247c159c9,
    0x6dd3b484471d4eb3,
    0xcd172f4701c1684d,
    0x0372e686ed8bb537,
    0xbc07f7ca65c5b071,
    0xab2b6170c3f78d9b,
    0xf3d74f461fe6f5b5,
    0xdbc13f4b31f3230b,
    0xd1420716e3f1572d,
    0xd5be2fd4d805464f,
    0xc68b97c136943851,
    0x9e27918af7cfb473,
    0x5ec8ab6c36ac7f41,
    0x964076331dd90979,
    0x30198eff77b002d7,
    0x3af7cb9583ece011,
    0x34ce06f643d9883b,
    0x79f767e528708c55,
    0x185332d2ef2313cd,
    0x43b611b84c8332a3,
    0xc2e215e4f43bb63d,
    0xf94b9dd22ce44e97,
    0xd895834a1db166a5,
    0x347d2f16d19b8d09,
    0x1b54d4dc45b7d98d,
    0x117ac30d9a044877,
    0x0e10b78a67a526e9,
    0x92da68a818688a9f,
    0xcf2b6c87f741f84b,
    0xd264f9bd41e18ed9,
    0x733cbeaa97166d8f,
    0xc9f475b021d22e81,
    0x731f76f2ec4c852b,
    0xdaf6f0c978f69945,
    0x749c8ad20c61ec93,
    0x09307ff8bd3c1261,
    0x334a69fb5a486e2d,
    0x1f36c7bf31578617,
    0x31ebbcc279ea6103,
    0x42e2aad119f466eb,
    0x106ec05a0ab1450d,
    0xb1b38db92a99e731,
    0x784ae377e67071e7,
    0x3e9e1471ba6671d7,
    0x82c29b59d4d73d0f,
    0xc23dd07128b5525d,
    0x4d4e5ce0e9245133,
    0xc8fd1057c09f8cc5,
    0xea1516e94f394035,
    0xb5e3319c564ee9df,
    0x126a69f90d822d8b,
    0x501ed6348857aa19,
    0xde344a324eee1c83,
    0x1dd9690cb2c406d1,
    0x08d6c5178d5e4387,
    0x4cea4050a3e8fdc1,
    0xc114a06acc83f777,
    0x20b060ebc0ea01db,
    0xfe50045acb78c99f,
    0x291a68705b196e91,
    0xc1042c724273e2bb,
    0x2cee680bb165b7c9,
    0xfd2ff9f12e0776d5,
    0x166a5da63af2cc6f,
    0xedd16a5930408d27,
    0x2adf30c26528844f,
    0x9a48d6572b5eec7b,
    0x6e8bf2877503cb9d,
    0xea27a191a7045389,
    0x6eb091f34dd45d3f,
    0xdc8a6cabb2937d41,
    0xbc2f04f254922a05,
    0x41431f4d6eb38631,
    0x7bd717435a08291f,
    0x4232df9c91fc1a55,
    0xa4651e1d5382eab7,
    0x7cfb5409de4cf3c5,
    0xcdd636fb068b9929,
    0xee8f95e740462c97,
    0x490f97b3a758b4a5,
    0x641431563c441287,
    0xb743dad3ec45916b,
    0x7b188be8f55c878d,
    0xd805648b2ca54ef9,
    0x76dbe6eef60123af,
    0x3711525e6a9e8867,
    0x85c2215cb383d8f5,
    0xe58f554c89825857,
    0x8fbd3b17c01dacd9,
    0x4c8c39dc7aedee65,
    0x653ac6dda86cd3b3,
    0x0d61c6791a9c2c81,
    0xb627a30090354237,
    0x83a89a539c527c23,
    0x28c8c09330e90d71,
    0xee1178d27b1f029b,
    0xcecc740b37860ab5,
    0x79736fde910c485f,
    0x6873d51f2487234f,
    0x2a112180614fb973,
    0xcb04cea98508f4eb,
    0xc2fcd2c527e28d7d,
    0x980203ee10393c69,
    0x3fa90a1d7d75681f,
    0xdbf3bfefef217cc3,
    0x66a17fd3087b41e5,
    0x962195d496fbbd3b,
    0xc705a86155443e49,
    0x3f298ee0be6febff,
    0xaa99b084e62fa613,
    0x1f000cb7d0b46fe1,
    0x9ed7858637c9b2cf,
    0x4d871aaf27c106fb,
    0x2e6a467cdc75a4f3,
    0xe9d938fb696dde8d,
    0x40ec71b0b1554485,
    0x3aae12f861e5f3e9,
    0xa97565873959f843,
    0x0b5a960c09fbca8f,
    0x463fe3d268012c91,
    0xe59a6bd5f5ee1bdd,
    0x6542e84d7775ce45,
    0x8b6eef58fd9effa9,
    0x58993dbb9f98075f,
    0x2997955a810acf61,
    0x76e3d2f5077db451,
    0xb37c1d2867e30907,
    0x53ce6e09bd8d8695,
    0x39db291ea2a6b0e3,
    0xddd265ab9c58847d,
    0x5beca8562dddd0cb,
    0xb69031c153ddbed7,
    0xd03c2271b42a6fcd,
    0xcd6fd19e63e40ea3,
    0xf7687aa8e4fd7bf1,
    0x649dfda112a272a7,
    0xecf7866a56d526df,
    0x72bbf1cfdaebfead,
    0x55f6a48df7055719,
    0x80060bffcfa00183,
    0x8a104f309919b087,
    0x98fa7db7652f6a15,
    0x5d7d1b3df70f7ac1,
    0x16ab7b5e04cc1f6b,
    0x78a5bfd2e5ececf9,
    0x6506392e171d869f,
    0xc3fc12e221ef146d,
    0xf8aa132822c33657,
    0x894496574f536f43,
    0x8b2546b08fb4cbd5,
    0x043bbb561bd1aa7f,
    0x2412c7cc4ea7a12b,
    0x6f0bd406dd71696f,
    0xad475c6988d54b71,
    0xd812e5d48dbbba27,
    0x22aaca437ba04893,
    0xdba6ff1fecd5f09d,
    0x13016d3396286773,
    0xc746494631bcfa41,
    0xd14888565bf6a10d,
    0xc002ef885f0adf05,
    0xe5a04da7fee6ade7,
    0xc114ce5468593bc7,
    0x0bb6747dd7f577b7,
    0x395ce5a20f285839,
    0x6eee8be66e8618ef,
    0x52acf64297f1241b,
    0x361dcc48a364093d,
    0x342d6f475d72e629,
    0x5e978bd46410d413,
    0xcc3433d75ba015ad,
    0x1c83b7628458d4fb,
    0xf9ca45637e38f809,
    0xcbed792ffaf6b115,
    0x9abd961d8c0e8c8d,
    0xe69572fa659340af,
    0x9187e7483a6436fd,
    0x1e9c726993bed9d9,
    0x243554db91976365,
    0x04d06ff994c0088f,
    0x25b76abcb74889dd,
    0x3a409642893c779b,
    0x8f8f620d8bc0c927,
    0x6f9f196b3369855f,
    0x92a522bb0638ed99,
    0x96270f1efdd7004f,
    0xb4844b380fdaa79d,
    0x108936aa5f9c1495,
    0x0b60f606f104c9eb,
    0xc663dfe8263b302f,
    0xc91a280b9110b15b,
    0x0904287118d10969,
    0x160d36a5d31bf553,
    0xe84f5fda3c67ad21,
    0xbd85701f72d4b6ed,
    0x4c50cf5924dee1c3,
    0x2455aaf1633bb6e5,
    0xd775b39f549b8ac7,
    0x87fcdda7a252cb49,
    0x53df2e3bd254a739,
    0x8915e69623a5f7ef,
    0x1ef24c80742dd08b,
    0xb4d87aaa6fb1e897,
    0x788573e8b92dbbfb,
    0x02527b137b0878c1,
    0x1870a7c8dee9f4f5,
    0x39b99e40910a224b,
    0x45821c0abd4df247,
    0x10fe2b2f50e02fb3,
    0x5762b90c043f0345,
    0x82a67b9193b27bbd,
    0xa6e914e28ec37693,
    0x835d9a4facaf445f,
    0x48def8175884f82d,
    0xae900e2d7c9a6f7b,
    0x1c08431bdd18be89,
    0xb370a66d684fd83f,
    0xb4be33e18f93b279,
    0x310c50872a7dd5e3,
    0x447ab1281276697d,
    0xc2f122216b2a6c21,
    0xab99c8b5ae1c3059,
    0xb78e17a2227d593b,
    0xabf97d03f7269c5d,
    0x867aefc9fdbfe7ff,
    0xf7f7ad182e47d5b7,
    0x50dff95a9847721b,
    0xe4cb8a0e83cb6a35,
    0x8da72ecdf9247a1d,
    0xc5b04bfc87f31d87,
    0xe2dcf622ea2b00f3,
    0xb9ce9f2e4972f46b,
    0x1ed785c911bf59f9,
    0x4ddb8a4eed70e085,
    0x81e93b4df68c24fd,
    0xee0d0812afcd8357,
    0xf62e3ba72268a891,
    0x3194d367c8154147,
    0xd096ede8e30c20d5,
    0xd68624d27b87a77f,
    0xb728fcdc11c8204d,
    0x9d6b6038077e066f,
    0xaa732d7a4a360d93,
    0x36af98a423972db5,
    0xc31d00da12940f17,
    0xed85352107410b25,
    0x829c85ee6db8567b,
    0xef60258952cc6d89,
    0xcf28c2e0da787741,
    0x57567d8494af28f7,
    0x2c7c98518f174031,
    0xb28b363a36825ae7,
    0xed1ffeb64f9ae769,
    0xcbbb0115e9b9a31f,
    0x8d3c5fecb7f9e4ed,
    0x816271698195cfc3,
    0x9ac939d1c2b1d35d,
    0xdd9fb7017b0ec455,
    0xc94cab1e57276e3d,
    0x8b8806b117c79913,
    0xa9e63292a3269fd1,
    0x76da5710f1e989fb,
    0xdded6688d83a918d,
    0x4e446b6a305428f9,
    0x4ddaca7a3696cfb1,
    0x7eadc4eb87f26ed3,
    0x76c13a0ff04c00c9,
    0xcbf800504d2a2681,
    0x0731dada6c4fec9b,
    0xbcb52a664e63f627,
    0xf1f9abda071c2aa9,
    0xf262ffa620ffe20b,
    0x93774a3d57199a99,
    0xfb3541cd467a1903,
    0x6828cab6b4fe8f51,
    0x12ac03e3d624cc9d,
    0x6363bd1e9bb7d7f7,
    0x334cfd676a484d2f,
    0xd511acd86f143a53,
    0x73fc2490e0062be5,
    0x10780dda36b78b55,
    0xabf601274064e0ab,
    0x3ef3e4ca27e4a2cd,
    0x9216a26e690a16f1,
    0xbae4849e6034bda7,
    0xf943a0520e01e9e1,
    0x7c89958f48f6658b,
    0xe67128750e0545a5,
    0xc6c9e1d414516ccf,
    0x805307f996e9e81d,
    0x3eddd2cff46ad5bf,
    0x35582c1aeb5aae85,
    0x4973c88573ef6eb1,
    0x3063f627c1e715d9,
    0x711ad679a8dcc243,
    0x51c224a17a3db4b3,
    0x612325ca50ddaed5,
    0x9929a7b6b7958b37,
    0xa78d222e5a857bb9,
    0x3ad0ffe3198d139b,
    0x08b4659ac547ed17,
    0x1752e8904aff1003,
    0x60745c37ee4e5925,
    0x29e2da1f6557ee51,
    0x80d78c24ac49cb89,
    0xc56c3b495c8d1f79,
    0xcf5bdf9f5088ac2f,
    0x8a44800e4fae4e7d,
    0xdd76384277e578e7,
    0x20b1562d2703facb,
    0xef56caf96e9d8e3b,
    0xf54061416aede033,
    0xe0bc78c21a26e4ff,
    0x0524f92731a179cd,
    0x5d3b4ad7deafec8b,
    0x508828f744da88ad,
    0x6e82014031710bcf,
    0xadf3b77a22595dd1,
    0x0d8f0c03f7ea8a87,
    0x2c49e3483c3a05f3,
    0xccecbc98c91274c1,
    0x273a08941bb71e77,
    0xdead5a1e3f341baf,
    0x83eee092593309fd,
    0x4af5f1bd3ae87ce9,
    0x4ca85ad2301c9e6d,
    0x1b19592cd31a3943,
    0x3e7aa05e6dcd81bb,
    0x86336cecb02ba47f,
    0xa96b30d0c8a44b2b,
    0xb7c63fa0cfca0571,
    0x8eaf59b405a642b5,
    0xdf29e9cbb536dc17,
    0xed14132c82c1d43f,
    0xaf68778e34caab0d,
    0xa4f04a3368941d31,
    0xe9960969357c07e7,
    0xeb47b62b7360b469,
    0x64c653d779ae730f,
    0x479702d3319915c7,
    0xef3c3eebc6803239,
    0x93807b1a2e3c0e1b,
    0x8167e33e3f478029,
    0x60cb76e38c339397,
    0xae34788ffe4bc283,
    0x4b6246a0c6c093a5,
    0x872e594b12b03efb,
    0xbc0ae83ce9045b15,
    0x0ad30a3917e0968d,
    0x124ef5a4e1c7cd63,
    0x5b98fe0e9fe17aaf,
    0x414306cfe45400fd,
    0xa06d1b4fd391e8bb,
    0x11939803a60c2381,
    0x668c11cc37ea6b23,
    0x83f9b2089dc10645,
    0x65dc8ae47af277a9,
    0x6e2368b9c685770b,
    0x3ea137aeba5a6b2d,
    0x735f57adca48f19d,
    0x69a8de0ba1b18107,
    0x8fb84bdf5822bd79,
    0xb8fab3b748562721,
    0xa6c658ea10a65c3b,
    0xe56381f33ab5e549,
    0xe3c224da14988139,
    0x438c253e6d99f513,
    0xc1b99f8841a3a6e1,
    0x63fa18c79c54fa8b,
    0xe7f6f609619d0d1d,
    0x7b39ef3b70afc109,
    0x73922c61ca7452bf,
    0x28d96828332372c1,
    0x6b6e92968c4e8463,
    0x571861f084962edb,
    0xd935c64f140f1ef5,
    0x96459f8fd72a4c4b,
    0x410ba9a2a18242d9,
    0xcf90979f89870391,
    0x10f94ff26bc00add,
    0xa6619fbb9da139b3,
    0x765a23334efb03d5,
    0x6f2f613b5e631837,
    0x666b99bfbcd368b9,
    0x922b78eb01ed45bd,
    0x7079a199c31de6a9,
    0xa181abcda167be5f,
    0x2f6dbbcab3a9822d,
    0xc5a83ff0e43eba17,
    0x28c68613dda7d97b,
    0x5cf33ed49efa5007,
    0x9125fdead661590d,
    0xaee67f478c7325e7,
    0x735b1274a0e89653,
    0x733b56eae1a4e621,
    0x1944ffb316ffe65d,
    0xf26bc3cfd2a01449,
    0xb5827ba68b83e201,
    0xac139507e48eefb7,
    0xeb7676b25834fda3,
    0xad898f4763da5c1b,
    0xea906f224398f9a7,
    0xa8aff3caca28cdad,
    0x46c53aa36b19b083,
    0x9ada32b09603e8cf,
    0xd31f842ef5d8e915,
    0x6124af44730a33f9,
    0x828ec4c2b6e64a85,
    0x3d6f49df999638af,
    0x7641460a0ea89b65,
    0x97703f98fb7fe291,
    0xd343c209e3e6b7b9,
    0x4e5fc01f6a41406f,
    0xb78a05b08aa4bcbd,
    0x3434a14919d34561,
    0xccead7dee120f525,
    0xe1375a2bccd87673,
    0xf727d51420a57141,
    0x2c3b68cfbcebb00d,
    0xda91e2f3e17542f7,
    0xb55f6100ae95d6e3,
    0x6a0c608e0bbaa975,
    0xac5f2fc151c016cb,
    0xb1e5af8146e4d00f,
    0x6e283d3b112602c7,
    0xf9a48bcb76c96e55,
    0xa776780ca4c0e101,
    0x8d40a2d47d99c7c5,
    0x4ed9d8a7aedcefef,
    0x55c5cf9586072313,
    0x62c640e386ef1f09,
    0xdb876e7feb8b02f9,
    0x5b85ac1558bdf263,
    0xb2b13930c2a889b1,
    0xdf53c897124f8c57,
    0x68a69390fdce78dd,
    0x5a1e8f0261e6e7b3,
    0xcce38a9ccaab014d,
    0xb0cd4811fe6a8171,
    0x911c24573e445027,
    0x9e86401e61cac4a9,
    0xfd2731405f265eb5,
    0x3f4c00205c05b02d,
    0xe92e3d0a829a974f,
    0xdec216e5aa47169d,
    0xa0397bf3448bcd73,
    0x9ecf538d7efa905b,
    0xb1037b5f84886421,
];
/// Limit of modular inverse muliplied by the target integer. Used together with SMALL_PRIMES_INV
#[cfg(feature = "big-table")]
pub const SMALL_PRIMES_INVLIM: [u64; 1024] = [
    0x7fffffffffffffff,
    0x5555555555555555,
    0x3333333333333333,
    0x2492492492492492,
    0x1745d1745d1745d1,
    0x13b13b13b13b13b1,
    0x0f0f0f0f0f0f0f0f,
    0x0d79435e50d79435,
    0x0b21642c8590b216,
    0x08d3dcb08d3dcb08,
    0x0842108421084210,
    0x06eb3e45306eb3e4,
    0x063e7063e7063e70,
    0x05f417d05f417d05,
    0x0572620ae4c415c9,
    0x04d4873ecade304d,
    0x0456c797dd49c341,
    0x04325c53ef368eb0,
    0x03d226357e16ece5,
    0x039b0ad12073615a,
    0x0381c0e070381c0e,
    0x033d91d2a2067b23,
    0x03159721ed7e7534,
    0x02e05c0b81702e05,
    0x02a3a0fd5c5f02a3,
    0x0288df0cac5b3f5d,
    0x027c45979c95204f,
    0x02647c69456217ec,
    0x02593f69b02593f6,
    0x0243f6f0243f6f02,
    0x0204081020408102,
    0x01f44659e4a42715,
    0x01de5d6e3f8868a4,
    0x01d77b654b82c339,
    0x01b7d6c3dda338b2,
    0x01b2036406c80d90,
    0x01a16d3f97a4b01a,
    0x01920fb49d0e228d,
    0x01886e5f0abb0499,
    0x017ad2208e0ecc35,
    0x016e1f76b4337c6c,
    0x016a13cd15372904,
    0x01571ed3c506b39a,
    0x015390948f40feac,
    0x014cab88725af6e7,
    0x0149539e3b2d066e,
    0x013698df3de07479,
    0x0125e22708092f11,
    0x0120b470c67c0d88,
    0x011e2ef3b3fb8744,
    0x0119453808ca29c0,
    0x0112358e75d30336,
    0x010fef010fef010f,
    0x0105197f7d734041,
    0x00ff00ff00ff00ff,
    0x00f92fb2211855a8,
    0x00f3a0d52cba8723,
    0x00f1d48bcee0d399,
    0x00ec979118f3fc4d,
    0x00e939651fe2d8d3,
    0x00e79372e225fe30,
    0x00dfac1f74346c57,
    0x00d578e97c3f5fe5,
    0x00d2ba083b445250,
    0x00d161543e28e502,
    0x00cebcf8bb5b4169,
    0x00c5fe740317f9d0,
    0x00c2780613c0309e,
    0x00bcdd535db1cc5b,
    0x00bbc8408cd63069,
    0x00b9a7862a0ff465,
    0x00b68d31340e4307,
    0x00b2927c29da5519,
    0x00afb321a1496fdf,
    0x00aceb0f891e6551,
    0x00ab1cbdd3e2970f,
    0x00a87917088e262b,
    0x00a513fd6bb00a51,
    0x00a36e71a2cb0331,
    0x00a03c1688732b30,
    0x009c69169b30446d,
    0x009baade8e4a2f6e,
    0x00980e4156201301,
    0x00975a750ff68a58,
    0x009548e4979e0829,
    0x0093efd1c50e726b,
    0x0091f5bcb8bb02d9,
    0x008f67a1e3fdc261,
    0x008e2917e0e702c6,
    0x008d8be33f95d715,
    0x008c55841c815ed5,
    0x0088d180cd3a4133,
    0x00869222b1acf1ce,
    0x0085797b917765ab,
    0x008355ace3c897db,
    0x00824a4e60b3262b,
    0x0080c121b28bd1ba,
    0x007dc9f3397d4c29,
    0x007d4ece8fe88139,
    0x0079237d65bcce50,
    0x0077cf53c5f7936c,
    0x0075a8accfbdd11e,
    0x007467ac557c228e,
    0x00732d70ed8db8e9,
    0x0072c62a24c3797f,
    0x007194a17f55a10d,
    0x006fa549b41da7e7,
    0x006e8419e6f61221,
    0x006d68b5356c207b,
    0x006d0b803685c01b,
    0x006bf790a8b2d207,
    0x006ae907ef4b96c2,
    0x006a37991a23aead,
    0x0069dfbdd4295b66,
    0x0067dc4c45c8033e,
    0x00663d80ff99c27f,
    0x0065ec17e3559948,
    0x00654ac835cfba5c,
    0x00645c854ae10772,
    0x006372990e5f901f,
    0x006325913c07beef,
    0x006160ff9e9f0061,
    0x0060cdb520e5e88e,
    0x005ff4017fd005ff,
    0x005ed79e31a4dccd,
    0x005d7d42d48ac5ef,
    0x005c6f35ccba5028,
    0x005b2618ec6ad0a5,
    0x005a2553748e42e7,
    0x0059686cf744cd5b,
    0x0058ae97bab79976,
    0x0058345f1876865f,
    0x005743d5bb24795a,
    0x005692c4d1ab74ab,
    0x00561e46a4d5f337,
    0x005538ed06533997,
    0x0054c807f2c0bec2,
    0x005345efbc572d36,
    0x00523a758f941345,
    0x005102370f816c89,
    0x0050cf129fb94acf,
    0x004fd31941cafdd1,
    0x004fa1704aa75945,
    0x004f3ed6d45a63ad,
    0x004f0de57154ebed,
    0x004e1cae8815f811,
    0x004cd47ba5f6ff19,
    0x004c78ae734df709,
    0x004c4b19ed85cfb8,
    0x004bf093221d1218,
    0x004aba3c21dc633f,
    0x004a6360c344de00,
    0x004a383e9f74d68a,
    0x0049e28fbabb9940,
    0x0048417b57c78cd7,
    0x0047f043713f3a2b,
    0x00474ff2a10281cf,
    0x00468b6f9a978f91,
    0x0045f13f1caff2e2,
    0x0045a5228cec23e9,
    0x0045342c556c66b9,
    0x0044c4a23feeced7,
    0x0043c5c20d3c9fe6,
    0x00437e494b239798,
    0x0043142d118e47cb,
    0x0042ab5c73a13458,
    0x004221950db0f3db,
    0x0041bbb2f80a4553,
    0x0040f391612c6680,
    0x0040b1e94173fefd,
    0x004050647d9d0445,
    0x004030241b144f3b,
    0x003f90c2ab542cb1,
    0x003f71412d59f597,
    0x003f137701b98841,
    0x003e79886b60e278,
    0x003e5b1916a7181d,
    0x003dc4a50968f524,
    0x003da6e4c9550321,
    0x003d4e4f06f1def3,
    0x003c4a6bdd24f9a4,
    0x003c11d54b525c73,
    0x003bf5b1c5721065,
    0x003bbdb9862f23b4,
    0x003b6a8801db5440,
    0x003b183cf0fed886,
    0x003aabe394bdc3f4,
    0x003a5ba3e76156da,
    0x003a0c3e953378db,
    0x0038f03561320b1e,
    0x0038d6ecaef5908a,
    0x003859cf221e6069,
    0x0037f7415dc9588a,
    0x00377df0d3902626,
    0x00373622136907fa,
    0x0036ef0c3b39b92f,
    0x0036915f47d55e6d,
    0x0036072cf3f866fd,
    0x0035d9b737be5ea8,
    0x0035961559cc81c7,
    0x0035531c897a4592,
    0x00353ceebd3e98a4,
    0x0034fad381585e5e,
    0x00347884d1103130,
    0x00340dd3ac39bf56,
    0x003351fdfecc140c,
    0x00333d72b089b524,
    0x0033148d44d6b261,
    0x0032d7aef8412458,
    0x0032c3850e79c0f1,
    0x00328766d59048a2,
    0x00325fa18cb11833,
    0x00324bd659327e22,
    0x0032246e784360f4,
    0x0031afa5f1a33a08,
    0x00319c63ff398e70,
    0x003162f7519a86a7,
    0x0030271fc9d3fc3c,
    0x002ff104ae89750b,
    0x002fbb62a236d133,
    0x002f74997d2070b4,
    0x002ed84aa8b6fce3,
    0x002e832df7a46dbd,
    0x002e0e0846857cab,
    0x002decfbdfb55ee6,
    0x002ddc876f3ff488,
    0x002dbbc1d4c482c4,
    0x002d8af0e0de0556,
    0x002d4a7b7d14b30a,
    0x002d2a85073bcf4e,
    0x002d1a9ab13e8be4,
    0x002ceb1eb4b9fd8b,
    0x002c8d503a79794c,
    0x002c404d708784ed,
    0x002c31066315ec52,
    0x002c1297d80f2664,
    0x002c037044c55f6b,
    0x002be5404cd13086,
    0x002bb845adaf0cce,
    0x002b5f62c639f16d,
    0x002b07e6734f2b88,
    0x002ace569d8342b7,
    0x002a791d5dbd4dcf,
    0x002a4eff8113017c,
    0x002a3319e156df32,
    0x002a0986286526ea,
    0x0029d29551d91e39,
    0x0029b7529e109f0a,
    0x00298137491ea465,
    0x0029665e1eb9f9da,
    0x002909752e019a5e,
    0x0028ef35e2e5efb0,
    0x0028c815aa4b8278,
    0x0028bb1b867199da,
    0x0028a13ff5d7b002,
    0x00287ab3f173e755,
    0x00286dead67713bd,
    0x002847bfcda6503e,
    0x002808c1ea6b4777,
    0x00278d0e0f23ff61,
    0x002768863c093c7f,
    0x0027505115a73ca8,
    0x00274441a61dc1b9,
    0x0026b5c166113cf0,
    0x00269e65ad07b18e,
    0x002692c25f877560,
    0x002658fa7523cd11,
    0x0026148710cf0f9e,
    0x002609363b22524f,
    0x0025d1065a1c1122,
    0x0025a48a382b863f,
    0x0025837190eccdbc,
    0x00256292e95d510c,
    0x002541eda98d068c,
    0x0024e15087fed8f5,
    0x0024c18b20979e5d,
    0x0024ac7b336de0c5,
    0x0024a1fc478c60bb,
    0x002463801231c009,
    0x0024300fd506ed33,
    0x0023f314a494da81,
    0x0023cadedd2fad3a,
    0x00237b7ed2664a03,
    0x0023372967dbaf1d,
    0x00231a308a371f20,
    0x002306fa63e1e600,
    0x0022fd6731575684,
    0x0022ea507805749c,
    0x0022e0cce8b3d720,
    0x0022b1887857d161,
    0x00227977fcc49cc0,
    0x00225db37b5e5f4f,
    0x0022421b91322ed6,
    0x0021f05b35f52102,
    0x0021e75de5c70d60,
    0x0021a01d6c19be96,
    0x0021974a6615c81a,
    0x00213767697cf36a,
    0x00211d9f7fad35f1,
    0x0020fb7d9dd36c18,
    0x0020e2123d661e0e,
    0x0020d135b66ae990,
    0x0020c8cded4d7a8e,
    0x0020b80b3f43ddbf,
    0x002096b9180f46a6,
    0x00207de7e28de5da,
    0x002054dec8cf1fb3,
    0x00204cb630b3aab5,
    0x00202428adc37beb,
    0x001fec0c7834def4,
    0x001fc46fae98a1d0,
    0x001facda430ff619,
    0x001f7e17dd8e15e5,
    0x001f765a3556a4ee,
    0x001f66ea49d802f1,
    0x001f5f3800faf9c0,
    0x001f38f4e6c0f1f9,
    0x001f0b8546752578,
    0x001f03ff83f001f0,
    0x001ec853b0a3883c,
    0x001ec0ee573723eb,
    0x001eaad38e6f6894,
    0x001e9c28a765fe53,
    0x001e94d8758c2003,
    0x001e707ba8f65e68,
    0x001e53a2a68f574e,
    0x001e1380a56b438d,
    0x001dbf9f513a3802,
    0x001db1d1d58bc600,
    0x001d9d358f53de38,
    0x001d81e6df6165c7,
    0x001d4bdf7fd40e30,
    0x001d452c7a1c958d,
    0x001d37cf9b902659,
    0x001d1d3a5791e97b,
    0x001ce89fe6b47416,
    0x001ce219f3235071,
    0x001cd516dcf92139,
    0x001cbb33bd1c2b8b,
    0x001ca7e7d2546688,
    0x001c94b5c1b3dbd3,
    0x001c87f7f9c241c1,
    0x001c6202706c35a9,
    0x001c5bb8a9437632,
    0x001c174343b4111e,
    0x001c04d0d3e46b42,
    0x001bfeb00fbf4308,
    0x001bec5dce0b202d,
    0x001be03444620037,
    0x001bce09c66f6fc3,
    0x001ba40228d02b30,
    0x001b9225b1cf8919,
    0x001b864a2ff3f53f,
    0x001b80604150e49b,
    0x001b6eb1aaeaacf3,
    0x001b62f48da3c8cc,
    0x001b516babe96092,
    0x001b2e9cef1e0c87,
    0x001b1d56bedc849b,
    0x001b0c267546aec0,
    0x001ae45f62024fa0,
    0x001ad917631b5f54,
    0x001ac83d18cb608f,
    0x001aa6c7ad8c063f,
    0x001a90a7b1228e2a,
    0x001a8027c03ba059,
    0x001a7533289deb89,
    0x001a2ed7ce16b49f,
    0x0019fefc0a279a73,
    0x0019e4b0cd873b5f,
    0x0019cfcdfd60e514,
    0x0019c56932d66c85,
    0x0019b5e1ab6fc7c2,
    0x0019b0b8a62f2a73,
    0x0019a149fc98942c,
    0x001969517ec25b85,
    0x00194b3083360ba8,
    0x00194631f4bebdc1,
    0x00191e84127268fd,
    0x00190adbb543984f,
    0x001901130bd18200,
    0x0018e3e6b889ac94,
    0x0018c233420e1ec1,
    0x0018aa5872d92bd6,
    0x0018a5989945ccf9,
    0x00189c1e60b57f60,
    0x0018893fbc8690b9,
    0x00187b2bb3e1041c,
    0x00186d27c9cdcfb8,
    0x001863d8bf4f2c1c,
    0x00185f33e2ad7593,
    0x001855ef75973e13,
    0x001848160153f134,
    0x001835b72e6f0656,
    0x00182c922d83eb39,
    0x0018280243c0365a,
    0x00181a5cd5898e73,
    0x001803c0961773aa,
    0x0017ff4005ffd001,
    0x0017e8d670433edb,
    0x0017d7066cf4bb5d,
    0x0017ce285b806b1f,
    0x0017af52cdf27e02,
    0x0017997d47d01039,
    0x00177f7ec2c6d0ba,
    0x00177b2f3cd00756,
    0x00176e4a22f692a0,
    0x001765b94271e11b,
    0x001761732b044ae4,
    0x00173f7a5300a2bc,
    0x001722112b48be1f,
    0x001719b7a16eb843,
    0x00170d3c99cc5052,
    0x0016fcad7aed3bb6,
    0x0016f051b8231ffd,
    0x0016e81beae20643,
    0x0016c3721584c1d8,
    0x0016b34c2ba09663,
    0x00169f3ce292ddcd,
    0x00169344b2220a0d,
    0x001687592593c1b1,
    0x00167787f1418ec9,
    0x001663e190395ff2,
    0x00164c7a4b6eb5b3,
    0x0016316a061182fd,
    0x001629ba914584e4,
    0x00161e3d57de21b2,
    0x001612cc01b977f0,
    0x00160efe30c525ff,
    0x0015da45249ec5de,
    0x0015d68ab4acff92,
    0x0015c3f989d1eb15,
    0x0015b535ad11b8f0,
    0x0015addb3f424ec1,
    0x00159445cb91be6b,
    0x00158d0199771e63,
    0x00157e87d9b69e04,
    0x001568f58bc01ac3,
    0x00155e3c993fda9b,
    0x001548eacc5e1e6e,
    0x001541d8f91ba6a7,
    0x00153747060cc340,
    0x001514569f93f7c4,
    0x00150309705d3d79,
    0x0014ff97020cf5bf,
    0x0014e42c114cf47e,
    0x0014b835bdcb6447,
    0x0014b182b53a9ab7,
    0x0014ae2ad094a3d3,
    0x00149a320ea59f96,
    0x001490441de1a2fb,
    0x001489aacce57200,
    0x001475f82ad6ff99,
    0x00146c2cfe53204f,
    0x00145f2ca490d4a1,
    0x001458b2aae0ec87,
    0x00144bcb0a3a3150,
    0x001428a1e65441d4,
    0x00142575a6c210d7,
    0x00141f2025ba5c46,
    0x00141bf6e35420fd,
    0x001409141d1d313a,
    0x0013dd8bc19c3513,
    0x0013da76f714dc8f,
    0x0013d13e50f8f49e,
    0x0013c80e37ca3819,
    0x0013bee69fa99ccf,
    0x0013b8d0ede55835,
    0x0013afb7680bb054,
    0x0013acb0c3841c96,
    0x00139a9c5f434fde,
    0x0013949cf33a0d9d,
    0x001382b4a00c31b0,
    0x00137fbbc0eedcbb,
    0x001370ecf047b069,
    0x00136df9790e3155,
    0x0013567dd8defd5b,
    0x0013539261fdbc34,
    0x00133c564292d28a,
    0x001333ae178d6388,
    0x0013170ad00d1fd7,
    0x0013005f01db0947,
    0x0012f51d40342210,
    0x0012ef815e4ed950,
    0x0012ecb4abccd827,
    0x0012e71dc1d3d820,
    0x0012e45389a16495,
    0x0012c5d9226476cc,
    0x0012badc391156fd,
    0x0012aa78e412f522,
    0x0012a251f5f47fd1,
    0x001294cb85c53534,
    0x0012921963beb65e,
    0x00128cb777c69ca8,
    0x001284aa6cf07294,
    0x001281fcf6ac7f87,
    0x001279f937367db9,
    0x00126cad0488be94,
    0x00126a06794646a2,
    0x00125a2f2bcd3e95,
    0x00124d108389e6b1,
    0x00124a73083771ac,
    0x00123d6acda0620a,
    0x00122b4b2917eafd,
    0x00122391bfce1e2f,
    0x00121e6f1ea579f2,
    0x001216c09e471568,
    0x00120c8cb9d93909,
    0x001204ed58e64ef9,
    0x0011fd546578f00c,
    0x0011e9310b8b4c9c,
    0x0011da3405db9911,
    0x0011d7b6f4eb055d,
    0x0011d2bee748c145,
    0x0011c1706ddce7a7,
    0x0011ba0fed2a4f14,
    0x0011b528538ed64a,
    0x0011ab61404242ac,
    0x00119f378ce81d2f,
    0x001195889ece79da,
    0x00118e4c65387077,
    0x001187161d70e725,
    0x00116cd6d1c85239,
    0x001165bbe7ce86b1,
    0x0011635ee344ce36,
    0x0011579767b6d679,
    0x00114734711e2b54,
    0x0011428b90147f05,
    0x00113b92f3021636,
    0x001126cabc886884,
    0x0011247eb1b85976,
    0x0011190bb01efd65,
    0x0011091de0fd679c,
    0x001104963c7e4e0b,
    0x00110253516420b0,
    0x0010f70db7c41797,
    0x0010e75ee2bf9ecd,
    0x0010e2e91c6e0676,
    0x0010da049b9d428d,
    0x0010c6248fe3b1a2,
    0x0010c1c03ed690eb,
    0x0010bb2e1379e3a2,
    0x0010b8fe7f61228e,
    0x0010b4a10d60a4f7,
    0x0010ae192681ec0f,
    0x0010abecfbe5b0ae,
    0x00109eefd568b96d,
    0x00109a9ff178b40c,
    0x00108531e22f9ff9,
    0x00106ddec1af4417,
    0x0010614174a4911d,
    0x00105f291f0448e7,
    0x00105afa0ef32891,
    0x001054b777bd2530,
    0x00104e79a97fb69e,
    0x00104c661eafd845,
    0x0010462ea939c933,
    0x00102f8baa442836,
    0x00102d7ff7e94004,
    0x0010275ff9f13c02,
    0x001017213fcbb4d3,
    0x00101112234579d1,
    0x00100501907d271c,
    0x00100300901b0510,
    0x000ffd008fe5050f,
    0x000ff10e02dd5084,
    0x000fe13b9c80c67f,
    0x000fdf4384be37ad,
    0x000fdb54cbe8766e,
    0x000fd5725ca6ff32,
    0x000fc7c84684c6fb,
    0x000fc3e5265dbaa8,
    0x000fc1f44e0cae12,
    0x000fb0921c50a7af,
    0x000f999fd70cbc6b,
    0x000f9023fd5339d0,
    0x000f8a78ce671475,
    0x000f8895fee86574,
    0x000f7f2ecb084b10,
    0x000f7d4eb7d10c29,
    0x000f73f52277a3c3,
    0x000f7217c598961c,
    0x000f68cbb1448f42,
    0x000f633d0276e4c5,
    0x000f6163ac20ec79,
    0x000f582ba2bc16c6,
    0x000f5654f43290a0,
    0x000f4d2a23810bc6,
    0x000f47af4d6a2f27,
    0x000f4066f2b6e652,
    0x000f2555048e3a92,
    0x000f1c64588a5bf6,
    0x000f1a9be09cb411,
    0x000f11b7d5259d39,
    0x000f0aa284e7f802,
    0x000f0556e5e3b7f2,
    0x000efc8bcbc808e5,
    0x000eecd1a690efbb,
    0x000ee79aed6d65f2,
    0x000edd386114d83a,
    0x000ed2e44366e5e2,
    0x000ed12cf8e17f64,
    0x000ec1cd284b2b2d,
    0x000ebcb44cadda1e,
    0x000eb9505943771d,
    0x000eb43d57efeadc,
    0x000eaf2dd4c00b03,
    0x000ea0141c1ba6a6,
    0x000e9e68805f05a7,
    0x000e96142b87e431,
    0x000e8a7acd811b8c,
    0x000e8587db3e001d,
    0x000e823d186d44dc,
    0x000e8098463ee194,
    0x000e7d4fbfb3ee1d,
    0x000e69bba6981ffa,
    0x000e681c5cf7d707,
    0x000e5e684930e334,
    0x000e5993247dc92d,
    0x000e4cbfee201016,
    0x000e465ee7daf979,
    0x000e4199de07af5c,
    0x000e3cd8031d4f40,
    0x000e2ea56c157eb2,
    0x000e221e5d4d3c73,
    0x000e208f09a841c7,
    0x000e1d716a945161,
    0x000e18c78ec8fd4d,
    0x000e173a4a162079,
    0x000e1294881bb494,
    0x000e0df1d5f24661,
    0x000e063ec7f50b1e,
    0x000e01a4313dc53d,
    0x000df8780f47c350,
    0x000def57e8eb9666,
    0x000de1bdf3f63d46,
    0x000de03cb5099809,
    0x000ddbbaecc84bc9,
    0x000dd8bb5ca73db6,
    0x000dcb4d529a6e07,
    0x000dc55da73dea60,
    0x000db3ad2585011f,
    0x000db0becf636a79,
    0x000daf481ca6fefb,
    0x000dac5ba7565dae,
    0x000da7fb4e419d19,
    0x000da6867a88d327,
    0x000d9dd005f50b02,
    0x000d9aeb01f763f7,
    0x000d90d31dd5804a,
    0x000d7b6453358f31,
    0x000d744e69d900e4,
    0x000d7011a317260e,
    0x000d67a0126e7c19,
    0x000d5dd39e775bd7,
    0x000d59a4f2990168,
    0x000d52b24cb6269d,
    0x000d4a6571da4f04,
    0x000d49044eac6581,
    0x000d4642e40d1129,
    0x000d4222e81fe723,
    0x000d3ca6e8c89f41,
    0x000d388ce29d4edc,
    0x000d31bc7b7d8013,
    0x000d306071c13fd5,
    0x000d2da935479b1a,
    0x000d2430aa043597,
    0x000d2025bc6c7db7,
    0x000d1c1d4ad1732b,
    0x000d196e5f46f8c8,
    0x000d156a0c9293e8,
    0x000d1413d26e0aee,
    0x000d0d68c6a4128f,
    0x000d0c142eaf3837,
    0x000d01792ab9d70d,
    0x000cf990317775bc,
    0x000cf44f8c38790a,
    0x000ce88d96d10e45,
    0x000ce5f39b07e906,
    0x000ce20e98148847,
    0x000cda4b9c30ccd7,
    0x000cd9015ae32495,
    0x000cd524244aca36,
    0x000cd14940099cf6,
    0x000ccd70ac089a07,
    0x000cbb9c535c4371,
    0x000cb7d0b46fe0ff,
    0x000caefe5d7135f4,
    0x000cac7b5f00f0cd,
    0x000ca7785ceddbea,
    0x000ca13a2a86e1db,
    0x000c9c4009753007,
    0x000c94d02e64bfab,
    0x000c89b8c9c875ef,
    0x000c87447737277e,
    0x000c860aaa2514e3,
    0x000c8397c813f1b9,
    0x000c74fa805d6d56,
    0x000c6db8a1f5cdfe,
    0x000c6a1add9e2398,
    0x000c68e6be826648,
    0x000c5f4e25fc9df0,
    0x000c5bb8bf2ad1cd,
    0x000c58256b316ced,
    0x000c4fd5ad917b5b,
    0x000c49ecb3ea4d7a,
    0x000c41b00b7d950a,
    0x000c3f57990b87a1,
    0x000c2ddcb31250f8,
    0x000c2a63b3651432,
    0x000c26ecae1db72e,
    0x000c2377a18c051e,
    0x000c1ede9efcec29,
    0x000c1b6e258d13a0,
    0x000c19243f5399bb,
    0x000c17ff9f400305,
    0x000c112865703b94,
    0x000c0dbfaea33225,
    0x000c0b7af12ddfb9,
    0x000c0a58e464462c,
    0x000c06f40512eef2,
    0x000bfa9275a2b247,
    0x000bf7367402cdf0,
    0x000bf61833f4f921,
    0x000bf3dc543a74a1,
    0x000be9d9302a7115,
    0x000be8bd6e051e01,
    0x000be6868804d5a6,
    0x000bdfe6c4359f0e,
    0x000bdeccdb0b5c3a,
    0x000bdb8058ee429a,
    0x000bd94e5c1b371f,
    0x000bcb1d293b1af3,
    0x000bc7db8db0c1a5,
    0x000bc49bbdfd2662,
    0x000bc2723240f402,
    0x000bbe217c2b7c13,
    0x000bb8c10aab27b2,
    0x000ba7ad528a7e79,
    0x000b9f3611b48c5e,
    0x000b9e2806e5e7c4,
    0x000b9aff0c4913fe,
    0x000b98e4aedd581c,
    0x000b97d7c94b7dc2,
    0x000b95be902d9d9e,
    0x000b94b23c872b90,
    0x000b8f77714d15a1,
    0x000b882d0beff6a1,
    0x000b850ff9852703,
    0x000b82fd86db8806,
    0x000b7edadd32f76c,
    0x000b79b3b4df3b7b,
    0x000b769e6d59833f,
    0x000b6c636b5141ff,
    0x000b6a59ceae8801,
    0x000b6955461e38f7,
    0x000b6648c2dc6bc2,
    0x000b572282260209,
    0x000b552072bde889,
    0x000b511e7552f9c4,
    0x000b4c1ff34a5c0e,
    0x000b4922f58d4aa2,
    0x000b46278c16b967,
    0x000b42301cd99b49,
    0x000b3f385dd77e4e,
    0x000b394d8ef8f0f6,
    0x000b375601507c14,
    0x000b3463f76be376,
    0x000b3368f6c4a07c,
    0x000b3078fc1c25f0,
    0x000b2e84854e93e5,
    0x000b2b971aa909a4,
    0x000b2a9da39d6bc8,
    0x000b25c0dc29a0fc,
    0x000b24c8698449a7,
    0x000b1d0ae579aefe,
    0x000b1a2698ea2f9e,
    0x000b108dc4186078,
    0x000b0ea463b00212,
    0x000b08ec37007962,
    0x000b024778cc023c,
    0x000af515df36a88e,
    0x000af24635f6561e,
    0x000ae8f1b92baeaf,
    0x000ae715eee11f8e,
    0x000adec0b0a3bb36,
    0x000adb10aa4c956f,
    0x000ad84e49752245,
    0x000ad6782597f0c2,
    0x000ad3b81a0d72fe,
    0x000acd52beced79e,
    0x000aca9755063254,
    0x000ac7dd4cafb12a,
    0x000ac354f80dca44,
    0x000ac26d5c2b8ad2,
    0x000abde997dabd3d,
    0x000ab883aa1100a0,
    0x000ab4ed637f5a0b,
    0x000ab074e9febf52,
    0x000aaf90778c2039,
    0x000aab1c7684f034,
    0x000aa78f20ebbb3e,
    0x000aa23f8dafd4cc,
    0x000a9dd69cad5934,
    0x000a935004a07302,
    0x000a9270690f3d14,
    0x000a90b1a0aa5d30,
    0x000a8d35c9d731e9,
    0x000a8a9a6a51f16c,
    0x000a88de370f596b,
    0x000a856786adae36,
    0x000a7da4c77d3161,
    0x000a7959f863d4a1,
    0x000a76c85e80c195,
    0x000a743806dc44c4,
    0x000a735d866dfa0a,
    0x000a70cedb02531e,
    0x000a6c8e842c770f,
    0x000a67791215dd74,
    0x000a66a0a51d363d,
    0x000a626893011861,
    0x000a5fe22c55c089,
    0x000a5d5cffb77275,
    0x000a5ad90c4186e5,
    0x000a578057e7c2eb,
    0x000a54ff3bb10e91,
    0x000a50d5683edc94,
    0x000a4e57854b3df4,
    0x000a4d8328c4b800,
    0x000a4b06e01d97b3,
    0x000a488bca2c4449,
    0x000a4611e6132ed5,
    0x000a41f40f39e646,
    0x000a3eab5c3e44e9,
    0x000a34ddd50561e0,
    0x000a326d60e94186,
    0x000a2985a81ce614,
    0x000a28b72e26f82e,
    0x000a217aa3479693,
    0x000a1fe05c62df4b,
    0x000a1cad538aebf9,
    0x000a18b05f490083,
    0x000a0ccc4c28fc31,
    0x000a09a544d01ffe,
    0x000a0294aa53e9a2,
    0x000a01041a6aaed5,
    0x000a003c01680870,
    0x0009fc5558a971c8,
    0x0009f9ff9c3c03e5,
    0x0009f9389b864ab9,
    0x0009f6e4534bdca8,
    0x0009f557687235c2,
    0x0009ee633c0391ab,
    0x0009eb4f28e0bb39,
    0x0009e6b49e92e4bb,
    0x0009dfd4ccbd0045,
    0x0009d9c0828536c1,
    0x0009d77ad449f777,
    0x0009d6b92b28ee48,
    0x0009d231a476ed51,
    0x0009cfef711bf120,
    0x0009cc2e1448b765,
    0x0009cb6e26cbc64d,
    0x0009c7b03b4a9c67,
    0x0009c6f0fd980ab1,
    0x0009c4b3f3a30c3f,
    0x0009c0fb29436687,
    0x0009bbca025b7aec,
    0x0009ba4f4421e52c,
    0x0009b1783809ff03,
    0x0009b0bc5b4d2eac,
    0x0009aae172fd8b9c,
    0x0009aa26954607ed,
    0x0009a681e758a022,
    0x0009a5c7b284942e,
    0x0009a2264ecc5558,
    0x00099aebb39be56f,
    0x0009997ae1a9faac,
    0x000998c2a22b6900,
    0x000997527603f8a8,
    0x00099473685e4d50,
    0x00098eba72512a13,
    0x00098c96d8dee9e1,
    0x00098a743453554e,
    0x000989be33c9e6bd,
    0x0009857c692e9a59,
    0x00097fd540c05c9e,
    0x00097d04302ed944,
    0x00097b9c48289935,
    0x0009798133ece717,
    0x00096f07c683689e,
    0x00096e55d6393fc5,
    0x00096addad861696,
    0x00096a2c5a2cf0cf,
    0x00096818fc825eba,
    0x000966b74027f48a,
    0x000964a56850b8ed,
    0x000962947990eb36,
    0x00095fd4a4c885e0,
    0x00095dc5d3954fde,
    0x00095c671ddfe516,
    0x0009584d6340ddf1,
    0x00095641de84afcc,
    0x000953893c386521,
    0x00094f7740d87794,
    0x00094e1cb70c9ce0,
    0x00094962ecbcc7ce,
    0x00094559c69059cf,
    0x000941ff7e640716,
    0x000939fd7a24b099,
    0x000937ff22c014bd,
    0x000934050872c09e,
    0x00093209446d56f6,
    0x0009316033b5bd22,
    0x00092a22b9a79374,
    0x000927838edba206,
    0x000921a2e7112833,
    0x00091e623d5660d0,
    0x00091c6fc0cab8b6,
    0x000917e7d88028eb,
    0x0009169d455585cd,
    0x000915f81ef2d529,
    0x0009140938595d3a,
    0x000910d2360a450e,
    0x00090e417104eabd,
    0x00090c55d0fdea28,
    0x00090b0e84c04f20,
    0x000909243fac6b70,
    0x0008ff9d0440d137,
    0x0008fb3192789d73,
    0x0008f80c0d5031e3,
    0x0008f76b3664f164,
    0x0008f3a80550abc3,
    0x0008f087c50e00c4,
    0x0008efe7fb408cc2,
    0x0008eaecce5c4fd7,
    0x0008ea4dccaaec0b,
    0x0008e4ba9fbc2ff0,
    0x0008dd5688a3b7d6,
    0x0008d7d3821fd94f,
    0x0008d5feb03c31d7,
    0x0008d12033cc9d30,
    0x0008cbac4dec6a82,
    0x0008c9dc80ab604b,
    0x0008c942115dcc96,
    0x0008c3d7df67b539,
    0x0008c2a4bc35cb3b,
    0x0008c0d8a4f1f264,
    0x0008c03f71cbf906,
    0x0008bd42abd9a107,
    0x0008bae051d7f6ff,
    0x0008b7e735068135,
    0x0008b61f82c5fb08,
    0x0008b4588a74a05a,
    0x0008b1fb0a7ed403,
    0x0008b0ccc5d8f5c8,
    0x0008af07f8ac5146,
    0x0008ae71328ffd49,
    0x0008ab8086624822,
    0x0008aaea3ab5ae89,
    0x0008a7661f7020fe,
    0x0008a63ab88aa8dd,
    0x0008a47a35d020f3,
    0x0008a2ba68a3cebf,
    0x0008a2254c852497,
    0x00089ea849898bb3,
    0x00089d7f3e285109,
    0x000899720af36739,
    0x00089442160d11dc,
    0x0008931bd5875a22,
    0x000891630877aedf,
    0x00088baaad83e38f,
    0x00088a86b9090aa4,
    0x0008883fb99bf244,
    0x0008868b45e727ee,
    0x00088568aef30d47,
    0x0008832468f0bcdd,
    0x00088202b9a4df76,
    0x00087e0f31872e9b,
    0x00087c5ecd731f42,
    0x00087b3eea3bb388,
    0x00087751a6c67d78,
    0x000873f6e2f9d34a,
    0x000872d938dcfc01,
    0x0008724a80151dba,
    0x000869f677f6cc1a,
    0x000868db701df58d,
    0x0008623f563a7d6d,
    0x00086099ef0c8886,
    0x00085ef52d38fe87,
    0x00085bad981c7847,
    0x0008586893de7cfc,
    0x0008549b491e9efe,
    0x000852fb3859bea4,
    0x000851e631fc08f8,
    0x0008515bc9cde5f1,
    0x000850472f6185b3,
    0x00084b6defbc166b,
    0x000849d17159854b,
    0x0008469a54a20645,
    0x00084476f9401ade,
    0x000842dd2e2dc25d,
    0x000841cc543f58cb,
    0x00083e9b6c3df688,
    0x00083e1382f22ff9,
    0x00083ae57a327933,
    0x000832fd15e00939,
    0x00082ecb9c6669ac,
    0x00082e45ba6652c4,
    0x00082cb47b00abaa,
    0x000826fa5d0ce5aa,
    0x000823598cfc6865,
    0x000821cc79da73f1,
    0x00081f37ff3d12c0,
    0x00081c21947b0acd,
    0x00081a97404af5f7,
    0x00081a13f02d110e,
    0x0008190d81c9877b,
    0x000817016c0b3ffd,
    0x00081473c50ac33e,
    0x000812ec59f2d11a,
    0x0008116582e237c8,
    0x00080b4fe85ec545,
    0x000807c7894d029a,
];

//////////////////// Tables for Miller-rabin bases of 32/64-bit integers using Bradley's hashing ////////////////////
/*
 * Copyright 2019 Bradley Berg   < (My last name) @ techneon.com >
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 */
#[cfg(feature = "big-table")]
pub const MILLER_RABIN_BASE32: [u32; 256] = [
    1216, 1836, 8885, 4564, 10978, 5228, 15613, 13941, 1553, 173, 3615, 3144, 10065, 9259, 233,
    2362, 6244, 6431, 10863, 5920, 6408, 6841, 22124, 2290, 45597, 6935, 4835, 7652, 1051, 445,
    5807, 842, 1534, 22140, 1282, 1733, 347, 6311, 14081, 11157, 186, 703, 9862, 15490, 1720,
    17816, 10433, 49185, 2535, 9158, 2143, 2840, 664, 29074, 24924, 1035, 41482, 1065, 10189, 8417,
    130, 4551, 5159, 48886, 786, 1938, 1013, 2139, 7171, 2143, 16873, 188, 5555, 42007, 1045, 3891,
    2853, 23642, 148, 3585, 3027, 280, 3101, 9918, 6452, 2716, 855, 990, 1925, 13557, 1063, 6916,
    4965, 4380, 587, 3214, 1808, 1036, 6356, 8191, 6783, 14424, 6929, 1002, 840, 422, 44215, 7753,
    5799, 3415, 231, 2013, 8895, 2081, 883, 3855, 5577, 876, 3574, 1925, 1192, 865, 7376, 12254,
    5952, 2516, 20463, 186, 5411, 35353, 50898, 1084, 2127, 4305, 115, 7821, 1265, 16169, 1705,
    1857, 24938, 220, 3650, 1057, 482, 1690, 2718, 4309, 7496, 1515, 7972, 3763, 10954, 2817, 3430,
    1423, 714, 6734, 328, 2581, 2580, 10047, 2797, 155, 5951, 3817, 54850, 2173, 1318, 246, 1807,
    2958, 2697, 337, 4871, 2439, 736, 37112, 1226, 527, 7531, 5418, 7242, 2421, 16135, 7015, 8432,
    2605, 5638, 5161, 11515, 14949, 748, 5003, 9048, 4679, 1915, 7652, 9657, 660, 3054, 15469,
    2910, 775, 14106, 1749, 136, 2673, 61814, 5633, 1244, 2567, 4989, 1637, 1273, 11423, 7974,
    7509, 6061, 531, 6608, 1088, 1627, 160, 6416, 11350, 921, 306, 18117, 1238, 463, 1722, 996,
    3866, 6576, 6055, 130, 24080, 7331, 3922, 8632, 2706, 24108, 32374, 4237, 15302, 287, 2296,
    1220, 20922, 3350, 2089, 562, 11745, 163, 11951,
];

#[cfg(feature = "big-table")]
pub const MILLER_RABIN_BASE64: [u32; 16384] = [
    1814, 1307, 1714, 6838, 1801, 179, 219, 1317, 6834, 3281, 8345, 4034, 1899, 5746, 4977, 117,
    12578, 18141, 4434, 4236, 543, 9046, 4005, 385, 4231, 3425, 859, 1450, 2199, 4262, 7025, 1562,
    717, 5611, 5298, 871, 1959, 258, 4543, 8969, 1075, 15030, 21041, 7917, 913, 15929, 3450, 9218,
    518, 11775, 7993, 175, 107, 407, 4738, 14388, 86, 6360, 3535, 3818, 8083, 11, 4337, 19601,
    32654, 2566, 974, 1549, 5047, 1542, 1941, 3887, 933, 295, 106, 823, 1529, 7999, 8424, 2958,
    17995, 1430, 8541, 5213, 341, 13034, 4437, 4058, 21281, 715, 1898, 434, 1298, 8930, 33, 3466,
    397, 14181, 2685, 2179, 15160, 271, 175, 5403, 1303, 2091, 4133, 466, 2143, 3908, 2819, 159,
    358, 302, 7778, 2001, 1029, 182, 7405, 13580, 26511, 11758, 743, 4841, 12448, 3585, 3123, 1959,
    3999, 1906, 597, 8293, 2508, 8231, 8676, 485, 4925, 12605, 22286, 1099, 1385, 382, 10310, 746,
    293, 415, 2947, 61, 295, 4797, 11775, 2902, 4254, 12326, 1335, 1541, 17775, 8067, 13739, 4254,
    685, 3963, 2258, 3827, 1151, 2962, 1981, 1043, 2910, 8555, 2478, 3181, 1399, 12907, 5843, 2056,
    7, 598, 17287, 609, 1823, 9968, 12866, 9614, 1177, 1799, 8135, 8861, 6242, 3655, 8700, 11136,
    3693, 229, 6637, 8587, 4915, 820, 6594, 8899, 3012, 833, 5439, 402, 387, 9998, 10074, 401,
    9281, 2246, 12470, 7919, 1533, 2725, 947, 6314, 6681, 2733, 10132, 1535, 7066, 17066, 1529,
    507, 29293, 5893, 6323, 551, 506, 1686, 394, 2355, 2394, 199, 8744, 3451, 1759, 12630, 8251,
    21127, 603, 1349, 16013, 1916, 657, 3143, 10920, 344, 1522, 3123, 10732, 635, 4927, 2641, 2786,
    554, 77, 8759, 2047, 8638, 233, 19015, 237, 29990, 4033, 110, 3949, 351, 293, 12792, 2370,
    3878, 9235, 1477, 14719, 4439, 4571, 3606, 6895, 1256, 8271, 2189, 73, 1006, 28711, 1970, 8162,
    1034, 1258, 3018, 5062, 2381, 1326, 19172, 17936, 17446, 4714, 7939, 975, 11261, 17179, 1138,
    9907, 7654, 8844, 213, 3378, 7937, 118, 3421, 1217, 7742, 52, 18125, 381, 1131, 545, 9748,
    8528, 1219, 2206, 415, 6235, 50334, 74, 397, 1773, 1150, 16385, 1419, 8899, 811, 3813, 2511,
    10359, 17270, 22652, 9646, 34497, 3474, 141, 1554, 1287, 4642, 1571, 934, 2834, 101, 749, 258,
    302, 26159, 1291, 5105, 1783, 7310, 415, 1558, 2393, 13393, 263, 1362, 7985, 9569, 8240, 2279,
    5018, 8430, 1961, 6005, 17635, 3622, 4213, 36293, 9985, 11, 1254, 2251, 27396, 239, 17364,
    3863, 4615, 7899, 3674, 5531, 199, 8996, 1293, 6098, 514, 8538, 17311, 2015, 6226, 4459, 5731,
    887, 5449, 8085, 1851, 206, 2925, 8337, 2363, 9490, 1085, 590, 9094, 78, 7190, 2843, 86, 497,
    1658, 113, 24990, 5302, 3141, 18856, 16074, 5754, 9497, 1757, 1353, 4013, 3739, 6930, 6179,
    813, 6894, 2527, 2218, 14648, 397, 2979, 3074, 2314, 6521, 18390, 2157, 10642, 1177, 5007, 391,
    1982, 4373, 26, 1917, 1567, 3091, 22352, 9374, 935, 3263, 8242, 3239, 3558, 18600, 9839, 2440,
    5325, 6414, 1303, 2995, 109, 5731, 9966, 1478, 8867, 9094, 284, 433, 1750, 277, 9773, 22660,
    181, 127, 279, 641, 851, 199, 539, 1190, 446, 10526, 2566, 677, 127, 2404, 9715, 8319, 15005,
    3086, 2779, 3951, 7, 6929, 461, 27337, 8909, 9543, 5150, 6703, 3327, 769, 901, 851, 2227, 819,
    1879, 3130, 4927, 2302, 802, 7599, 1163, 71, 33, 925, 5550, 2971, 8201, 23865, 114, 5886, 1145,
    19851, 6219, 1877, 613, 827, 11302, 5726, 1237, 769, 590, 9156, 9048, 6049, 7379, 5462, 12522,
    5706, 823, 8431, 3349, 1623, 2245, 5198, 1329, 14794, 6193, 2266, 9722, 3716, 10294, 2009,
    1193, 233, 3979, 1077, 418, 941, 4613, 8635, 26786, 1507, 8508, 258, 1419, 5225, 7485, 1051,
    13020, 31711, 22593, 3331, 3338, 12526, 161, 5338, 7302, 78, 23889, 571, 26734, 8013, 8305,
    9053, 1566, 2482, 191, 12703, 1703, 3385, 654, 26390, 3647, 2639, 8498, 3813, 1161, 7777, 117,
    2265, 813, 8069, 2053, 2091, 5033, 6637, 8588, 15954, 325, 1713, 9719, 1983, 619, 107, 419,
    9104, 9019, 8688, 577, 1773, 237, 3379, 2946, 1151, 13815, 1119, 1393, 11565, 2177, 530, 265,
    1663, 4058, 11031, 12256, 2443, 173, 1405, 1258, 8592, 4364, 4883, 1201, 2470, 119, 2775,
    37084, 6690, 1067, 10479, 337, 4147, 19565, 853, 4005, 1790, 855, 2329, 481, 585, 2616, 1741,
    10908, 17057, 9267, 4403, 10597, 4963, 7154, 3079, 8245, 3271, 10807, 1513, 691, 2821, 5285,
    681, 9822, 690, 566, 46, 639, 141, 6706, 9308, 1009, 7433, 24590, 29, 25809, 710, 4839, 2857,
    17910, 383, 4630, 2993, 1059, 39, 13218, 10597, 6607, 3466, 3029, 8639, 1646, 8895, 231, 4885,
    3305, 5734, 10310, 5954, 7773, 3329, 587, 14302, 9806, 2130, 189, 1267, 7397, 12364, 953, 299,
    5297, 3354, 1921, 2486, 6827, 2202, 2369, 15279, 871, 713, 8273, 1227, 9691, 3475, 469, 865,
    2230, 353, 6249, 9310, 1546, 38417, 2199, 1789, 9051, 6555, 3766, 1701, 10210, 1270, 3279,
    8620, 3039, 929, 2029, 618, 6641, 7122, 4215, 34, 89, 5251, 1293, 6401, 409, 537, 935, 26421,
    2882, 4129, 55, 7427, 369, 12218, 9918, 1870, 4888, 1449, 1430, 4430, 5291, 1699, 2074, 67,
    12096, 6687, 8414, 573, 2803, 887, 2435, 1138, 2413, 4129, 205, 2190, 1654, 3697, 3170, 3538,
    1574, 555, 1338, 2822, 1077, 610, 8391, 1539, 11, 6637, 1053, 179, 698, 1495, 1217, 1436, 9935,
    778, 310, 801, 718, 1066, 3066, 261, 839, 651, 3293, 20997, 8527, 8721, 2146, 37272, 3607,
    9206, 698, 9983, 28805, 221, 3686, 11549, 348, 181, 1293, 3846, 2717, 6926, 5133, 137, 1785,
    9727, 2325, 2723, 8678, 607, 3010, 8749, 8855, 10668, 2675, 301, 7899, 8191, 9141, 6622, 9435,
    1579, 1539, 3261, 5773, 218, 1654, 4466, 739, 3366, 2047, 557, 3189, 7558, 2867, 2810, 3842,
    1663, 4635, 34027, 269, 9256, 3533, 4074, 1022, 3365, 1633, 15096, 8323, 9249, 1767, 1326, 39,
    9211, 165, 8797, 9519, 1006, 5738, 18521, 323, 1746, 10996, 1517, 8204, 6547, 4359, 1834, 2966,
    716, 715, 2470, 1443, 8518, 2938, 1962, 2387, 10961, 851, 5533, 553, 5150, 934, 554, 157, 4429,
    15949, 2194, 1657, 1449, 7391, 377, 2065, 3307, 6695, 2731, 1141, 813, 3238, 8318, 2255, 9003,
    618, 3697, 24981, 4401, 1523, 219, 721, 6284, 6220, 7747, 333, 12835, 5619, 949, 4346, 1206,
    846, 8063, 6167, 1383, 6843, 10299, 3085, 9018, 277, 221, 12774, 1173, 1926, 21591, 4182, 2194,
    8223, 3983, 550, 2254, 8371, 606, 1202, 29, 1273, 6557, 8808, 7918, 3797, 2941, 1566, 15003,
    1955, 2796, 2583, 233, 10105, 1418, 2889, 24939, 9590, 3573, 613, 70, 247, 934, 1822, 895,
    4197, 30570, 8511, 9367, 11295, 4715, 3145, 3103, 57, 5810, 11354, 6406, 1111, 30851, 2838,
    2639, 8673, 18562, 2999, 9448, 11625, 1773, 9272, 13946, 1498, 4613, 21674, 30566, 5053, 11644,
    3701, 2654, 4693, 19, 1986, 14647, 1615, 1711, 1259, 4569, 2798, 1070, 25016, 1461, 358, 1457,
    807, 1018, 9256, 1249, 9589, 8615, 2290, 2073, 21, 1363, 321, 3059, 6965, 865, 12695, 287,
    4322, 6173, 8206, 1967, 457, 59, 1417, 15449, 5635, 2734, 5122, 1630, 1115, 4194, 3458, 10494,
    5373, 2266, 7884, 8637, 6114, 8655, 4279, 6438, 1383, 3838, 2078, 1721, 2485, 2447, 541, 8289,
    2291, 221, 138, 567, 7194, 1141, 1941, 1311, 1522, 1687, 3510, 2446, 8578, 10349, 25626, 32066,
    1654, 413, 129, 11851, 4902, 547, 2014, 4427, 11738, 1661, 2081, 10933, 3226, 438, 4433, 4679,
    339, 617, 9451, 6134, 6822, 8957, 5171, 14095, 11767, 2697, 3351, 9461, 2207, 1501, 930, 617,
    435, 8966, 3185, 6603, 10993, 908, 881, 1033, 254, 1538, 1383, 7890, 1489, 18615, 1070, 20332,
    3867, 5666, 1513, 2661, 7297, 3263, 37879, 5883, 11163, 1081, 2589, 5033, 10067, 56007, 8629,
    53493, 5443, 5749, 3982, 791, 5573, 3967, 549, 3371, 727, 6788, 4382, 607, 4111, 2837, 4565,
    2273, 2567, 286, 39, 1441, 4254, 27375, 5061, 442, 3597, 8308, 8035, 3041, 967, 27275, 1939,
    1034, 1038, 913, 4649, 426, 87, 36883, 4918, 2381, 478, 3965, 883, 3855, 4366, 7214, 3562, 681,
    763, 8538, 1229, 3223, 5371, 773, 1825, 2061, 446, 629, 8832, 2269, 3393, 6712, 9365, 11001,
    1459, 9795, 1247, 1966, 1117, 717, 1547, 19975, 3428, 253, 421, 10245, 933, 926, 3789, 3351,
    5138, 262, 2193, 1910, 1183, 193, 1497, 151, 1343, 7343, 2817, 351, 1767, 6540, 6179, 1983,
    2985, 14, 2811, 17706, 1271, 737, 10887, 305, 2794, 4717, 12567, 1507, 17602, 1523, 5209, 463,
    8866, 11923, 4955, 5265, 5288, 6043, 5641, 595, 282, 446, 6497, 911, 1661, 6946, 609, 8322,
    457, 3393, 7889, 649, 5663, 2237, 8149, 5867, 1249, 7135, 1079, 17165, 32290, 26529, 26547,
    483, 4357, 4110, 1893, 4482, 1930, 3294, 9811, 35577, 2541, 11035, 1091, 129, 513, 445, 12330,
    3553, 2689, 9003, 29917, 23793, 8814, 8380, 738, 6846, 1693, 127, 2667, 5245, 787, 2058, 217,
    1073, 462, 311, 5087, 10108, 291, 2431, 951, 41, 16869, 9010, 10398, 1531, 4053, 2456, 727,
    4636, 2689, 1526, 2126, 3736, 279, 4038, 1005, 2186, 9358, 5537, 3729, 693, 530, 749, 4379,
    1583, 34038, 2590, 457, 311, 3077, 12423, 1006, 1923, 3071, 811, 10642, 8501, 4835, 1735, 9606,
    35324, 85, 8232, 2425, 177, 8384, 933, 341, 2481, 2017, 3694, 1381, 131, 713, 4213, 13802, 959,
    1343, 3674, 325, 4090, 265, 1147, 4209, 785, 1970, 110, 4262, 297, 6074, 4769, 5299, 2309,
    4433, 3833, 17427, 6914, 9188, 2246, 7726, 8747, 7501, 10637, 278, 655, 10539, 349, 414, 2871,
    31558, 3958, 10727, 3338, 1703, 11716, 22687, 8919, 17340, 2461, 231, 1994, 797, 12341, 307,
    5114, 691, 20111, 532, 2286, 12881, 4121, 2453, 874, 2947, 5494, 1866, 2543, 665, 3864, 1581,
    11513, 35676, 4961, 2715, 447, 73, 5211, 818, 4286, 5691, 6021, 466, 5606, 1077, 5970, 1293,
    4499, 11210, 2006, 9794, 1141, 4861, 21132, 1068, 2033, 251, 6987, 5718, 3218, 629, 755, 453,
    3259, 457, 5181, 6482, 25951, 8372, 233, 1397, 511, 7157, 10090, 8219, 10289, 9257, 29842, 670,
    4025, 2471, 35572, 1035, 354, 831, 1713, 2215, 3393, 16440, 16076, 32137, 2806, 723, 1223,
    2007, 137, 463, 38, 1195, 3566, 1462, 33400, 101, 785, 89, 6471, 3595, 187, 7006, 3323, 322,
    20079, 413, 3115, 12292, 575, 2635, 5479, 9336, 5319, 10302, 1801, 2925, 1530, 1139, 6043,
    5654, 565, 6333, 842, 67, 16995, 3923, 9546, 1697, 4877, 239, 55, 5989, 9439, 366, 3723, 839,
    3097, 10983, 395, 4135, 40693, 619, 1310, 19017, 7057, 9101, 7513, 2173, 1314, 9023, 13279,
    281, 82, 1007, 8011, 706, 3061, 6470, 7503, 1547, 33138, 183, 345, 2533, 118, 10472, 563, 505,
    11464, 5123, 8943, 4149, 2203, 489, 2511, 682, 8525, 4387, 1127, 4066, 14636, 2982, 5560, 2310,
    7825, 497, 2938, 3787, 1306, 3791, 10147, 215, 9734, 12660, 1397, 1079, 1907, 10889, 97, 11659,
    3126, 8857, 2495, 1895, 8348, 1209, 42, 4622, 11471, 313, 1022, 1185, 1039, 5813, 15838, 2995,
    7481, 8107, 2118, 8554, 117, 13263, 1122, 38, 6378, 767, 7859, 1361, 1854, 9849, 26881, 1763,
    2830, 6158, 1041, 1789, 1939, 756, 1075, 239, 1054, 1823, 50612, 11794, 19210, 6947, 33782,
    4174, 5451, 786, 2006, 2653, 1909, 963, 14187, 2429, 1199, 2971, 11327, 2971, 4419, 31837, 195,
    2493, 2237, 509, 119, 1023, 11335, 6753, 6021, 8385, 1474, 11087, 8289, 237, 3473, 998, 19,
    1106, 13188, 159, 413, 2579, 8123, 4488, 1839, 1770, 3765, 1842, 237, 1026, 5031, 23543, 5823,
    6053, 295, 1930, 1685, 2193, 4344, 5410, 1097, 4995, 3765, 3055, 13144, 1283, 2734, 1147, 9486,
    17, 1339, 3065, 7703, 3295, 5405, 1071, 617, 938, 2505, 437, 3921, 671, 5842, 12540, 3681, 11,
    5790, 534, 3755, 313, 1334, 6903, 4343, 1101, 7133, 885, 6518, 6879, 4589, 469, 1625, 1774,
    737, 15813, 13743, 266, 775, 131, 2799, 97, 1845, 5810, 593, 10268, 4270, 3309, 6339, 2702,
    9644, 1209, 1073, 7107, 2903, 253, 1978, 643, 3573, 233, 5741, 4018, 3811, 326, 5714, 1574,
    6794, 122, 843, 661, 1771, 2651, 2858, 16330, 4957, 8522, 5097, 2353, 1022, 5733, 3676, 7539,
    609, 1351, 78, 2735, 971, 1931, 2539, 4190, 1782, 8817, 3710, 4515, 3725, 7182, 5795, 13535,
    4921, 9587, 1785, 9627, 2295, 5750, 2849, 1537, 23868, 327, 899, 9475, 10305, 241, 5939, 4223,
    3187, 171, 4191, 3480, 530, 5411, 674, 2931, 679, 15237, 1378, 1426, 1714, 33684, 3330, 2195,
    14478, 27430, 1275, 4773, 1117, 8516, 415, 1441, 5859, 8501, 547, 1108, 13905, 1422, 8908,
    5698, 10789, 1779, 1385, 2635, 14718, 10378, 6153, 2606, 9453, 2246, 7302, 493, 10989, 21648,
    1127, 8798, 24057, 330, 1487, 8675, 739, 1811, 4445, 3301, 3065, 5115, 1415, 7959, 267, 3129,
    14051, 6209, 382, 12483, 522, 1113, 10089, 9521, 495, 6150, 11568, 3754, 13268, 377, 5559, 933,
    3819, 1205, 1077, 8766, 2076, 9137, 31061, 1954, 53, 5651, 4578, 9272, 11578, 497, 309, 747,
    6139, 8729, 10329, 4226, 563, 59, 3143, 9909, 2246, 1727, 39, 17041, 9150, 51, 629, 8416, 7,
    1287, 781, 7213, 16098, 9173, 965, 2581, 558, 467, 2525, 6350, 8344, 4895, 2474, 119, 875,
    1261, 753, 9806, 15075, 16324, 9171, 1286, 2177, 9456, 5126, 18290, 1771, 6631, 21720, 8167,
    4961, 6151, 5018, 33874, 8865, 4519, 7159, 1447, 4395, 3021, 327, 3634, 1002, 7650, 3822, 6341,
    2194, 1275, 391, 4459, 7047, 3450, 4442, 36922, 12597, 493, 33779, 825, 5291, 914, 8348, 1765,
    1589, 222, 5454, 1971, 1163, 1686, 8996, 12473, 15722, 8998, 175, 15597, 24939, 1651, 7307,
    19984, 9696, 8462, 39, 1367, 1613, 5949, 14266, 1082, 1883, 40249, 2402, 2015, 598, 595, 22769,
    9766, 1577, 3587, 2687, 4298, 1282, 17860, 4337, 3388, 9544, 9354, 4267, 431, 3235, 215, 3639,
    2697, 8034, 739, 3783, 6515, 8602, 4357, 9085, 1046, 9463, 10105, 230, 233, 708, 3571, 958,
    663, 9174, 26220, 1573, 939, 1001, 30407, 30199, 1398, 1793, 297, 8259, 788, 1354, 714, 10163,
    9262, 1334, 3673, 3162, 262, 2351, 2454, 634, 9888, 395, 3559, 3582, 447, 4889, 665, 549, 7653,
    26390, 2295, 24008, 1633, 9844, 12067, 12190, 7551, 783, 1117, 1013, 1037, 4387, 3831, 1138,
    2870, 186, 697, 463, 42723, 1145, 553, 3938, 6759, 8930, 2799, 36308, 36438, 2725, 42, 1239,
    173, 4602, 2487, 1617, 5117, 309, 8552, 14077, 586, 836, 4454, 2467, 257, 4959, 7583, 1193,
    4358, 1509, 1817, 5155, 149, 425, 12379, 369, 2101, 9459, 2394, 86, 12155, 6134, 3214, 298,
    306, 4315, 2825, 4293, 1891, 16369, 683, 10840, 4667, 2701, 587, 1505, 7923, 9416, 5773, 3259,
    2331, 11122, 5395, 1419, 58, 891, 4811, 202, 3553, 2077, 1809, 2158, 3711, 6714, 510, 6598,
    7886, 951, 618, 402, 11766, 7775, 22833, 8300, 3505, 10198, 8075, 1002, 6285, 8594, 5978, 8843,
    6469, 14349, 241, 2094, 4745, 474, 1657, 337, 6702, 3382, 4769, 37346, 698, 6654, 25229, 7710,
    2623, 2254, 6547, 5689, 395, 7087, 1130, 1091, 562, 5539, 7751, 7313, 9250, 19891, 1546, 1613,
    618, 1558, 2593, 1623, 4433, 7893, 11064, 326, 1797, 12623, 8202, 10434, 10708, 37382, 1191,
    1194, 9848, 1314, 633, 6040, 3406, 12332, 258, 4457, 16136, 301, 323, 5859, 13959, 235, 39884,
    2137, 1197, 1277, 194, 446, 1201, 5813, 2339, 4690, 1391, 8238, 203, 2991, 5099, 2990, 6082,
    9708, 8726, 6583, 10146, 26158, 63, 10238, 3027, 3771, 2353, 1098, 631, 3460, 23197, 4667,
    2866, 618, 1779, 1579, 9118, 8157, 703, 5327, 7355, 10858, 2709, 4483, 627, 965, 9160, 2531,
    2049, 899, 61, 3467, 3802, 5255, 1561, 1071, 16171, 2199, 1413, 14344, 611, 2195, 12869, 3357,
    804, 3062, 2314, 2811, 18751, 515, 929, 4110, 1675, 42, 427, 5573, 2259, 9557, 1339, 10390,
    8577, 230, 9168, 1141, 3961, 871, 8420, 20283, 1515, 517, 6293, 11102, 1611, 390, 5177, 2447,
    1315, 663, 2449, 475, 2027, 646, 3834, 3827, 781, 8977, 418, 9111, 9859, 426, 91, 12846, 1063,
    770, 3490, 8514, 3347, 997, 8977, 11043, 1306, 2622, 2577, 1629, 69, 7413, 3215, 2579, 3069,
    1383, 2091, 1311, 1674, 4042, 7946, 3803, 479, 1819, 7459, 18973, 445, 653, 3867, 5101, 3039,
    8274, 11634, 3449, 514, 394, 3519, 949, 4981, 574, 401, 1235, 67, 8170, 455, 2743, 251, 546,
    12009, 1709, 3859, 3287, 577, 7854, 16114, 9549, 4082, 2550, 2059, 167, 488, 267, 581, 4065,
    1411, 3847, 21190, 1245, 2198, 11123, 862, 114, 3949, 15247, 4743, 35399, 4162, 725, 1476,
    13148, 3917, 206, 12255, 347, 687, 1293, 8266, 9033, 1219, 4573, 2449, 8535, 4527, 1987, 417,
    957, 5311, 1265, 5650, 2517, 7873, 7498, 839, 9386, 249, 4850, 8543, 751, 3527, 1825, 10317,
    1783, 9334, 2127, 3661, 14659, 11433, 2114, 13388, 137, 1837, 4443, 3900, 938, 1226, 5603,
    8117, 10264, 7001, 1755, 2501, 10772, 5086, 1649, 3169, 2211, 1659, 453, 2594, 501, 6654,
    18817, 3429, 3661, 7, 885, 7502, 10083, 5807, 2830, 267, 1001, 411, 8372, 2831, 3614, 2655,
    6689, 14038, 893, 1685, 1195, 2550, 1207, 10292, 110, 1830, 4846, 3643, 967, 8456, 1630, 57,
    742, 805, 2961, 13600, 3798, 6915, 870, 2946, 16517, 173, 11537, 4598, 963, 573, 8726, 10825,
    1881, 8082, 1887, 86, 17, 7595, 3190, 3975, 2909, 12673, 1517, 4855, 641, 3262, 8263, 6967,
    6011, 1189, 1571, 257, 1315, 77, 8267, 3467, 327, 846, 264, 913, 2242, 3433, 671, 1527, 2906,
    6153, 1827, 8189, 789, 5359, 2982, 2529, 4874, 2643, 563, 3279, 3389, 13080, 390, 790, 4329,
    1018, 965, 19849, 3150, 8249, 6066, 7167, 1002, 4491, 3695, 759, 5705, 6221, 555, 1949, 12784,
    701, 1073, 4209, 3175, 14126, 4345, 5795, 22149, 1027, 946, 8944, 8358, 9324, 3558, 995, 27113,
    8524, 47, 1379, 1549, 601, 7491, 2903, 16198, 759, 11501, 11500, 11814, 8453, 7286, 1478, 8931,
    2082, 12974, 1763, 347, 9355, 13892, 8893, 24825, 757, 2132, 1158, 3180, 1398, 1129, 4631,
    6898, 4737, 319, 10443, 1199, 23935, 8504, 12169, 1486, 8178, 573, 4097, 5029, 3939, 3369,
    19764, 325, 2174, 595, 1254, 897, 4815, 3606, 3337, 2873, 6127, 2942, 2337, 9571, 103, 1957,
    1045, 14285, 2657, 873, 3869, 8366, 2182, 3381, 5063, 2647, 2975, 23863, 2263, 101, 2849, 9470,
    4109, 1108, 151, 2599, 1015, 2005, 785, 733, 2227, 5655, 13902, 1714, 2282, 3243, 6619, 16141,
    129, 18228, 7298, 7093, 1063, 6823, 293, 5473, 627, 6214, 3317, 6771, 6165, 38900, 1937, 1746,
    1269, 2329, 6082, 11224, 9379, 4795, 13866, 8956, 57, 5566, 6489, 3943, 233, 1847, 967, 530,
    8733, 606, 2279, 4942, 9920, 3239, 10307, 6246, 10483, 9803, 4429, 695, 25931, 2564, 46, 983,
    3847, 1989, 1834, 9705, 1434, 202, 9976, 8081, 8452, 3414, 733, 3821, 3333, 235, 167, 909,
    9333, 1318, 438, 10315, 414, 5023, 3955, 1181, 3917, 1415, 1959, 2497, 1647, 2755, 879, 4652,
    2483, 6246, 282, 31, 4319, 7063, 783, 2797, 23, 5899, 4177, 62, 9908, 3337, 11189, 4829, 9221,
    16324, 6483, 7851, 5371, 761, 5237, 8922, 3055, 4417, 5103, 290, 1357, 1822, 91, 302, 341,
    2555, 11573, 9424, 911, 14920, 6234, 12781, 9614, 935, 9285, 8612, 730, 930, 379, 8435, 681,
    11055, 7465, 8658, 7713, 7702, 9099, 5203, 5379, 4263, 8561, 413, 2879, 7978, 15744, 9028,
    3512, 873, 3490, 1766, 633, 5802, 2775, 14459, 10577, 2141, 5493, 553, 14709, 9927, 1417, 1953,
    1567, 825, 15910, 618, 743, 491, 12034, 13580, 5527, 426, 1027, 653, 2246, 3845, 3229, 2894,
    1553, 11580, 8707, 4040, 13748, 10361, 9733, 4958, 1742, 265, 550, 238, 202, 3795, 1051, 5658,
    506, 7594, 3198, 2091, 2294, 935, 16882, 3201, 9416, 5213, 2127, 20447, 8697, 5386, 11993,
    1507, 2510, 1087, 55335, 8709, 9199, 8866, 5014, 687, 13, 17066, 811, 3991, 846, 457, 73,
    11930, 5761, 1186, 1169, 37, 615, 4593, 1026, 3778, 5699, 6297, 9825, 1337, 1859, 5881, 2778,
    12511, 773, 1077, 7301, 137, 4731, 3667, 1262, 12028, 25454, 16558, 13560, 1598, 5126, 8409,
    229, 1121, 1901, 514, 8789, 13127, 2699, 1090, 8327, 113, 2233, 537, 866, 1077, 20083, 1255,
    47, 17516, 939, 2101, 1553, 9535, 206, 12130, 802, 13290, 6747, 8003, 10517, 1463, 183, 666,
    1567, 2002, 153, 805, 629, 1094, 201, 5135, 25282, 7605, 87, 3135, 3521, 11759, 467, 2722,
    6878, 10398, 6786, 1953, 9705, 9372, 7025, 902, 5838, 59, 1847, 193, 11466, 4729, 10970, 16215,
    6321, 1209, 3211, 8629, 1243, 10586, 9955, 4002, 359, 8848, 5218, 12110, 598, 5086, 83, 11617,
    10312, 6661, 2635, 4674, 3973, 802, 4703, 365, 1195, 11558, 2534, 1401, 874, 8972, 5011, 4079,
    875, 7418, 1915, 2701, 36501, 15472, 7298, 47, 7287, 813, 905, 5062, 609, 12056, 3777, 8965,
    531, 325, 1167, 1557, 1935, 9162, 702, 4539, 3663, 197, 4978, 26779, 8005, 3843, 157, 14203,
    2135, 4702, 4129, 4259, 1410, 3821, 10397, 5794, 7841, 2786, 9906, 2562, 12754, 3955, 8611,
    721, 3353, 783, 7, 1821, 1138, 8238, 14464, 869, 662, 43, 2957, 1005, 17632, 406, 4090, 822,
    1527, 6209, 3349, 865, 3039, 5455, 6874, 2999, 14114, 5829, 2751, 11968, 3533, 6987, 1726, 271,
    5609, 2191, 8495, 4259, 705, 7475, 9945, 4209, 10238, 6104, 21039, 9559, 905, 16581, 2357,
    2237, 7981, 20161, 1021, 1155, 27047, 6559, 2485, 1318, 27706, 23, 5589, 33998, 11175, 3685,
    949, 13682, 5542, 1295, 745, 2621, 1614, 3291, 2263, 143, 6819, 8629, 15753, 12906, 13988,
    5135, 8711, 10817, 1429, 27903, 7673, 2906, 1229, 3347, 119, 2915, 3223, 53, 1282, 37910, 1605,
    2518, 1165, 4131, 3502, 3123, 1185, 238, 9935, 10470, 1695, 1149, 14237, 5054, 5291, 11489,
    1045, 2929, 414, 2223, 10223, 2966, 6545, 563, 1286, 14409, 11185, 12491, 3010, 10548, 1307,
    5458, 2065, 3297, 3357, 8111, 9016, 9077, 145, 651, 8528, 3250, 451, 2015, 5755, 1498, 1631,
    2393, 8545, 9415, 2203, 921, 9723, 6077, 2917, 7459, 5486, 563, 2105, 29726, 8745, 5577, 710,
    30187, 10675, 1321, 123, 9696, 2855, 9497, 31085, 12340, 3729, 374, 1393, 2491, 5015, 8972,
    6741, 2126, 3627, 2003, 8514, 2414, 2006, 753, 633, 3395, 5730, 10219, 871, 4499, 2833, 4241,
    1955, 1807, 15988, 8726, 4207, 6014, 10213, 16722, 5717, 8636, 365, 3691, 62, 8212, 14814,
    18465, 989, 1121, 13774, 927, 202, 2053, 1593, 2181, 29, 1867, 30613, 8467, 1773, 101, 3062,
    2613, 3653, 329, 28245, 2589, 982, 4060, 16311, 30745, 1119, 6010, 835, 378, 2994, 897, 9784,
    1761, 8749, 3386, 7918, 295, 2493, 17637, 11666, 9997, 2685, 9932, 1030, 573, 2954, 377, 6009,
    4897, 21, 709, 1122, 921, 77, 7061, 13576, 14489, 6062, 5031, 451, 691, 290, 1074, 5409, 9630,
    1433, 10274, 1413, 4982, 5602, 5995, 1638, 449, 5785, 264, 1797, 5006, 4698, 5183, 505, 87,
    2171, 10058, 1045, 501, 20861, 34822, 14, 11442, 1603, 3038, 9343, 8990, 8641, 8325, 94, 279,
    2307, 3686, 1993, 4382, 15155, 1410, 781, 2379, 3898, 10256, 7110, 97, 8710, 6809, 482, 3034,
    133, 3145, 1410, 3013, 1618, 6349, 414, 2214, 1010, 22666, 821, 1313, 9340, 14471, 12239, 8847,
    1011, 381, 1045, 5457, 7503, 1414, 943, 394, 13675, 823, 2521, 402, 18213, 2701, 689, 1302, 61,
    1857, 2718, 575, 8842, 1791, 442, 12622, 3901, 5723, 1187, 10006, 5825, 9071, 5650, 95, 3963,
    9193, 191, 2570, 8271, 2246, 1538, 395, 5571, 12520, 1758, 1223, 10003, 3241, 4551, 427, 5003,
    39, 2798, 5739, 842, 6010, 89, 2794, 5395, 9557, 3446, 5694, 41868, 5679, 32901, 2569, 1399,
    4565, 9746, 40747, 5403, 1362, 2851, 26, 886, 442, 55, 3053, 4895, 3871, 739, 2550, 5690, 505,
    642, 1505, 2270, 483, 13414, 4307, 16633, 1583, 847, 12303, 5367, 159, 87, 701, 10519, 17861,
    3958, 3691, 15220, 2397, 282, 161, 1315, 8977, 5734, 351, 1209, 2285, 8804, 4451, 2393, 4807,
    1267, 1610, 10303, 1002, 12929, 6767, 2339, 7550, 7741, 8209, 369, 7914, 8074, 1686, 1918,
    5909, 2081, 18317, 3356, 4491, 197, 10102, 7895, 1624, 922, 7383, 1002, 8880, 138, 6581, 370,
    13034, 894, 25887, 2325, 1494, 1773, 383, 3099, 849, 1877, 25954, 3202, 1059, 2431, 9101, 4681,
    18662, 13765, 3605, 4063, 11196, 8445, 2478, 6899, 6939, 11226, 11177, 979, 11548, 23549, 1513,
    417, 7742, 379, 2961, 3497, 2869, 3830, 1657, 52, 8248, 11582, 8285, 389, 32764, 8021, 4138,
    1229, 7167, 9013, 3771, 2998, 1335, 1611, 46, 501, 1374, 8470, 15179, 3035, 28253, 886, 7479,
    4641, 70, 8194, 8633, 2141, 7881, 1891, 20741, 1747, 9536, 8202, 9296, 917, 1247, 3954, 1971,
    730, 1779, 8967, 971, 2695, 3006, 8217, 4887, 20668, 3661, 15431, 11327, 381, 742, 6026, 5730,
    8637, 1501, 358, 9862, 1829, 5231, 27105, 858, 366, 3143, 1046, 574, 62, 1218, 9024, 349,
    10419, 1180, 4561, 159, 2257, 349, 5095, 107, 9573, 653, 18543, 5974, 14512, 8637, 8215, 5393,
    3575, 657, 5965, 239, 153, 407, 305, 5491, 5463, 12680, 8734, 334, 8331, 4754, 1337, 3762,
    2935, 111, 3891, 2957, 137, 451, 627, 4439, 1827, 31, 3727, 4949, 6078, 2425, 2522, 1282, 3113,
    1583, 4573, 6745, 4878, 6647, 2665, 4627, 51, 2743, 341, 6714, 2261, 397, 1999, 1633, 7221,
    11216, 7299, 7259, 1961, 3106, 951, 2730, 1732, 2382, 1391, 2470, 4338, 591, 7507, 4565, 1208,
    2207, 353, 1671, 3807, 853, 11611, 645, 4703, 4021, 501, 2232, 2419, 497, 12029, 3099, 893,
    2062, 5942, 4934, 13277, 985, 873, 2821, 8330, 8327, 10300, 451, 723, 8459, 1974, 28847, 2910,
    839, 2915, 12869, 4722, 981, 4333, 3005, 17779, 2114, 8429, 2943, 10274, 9504, 767, 65, 358,
    582, 3753, 2626, 6373, 974, 295, 1633, 12690, 6353, 35178, 1413, 1243, 755, 173, 1967, 13079,
    2022, 1019, 9769, 187, 11060, 28711, 9109, 4595, 7191, 803, 99, 11030, 275, 11962, 6434, 8528,
    4039, 6034, 123, 1244, 2513, 2986, 3873, 65, 36075, 8717, 21884, 429, 6963, 4582, 149, 9695,
    9616, 1413, 2915, 1094, 9006, 119, 2201, 367, 2170, 2605, 2061, 1634, 925, 2665, 1913, 11438,
    17811, 12610, 6675, 1671, 1711, 3329, 7053, 5223, 28562, 1981, 2382, 8039, 12156, 2747, 319,
    10973, 4918, 3694, 499, 843, 2497, 171, 4953, 1031, 643, 14139, 1911, 7190, 572, 1613, 677,
    1463, 3735, 17967, 10549, 1979, 6554, 2648, 1427, 615, 17315, 10873, 3142, 430, 1893, 1911,
    1882, 8713, 4782, 8145, 113, 8242, 937, 5890, 925, 7062, 1911, 8542, 20615, 5831, 22307, 2255,
    9299, 463, 714, 2146, 627, 4314, 227, 174, 9569, 14120, 12664, 383, 9932, 299, 982, 8029, 8332,
    3854, 8741, 7014, 1695, 3781, 26865, 6086, 2547, 3598, 1181, 2303, 1067, 10842, 3655, 398,
    9584, 12828, 590, 1367, 3059, 1114, 5014, 479, 7647, 1623, 9567, 5076, 495, 4086, 14425, 6711,
    1307, 715, 822, 933, 1195, 354, 36824, 8230, 8340, 3347, 16803, 12920, 2821, 8551, 2374, 17680,
    10366, 7287, 1265, 1751, 14409, 1599, 1697, 9785, 17774, 7194, 6335, 1917, 8883, 1579, 5621,
    210, 221, 7342, 9521, 2051, 393, 12370, 3804, 3801, 9565, 219, 86, 6658, 3949, 2917, 5790, 89,
    3395, 6371, 3219, 1306, 3046, 3651, 8561, 7806, 2771, 5493, 2111, 7182, 57, 155, 4849, 4505,
    1163, 10309, 247, 15878, 6307, 9605, 7194, 3357, 9719, 253, 46, 2677, 30597, 2623, 4935, 418,
    126, 1242, 2107, 561, 922, 277, 11177, 10390, 1333, 3145, 14550, 2369, 789, 9821, 185, 1229,
    4695, 1333, 3382, 2679, 4817, 2747, 737, 1322, 218, 7237, 1683, 8725, 10068, 8769, 189, 25475,
    5293, 3198, 3731, 10601, 6277, 2797, 5979, 9052, 45502, 9507, 3171, 13942, 263, 3673, 2739,
    1334, 4293, 4626, 313, 10054, 12940, 1725, 454, 3198, 365, 4377, 10102, 3183, 9266, 5710, 37,
    4549, 222, 16997, 13589, 149, 5643, 17881, 2990, 1414, 3789, 421, 2746, 1157, 975, 1074, 233,
    1580, 237, 359, 7186, 2051, 6747, 8676, 1325, 3857, 1286, 17648, 2823, 10024, 1576, 558, 2517,
    13656, 6330, 2995, 787, 1926, 11892, 6353, 14206, 998, 3448, 122, 987, 373, 4290, 2669, 615,
    2577, 489, 413, 4582, 26571, 374, 1493, 13595, 703, 7612, 2006, 8433, 3797, 10567, 5550, 16144,
    7753, 6635, 805, 3715, 430, 685, 893, 3633, 11093, 3293, 59, 1063, 2369, 6891, 19833, 6874,
    8103, 5725, 1755, 1247, 2954, 11876, 2862, 590, 2819, 302, 8975, 285, 1999, 2577, 7539, 117,
    8952, 6281, 10643, 373, 7538, 2629, 946, 1889, 89, 6275, 1251, 8555, 11469, 3297, 993, 10921,
    3242, 3964, 1913, 949, 1262, 3, 10139, 916, 7571, 1599, 730, 262, 8596, 757, 965, 630, 4189,
    8286, 16811, 1755, 12610, 5961, 7799, 71, 1966, 6010, 29047, 26831, 35, 9879, 3693, 4785, 1953,
    357, 238, 3406, 714, 1064, 6518, 3729, 1787, 2718, 167, 1819, 2819, 6329, 301, 310, 4497, 4421,
    834, 8267, 9281, 3787, 2322, 3754, 102, 3129, 2163, 9946, 1757, 5022, 8035, 1229, 145, 103,
    399, 417, 599, 1627, 1627, 1221, 4150, 6327, 10037, 339, 2455, 259, 14586, 790, 23973, 3126,
    1453, 1114, 89, 3801, 319, 13816, 8482, 16661, 5359, 331, 2773, 2301, 11510, 9440, 3203, 1141,
    786, 7451, 5354, 2489, 12788, 3562, 1093, 14944, 1101, 1793, 9152, 263, 167, 10356, 8370, 6545,
    51, 2004, 2031, 498, 2027, 2657, 2358, 183, 1569, 1084, 4003, 12548, 241, 12409, 4161, 218,
    481, 1543, 1115, 1311, 5409, 9748, 2839, 21673, 6149, 8001, 25953, 6043, 3965, 1223, 2334, 745,
    16238, 865, 89, 1047, 5284, 1816, 477, 12767, 11022, 10422, 2523, 5139, 14177, 823, 2108,
    25035, 983, 73, 4089, 5113, 6821, 3595, 29, 1595, 833, 1323, 1051, 4677, 26134, 22220, 2635,
    3907, 2417, 1574, 5134, 7999, 9241, 1754, 5441, 14097, 5378, 8468, 3562, 5091, 8370, 442, 4235,
    8125, 34286, 12416, 3547, 510, 4351, 15718, 82, 2751, 4807, 2297, 4923, 287, 114, 1404, 1161,
    909, 19956, 13860, 2434, 470, 2232, 1277, 6811, 819, 4038, 654, 51464, 5854, 4159, 1942, 1965,
    215, 6530, 671, 617, 2872, 5135, 709, 7311, 3244, 3678, 1857, 4465, 8558, 13793, 5542, 386,
    16310, 1911, 2130, 10441, 1290, 2565, 3061, 1013, 890, 947, 763, 4793, 7436, 5674, 106, 1499,
    1981, 10536, 5271, 3058, 1410, 10060, 91, 7474, 9137, 7443, 3685, 4477, 8368, 1079, 5101, 2274,
    110, 2226, 8528, 4826, 19291, 815, 745, 2173, 3307, 5798, 817, 1230, 21666, 296, 1263, 24593,
    10068, 9621, 9965, 8668, 7498, 7, 5789, 10364, 14145, 5510, 18956, 1147, 489, 4091, 146, 3745,
    9559, 883, 545, 1097, 2963, 1625, 4574, 713, 1758, 3725, 3302, 6190, 11530, 2233, 6217, 17799,
    6635, 9811, 20174, 137, 506, 23655, 247, 1478, 3931, 449, 5637, 2243, 161, 3597, 8515, 9857,
    10421, 5223, 2239, 970, 19578, 2094, 11281, 449, 9749, 10105, 1929, 59, 33, 2266, 195, 7854,
    5387, 19083, 3437, 4111, 5013, 2444, 5442, 2839, 1479, 799, 693, 571, 126, 307, 205, 3837,
    7781, 6036, 8969, 483, 555, 3593, 35838, 55, 594, 265, 10451, 1285, 2771, 1314, 3207, 8218,
    3501, 339, 263, 14423, 1157, 610, 5861, 4445, 12435, 2614, 8441, 4642, 10842, 385, 9606, 3963,
    1461, 37, 4786, 253, 1733, 13, 5727, 341, 1693, 7494, 11075, 385, 15670, 2033, 579, 927, 16444,
    151, 2563, 3114, 7149, 7964, 6299, 12218, 4604, 1966, 522, 7119, 36972, 178, 5270, 443, 1483,
    1793, 6810, 2675, 9798, 202, 3762, 621, 39118, 2367, 333, 2907, 3710, 945, 10161, 12315, 3663,
    16115, 8345, 755, 1997, 1895, 6485, 17010, 1463, 33579, 565, 24958, 2113, 1834, 20989, 603,
    1431, 13, 4211, 1453, 10090, 18288, 12109, 2378, 1363, 26914, 1478, 797, 654, 6415, 891, 1722,
    1923, 2270, 235, 3859, 469, 4207, 2516, 1119, 1630, 1159, 2814, 1483, 37797, 2774, 3037, 1861,
    1723, 9514, 9593, 1583, 9644, 37, 8409, 513, 3809, 111, 2005, 1054, 5804, 1571, 69, 1921, 7069,
    2298, 3473, 1071, 18609, 12922, 9594, 1582, 218, 3931, 3317, 2487, 2885, 25991, 333, 518, 7999,
    1238, 8163, 99, 8469, 2153, 12751, 5492, 10292, 40456, 1251, 339, 951, 6814, 6401, 1373, 2977,
    425, 755, 12431, 3399, 1378, 8745, 14073, 1154, 8391, 221, 8023, 9781, 4166, 8607, 391, 8802,
    6589, 9945, 8424, 4347, 5771, 873, 1769, 16271, 1006, 7858, 21522, 11540, 4867, 7687, 27993,
    6762, 773, 2845, 4705, 1927, 9806, 2099, 3683, 9130, 3226, 2873, 611, 1050, 70, 13307, 7191,
    1775, 9649, 101, 949, 2231, 1745, 914, 2445, 27670, 398, 1090, 6870, 1097, 3467, 1143, 12770,
    1649, 7643, 3013, 315, 11541, 3995, 412, 2310, 197, 1175, 8302, 9892, 4213, 339, 34843, 3131,
    1079, 1442, 1443, 5517, 935, 9399, 2454, 1661, 589, 1869, 5577, 11259, 7527, 1651, 4011, 9601,
    8264, 1486, 2498, 787, 20393, 5443, 3483, 373, 38723, 8183, 6503, 1457, 2175, 10699, 287, 9257,
    21455, 2033, 443, 10893, 2977, 349, 1869, 2195, 23932, 3242, 1740, 651, 4413, 469, 15581,
    23742, 16439, 4841, 1199, 2731, 2005, 1258, 442, 753, 446, 8255, 741, 2435, 1526, 7585, 4383,
    1647, 573, 1429, 1367, 377, 351, 1693, 3854, 3255, 6761, 2433, 10323, 5111, 3767, 869, 1501,
    15477, 4473, 11073, 13655, 1363, 18339, 549, 145, 12270, 1601, 2575, 20056, 1162, 603, 693,
    2659, 3210, 6107, 3213, 6453, 2565, 5529, 883, 4701, 107, 6318, 378, 8884, 5825, 8424, 865,
    8293, 11579, 301, 838, 303, 137, 6474, 3563, 2793, 2533, 2293, 153, 10827, 9226, 2157, 1507,
    9695, 2434, 8934, 5081, 14100, 18396, 1914, 12594, 39, 1886, 195, 1605, 877, 2821, 19912, 23,
    2777, 3306, 366, 8418, 462, 2843, 786, 6381, 3837, 510, 2865, 358, 233, 913, 2607, 663, 2691,
    3669, 2069, 2543, 1478, 7780, 149, 1769, 1547, 3900, 1958, 1295, 1745, 1481, 2619, 7594, 7039,
    1428, 7438, 4007, 9686, 3410, 2357, 44, 9595, 217, 3751, 1911, 19066, 10085, 3083, 8634, 2681,
    2609, 1545, 9774, 129, 114, 71, 11211, 758, 3751, 19929, 9500, 826, 2111, 977, 13772, 7941,
    1831, 28299, 12462, 10282, 4159, 1547, 2317, 527, 1893, 3862, 33473, 11277, 3342, 521, 28677,
    2994, 1870, 975, 26609, 321, 5303, 7943, 1166, 1477, 8527, 615, 10684, 1117, 981, 209, 1617,
    5453, 4364, 3950, 6922, 12727, 14739, 34165, 12715, 954, 3733, 6222, 1595, 6119, 239, 11497,
    23844, 1489, 11647, 482, 9127, 2717, 370, 239, 1247, 1374, 12559, 10695, 1153, 2619, 4120,
    14839, 29918, 6411, 6622, 8660, 9829, 4222, 1203, 1663, 10390, 19209, 1959, 297, 2650, 4838,
    1291, 1147, 26050, 14069, 6563, 2162, 5277, 4979, 851, 10544, 12007, 3163, 4365, 3978, 6162,
    119, 290, 6039, 5099, 8609, 17077, 4610, 7529, 1063, 178, 1147, 643, 3651, 438, 4334, 142,
    15128, 1537, 1839, 1905, 323, 4927, 9408, 12163, 1026, 437, 1559, 15830, 18505, 330, 1485, 47,
    73, 3547, 4286, 57, 521, 15517, 1179, 41, 3329, 17751, 7325, 319, 2797, 396, 8359, 7482, 9615,
    10750, 3532, 3999, 2521, 4327, 41, 10332, 257, 2238, 2574, 14066, 14582, 23, 215, 1391, 2182,
    3489, 526, 2950, 13569, 609, 8707, 6325, 842, 118, 1507, 2568, 719, 49509, 1771, 1926, 818,
    4733, 2938, 73, 449, 19066, 1393, 4955, 526, 2447, 11433, 5590, 1349, 951, 4233, 10278, 118,
    14668, 1707, 561, 18388, 2067, 346, 2078, 228, 8198, 1066, 690, 3205, 7863, 5073, 4511, 1291,
    5861, 4475, 9566, 4151, 5971, 1479, 6073, 502, 2351, 15292, 59, 1502, 302, 14837, 1150, 621,
    783, 9658, 2594, 9436, 11140, 8083, 8614, 519, 5522, 7827, 630, 15445, 679, 9485, 10167, 6249,
    4730, 1773, 8082, 12915, 1305, 8127, 17144, 4817, 3245, 527, 4601, 663, 8408, 3771, 7363, 683,
    9157, 157, 9047, 798, 1119, 509, 12457, 2331, 1054, 1251, 3111, 3946, 1974, 2898, 2307, 1665,
    5749, 1551, 2062, 1069, 1065, 3853, 903, 1973, 249, 30830, 3182, 130, 2056, 903, 2226, 8025,
    873, 969, 205, 2095, 1569, 11114, 1791, 46, 1357, 5907, 15633, 4738, 6515, 865, 5613, 6547,
    7527, 3865, 440, 5559, 6674, 27439, 9853, 1223, 221, 9047, 3986, 478, 2309, 3015, 1299, 547,
    13990, 10117, 3494, 2622, 8946, 401, 9050, 4762, 9835, 1187, 1405, 801, 365, 4129, 261, 10057,
    37933, 15284, 1407, 8756, 8206, 2670, 8181, 337, 775, 5243, 374, 3379, 323, 1534, 1638, 10553,
    4343, 977, 406, 95, 1355, 1271, 3349, 3329, 1405, 6513, 5779, 86, 22289, 25119, 1455, 1095,
    20870, 1054, 12848, 26989, 1823, 9005, 2485, 16240, 7718, 2326, 4115, 43, 606, 2029, 3434,
    4470, 1638, 4081, 966, 1569, 389, 55, 2435, 9878, 7378, 3229, 11218, 478, 914, 1634, 607, 1391,
    7947, 2753, 197, 397, 2571, 4191, 5317, 11120, 1269, 11016, 13372, 1395, 463, 9008, 4741, 91,
    1499, 10647, 1035, 4638, 10893, 9570, 1175, 3185, 934, 161, 8329, 25714, 19077, 2279, 3483,
    4573, 775, 2822, 1111, 9282, 1661, 621, 8741, 982, 1081, 534, 2885, 919, 9879, 1685, 629, 6489,
    4671, 10515, 3185, 846, 807, 2626, 5662, 958, 2963, 4065, 1355, 545, 1423, 6246, 170, 153, 215,
    43, 5849, 6989, 550, 1054, 1555, 4430, 10544, 40324, 783, 17382, 4198, 1059, 1422, 354, 2301,
    7095, 2305, 398, 634, 10147, 1453, 2782, 577, 9793, 2255, 290, 3839, 13838, 14932, 11919, 6987,
    3535, 89, 973, 93, 1695, 805, 2687, 4051, 3551, 3989, 17180, 2749, 651, 7627, 8398, 4199, 2606,
    721, 305, 8642, 10101, 4439, 30733, 3602, 3771, 1902, 2987, 1927, 2687, 4049, 8213, 6307,
    11400, 11612, 818, 2777, 4021, 40185, 6593, 13200, 670, 5985, 11212, 66, 890, 3411, 437, 13445,
    8749, 1379, 1879, 2138, 907, 1421, 3261, 1703, 10645, 3275, 1079, 9765, 4679, 1261, 11870,
    8414, 87, 7522, 7039, 10589, 2616, 298, 1249, 6941, 10817, 7317, 2509, 9977, 1382, 179, 3503,
    2658, 3961, 1249, 473, 31983, 9819, 6623, 19379, 2263, 37, 66, 4758, 1301, 1625, 2451, 20156,
    10907, 1893, 1986, 2964, 3455, 12563, 1987, 8673, 6974, 13852, 5550, 1153, 15110, 4797, 9311,
    9230, 3799, 398, 8447, 8494, 603, 2129, 3917, 2338, 2058, 6137, 6267, 4734, 7665, 1462, 10183,
    2274, 4783, 2073, 8207, 14275, 9197, 247, 1555, 1602, 179, 1053, 1190, 15233, 12075, 14577,
    2951, 1409, 2737, 1745, 11054, 24941, 2854, 9399, 347, 1019, 117, 349, 4095, 6808, 5281, 909,
    2311, 2203, 513, 1833, 31731, 8459, 1971, 8550, 3382, 4123, 683, 3577, 1037, 6875, 8483, 17092,
    483, 6993, 633, 3399, 218, 1459, 1630, 829, 10455, 403, 8291, 24536, 599, 271, 27199, 3023,
    8503, 626, 566, 9493, 5238, 3781, 5567, 382, 12526, 11596, 9404, 10863, 8311, 2862, 1102, 89,
    3857, 15788, 3081, 1827, 13293, 1457, 1949, 4183, 12474, 1325, 2237, 3813, 3927, 701, 1405, 87,
    3258, 319, 5803, 1731, 8426, 542, 423, 2247, 9539, 4690, 13762, 8167, 379, 3531, 965, 1881,
    897, 2279, 21513, 16587, 11605, 5462, 4619, 5099, 33763, 8591, 437, 4849, 626, 3397, 889, 74,
    3779, 8476, 13412, 6711, 475, 5186, 859, 2414, 3302, 725, 823, 4527, 10127, 1098, 1703, 891,
    15887, 1683, 9632, 1137, 892, 1935, 34001, 1955, 5393, 1685, 513, 644, 921, 3634, 1502, 8676,
    5257, 783, 4599, 7149, 6567, 1343, 844, 1326, 2874, 6733, 3410, 26710, 10845, 3782, 16851, 510,
    106, 8566, 2170, 3293, 12036, 4353, 1746, 4006, 15690, 421, 2079, 8915, 59, 82, 905, 3715,
    11867, 1213, 3739, 4747, 1535, 1207, 4710, 1737, 7055, 2173, 2261, 10510, 11921, 2098, 6698,
    12883, 887, 536, 2729, 869, 4879, 5786, 794, 10244, 3434, 483, 19650, 1141, 401, 898, 1592,
    2650, 7205, 1338, 1007, 1471, 10729, 2565, 1218, 11544, 1519, 2522, 1075, 611, 3049, 16378,
    15818, 1891, 3203, 4234, 373, 5363, 1010, 598, 5794, 1187, 2358, 9752, 4490, 901, 8639, 15994,
    8821, 830, 4233, 16956, 2121, 157, 261, 133, 51, 4818, 3094, 354, 9414, 2271, 1746, 9036, 401,
    11165, 14693, 1175, 4198, 9202, 8891, 7081, 906, 145, 2643, 3571, 1633, 1173, 30191, 5135,
    4201, 19172, 437, 15320, 4420, 102, 2547, 3655, 3121, 535, 497, 493, 3994, 5911, 1498, 143,
    7135, 9386, 3038, 27732, 4663, 13104, 1295, 1899, 2709, 12022, 23117, 14947, 627, 5470, 4340,
    5787, 869, 929, 1575, 11354, 157, 962, 6809, 9877, 559, 923, 5923, 1269, 13261, 2367, 4801,
    371, 4179, 903, 509, 2082, 2481, 655, 233, 6782, 8979, 806, 743, 2678, 1462, 13515, 6119,
    14899, 8872, 7517, 709, 1010, 862, 477, 2621, 1429, 6131, 14911, 603, 133, 19164, 6842, 603,
    1022, 76, 1987, 2631, 4965, 747, 5474, 1227, 371, 11016, 9862, 6775, 9580, 23309, 8884, 20046,
    8318, 1342, 37, 2022, 101, 2058, 8226, 101, 2442, 29389, 1482, 773, 1957, 2685, 325, 6044,
    11425, 1007, 3265, 4095, 907, 2941, 422, 4641, 1177, 2333, 32686, 1473, 2655, 10225, 6331,
    2323, 793, 14119, 15186, 6557, 677, 991, 606, 11522, 339, 717, 2118, 1702, 5409, 2968, 2147,
    3555, 897, 8343, 3057, 5461, 170, 884, 255, 111, 2145, 13382, 3657, 4548, 13812, 6090, 8519,
    9254, 10108, 3210, 825, 897, 23, 2726, 10008, 923, 537, 4318, 1724, 3484, 585, 8085, 1747,
    4078, 167, 16127, 2649, 16040, 519, 1295, 1194, 1353, 982, 1653, 445, 525, 1501, 1183, 23072,
    894, 15270, 18425, 2013, 2863, 2245, 1017, 217, 3041, 2813, 7972, 9059, 11248, 281, 4274, 387,
    1730, 8271, 1435, 4631, 10011, 443, 457, 6991, 1198, 266, 554, 6550, 3505, 203, 19580, 1686,
    24714, 3185, 29, 3394, 6999, 543, 7750, 118, 3080, 1661, 1205, 381, 13694, 157, 1593, 122,
    2641, 7385, 24921, 9386, 2371, 762, 171, 4278, 78, 8697, 787, 1078, 789, 8794, 1066, 7898, 171,
    9777, 415, 4623, 1385, 159, 22692, 167, 1395, 253, 6791, 957, 4831, 21354, 3745, 1113, 1935,
    1033, 1861, 7304, 7317, 8609, 402, 2402, 8974, 26162, 8318, 957, 3076, 2687, 234, 3449, 869,
    2445, 267, 3605, 787, 771, 2327, 4058, 1871, 13607, 265, 2006, 1626, 917, 6579, 14284, 13292,
    990, 15425, 12117, 4411, 5086, 29949, 67, 1590, 627, 3769, 3815, 2939, 42, 1311, 5855, 2161,
    8900, 5663, 2434, 9405, 5305, 8730, 339, 3289, 934, 190, 2254, 4565, 17751, 123, 8771, 982,
    5857, 1009, 2221, 5934, 1842, 4985, 7406, 9149, 8689, 34907, 4286, 331, 559, 2623, 774, 1165,
    970, 3479, 34963, 2793, 906, 18296, 547, 439, 29, 5199, 117, 645, 546, 13966, 5634, 2391, 589,
    6619, 2285, 1021, 1547, 14562, 3798, 2695, 2314, 893, 970, 11437, 55, 1123, 15025, 9319, 130,
    333, 12497, 2089, 10216, 4311, 4043, 273, 6423, 2477, 2591, 1563, 2211, 1689, 6558, 1689, 7411,
    6686, 4459, 24641, 755, 7035, 251, 929, 1465, 1749, 595, 4395, 3971, 6079, 745, 4943, 9910,
    7731, 699, 299, 521, 2338, 2753, 9228, 2091, 2837, 519, 103, 20129, 1077, 329, 3689, 3483,
    1993, 10022, 330, 9251, 4078, 19689, 9570, 1061, 1953, 2710, 1019, 7571, 7945, 123, 1399, 2015,
    1266, 3545, 36069, 85, 4993, 10192, 4195, 1047, 1322, 2565, 2638, 257, 2999, 22664, 709, 23,
    24060, 491, 5907, 7529, 55, 8134, 10110, 5731, 10379, 3414, 195, 10983, 28705, 3445, 3341,
    9242, 8220, 369, 2142, 8873, 583, 5713, 2263, 341, 6426, 2917, 133, 7606, 27717, 8690, 5107,
    5495, 22065, 377, 6202, 10001, 15505, 3742, 3766, 1913, 2819, 1825, 4799, 5377, 7385, 1533,
    2129, 10118, 8711, 29298, 1316, 11677, 54862, 3514, 2235, 6815, 8121, 2293, 2810, 565, 6410,
    1130, 9130, 489, 3022, 1671, 4491, 3790, 1395, 15396, 1222, 6070, 452, 9601, 13623, 6935, 4118,
    862, 2438, 3854, 1604, 12305, 2750, 273, 22, 13840, 127, 9788, 4054, 3689, 3286, 935, 6019,
    2518, 2525, 3245, 14826, 9275, 14709, 2525, 14227, 12697, 2390, 7149, 1991, 1565, 8038, 2893,
    1261, 5947, 181, 6799, 4973, 3791, 3478, 5197, 1531, 559, 142, 23379, 650, 206, 1355, 129,
    11545, 10962, 5726, 8517, 9162, 5467, 407, 2931, 5341, 1197, 5473, 2290, 1537, 6024, 133, 2419,
    17301, 73, 82, 806, 2918, 11242, 12158, 3301, 182, 7227, 9626, 37807, 7674, 3097, 10453, 31919,
    9515, 297, 11208, 4826, 2164, 3417, 11280, 1122, 8042, 1173, 595, 847, 993, 8297, 274, 5887,
    1281, 8320, 5281, 1438, 569, 107, 719, 26808, 714, 3915, 2925, 205, 611, 339, 4917, 14016,
    8115, 682, 7423, 6747, 7054, 1821, 21073, 6905, 1043, 11643, 3327, 8373, 1130, 520, 1642, 1287,
    1382, 2225, 342, 723, 741, 4385, 9087, 719, 3810, 1330, 14038, 302, 6681, 4102, 433, 1119,
    16577, 2247, 6009, 4599, 2821, 2589, 2307, 538, 1418, 2249, 322, 27193, 4115, 2516, 20100,
    2807, 20535, 15819, 2518, 399, 2030, 182, 24333, 14385, 3166, 6619, 8963, 1902, 1837, 2615,
    822, 773, 526, 4723, 2726, 373, 5131, 771, 6095, 2927, 3241, 943, 367, 1831, 901, 298, 12393,
    3354, 2374, 3178, 979, 458, 5953, 2287, 4407, 1166, 3751, 114, 9393, 2251, 863, 7979, 277,
    13168, 6229, 909, 1986, 1417, 4259, 12953, 1570, 5479, 2567, 7471, 1787, 782, 851, 611, 3318,
    12418, 4990, 2066, 5559, 9621, 13700, 4969, 863, 1769, 1179, 235, 1809, 797, 11136, 3561, 9594,
    505, 1368, 330, 40770, 1529, 1782, 4317, 8371, 5947, 7619, 411, 4593, 1391, 8270, 6290, 9685,
    4295, 526, 11247, 36028, 3471, 731, 7501, 995, 2253, 12660, 1621, 6662, 568, 6054, 577, 4076,
    23138, 3485, 63, 1270, 9499, 6019, 365, 871, 11, 11477, 2206, 1662, 542, 13541, 527, 2054,
    1037, 3369, 7314, 7799, 7403, 307, 807, 3714, 10753, 823, 697, 239, 1453, 11305, 2349, 8890,
    7953, 4971, 23987, 3823, 3091, 5555, 15859, 1689, 697, 3621, 19982, 8050, 1491, 22508, 1462,
    603, 4349, 3903, 1179, 1283, 1418, 2267, 3537, 689, 4962, 1059, 5890, 2267, 12155, 4865, 1365,
    26935, 3195, 191, 40521, 9127, 27623, 67, 2902, 8511, 47, 15914, 8984, 7613, 3511, 8678, 439,
    1554, 2526, 73, 2874, 2471, 8861, 117, 1124, 9084, 6043, 7698, 1035, 2366, 5009, 515, 24582,
    2229, 263, 1713, 5391, 21108, 46805, 7343, 1199, 1342, 813, 1063, 173, 2243, 945, 1517, 14286,
    321, 7561, 1683, 9676, 76, 1202, 8700, 6605, 4551, 1847, 53, 322, 4597, 1023, 715, 15929, 2643,
    11978, 4383, 4773, 2825, 57371, 61, 386, 1548, 2677, 5291, 2655, 38702, 9825, 587, 193, 6359,
    1651, 9860, 8989, 14872, 7036, 8038, 3877, 52242, 1043, 4583, 6571, 4914, 18419, 1962, 17405,
    295, 8411, 855, 779, 498, 9139, 3685, 14424, 19, 8131, 29477, 6331, 681, 1050, 2041, 9256,
    16083, 4615, 8365, 2885, 3014, 975, 1553, 5759, 29006, 2623, 9574, 3859, 1769, 3339, 3330,
    4265, 271, 38434, 39, 5898, 26227, 8563, 8734, 9987, 2967, 770, 8305, 14856, 1109, 9109, 553,
    5818, 17466, 2567, 8540, 6623, 1227, 4242, 2739, 1911, 2105, 881, 796, 12709, 923, 4709, 6851,
    26911, 895, 5179, 751, 182, 2305, 11650, 733, 3442, 9766, 3203, 8386, 33, 1385, 2069, 6731,
    9040, 12617, 3926, 91, 3519, 3131, 1441, 1319, 2097, 83, 1203, 4780, 2506, 2287, 5829, 2179,
    3211, 7382, 3434, 1045, 3734, 2990, 4739, 3158, 63, 1939, 1314, 2790, 5827, 7193, 1207, 2018,
    1079, 1033, 4105, 1642, 7371, 7065, 1098, 1475, 1674, 2930, 5427, 655, 7946, 9651, 2951, 766,
    397, 8643, 4979, 341, 4925, 193, 70, 3589, 9921, 19, 775, 14671, 3107, 2909, 1073, 170, 9447,
    13390, 3803, 8178, 10608, 2763, 3150, 809, 501, 1636, 8811, 2163, 941, 7045, 7770, 4721, 1406,
    7862, 3735, 3573, 1651, 3293, 822, 373, 8534, 7869, 505, 5085, 2569, 2106, 9380, 4382, 942,
    1409, 6689, 5697, 2103, 9774, 7094, 4459, 399, 8650, 4527, 666, 2967, 1019, 2741, 347, 2494,
    9470, 4011, 5945, 5370, 3891, 6207, 7117, 3073, 2210, 3809, 10774, 1379, 14140, 7123, 1035,
    1149, 238, 747, 6076, 849, 915, 1810, 161, 20518, 2266, 3431, 4089, 6851, 17348, 1143, 3077,
    281, 17372, 903, 2107, 2633, 182, 997, 467, 489, 495, 333, 7678, 1497, 269, 12609, 33, 1443,
    1631, 397, 2554, 499, 4550, 286, 741, 193, 2967, 2695, 5362, 395, 3778, 1329, 10132, 13834,
    6215, 311, 1778, 4626, 13933, 62, 465, 12052, 9853, 4904, 2373, 274, 937, 253, 461, 59, 2686,
    809, 4594, 9303, 2838, 446, 14486, 2917, 1290, 175, 7907, 285, 1830, 26995, 267, 1743, 1231,
    219, 3815, 9634, 110, 6751, 3519, 1168, 4957, 2945, 109, 2001, 5042, 9565, 4059, 2072, 4457,
    202, 3441, 4363, 839, 3615, 9345, 4031, 2161, 618, 3035, 889, 5209, 5474, 3967, 154, 538, 2181,
    7118, 1941, 3763, 9393, 453, 149, 12456, 431, 654, 145, 1274, 226, 65, 5459, 649, 779, 13811,
    9822, 47855, 3087, 1505, 342, 1341, 2685, 801, 826, 2171, 32511, 1059, 10653, 1135, 345, 16271,
    2538, 593, 9058, 15071, 3223, 5238, 1003, 5701, 17488, 962, 8803, 7137, 222, 804, 2609, 15770,
    1029, 5527, 22081, 586, 1390, 5445, 3922, 8273, 1414, 9341, 923, 433, 28522, 1247, 5179, 9045,
    503, 4109, 299, 395, 2658, 2909, 6078, 3640, 2549, 1189, 5919, 17749, 12627, 843, 755, 374,
    799, 25971, 10758, 2713, 9154, 148, 747, 342, 2669, 1678, 7153, 1162, 2978, 674, 606, 435,
    6027, 1357, 355, 10008, 3901, 10330, 5946, 2410, 8216, 1966, 20319, 1683, 6781, 449, 1011, 39,
    3111, 5457, 3910, 2709, 8531, 9421, 7577, 14622, 322, 12918, 36524, 14046, 321, 2799, 1917,
    1801, 6190, 3247, 1590, 3278, 10549, 6709, 711, 1317, 927, 10035, 4067, 1189, 670, 5245, 7286,
    2139, 6783, 694, 2423, 9812, 2301, 9187, 189, 1469, 409, 83, 9407, 5927, 1486, 19888, 127,
    5817, 991, 4534, 355, 10037, 8431, 4747, 2491, 547, 3035, 67, 2035, 14503, 290, 9555, 411,
    3942, 4574, 3746, 1699, 3653, 12261, 5837, 8398, 107, 1653, 202, 7094, 1543, 1318, 4403, 1557,
    899, 5134, 5898, 4954, 2861, 370, 10007, 4438, 8844, 1322, 1806, 15540, 651, 985, 10316, 8791,
    5265, 4358, 11917, 2234, 1419, 4619, 1579, 1479, 5183, 2597, 22, 5591, 2925, 2552, 13786, 1319,
    10397, 5417, 1287, 1463, 218, 4971, 5157, 8675, 1923, 1683, 1529, 2713, 1510, 683, 3841, 4043,
    711, 62, 8435, 829, 4775, 693, 3107, 4997, 339, 1810, 339, 1437, 2303, 2422, 93, 249, 10867,
    1689, 609, 1162, 2697, 394, 6253, 4465, 1863, 337, 3914, 19686, 2294, 2227, 1637, 613, 10573,
    5094, 15483, 369, 1635, 6043, 82, 2030, 6468, 9803, 1597, 6758, 3678, 1798, 491, 12148, 364,
    1381, 789, 6971, 2105, 95, 350, 193, 13234, 3180, 5201, 6044, 285, 1406, 4643, 741, 8730, 527,
    4091, 15388, 2692, 15028, 74, 13039, 1635, 1853, 2570, 6139, 1741, 3747, 4045, 513, 3679, 286,
    2635, 1950, 1901, 4954, 3043, 232, 2469, 8470, 1575, 1642, 3347, 1145, 1674, 1194, 9533, 4781,
    446, 859, 385, 4193, 1562, 73, 660, 501, 582, 2125, 5691, 13373, 3372, 5429, 3775, 2431, 2269,
    13287, 16654, 3438, 6238, 911, 4030, 463, 3110, 15501, 210, 8399, 6351, 11563, 2139, 122, 1478,
    15307, 7661, 11932, 7553, 3197, 371, 23131, 5229, 3315, 1766, 1506, 10929, 7162, 3260, 2466,
    4934, 5685, 4899, 2981, 29, 2378, 913, 12329, 373, 185, 1462, 425, 1767, 9326, 33, 6063, 22818,
    4217, 7999, 8355, 20228, 10667, 49583, 3447, 1522, 489, 20148, 5467, 5866, 1309, 229, 239, 862,
    3101, 431, 3745, 2894, 3298, 2449, 8454, 10880, 1277, 7015, 2743, 1229, 1317, 2451, 2079, 4006,
    8210, 1374, 1179, 1229, 1717, 2333, 7389, 395, 5745, 12014, 4018, 317, 6447, 8587, 9440, 1097,
    515, 10104, 1673, 2607, 181, 10081, 474, 6927, 10056, 19017, 2350, 3609, 27185, 9227, 207,
    4201, 14624, 357, 3111, 58, 1055, 9607, 2949, 1101, 4771, 30703, 2630, 79, 3587, 2342, 11631,
    3582, 1517, 230, 1165, 6773, 2077, 35618, 649, 3522, 1674, 1654, 1031, 233, 14118, 1574, 8877,
    3257, 10400, 6959, 2785, 7715, 6649, 2707, 274, 833, 9277, 624, 3514, 2979, 247, 5197, 8459,
    889, 18575, 1851, 8337, 31, 1703, 1442, 17048, 7729, 1969, 1351, 10114, 1443, 1165, 807, 1262,
    3867, 24831, 1390, 13939, 4021, 2442, 8577, 26737, 5110, 3614, 257, 2455, 9174, 2027, 7903,
    1743, 129, 1209, 12840, 9274, 8019, 1902, 11917, 371, 1703, 1746, 8897, 8691, 38333, 1537,
    1202, 1954, 483, 9852, 7315, 1978, 103, 2598, 446, 175, 8149, 2186, 7787, 155, 5077, 1546,
    5986, 6349, 6025, 385, 3679, 8377, 5899, 693, 3029, 966, 1047, 11, 13947, 14610, 10278, 4586,
    226, 3171, 2379, 6511, 31, 10692, 9355, 9250, 9359, 20510, 487, 998, 6487, 8936, 3247, 840,
    8678, 89, 4955, 1859, 2054, 9528, 1015, 157, 458, 695, 130, 9122, 19072, 470, 5157, 7117, 1286,
    3333, 11906, 9496, 123, 7438, 6563, 1599, 6547, 538, 8399, 449, 463, 8099, 3014, 2226, 15461,
    47, 1281, 7461, 1115, 6970, 139, 5646, 1105, 14044, 2951, 5369, 8489, 33253, 4758, 13469, 309,
    2838, 1859, 2465, 11666, 1277, 11864, 4887, 4111, 3866, 1409, 1209, 3857, 2147, 1139, 3623,
    14799, 7041, 3411, 438, 3021, 3023, 9574, 3577, 723, 2955, 161, 1951, 415, 682, 4398, 4971,
    419, 17664, 2639, 1377, 10449, 103, 4559, 27115, 28468, 4417, 333, 14904, 25458, 2102, 3642,
    3749, 2329, 7471, 3559, 2226, 86, 1206, 8334, 5271, 9915, 6105, 587, 26801, 3914, 7399, 6322,
    1382, 799, 74, 7013, 10693, 274, 12410, 7763, 354, 1063, 183, 1347, 1349, 3638, 3326, 15180,
    2174, 1192, 1446, 3497, 2395, 3117, 6647, 5111, 595, 1523, 443, 6054, 2631, 1886, 6549, 758,
    4709, 1294, 1123, 5215, 137, 335, 1334, 1333, 4810, 4478, 438, 2387, 4031, 741, 4401, 2261,
    1241, 411, 403, 2331, 171, 814, 8354, 1525, 6051, 1418, 4607, 12084, 2103, 4249, 11626, 3098,
    21244, 4325, 7061, 2109, 1735, 213, 833, 473, 126, 117, 5229, 7151, 39, 1859, 8443, 2471, 505,
    4754, 970, 1820, 285, 47682, 1897, 4869, 4187, 1113, 13768, 3735, 935, 1139, 8229, 3197, 994,
    1302, 2371, 4333, 6110, 15553, 1267, 179, 285, 7429, 667, 6645, 895, 2811, 14, 140, 2318, 1386,
    117, 11149, 2318, 1189, 215, 4781, 12607, 8236, 4958, 14578, 8654, 37, 1309, 2289, 1113, 8538,
    105, 127, 5318, 11727, 8730, 1439, 321, 6133, 1395, 367, 2613, 1877, 6071, 10521, 1795, 8359,
    1005, 858, 12404, 2453, 7490, 2703, 3889, 2066, 562, 18950, 15869, 703, 3466, 5883, 257, 5393,
    10612, 2402, 2633, 362, 783, 1786, 8408, 1473, 813, 27802, 1706, 9446, 3063, 2354, 5995, 3577,
    14446, 4817, 2629, 2946, 254, 145, 11371, 5258, 2979, 14220, 147, 1915, 8505, 1699, 402, 10499,
    95, 359, 970, 3731, 4567, 990, 3495, 7116, 5419, 1935, 1086, 6577, 3277, 185, 9799, 2551,
    15059, 319, 5918, 7501, 11205, 2125, 8309, 10646, 6881, 3474, 348, 42908, 970, 277, 5022, 2129,
    377, 9896, 6943, 443, 9066, 1255, 1318, 1833, 2119, 18553, 9716, 10022, 1094, 850, 8320, 8855,
    3291, 662, 2202, 1475, 1762, 5257, 986, 1603, 2743, 2941, 2489, 2549, 6863, 8133, 8137, 3117,
    3815, 5799, 779, 766, 8752, 2351, 13977, 16855, 9069, 339, 9230, 9328, 9203, 6742, 3335, 4002,
    1459, 2635, 3189, 1229, 731, 3107, 1541, 154, 3545, 13304, 2931, 4343, 655, 3507, 16263, 6987,
    533, 15688, 12163, 10170, 1269, 1877, 1293, 274, 5739, 5521, 2046, 7775, 24676, 5727, 2110,
    10528, 85, 13, 14210, 581, 1510, 2146, 33, 2202, 1519, 527, 4566, 902, 2748, 7114, 11962,
    14540, 22072, 801, 12324, 1862, 1101, 5259, 3679, 1929, 1333, 4989, 34918, 3023, 449, 4433,
    3795, 2763, 5457, 10021, 2943, 1785, 11181, 3007, 1543, 6429, 3363, 5315, 11071, 12203, 1361,
    370, 2358, 34741, 782, 2165, 4307, 3890, 12440, 2929, 4013, 1451, 5279, 4910, 305, 12146, 4497,
    10214, 14631, 4479, 1974, 3963, 2606, 1333, 1582, 1640, 1222, 10582, 2529, 1233, 3127, 4821,
    4143, 2310, 1015, 13400, 9325, 34259, 3179, 1594, 3513, 8826, 4104, 9417, 8449, 6017, 10718,
    7563, 1251, 5756, 5469, 15173, 1330, 1554, 1321, 1302, 3283, 373, 109, 3810, 4155, 5849, 3389,
    1971, 8506, 313, 10045, 617, 743, 357, 366, 7485, 6342, 2243, 1859, 8690, 157, 15486, 1967,
    2010, 3451, 27013, 2866, 485, 7083, 1282, 4794, 573, 2162, 518, 16566, 222, 789, 3109, 4983,
    2722, 527, 391, 962, 4185, 9480, 6005, 1413, 2177, 17510, 3994, 10121, 399, 4138, 5794, 251,
    12083, 517, 2075, 2357, 8111, 339, 7375, 15860, 8574, 2243, 14859, 1407, 1155, 8262, 1897,
    3062, 3141, 308, 8384, 9033, 1337, 2249, 254, 374, 1998, 11342, 11044, 23, 413, 3635, 504, 455,
    8494, 95, 6147, 2395, 247, 3305, 6891, 669, 12226, 43544, 181, 9971, 2533, 8568, 4697, 230,
    1511, 133, 362, 727, 2869, 1003, 10855, 253, 402, 4582, 11298, 10929, 215, 7209, 10601, 482,
    2085, 3455, 9761, 233, 14870, 8417, 1731, 621, 12763, 1134, 13976, 306, 4354, 6499, 1694, 2350,
    937, 11305, 991, 11800, 1299, 5309, 873, 1357, 5909, 28187, 181, 1335, 3858, 325, 9707, 318,
    2493, 7815, 3279, 3321, 511, 2138, 5035, 11146, 397, 819, 1717, 2557, 14642, 9775, 4054, 2024,
    8979, 4967, 13415, 1105, 8972, 35, 1446, 9868, 73, 4383, 6711, 2651, 2109, 1046, 1257, 499,
    2247, 6835, 1755, 7941, 14130, 21203, 8469, 6782, 716, 1747, 7894, 2241, 2301, 9206, 5782,
    5430, 5742, 899, 1347, 602, 11241, 1621, 74, 863, 311, 1693, 626, 1794, 5243, 1223, 1423, 1763,
    8996, 2067, 12458, 555, 6965, 5027, 1987, 5026, 1719, 953, 1341, 654, 1988, 815, 11669, 2947,
    4055, 1173, 2525, 783, 6595, 9856, 3939, 8645, 813, 3877, 2174, 6513, 4214, 537, 301, 6850,
    3485, 7623, 517, 18742, 1385, 399, 7826, 897, 33146, 7021, 6086, 1174, 3115, 4278, 8094, 1289,
    1657, 2578, 2241, 39, 39974, 582, 9018, 2993, 3378, 16527, 3597, 1349, 9484, 2202, 585, 498,
    8213, 937, 19053, 554, 637, 4999, 318, 199, 91, 791, 5794, 1762, 3205, 10056, 8886, 2163, 9902,
    1143, 1667, 5551, 6703, 8784, 741, 1378, 11076, 1528, 766, 7275, 9554, 6333, 3571, 5218, 1687,
    6631, 9539, 473, 10893, 5802, 342, 3086, 5474, 14916, 878, 1213, 3157, 4025, 9185, 8399, 7093,
    21371, 2930, 8555, 21134, 4923, 1598, 9528, 2117, 8697, 5813, 1262, 5351, 14227, 10024, 16760,
    8309, 5398, 322, 74, 9981, 11504, 6095, 1874, 37638, 2115, 2346, 326, 13138, 25275, 1097,
    10315, 12230, 489, 333, 11167, 739, 3641, 521, 1621, 11779, 7235, 731, 677, 9663, 1255, 12508,
    634, 2950, 16475, 7138, 1261, 4559, 7565, 1357, 4795, 813, 1051, 2277, 12431, 4955, 611, 10575,
    26482, 1043, 12469, 7439, 6783, 1498, 5509, 3195, 7370, 1281, 1163, 5322, 16860, 1861, 2267,
    1833, 1163, 1806, 3181, 4579, 2411, 1821, 3117, 7659, 1698, 438, 9745, 533, 9273, 6177, 2862,
    3011, 1177, 2866, 5391, 3406, 455, 10889, 10236, 37, 1467, 1603, 1107, 4933, 974, 9404, 115,
    2141, 442, 9723, 2315, 13, 7718, 971, 481, 4615, 187, 6898, 3458, 3961, 1473, 9222, 1870, 1485,
    8338, 3930, 522, 14163, 12509, 1854, 1241, 7301, 12462, 8254, 1949, 319, 1282, 10822, 31018,
    1045, 4810, 8198, 145, 13982, 13441, 7245, 3414, 2269, 10344, 9899, 3757, 1861, 674, 1467,
    2232, 2005, 2225, 2241, 153, 2397, 2573, 61, 1861, 1285, 8889, 13781, 8672, 2819, 13866, 541,
    318, 19373, 319, 3003, 1061, 9303, 954, 2493, 1817, 11212, 2577, 858, 103, 5003, 571, 1693,
    10243, 2331, 1097, 9877, 4177, 9255, 3050, 8260, 18974, 5963, 3189, 2823, 2967, 1418, 11386,
    6188, 29154, 7939, 926, 5549, 78, 979, 4338, 259, 2330, 2666, 5901, 5187, 803, 8222, 86, 9079,
    1271, 5391, 4279, 3798, 7359, 1385, 487, 4771, 67, 8175, 941, 259, 11494, 4110, 5379, 1206,
    522, 1890, 226, 24149, 4011, 114, 1707, 261, 2515, 8358, 1771, 850, 365, 663, 19574, 2482,
    8724, 9203, 475, 563, 8925, 830, 4037, 4973, 7211, 291, 10869, 15689, 234, 482, 1093, 13607,
    2363, 3114, 3465, 93, 818, 1649, 1974, 1351, 2402, 381, 681, 1940, 534, 4699, 3855, 1233, 397,
    8475, 22934, 15726, 523, 7653, 3369, 9177, 1997, 3014, 3729, 878, 7231, 2294, 13194, 31, 2591,
    4718, 213, 4235, 5257, 8179, 533, 1405, 1127, 5673, 1611, 727, 153, 531, 1053, 5354, 4415,
    9494, 1015, 5933, 1461, 8264, 3913, 2737, 2509, 199, 102, 3193, 287, 470, 9547, 1543, 1546,
    4745, 4959, 8644, 4069, 12063, 2805, 3130, 834, 246, 3911, 751, 5963, 28947, 129, 3615, 43, 37,
    4453, 3, 3826, 1562, 8399, 1434, 730, 6961, 4977, 2473, 15820, 8329, 1530, 623, 10531, 6653,
    21731, 731, 1074, 1041, 1430, 9721, 2921, 251, 1115, 222, 8393, 4211, 163, 4258, 7385, 2533,
    1499, 99, 2203, 1407, 1357, 41, 2509, 6949, 3145, 4003, 1446, 7611, 9415, 5447, 817, 5207,
    4325, 5717, 427, 2007, 8159, 87, 1848, 15378, 9005, 5170, 398, 103, 9519, 13692, 4015, 12125,
    3178, 3286, 8779, 1687, 563, 9770, 278, 2129, 25951, 24657, 5030, 682, 3250, 3743, 5659, 34,
    3623, 1939, 3185, 11304, 58, 17944, 1886, 166, 1002, 5593, 881, 443, 542, 1714, 5301, 1689,
    1438, 3694, 12812, 518, 906, 850, 755, 6251, 2649, 327, 2653, 403, 8677, 530, 110, 290, 5645,
    2695, 670, 8318, 3149, 1958, 4705, 894, 1503, 12507, 159, 177, 83, 609, 7035, 2915, 6483, 6749,
    5694, 139, 3317, 149, 3515, 1162, 2577, 11452, 14934, 1779, 1510, 1052, 36540, 9387, 743, 3879,
    1335, 5123, 8486, 3767, 183, 7522, 1449, 33329, 637, 941, 7581, 449, 6114, 609, 938, 1594,
    5627, 5361, 1146, 2253, 9878, 1273, 2737, 770, 2007, 4474, 8346, 373, 3615, 14282, 31, 755,
    6034, 5438, 34365, 1622, 7157, 4298, 11043, 554, 15964, 1677, 1507, 9169, 1609, 24594, 3745,
    805, 387, 4457, 5891, 2158, 10954, 1699, 2229, 3695, 5958, 3255, 351, 1642, 4079, 8827, 762,
    1062, 3016, 8501, 12296, 2874, 8213, 2549, 19964, 1306, 4187, 565, 3886, 3563, 12387, 214,
    13743, 2795, 11443, 2067, 1741, 3753, 234, 691, 351, 7268, 253, 5287, 6175, 11736, 2341, 1071,
    1395, 3525, 8719, 4095, 1546, 3585, 1669, 1171, 3997, 6958, 12762, 9308, 664, 1009, 1775, 4383,
    5635, 1141, 1313, 2259, 2635, 9023, 1537, 491, 1038, 1373, 2005, 5274, 4719, 351, 5597, 241,
    2167, 14539, 3814, 3666, 2113, 399, 109, 1929, 29930, 2647, 1142, 1941, 781, 7114, 2743, 11414,
    1415, 217, 83, 1839, 5764, 1535, 21484, 2461, 1249, 3463, 4701, 11678, 1065, 6953, 8981, 5141,
    5990, 10440, 874, 7163, 2053, 9647, 522, 3565, 919, 8959, 3321, 13, 93, 9736, 2883, 313, 860,
    4565, 27869, 1610, 11508, 1581, 24114, 10586, 1162, 849, 274, 142, 16797, 303, 7073, 8222,
    1315, 89, 2929, 11501, 4363, 1790, 671, 2520, 366, 617, 3349, 1061, 14962, 15408, 1127, 4202,
    8493, 1767, 765, 1353, 2318, 5579, 1290, 13840, 191, 797, 1357, 3559, 2556, 13705, 2798, 277,
    2725, 2138, 4227, 4517, 1731, 35782, 7298, 7102, 863, 3449, 8479, 8531, 585, 3003, 2103, 13725,
    254, 2319, 6935, 11724, 10846, 341, 13048, 25829, 1686, 539, 6523, 1769, 1466, 1153, 1866, 203,
    24088, 503, 5403, 2510, 7045, 37, 14181, 3629, 751, 1469, 8400, 4935, 13354, 2515, 6118, 2949,
    8357, 185, 2826, 21389, 1053, 3885, 606, 3179, 334, 379, 4595, 66, 2920, 2687, 8827, 1705,
    2091, 691, 1930, 4435, 921, 126, 219, 434, 5950, 12683, 3305, 7778, 7283, 9113, 4038, 122,
    17279, 794, 10620, 1289, 509, 806, 8848, 11960, 1045, 11377, 827, 2442, 26077, 418, 21117, 745,
    9134, 1895, 13771, 17855, 753, 9024, 3451, 743, 3651, 406, 9335, 5394, 1034, 642, 1546, 7713,
    6319, 8194, 73, 993, 19069, 638, 337, 4329, 833, 7335, 926, 7659, 4431, 1434, 7913, 8343, 1373,
    2666, 7413, 1837, 9301, 8685, 805, 35769, 3923, 1338, 12778, 1175, 12479, 754, 5270, 8470,
    6629, 1570, 9193, 28873, 2251, 9362, 10675, 201, 965, 185, 39, 9860, 6935, 12368, 9949, 2042,
    4617, 9860, 5543, 1401, 9161, 1878, 12106, 23743, 2426, 366, 194, 10630, 8183, 6882, 7969,
    33211, 1563, 3571, 339, 11217, 7671, 323, 1006, 185, 859, 10823, 4977, 7689, 1465, 1098, 8895,
    2021, 15633, 1271, 12530, 6558, 152, 21710, 1373, 5803, 5673, 8923, 826, 5195, 8125, 475, 1722,
    3049, 6012, 9371, 10213, 1878, 10139, 3818, 11028, 1358, 3225, 21639, 63, 31238, 1923, 178,
    15822, 3194, 9818, 3474, 1535, 3391, 549, 993, 62, 1049, 2274, 2982, 8191, 473, 12698, 2710,
    8198, 198, 3293, 313, 985, 574, 3432, 1165, 7910, 25189, 2429, 3901, 11373, 1276, 4385, 173,
    1431, 719, 1733, 1107, 58693, 1946, 4507, 3369, 2902, 2870, 1086, 2066, 1630, 4635, 3601, 291,
    749, 2721, 4393, 2403, 2015, 7631, 5574, 1614, 311, 12052, 247, 1387, 15612, 3429, 2279, 8055,
    9127, 70, 1149, 586, 2687, 61, 399, 4154, 6019, 4625, 1623, 499, 7142, 429, 4645, 20059, 1099,
    777, 369, 1806, 55, 9257, 3230, 13910, 1927, 2731, 1443, 2686, 5586, 265, 12276, 2977, 2143,
    481, 11223, 598, 12552, 811, 1095, 762, 1067, 551, 1103, 4495, 12450, 25633, 1535, 714, 258,
    2525, 8162, 2046, 3464, 237, 13, 2979, 1179, 1105, 101, 2194, 178, 2834, 786, 1939, 259, 6629,
    12329, 1034, 4206, 11750, 6570, 6634, 2713, 1346, 16681, 11996, 11214, 4850, 16490, 11527, 261,
    9615, 16830, 5179, 6685, 17351, 1207, 3638, 13420, 1783, 301, 483, 539, 3322, 14839, 2357, 994,
    11433, 109, 5854, 12331, 7558, 1133, 7715, 2877, 8007, 29, 5167, 6109, 51, 3713, 3886, 2241,
    8123, 3687, 283, 5729, 749, 1329, 4667, 11070, 2513, 2619, 4246, 1895, 1295, 1311, 235, 3205,
    2063, 7594, 92, 1263, 2577, 537, 947, 3959, 1739, 1219, 13572, 2162, 2386, 11758, 8349, 102,
    46, 6575, 19440, 3681, 6986, 778, 2325, 94, 13322, 231, 2219, 3030, 4065, 2717, 255, 3239,
    1226, 82, 2570, 445, 521, 3449, 7782, 6361, 617, 3641, 2158, 234, 3179, 971, 9821, 2630, 213,
    1357, 11744, 7218, 14618, 9085, 5723, 4921, 8761, 411, 2266, 31, 8215, 9098, 6454, 12298, 4834,
    2882, 2698, 4143, 2694, 8241, 383, 1293, 174, 2077, 15428, 11321, 2431, 3311, 3255, 10483,
    5206, 18205, 1078, 5514, 930, 3997, 4461, 1725, 6194, 2453, 282, 8567, 8183, 23128, 7234,
    11685, 1482, 1589, 24678, 330, 11755, 35928, 5194, 4043, 5477, 2961, 7958, 2082, 38693, 10293,
    10738, 1042, 861, 1318, 8762, 235, 3451, 4866, 6419, 1751, 313, 177, 8373, 415, 4187, 30810,
    4413, 286, 10491, 15662, 9524, 1357, 4101, 3955, 3385, 3286, 7105, 138, 1915, 9152, 18705,
    2266, 1525, 93, 9308, 12606, 1970, 317, 333, 952, 891, 1667, 222, 11848, 781, 34257, 183, 477,
    855, 32935, 347, 442, 9003, 4529, 1233, 9748, 4632, 35483, 1030, 5005, 763, 8531, 9164, 1222,
    5530, 8238, 3407, 1207, 1918, 3746, 542, 5463, 2412, 7411, 2602, 1265, 2837, 13905, 353, 9441,
    407, 1694, 4206, 137, 859, 155, 847, 10435, 2923, 85, 18721, 231, 505, 10994, 16233, 9906, 325,
    1219, 5835, 947, 331, 17, 2267, 14496, 6355, 927, 5094, 677, 3478, 12290, 24969, 9351, 1214,
    2677, 36327, 195, 1739, 3115, 6971, 8054, 1049, 141, 1034, 967, 4329, 4101, 427, 4467, 10061,
    11294, 1270, 9108, 1187, 353, 487, 16360, 2001, 535, 263, 1845, 254, 593, 577, 4659, 423, 4505,
    4833, 1453, 8669, 1133, 1642, 7016, 11332, 9186, 99, 6065, 367, 513, 3477, 67, 2518, 101, 623,
    1359, 193, 1443, 3595, 107, 8126, 6923, 11862, 1546, 6570, 7033, 1158, 4187, 1190, 635, 1242,
    3709, 3379, 3230, 7647, 26233, 3827, 707, 913, 67, 4707, 1263, 18347, 1797, 4593, 117, 194,
    4157, 1529, 8359, 2403, 9093, 2123, 197, 2083, 3322, 2009, 137, 3829, 1263, 12055, 1905, 2461,
    2381, 653, 199, 9278, 3018, 3187, 4978, 472, 65, 25817, 109, 1754, 5946, 13835, 2159, 1202,
    5991, 2193, 860, 1966, 15881, 36127, 4651, 21, 5450, 5322, 3594, 3007, 5657, 3441, 9479, 1087,
    9789, 7582, 5141, 387, 234, 443, 86, 1453, 11729, 718, 4166, 9414, 1757, 971, 10884, 8946, 783,
    725, 8403, 10891, 4707, 4298, 14380, 1867, 61, 2590, 1630, 2715, 646, 2903, 326, 1533, 4302,
    951, 609, 2707, 5164, 3709, 661, 3170, 670, 4142, 14086, 11572, 3639, 11756, 5771, 881, 667,
    4071, 13385, 9141, 835, 1342, 3194, 2267, 8800, 2517, 10023, 4026, 211, 4647, 614, 1279, 6245,
    669, 1013, 14198, 2565, 8541, 1117, 2117, 27714, 2236, 4644, 1593, 57, 239, 711, 11030, 27875,
    2761, 741, 7910, 11041, 17570, 29830, 562, 2321, 4526, 6113, 482, 997, 2443, 6934, 1627, 10643,
    1607, 2847, 804, 5233, 2855, 4406, 161, 1846, 2391, 1011, 4683, 209, 661, 1538, 3869, 1702,
    15126, 2438, 4674, 2766, 4777, 2021, 1199, 4526, 2041, 2490, 9554, 685, 395, 683, 21711, 21509,
    5977, 1529, 1873, 99, 2845, 7493, 534, 8644, 27397, 8783, 19343, 9212, 13377, 3419, 11585,
    9852, 454, 542, 10056, 502, 11494, 8523, 2287, 1171, 1583, 14926, 3774, 803, 2915, 29850, 3469,
    5218, 2631, 32897, 738, 15454, 2379, 1307, 11909, 13919, 8403, 3407, 1123, 11, 9108, 5059,
    11624, 2082, 4717, 514, 21778, 257, 4991, 295, 2305, 1403, 4197, 326, 14069, 6290, 8775, 161,
    8319, 199, 197, 1261, 2273, 4359, 1262, 3213, 327, 4937, 2301, 1607, 2521, 659, 237, 32386,
    1430, 2735, 11385, 445, 9992, 971, 8271, 317, 2286, 3818, 83, 876, 1954, 614, 10870, 5946, 541,
    9885, 1349, 5554, 12635, 1321, 9852, 7471, 4407, 4501, 489, 13175, 1048, 6909, 2233, 11852,
    3842, 5467, 4586, 25179, 3367, 2073, 1827, 2757, 3883, 1826, 3989, 4081, 1070, 9441, 417, 3583,
    8631, 8814, 4419, 5169, 2249, 10389, 7831, 969, 3537, 253, 9048, 6597, 914, 1218, 3099, 6513,
    13284, 691, 667, 1771, 9396, 1489, 8965, 817, 4117, 2854, 1385, 549, 5465, 8337, 1453, 13,
    1547, 8019, 4053, 9184, 4643, 4271, 8610, 1338, 747, 789, 4605, 791, 9188, 11602, 8387, 286,
    21007, 1018, 2701, 6014, 82, 7074, 437, 13432, 6119, 303, 1237, 149, 34014, 3806, 55, 18351,
    4697, 8723, 1495, 1654, 4979, 3122, 9799, 3539, 19723, 5578, 4098, 741, 189, 1615, 1627, 3067,
    5378, 2029, 12828, 11611, 801, 4162, 3046, 536, 6653, 247, 7186, 4619, 109, 6385, 22852, 1161,
    4502, 178, 353, 511, 1271, 329, 333, 9194, 723, 9890, 202, 1159, 987, 23528, 366, 885, 278,
    1214, 1978, 2201, 14191, 461, 1203, 2261, 3391, 166, 419, 32619, 1449, 1613, 183, 23, 17873,
    4971, 19366, 6815, 1835, 1466, 207, 8276, 12774, 2210, 11613, 97, 8171, 8280, 639, 1349, 29552,
    9184, 203, 353, 667, 1573, 4857, 114, 10089, 1067, 390, 295, 440, 3801, 3625, 9350, 6521, 102,
    1913, 1801, 2379, 2042, 7193, 922, 2785, 2493, 1133, 2150, 629, 2547, 4501, 2949, 5073, 9403,
    3589, 333, 8702, 1719, 2158, 1061, 4385, 6917, 23764, 696, 425, 8760, 2299, 10706, 5458, 951,
    1730, 10615, 8335, 1459, 291, 1469, 8413, 13188, 71, 994, 139, 2402, 743, 14043, 877, 1309,
    4581, 466, 4035, 3858, 551, 2106, 7265, 1377, 1067, 2487, 14933, 1134, 1198, 1917, 5742, 215,
    1491, 1930, 8827, 1413, 16793, 305, 5665, 1911, 817, 15645, 963, 774, 3115, 8800, 7402, 1506,
    394, 2319, 4362, 1382, 9316, 3250, 109, 14028, 3714, 9312, 1257, 3597, 6849, 1155, 35901, 5377,
    3695, 481, 2449, 2495, 3439, 2997, 2567, 3082, 6551, 599, 630, 933, 1847, 1205, 10562, 43,
    8984, 1127, 4685, 1196, 3849, 11071, 3385, 4711, 5365, 1134, 9285, 1719, 239, 185, 201, 7237,
    9199, 3154, 563, 5405, 1958, 455, 201, 4063, 663, 10272, 138, 629, 985, 7550, 494, 15154, 4679,
    14222, 6329, 69, 1149, 7817, 5023, 649, 1534, 8606, 7067, 24901, 2865, 2739, 17569, 1051, 471,
    15756, 3762, 1700, 1450, 9968, 11550, 4019, 11785, 4823, 3977, 11204, 1219, 175, 26853, 11613,
    8785, 14340, 6111, 158, 12594, 7311, 3115, 1937, 965, 567, 6062, 1963, 34747, 4607, 1519, 8438,
    12579, 1759, 5878, 1497, 1037, 690, 583, 1911, 3617, 10944, 407, 39, 553, 732, 765, 8623, 337,
    3309, 12455, 158, 1729, 1006, 1075, 9761, 12702, 5989, 759, 3271, 265, 5103, 5733, 1946, 4435,
    23113, 3474, 525, 994, 4627, 5435, 14428, 23, 10926, 957, 533, 10964, 8687, 3094, 5565, 989,
    1138, 1273, 6429, 24421, 3897, 1003, 18936, 6089, 4043, 15893, 11195, 2579, 1442, 6966, 520,
    17582, 10996, 1645, 129, 6513, 6221, 8452, 8697, 10664, 3347, 4118, 8066, 291, 551, 1301, 9749,
    1863, 271, 2510, 1767, 410, 2605, 12413, 3066, 6983, 461, 407, 2010, 9108, 4057, 333, 2214,
    521, 39, 581, 934, 8295, 3406, 2689, 9172, 485, 12099, 6823, 3453, 11032, 3121, 1419, 277, 761,
    929, 1214, 626, 20282, 1970, 4364, 1810, 489, 1202, 4762, 7474, 506, 9733, 1319, 1162, 299,
    28650, 4097, 9107, 6317, 311, 339, 11793, 13427, 9010, 874, 9568, 2719, 2146, 38, 533, 4274,
    3301, 10134, 74, 1685, 3391, 2454, 2550, 5238, 4878, 13093, 8878, 4085, 5685, 4226, 8887, 1433,
    1777, 986, 803, 11192, 9400, 461, 6385, 621, 3094, 157, 1234, 1610, 325, 1755, 3317, 55, 1657,
    3179, 262, 2023, 1446, 8609, 38923, 8562, 473, 12488, 9200, 1610, 258, 9010, 1411, 5685, 1775,
    3237, 3679, 1495, 20745, 241, 9495, 1570, 1047, 5335, 133, 202, 4740, 4270, 2994, 5971, 12360,
    3141, 3031, 10461, 15909, 4650, 18960, 713, 10684, 887, 6365, 95, 7258, 13463, 7469, 2534, 649,
    3127, 1387, 3789, 678, 1034, 873, 591, 259, 971, 10716, 1313, 4087, 3322, 763, 6262, 7501, 466,
    44, 909, 5903, 573, 403, 5890, 14954, 2274, 8715, 11968, 7010, 491, 685, 641, 8330, 607, 5567,
    710, 719, 8546, 583, 7078, 1366, 1047, 12233, 442, 2527, 10467, 14994, 4039, 2219, 927, 1795,
    12713, 2870, 5146, 3891, 2255, 590, 2947, 26, 641, 1062, 9131, 813, 11201, 1967, 869, 8317,
    654, 191, 1510, 3853, 15723, 557, 1411, 4198, 173, 527, 10994, 1335, 1398, 3242, 21465, 1618,
    2709, 929, 3999, 21022, 4422, 2245, 9174, 369, 3682, 1926, 13789, 35, 7574, 3367, 4143, 1434,
    3898, 531, 2375, 9215, 1194, 925, 7502, 4486, 309, 52035, 11213, 4447, 1337, 3591, 8910, 353,
    919, 4538, 36686, 9352, 10808, 77, 3265, 20242, 1339, 2427, 3119, 4988, 1446, 2857, 1625,
    11506, 12261, 1926, 1230, 12498, 4227, 3453, 1831, 3431, 187, 1793, 457, 15834, 6655, 2041,
    473, 8765, 2278, 9182, 1513, 32409, 3229, 2594, 3565, 32307, 1995, 2783, 115, 3257, 17808,
    8458, 959, 5770, 13365, 6771, 67, 2414, 8858, 2447, 1322, 3619, 310, 1229, 5367, 1371, 33319,
    2037, 3861, 8487, 6777, 2595, 119, 1767, 1754, 3517, 1679, 118, 21137, 4282, 2134, 2306, 2550,
    430, 4069, 1847, 12552, 1687, 16639, 6466, 3829, 9885, 6095, 2787, 1205, 4515, 1221, 171, 3137,
    11388, 557, 8453, 2828, 293, 8307, 1951, 32179, 2206, 73, 13694, 17540, 149, 4817, 17271, 3766,
    9260, 27874, 2378, 3194, 11078, 927, 1283, 3989, 4243, 221, 6182, 20619, 3758, 865, 3489,
    16486, 4023, 436, 1611, 978, 187, 8873, 9285, 7306, 703, 8992, 199, 12971, 3270, 1991, 10021,
    266, 391, 835, 1393, 8870, 27321, 4105, 2651, 403, 1266, 8349, 2578, 6147, 4313, 11667, 7438,
    2173, 6533, 11817, 8675, 19495, 1518, 1978, 943, 7065, 13299, 2543, 311, 1437, 5901, 1819,
    1862, 32323, 5047, 385, 65, 543, 1802, 4861, 1383, 1902, 9700, 451, 705, 1943, 3909, 1219,
    4534, 1573, 1621, 6085, 15544, 3713, 522, 385, 1270, 11328, 1685, 11479, 12606, 13831, 8928,
    15019, 2023, 1597, 9531, 2921, 2239, 570, 1107, 5326, 386, 16087, 5582, 111, 6098, 70, 2837,
    11720, 4067, 3350, 11065, 618, 13184, 2763, 1229, 1419, 19658, 30146, 9472, 12700, 1133, 474,
    9130, 1639, 646, 1425, 3158, 7254, 1903, 6823, 9588, 7705, 3413, 2630, 6451, 2453, 929, 12074,
    1658, 3794, 581, 177, 1018, 1469, 818, 2097, 1178, 2142, 12607, 2853, 7229, 539, 897, 20441,
    618, 951, 2342, 2317, 926, 1929, 599, 1897, 13327, 1278, 2995, 5614, 9756, 403, 3315, 26191,
    5956, 4213, 22599, 10279, 1010, 226, 1988, 609, 8992, 693, 2898, 14284, 6033, 4327, 6753, 2451,
    17639, 489, 5605, 5105, 2142, 1705, 13250, 12015, 829, 2873, 7779, 702, 211, 1038, 1539, 9882,
    5298, 495, 7559, 315, 9483, 2875, 3093, 5559, 39349, 734, 194, 3479, 761, 3503, 2931, 1179,
    40618, 140, 1559, 2097, 8609, 340, 247, 40061, 9475, 195, 4533, 2411, 183, 9202, 4486, 131,
    8973, 819, 719, 2003, 1435, 893, 2627, 61, 419, 3047, 434, 5214, 5089, 763, 2919, 985, 2449,
    1197, 622, 797, 2481, 5119, 8502, 3959, 861, 9035, 4837, 1374, 6159, 5009, 881, 1142, 6305,
    9785, 899, 9258, 13854, 342, 1171, 8799, 8549, 8847, 335, 5411, 211, 5889, 1263, 5263, 1771,
    14555, 899, 1226, 33071, 3269, 8872, 2415, 3641, 229, 8388, 5726, 2278, 3985, 857, 4227, 3030,
    10879, 6187, 8619, 2386, 5639, 10934, 2358, 4433, 4939, 833, 91, 222, 7197, 10821, 2819, 6493,
    981, 3634, 468, 8470, 3441, 2747, 3058, 10208, 115, 1505, 545, 2259, 3943, 9555, 517, 8943,
    2302, 9664, 3109, 899, 2650, 754, 30565, 13597, 62, 12733, 11205, 311, 2826, 5949, 5179, 3439,
    10219, 17006, 665, 3749, 1207, 5515, 2519, 3397, 5277, 8256, 12441, 999, 3067, 13321, 11753,
    2877, 4241, 2431, 12999, 9164, 21694, 8203, 2897, 7002, 9232, 2145, 4141, 2939, 953, 7317, 189,
    1655, 261, 906, 31970, 4550, 38284, 327, 377, 151, 1282, 778, 7845, 2293, 3867, 61, 3809, 1195,
    3090, 2161, 1186, 4243, 197, 31115, 3114, 4051, 1389, 386, 9608, 227, 8488, 2734, 4849, 851,
    5612, 2497, 359, 16856, 1507, 2766, 17500, 7695, 2202, 1647, 2249, 1873, 221, 2899, 587, 1063,
    2619, 301, 11975, 11132, 11989, 8826, 3839, 311, 571, 711, 35637, 7023, 5689, 8813, 4139,
    17493, 3753, 2253, 3965, 2246, 1454, 875, 10421, 806, 279, 15554, 1295, 5525, 2510, 10653, 671,
    10600, 683, 9446, 11788, 9291, 4239, 4479, 1279, 1795, 269, 2266, 753, 5733, 3492, 2861, 1451,
    1745, 1782, 12176, 1274, 5315, 12401, 6033, 219, 11585, 2877, 417, 505, 778, 2782, 5169, 11119,
    1943, 1563, 2722, 703, 6651, 8363, 8057, 453, 2455, 711, 4042, 11509, 2965, 826, 13133, 13170,
    1489, 1249, 1091, 1543, 11279, 14055, 31323, 5399, 319, 21263, 3101, 2338, 670, 531, 257, 145,
    2145, 2134, 8034, 14369, 13771, 4463, 3726, 4330, 1430, 1475, 10342, 2507, 9502, 9616, 1429,
    3823, 193, 6717, 521, 483, 5638, 16948, 846, 279, 11340, 3849, 5001, 4191, 3945, 1791, 569,
    533, 3501, 1021, 923, 22341, 14006, 5587, 3001, 1198, 1961, 3498, 1739, 9407, 7291, 3231, 333,
    3043, 583, 2511, 8508, 2087, 154, 1962, 10975, 903, 2221, 482, 229, 4097, 9254, 3725, 7799,
    4493, 15180, 743, 10180, 1974, 1607, 8358, 12614, 8683, 10502, 1234, 32771, 3699, 91, 631,
    5627, 937, 2997, 12524, 11404, 378, 10053, 38675, 11696, 5277, 1554, 5425, 4591, 9374, 6366,
    523, 890, 5898, 183, 6141, 6749, 979, 1662, 5603, 2829, 9253, 239, 1871, 9576, 10167, 341,
    12252, 5770, 14841, 1795, 2753, 7411, 739, 79, 4243, 13605, 4841, 14797, 4219, 8063, 6156,
    1855, 7347, 1590, 10808, 85, 213, 3202, 851, 28427, 9457, 8885, 26285, 190, 943, 2417, 1619,
    8321, 4454, 59, 3014, 138, 1235, 1738, 689, 4534, 1219, 131, 9912, 1673, 171, 1105, 833, 6547,
    257, 1155, 553, 1274, 6393, 2298, 17456, 9465, 9626, 2058, 942, 5110, 2027, 26437, 37327, 1726,
    5066, 2903, 8330, 9328, 1158, 3009, 14902, 893, 3243, 419, 4681, 7451, 5927, 473, 6311, 1741,
    1554, 2194, 690, 587, 5326, 222, 8835, 1777, 531, 654, 1247, 9420, 6333, 1550, 8163, 1160, 347,
    717, 549, 8431, 10492, 943, 637, 8228, 1577, 10335, 7851, 970, 14331, 167, 963, 4433, 3313, 61,
    220, 373, 5934, 6713, 926, 2141, 663, 11758, 25155, 13942, 6985, 1959, 3875, 3578, 1617, 7373,
    1835, 269, 107, 991, 3911, 910, 1034, 1741, 293, 4819, 193, 597, 11559, 6474, 2146, 16210,
    1811, 573, 16510, 2591, 7722, 549, 2834, 1874, 6581, 2246, 1295, 463, 15572, 18665, 354, 319,
    5279, 723, 1449, 7457, 499, 1883, 7489, 199, 3882, 3302, 2275, 1279, 7865, 1574, 6835, 466,
    2055, 3927, 602, 8980, 915, 2101, 463, 2751, 223, 15083, 1949, 1405, 5065, 6831, 897, 3651,
    1742, 7743, 909, 2118, 18651, 5233, 11864, 3198, 2295, 7326, 769, 2613, 14455, 8315, 139, 3258,
    8119, 12148, 850, 6551, 7343, 8847, 1033, 21435, 9241, 835, 4693, 1422, 185, 1162, 1173, 8367,
    238, 4645, 494, 7147, 33, 179, 2074, 22, 395, 6907, 799, 9877, 24701, 3171, 209, 4155, 2142,
    1507, 4091, 1817, 753, 235, 10466, 4790, 1635, 5582, 14739, 2389, 2498, 1931, 11797, 1045,
    1306, 4706, 18007, 597, 2195, 3734, 5886, 1659, 939, 2783, 7330, 1254, 639, 4627, 4137, 2630,
    2869, 411, 12782, 3235, 4878, 20995, 3677, 9116, 12037, 7433, 25103, 107, 237, 821, 3090, 2839,
    11347, 978, 7573, 3673, 1832, 3670, 1022, 8994, 1217, 3134, 5570, 2157, 4717, 7391, 3334, 7055,
    7402, 115, 3983, 300, 221, 318, 907, 8354, 1094, 761, 401, 1645, 6874, 1218, 33051, 295, 5995,
    843, 127, 6109, 4671, 94, 30514, 2863, 6873, 11961, 1045, 1647, 12889, 4547, 8906, 3868, 2363,
    1923, 5097, 11199, 2898, 1487, 667, 6439, 4439, 878, 1689, 7181, 2773, 5267, 4459, 1893, 12715,
    609, 4269, 3767, 3102, 29273, 3854, 17937, 385, 2290, 1569, 4953, 6270, 535, 1437, 2330, 8559,
    2282, 4691, 730, 8026, 5523, 7339, 19277, 3637, 5170, 8814, 10535, 983, 2422, 31423, 73, 109,
    1910, 16379, 435, 1272, 6727, 3454, 3659, 1758, 7617, 12092, 1822, 850, 6670, 8741, 4630, 1171,
    10286, 2955, 1274, 2053, 10141, 2891, 4122, 8098, 1539, 5167, 386, 9786, 2631, 2491, 2442,
    5038, 4629, 7466, 10181, 843, 11705, 1753, 435, 4973, 2774, 267, 4091, 959, 742, 374, 258,
    2974, 2217, 3942, 1143, 786, 267, 4086, 1810, 8826, 3207, 3074, 86, 3469, 123, 8377, 6342,
    15830, 8824, 5427, 4043, 399, 2713, 1053, 2655, 2771, 3991, 2949, 2461, 219, 3999, 609, 1353,
    1007, 7978, 757, 3013, 6958, 2546, 671, 2571, 3153, 2189, 1731, 4641, 8444, 3225, 8054, 2557,
    8638, 28335, 903, 3149, 3515, 1275, 659, 2388, 8912, 3842, 4695, 6349, 14610, 5663, 8966, 5233,
    1047, 8669, 1407, 493, 34818, 7507, 6183, 10270, 3303, 2353, 146, 2710, 2058, 681, 38380, 9298,
    478, 1419, 377, 10467, 1237, 927, 3223, 946, 1570, 747, 47, 2231, 785, 1277, 4697, 577, 667,
    581, 114, 281, 3089, 10880, 4123, 3091, 2135, 8292, 12459, 1337, 2018, 8593, 1693, 2226, 1903,
    379, 6231, 39748, 8336, 5330, 1854, 1309, 94, 6143, 4171, 1831, 4550, 491, 4151, 2421, 2071,
    8773, 2295, 2858, 2257, 2322, 1109, 3358, 32545, 651, 1553, 113, 314, 13574, 163, 5031, 8518,
    9357, 3157, 7593, 834, 3181, 3627, 2795, 1407, 15592, 8354, 5947, 11567, 251, 771, 14684, 5066,
    177, 2519, 523, 1294, 5897, 9646, 1517, 511, 4309, 64850, 2413, 1625, 5489, 3214, 4582, 3033,
    567, 857, 8933, 706, 11197, 8314, 2697, 1061, 5459, 7599, 2765, 1133, 6249, 446, 5577, 5125,
    778, 582, 2267, 1305, 2657, 1903, 1591, 325, 981, 2063, 4801, 3071, 3166, 79, 3077, 354, 1063,
    385, 8724, 3246, 703, 1061, 4581, 58, 385, 3101, 13386, 3747, 333, 5085, 10739, 1935, 5335,
    3427, 8400, 22852, 1757, 77, 8334, 199, 1015, 8956, 7262, 7963, 2605, 393, 222, 3905, 17976,
    8934, 2019, 2313, 2607, 2273, 4215, 319, 16672, 415, 2550, 5727, 3567, 14742, 462, 2733, 831,
    5715, 7159, 2175, 621, 7815, 25545, 4022, 863, 2135, 15157, 2519, 7243, 5151, 1205, 11409,
    12115, 1637, 373, 974, 139, 2927, 217, 14, 488, 2811, 11180, 298, 15520, 1673, 24691, 10854,
    4338, 431, 1153, 8286, 10135, 831, 4467, 582, 3157, 5959, 9402, 842, 673, 4306, 171, 2042, 397,
    1207, 3930, 5535, 8874, 502, 391, 149, 9855, 21, 545, 3759, 713, 19830, 3095, 2674, 3194, 8861,
    23942, 7413, 10074, 1781, 11391, 3503, 20615, 6267, 7226, 71, 305, 4914, 1477, 94, 22012, 4010,
    3385, 4857, 2473, 2938, 319, 1359, 1142, 10037, 5039, 12645, 766, 350, 3194, 594, 2286, 10499,
    3197, 5131, 13639, 35366, 249, 839, 9232, 8098, 28263, 1753, 1099, 454, 4231, 197, 2759, 13553,
    567, 7317, 1495, 3690, 5817, 57, 4442, 8227, 1781, 9688, 1635, 3101, 15325, 271, 411, 1290,
    2720, 2574, 9782, 7465, 8404, 621, 1619, 791, 2767, 9556, 3971, 74, 5390, 743, 623, 2857, 4268,
    2521, 1326, 4151, 171, 7227, 2497, 6089, 10076, 6058, 11036, 10870, 4250, 18948, 646, 11109,
    910, 2179, 11924, 20986, 19, 4637, 499, 8469, 3425, 226, 1615, 4499, 921, 299, 5330, 1258,
    6671, 2371, 223, 8014, 4583, 5003, 918, 827, 11758, 2402, 8516, 199, 57, 10613, 9382, 8940,
    5691, 299, 1221, 1145, 2113, 3529, 1246, 3324, 2711, 29761, 8517, 51348, 347, 438, 14313, 2225,
    5861, 37, 1118, 6769, 35452, 936, 622, 859, 214, 931, 11008, 30545, 2286, 1763, 910, 1385, 551,
    1155, 2239, 5979, 1831, 19878, 2614, 303, 4711, 117, 2463, 39994, 1943, 3133, 3030, 317, 633,
    11043, 8682, 18630, 1988, 114, 313, 339, 650, 3755, 1783, 4822, 899, 6907, 3739, 9331, 8129,
    12933, 2967, 2003, 6255, 3011, 1915, 8877, 492, 4682, 7086, 974, 10743, 306, 13655, 1397, 373,
    1566, 4721, 13771, 1913, 707, 811, 9448, 13180, 35, 629, 9023, 3309, 646, 8325, 863, 327, 603,
    6614, 53, 393, 8439, 1055, 451, 9700, 1867, 5322, 3669, 13, 8216, 2578, 226, 1126, 1973, 3110,
    8212, 1963, 6997, 6339, 3879, 2449, 951, 2108, 6377, 233, 590, 1629, 95, 5993, 526, 730, 939,
    615, 767, 622, 3103, 11088, 4937, 2423, 7346, 1881, 7541, 9955, 95, 1277, 6697, 1318, 787,
    5466, 24557, 29726, 19263, 1927, 12989, 8760, 2417, 865, 2714, 5773, 3670, 349, 11670, 11241,
    471, 4071, 2893, 4227, 679, 1378, 730, 2750, 471, 7835, 473, 1117, 37, 10378, 613, 7027, 10567,
    6877, 10298, 24918, 16518, 2123, 1542, 3207, 1074, 1902, 19348, 327, 2203, 2397, 1061, 5537,
    2455, 9511, 219, 2131, 3715, 15407, 802, 8290, 9170, 5131, 127, 6439, 14725, 26917, 239, 5615,
    3471, 797, 74, 155, 1159, 705, 2578, 16566, 8769, 3211, 106, 2443, 12704, 907, 15497, 4501,
    1037, 574, 5123, 8675, 109, 3293, 1265, 7374, 4150, 8429, 27555, 91, 8599, 417, 533, 3821,
    52443, 619, 2214, 4413, 1675, 13537, 2050, 881, 2698, 567, 755, 24281, 6266, 11496, 445, 430,
    223, 657, 5257, 2923, 139, 3151, 7818, 3159, 3338, 499, 979, 6546, 1899, 463, 3005, 3091, 9921,
    5750, 369, 31, 347, 1279, 1374, 8694, 379, 2673, 1319, 623, 3505, 641, 3814, 3110, 2226, 795,
    61, 3997, 2003, 12244, 23, 9717, 1827, 1471, 3035, 8585, 5403, 4266, 155, 38, 1841, 10537,
    2569, 473, 1617, 727, 9935, 2718, 359, 7093, 683, 386, 2379, 25013, 2563, 413, 1773, 6986,
    1274, 3007, 1966, 4149, 1099, 12267, 1569, 1107, 5814, 2082, 2514, 569, 15238, 17613, 3238,
    2006, 9902, 1498, 17398, 29281, 2611, 4397, 141, 373, 6459, 691, 1039, 4389, 1135, 5858, 1053,
    3567, 9654, 3707, 2395, 13106, 1186, 769, 223, 913, 8827, 7294, 3418, 5141, 1173, 290, 2788,
    107, 770, 2419, 4009, 327, 14100, 3695, 1766, 12925, 443, 2234, 1869, 2619, 3965, 278, 4430,
    8395, 1685, 7467, 13234, 3442, 6206, 2575, 699, 2046, 1067, 5305, 359, 8308, 1506, 3503, 769,
    457, 2724, 257, 2215, 3073, 811, 14258, 1430, 1706, 3109, 933, 1871, 5543, 1361, 22895, 3434,
    149, 526, 521, 25911, 1449, 3507, 3677, 3865, 11, 8946, 183, 2326, 10565, 623, 3359, 85, 11290,
    2014, 2359, 4037, 7949, 33916, 11639, 5253, 6091, 1355, 8695, 1101, 4510, 2513, 8242, 8807,
    3625, 63, 1247, 3361, 165, 5489, 8298, 3234, 12760, 15322, 9993, 4821, 3713, 691, 4993, 1309,
    2354, 1365, 3459, 541, 8697, 4417, 833, 549, 4110, 2651, 8514, 14666, 871, 22999, 1191, 2722,
    6910, 8521, 4731, 2905, 8038, 3851, 5678, 3818, 4795, 123, 21052, 214, 9169, 607, 7489, 1157,
    190, 20594, 1259, 1663, 2387, 219, 215, 4341, 7011, 13393, 2845, 1203, 209, 694, 1177, 903,
    8757, 2253, 3477, 19033, 353, 1735, 7979, 5249, 1973, 1251, 1654, 505, 483, 143, 725, 325,
    8369, 991, 1245, 2815, 11071, 2854, 10759, 15426, 183, 3270, 21723, 419, 631, 803, 759, 8207,
    7734, 407, 1929, 6299, 3061, 4438, 693, 5381, 11421, 2669, 42688, 564, 4622, 4226, 2138, 8511,
    1857, 3961, 2153, 337, 4271, 209, 1954, 8390, 671, 10024, 526, 2747, 2942, 5748, 4285, 1926,
    7333, 323, 4918, 57, 2381, 1061, 9884, 5006, 817, 3147, 1226, 4767, 9497, 701, 1615, 1614, 859,
    369, 8330, 5163, 10834, 2747, 11715, 1955, 422, 1055, 2381, 4562, 9564, 165, 1463, 2290, 1298,
    2939, 11956, 1863, 8501, 3652, 199, 2854, 3081, 1807, 543, 2886, 13930, 1832, 7, 1078, 21211,
    1495, 1999, 6739, 1743, 32510, 627, 8105, 246, 3251, 2379, 3423, 14117, 1177, 367, 11697, 102,
    12632, 16279, 6293, 11814, 13366, 1543, 11556, 22, 10255, 309, 3049, 2106, 9265, 933, 123, 393,
    15557, 159, 483, 2223, 949, 16484, 10589, 5997, 1191, 4402, 674, 1592, 554, 9641, 9889, 11547,
    24548, 8263, 1335, 381, 15222, 115, 23229, 14923, 5067, 390, 1245, 9561, 922, 1543, 6689, 717,
    3395, 423, 3784, 3707, 6821, 590, 665, 3938, 2323, 3017, 26038, 2770, 201, 9190, 1729, 8629,
    9065, 4716, 3365, 1293, 3795, 1557, 2089, 1848, 6602, 4743, 1953, 969, 687, 8022, 4059, 2050,
    5510, 8518, 10274, 939, 1071, 861, 1619, 553, 6373, 1547, 1579, 1283, 11477, 11658, 1405,
    20153, 2599, 374, 5878, 12764, 1026, 1353, 12673, 24794, 1067, 8722, 9613, 2516, 101, 6550,
    1069, 809, 20469, 1139, 2262, 1187, 1954, 4442, 502, 302, 18284, 1602, 663, 1051, 927, 2327,
    63, 1387, 115, 4603, 342, 231, 3911, 193, 678, 658, 2599, 36738, 1064, 1393, 2621, 2382, 4569,
    482, 2507, 1069, 1234, 934, 469, 637, 15058, 1957, 7143, 15990, 6047, 1325, 923, 3327, 2789,
    9374, 259, 12701, 10977, 505, 3781, 6874, 7193, 1949, 5951, 693, 9064, 986, 549, 1822, 406,
    1751, 6073, 7159, 4435, 869, 2655, 13, 10487, 8725, 15267, 146, 119, 9016, 1485, 1509, 9869,
    546, 3793, 278, 2849, 1902, 12920, 238, 12712, 21663, 3363, 6237, 381, 3521, 741, 6413, 1155,
    4202, 2283, 1220, 2410, 2333, 277, 3942, 8309, 2714, 8556, 11658, 8902, 1901, 1081, 7177, 1094,
    7611, 9370, 2330, 131, 1221, 3855, 2543, 2067, 3033, 4682, 9284, 2487, 4681, 4150, 26114, 6762,
    12069, 3429, 282, 895, 3586, 1013, 1346, 1279, 467, 11085, 2729, 787, 1079, 3406, 11489, 489,
    8693, 13470, 159, 1573, 4100, 786, 107, 13943, 1459, 4677, 8937, 2189, 10152, 5351, 481, 7171,
    2641, 3111, 171, 10735, 843, 5431, 274, 18017, 4497, 4287, 185, 1485, 2438, 954, 9029, 2526,
    5333, 2349, 502, 10171, 10018, 8221, 1766, 913, 16499, 1555, 10509, 9111, 714, 17345, 266,
    3795, 8439, 16355, 5854, 1115, 94, 166, 2713, 4073, 2790, 14312, 34111, 8268, 2173, 3085, 1209,
    7403, 2631, 1061, 1829, 2427, 1567, 119, 8651, 2851, 1126, 554, 1467, 5982, 8723, 6843, 1846,
    3735, 10948, 3033, 29321, 16458, 526, 7715, 1430, 2571, 2515, 8695, 1667, 1751, 3907, 8415,
    530, 714, 902, 2050, 346, 9440, 17954, 3517, 8375, 1027, 1131, 2127, 535, 3305, 2643, 2506,
    2811, 5239, 3475, 6985, 1035, 5714, 731, 4330, 9588, 31372, 8854, 5950, 1661, 20185, 3293, 146,
    18010, 8629, 9746, 13169, 6343, 1727, 65, 103, 1007, 4878, 14265, 137, 9490, 5785, 58, 673,
    2555, 1286, 397, 1705, 36761, 104, 6149, 9852, 2383, 161, 619, 10874, 12903, 56731, 13423,
    4793, 5369, 10942, 13745, 707, 83, 21707, 5943, 2495, 5737, 645, 4763, 1300, 2391, 2833, 4494,
    1211, 3115, 6951, 635, 3230, 8832, 303, 2063, 61, 1450, 2065, 3085, 426, 1345, 2473, 1285,
    5797, 9872, 5530, 227, 2409, 1415, 26, 8914, 7515, 730, 10211, 3483, 6754, 203, 163, 174, 2518,
    27165, 15711, 12886, 4514, 13810, 1942, 1389, 471, 9254, 2753, 1281, 3226, 9702, 1607, 1910,
    9337, 8301, 664, 949, 12623, 1843, 3411, 3543, 3619, 1670, 43, 3373, 4027, 590, 319, 1041,
    8210, 5246, 15456, 803, 14536, 21487, 371, 8821, 2358, 6423, 7739, 2497, 2269, 975, 11092,
    1537, 2713, 3238, 366, 1785, 861, 637, 1131, 2211, 1490, 11652, 2055, 689, 1065, 2407, 28010,
    2071, 138, 799, 183, 27759, 1051, 31, 2921, 9971, 5491, 981, 8597, 2322, 670, 1256, 617, 234,
    1819, 12932, 3851, 8199, 8387, 63, 499, 2539, 1494, 1073, 1983, 387, 485, 1821, 10421, 793,
    573, 2203, 1266, 545, 4149, 1189, 1786, 2417, 12255, 195, 3298, 11968, 1547, 616, 190, 7358,
    28642, 398, 7561, 335, 2043, 119, 33, 7061, 1604, 3053, 13082, 4291, 5981, 482, 3541, 1873,
    3494, 15780, 8429, 263, 403, 10729, 537, 9225, 7358, 209, 10539, 3253, 385, 667, 15674, 755,
    1219, 8226, 4355, 667, 1081, 141, 541, 8516, 8114, 3599, 4199, 787, 5629, 3229, 181, 383, 6374,
    4850, 583, 13864, 991, 2333, 478, 4162, 1827, 757, 1175, 9534, 2023, 2106, 9615, 2117, 1259,
    4435, 1705, 1987, 411, 13931, 1013, 8520, 199, 498, 9790, 2314, 1561, 3083, 322, 1401, 8700,
    3769, 13119, 11458, 10670, 2085, 3278, 6911, 2871, 12023, 5336, 157, 9477, 2773, 1547, 9386,
    8606, 2878, 5626, 4417, 3438, 322, 3786, 1206, 2849, 2773, 11429, 3263, 10017, 5814, 69, 534,
    3525, 1213, 2846, 1041, 5187, 147, 583, 5030, 4637, 1933, 9142, 7099, 971, 555, 7559, 1015,
    889, 6659, 599, 1157, 2177, 6898, 786, 199, 6375, 1551, 1979, 16801, 315, 11519, 4253, 2553,
    5050, 1635, 2219, 850, 4058, 11347, 279, 834, 885, 8926, 262, 3497, 2438, 2287, 1409, 6369,
    9909, 28468, 8915, 425, 8002, 305, 8913, 1487, 1423, 2171, 10114, 6127, 1601, 3438, 40134,
    15886, 5843, 2459, 10131, 7013, 3517, 21767, 277, 4349, 403, 28038, 19301, 131, 1233, 33925,
    4329, 3878, 1433, 1399, 2193, 31, 1359, 4854, 8387, 2913, 1263, 4748, 31895, 30669, 4873, 927,
    11268, 5999, 157, 3175, 91, 1228, 1027, 127, 8512, 471, 925, 1195, 8284, 9319, 3113, 5006, 214,
    2995, 4037, 32185, 3587, 23182, 2573, 309, 1143, 2539, 15576, 2361, 7807, 106, 3473, 8238,
    5814, 3143, 2221, 998, 2225, 409, 8539, 9340, 838, 8932, 957, 22832, 59, 14862, 2527, 25938,
    2069, 5218, 1075, 4961, 3159, 2595, 2425, 2265, 6093, 2010, 10451, 5862, 133, 1650, 1917, 620,
    5831, 247, 2119, 10366, 3827, 1251, 11192, 9668, 1737, 199, 1543, 3205, 13474, 9411, 1991, 442,
    6378, 3319, 15449, 335, 2165, 1494, 3889, 4789, 1410, 2938, 8686, 6401, 51, 7782, 2321, 1611,
    2465, 10338, 8655, 17946, 1217, 1529, 2930, 4239, 142, 1342, 13711, 1226, 11451, 1574, 6231,
    1601, 8305, 4519, 7247, 886, 4317, 38430, 17965, 2937, 3, 9592, 1921, 1661, 3788, 1146, 4169,
    2929, 547, 2402, 1647, 1107, 2935, 6247, 1631, 355, 2373, 2141, 1169, 2455, 14720, 369, 5355,
    1053, 10699, 2123, 4021, 7361, 922, 39, 4459, 3505, 2243, 2267, 3195, 2334, 3129, 5379, 790,
    12504, 5089, 7682, 9985, 29227, 5651, 902, 9647, 32794, 2109, 19794, 233, 253, 2337, 1585, 601,
    1255, 2018, 4254, 4009, 1725, 857, 65, 1165, 4693, 9262, 25157, 6225, 2665, 11374, 1349, 19697,
    1965, 3491, 7526, 2842, 30969, 1422, 967, 34364, 637, 3819, 3210, 4919, 21444, 33354, 167,
    1450, 2542, 1530, 14684, 4005, 214, 3215, 7421, 2455, 191, 1057, 8631, 1377, 575, 2981, 3027,
    5942, 1485, 11218, 5998, 7644, 1318, 2803, 9428, 9625, 1267, 1915, 1702, 1201, 19335, 1172,
    2861, 1011, 16163, 524, 747, 5639, 6195, 1971, 223, 6469, 11985, 8510, 259, 1791, 18567, 9267,
    4843, 1252, 3994, 1767, 10614, 8416, 263, 23459, 10837, 4173, 1669, 2379, 4033, 11705, 6749,
    67, 506, 1989, 5225, 5074, 74, 2618, 35, 419, 34, 467, 2305, 5589, 5777, 2599, 4957, 29, 37712,
    2462, 5563, 1535, 2290, 7386, 1642, 5821, 990, 6947, 749, 1087, 110, 635, 16080, 3003, 4758,
    3542, 651, 663, 1997, 1981, 2855, 4413, 15183, 435, 591, 285, 2099, 15978, 9906, 10479, 2595,
    9196, 11464, 25723, 4218, 29515, 10660, 24195, 6843, 11805, 733, 255, 838, 2815, 6717, 9791,
    14339, 4709, 4702, 4967, 5075, 9512, 1346, 7126, 16150, 5277, 1327, 4135, 161, 563, 11, 1259,
    2435, 275, 1686, 186, 3023, 8819, 10050, 1863, 4578, 218, 12108, 4869, 1031, 7467, 6337, 861,
    2834, 4775, 5371, 782, 11457, 1543, 12694, 705, 9143, 1871, 1704, 251, 217, 117, 809, 293,
    9077, 3211, 6837, 2982, 6199, 4126, 907, 8033, 15097, 1894, 5826, 12785, 241, 3715, 1583, 2857,
    16459, 657, 1653, 9582, 1795, 695, 10461, 1314, 4062, 511, 12379, 3022, 795, 365, 5719, 1039,
    4497, 4926, 7013, 5401, 14153, 2289, 667, 3268, 9070, 8906, 5425, 71, 1820, 8474, 1102, 2994,
    4849, 7430, 629, 16702, 1559, 5387, 790, 6742, 8634, 1255, 5279, 1926, 13868, 1347, 11504,
    2354, 1219, 1041, 3447, 1821, 894, 1238, 697, 3706, 1135, 9157, 1038, 5001, 950, 927, 1122,
    2461, 959, 287, 9209, 11219, 790, 7709, 266, 183, 10857, 7306, 34, 185, 303, 6413, 3153, 285,
    2059, 2707, 10863, 673, 3187, 6791, 8766, 9605, 7506, 1879, 16066, 371, 15962, 885, 8528, 359,
    8965, 2153, 178, 695, 7233, 29898, 11600, 5910, 20985, 7341, 9215, 10928, 3297, 1341, 19247,
    3037, 11758, 610, 1313, 969, 743, 660, 8441, 12867, 788, 9012, 175, 883, 577, 522, 3902, 3357,
    7222, 6279, 18376, 11043, 453, 266, 3290, 1335, 1651, 381, 5449, 2301, 24103, 2163, 5767, 2177,
    2847, 1798, 7978, 182, 1267, 6582, 1241, 14002, 3839, 381, 1465, 53, 6285, 2867, 4011, 2375,
    175, 13959, 2995, 9991, 707, 10411, 1546, 3614, 851, 4836, 807, 2417, 154, 1870, 322, 9944,
    1102, 247, 7363, 1006, 24838, 974, 10099, 10034, 634, 12851, 9298, 2101, 690, 2065, 8935, 5377,
    4922, 15273, 2149, 2595, 16671, 1854, 1963, 1193, 4290, 1599, 11067, 28, 8114, 1569, 341, 263,
    2698, 10930, 2149, 3339, 1911, 143, 8337, 29431, 29, 3945, 1675, 434, 555, 3413, 8437, 4997,
    233, 833, 539, 8998, 9057, 995, 163, 4237, 11454, 663, 1795, 1722, 3495, 16491, 8726, 7727,
    1147, 2503, 2573, 2930, 5627, 4105, 7385, 4634, 402, 8517, 1578, 10040, 3003, 725, 111, 8519,
    477, 271, 1235, 4111, 9612, 1480, 546, 699, 3842, 1927, 3321, 201, 9133, 1033, 1286, 9715,
    13596, 11580, 2717, 11271, 533, 8167, 5306, 8220, 1338, 4115, 3517, 6905, 28417, 2175, 3757,
    3207, 5791, 5826, 1023, 797, 2323, 7595, 2071, 761, 7636, 7447, 4799, 1057, 470, 2329, 2006,
    201, 635, 538, 1679, 371, 5579, 1140, 1097, 1299, 6481, 11547, 12056, 14638, 25754, 1529, 1146,
    11408, 715, 428, 983, 3053, 2886, 2527, 6702, 542, 11825, 7373, 3133, 638, 1337, 977, 1053,
    9042, 22742, 1873, 8707, 309, 1937, 4137, 253, 2991, 919, 1203, 1955, 4754, 6829, 9052, 351,
    801, 2405, 17945, 3081, 5098, 8528, 342, 5995, 17229, 10371, 1117, 15692, 5642, 3405, 6477,
    4893, 7035, 873, 8316, 3314, 1077, 1037, 1081, 1591, 9941, 4978, 1191, 635, 11027, 3955, 2959,
    2466, 16160, 433, 861, 5982, 1477, 142, 1407, 14880, 241, 455, 1410, 3749, 3005, 366, 847,
    10527, 777, 7715, 9691, 3541, 46, 2007, 1793, 1115, 222, 8421, 4325, 2227, 133, 569, 4701,
    3349, 1122, 9930, 8196, 191, 2314, 2461, 655, 2049, 319, 9320, 634, 1199, 22, 2049, 8747,
    10096, 1489, 10390, 3142, 1593, 1341, 10033, 1230, 1791, 1738, 1522, 10984, 3194, 14522, 2927,
    1733, 1093, 13680, 1469, 907, 990, 4822, 403, 1743, 620, 9855, 1953, 5030, 1327, 8635, 3065,
    2989, 5157, 2797, 8637, 4377, 759, 1174, 517, 5307, 1626, 7422, 4195, 14469, 2086, 6205, 20548,
    5369, 478, 8792, 63, 2387, 11896, 689, 1743, 4897, 15664, 593, 5794, 70, 1046, 3277, 998, 6585,
    6402, 7550, 3377, 13446, 6297, 3813, 3063, 4607, 42559, 701, 1914, 9970, 3631, 287, 573, 1597,
    1491, 3873, 1794, 1271, 5611, 2150, 153, 5691, 5039, 325, 1142, 2253, 62, 2050, 1142, 18879,
    95, 4515, 523, 4967, 3725, 9098, 18274, 2290, 2871, 633, 14582, 269, 7923, 8149, 3053, 1982,
    2039, 15796, 2983, 12818, 498, 2834, 2665, 2158, 426, 1938, 6567, 466, 82, 18667, 909, 15857,
    2035, 291, 241, 654, 31194, 167, 349, 819, 4809, 12879, 9479, 386, 1750, 3283, 141, 2427, 2873,
    1449, 9942, 154, 987, 4365, 3967, 11048, 659, 2827, 6573, 2121, 22450, 8014, 5939, 1107, 1491,
    1069, 1147, 22, 3334, 5273, 13168, 5217, 78, 2889, 8742, 166, 677, 249, 246, 1240, 3954, 7174,
    12082, 4307, 12917, 823, 821, 3686, 593, 28979, 5650, 9321, 12380, 997, 950, 3043, 1813, 4593,
    7292, 1737, 309, 3399, 3955, 11160, 8434, 6302, 2918, 31003, 5181, 5146, 459, 1633, 1787, 2339,
    3047, 14577, 36985, 1983, 4757, 1699, 24406, 3819, 2654, 433, 1015, 12878, 1302, 10336, 1741,
    3373, 7170, 5785, 1053, 13374, 11913, 533, 941, 599, 5086, 9586, 2747, 4458, 1666, 11487, 5645,
    40577, 7163, 9925, 279, 2009, 407, 2693, 9102, 5273, 10107, 1422, 115, 12049, 119, 11801, 341,
    9129, 8514, 1614, 1043, 2349, 366, 903, 3702, 3609, 2705, 2328, 1414, 2293, 2727, 2113, 3127,
    19308, 2329, 1023, 10175, 925, 8544, 4115, 2194, 1359, 9252, 138, 817, 7897, 918, 323, 8530,
    2829, 1751, 5581, 25658, 1517, 445, 2397, 163, 241, 14777, 171, 2782, 3930, 39974, 1277, 318,
    12735, 12044, 4934, 2323, 6697, 698, 6036, 4694, 5269, 587, 8445, 2218, 4630, 1954, 3981, 8067,
    8187, 4522, 1041, 10402, 37117, 1182, 1543, 11430, 9242, 834, 3537, 11522, 33957, 4995, 491,
    2279, 4122, 637, 8722, 3317, 6163, 1349, 455, 843, 995, 681, 42, 1783, 959, 3095, 1469, 28598,
    9902, 5909, 1163, 8572, 1722, 566, 9001, 2725, 3226, 7830, 2051, 1354, 1151, 8228, 1735, 19522,
    6446, 178, 1223, 5197, 51, 1074, 893, 381, 5286, 8925, 27691, 681, 1239, 9390, 1159, 7178, 598,
    10187, 10721, 12067, 891, 9191, 783, 2377, 2889, 137, 5050, 2581, 394, 5383, 1110, 6173, 9401,
    9059, 1275, 10477, 2108, 2211, 510, 8195, 4039, 13323, 4398, 4829, 1828, 5146, 5553, 9105, 209,
    9466, 201, 941, 8550, 3713, 2761, 12454, 3582, 3270, 15090, 3446, 8658, 7411, 2694, 419, 9631,
    8598, 5405, 911, 2447, 2630, 5381, 2755, 4395, 3385, 8224, 3201, 10432, 2437, 4091, 8499, 2335,
    777, 20517, 643, 2641, 5426, 2302, 1266, 386, 1115, 2494, 165, 2802, 4617, 695, 1625, 922, 701,
    1757, 8954, 510, 2072, 4207, 1966, 7815, 914, 26081, 2099, 917, 4625, 6695, 7186, 6210, 117,
    5026, 1271, 5254, 1715, 4462, 5370, 8965, 1593, 1116, 2115, 673, 1307, 62, 3722, 4870, 1913,
    253, 8632, 550, 876, 205, 943, 20717, 1486, 1779, 3334, 3878, 1266, 499, 7989, 15711, 7187,
    4325, 8247, 658, 5398, 2698, 3269, 1706, 5141, 30431, 1649, 1094, 13996, 3847, 4390, 3694,
    15908, 17573, 20205, 2870, 14429, 2837, 383, 2124, 2607, 5042, 5186, 1502, 2006, 8628, 2963,
    2653, 442, 1579, 1353, 3183, 893, 3977, 8346, 11853, 714, 417, 466, 762, 967, 495, 275, 10096,
    3197, 3982, 1911, 2821, 21737, 2510, 9330, 1671, 6513, 8563, 6061, 2022, 2153, 4302, 8330,
    1195, 5967, 10342, 543, 766, 519, 2371, 511, 417, 1446, 14731, 389, 2687, 431, 319, 117, 561,
    638, 2563, 4293, 1303, 1289, 5850, 467, 2709, 1828, 154, 24097, 419, 178, 2094, 587, 653, 9920,
    6697, 3799, 3747, 323, 1437, 2619, 1630, 1379, 924, 3523, 8811, 1134, 1299, 991, 1659, 11179,
    3105, 34590, 8896, 362, 2595, 1254, 1393, 379, 6677, 571, 155, 379, 4726, 501, 1186, 1025, 447,
    10505, 975, 754, 1386, 2117, 20220, 5143, 7843, 2795, 3011, 713, 389, 1101, 767, 1116, 447,
    1307, 2059, 4058, 3598, 610, 2921, 2353, 166, 4433, 199, 1477, 4025, 9850, 301, 5006, 14630,
    693, 10889, 12815, 151, 2526, 4595, 1159, 4087, 1795, 7707, 30981, 10557, 10197, 5281, 6233,
    7310, 1823, 2131, 4621, 2283, 3510, 1310, 1449, 4606, 2231, 326, 2062, 9019, 10527, 3577, 6812,
    4229, 323, 669, 2413, 3601, 4041, 2151, 8904, 10171, 5683, 1349, 2763, 7486, 1189, 771, 1139,
    3803, 12249, 431, 13170, 1091, 37694, 3891, 9607, 2221, 5779, 7870, 11678, 998, 969, 2133,
    8933, 14255, 11624, 2465, 15753, 2838, 4187, 3367, 1286, 5679, 2186, 715, 2143, 998, 342, 751,
    12177, 6259, 1302, 571, 8110, 7923, 12334, 8286, 1113, 1222, 8769, 371, 876, 7991, 2706, 4717,
    2705, 541, 394, 1883, 7507, 643, 835, 357, 929, 8694, 8122, 1973, 2503, 357, 130, 937, 8681,
    975, 91, 123, 13290, 4583, 2078, 3270, 2682, 8936, 8249, 1775, 8519, 1439, 6295, 4151, 10817,
    8942, 3263, 18933, 1293, 183, 4601, 9155, 5481, 481, 39192, 358, 16721, 419, 2142, 4135, 16945,
    10998, 6490, 1175, 8490, 846, 6409, 12493, 23326, 2899, 415, 36491, 13051, 866, 161, 22, 2150,
    849, 4670, 3849, 489, 833, 549, 5675, 14716, 8764, 1685, 161, 4785, 30101, 1067, 635, 126,
    1387, 5495, 985, 1942, 166, 453, 2101, 8602, 1389, 6802, 193, 4307, 1302, 6233, 3138, 1199,
    3111, 10278, 2852, 8693, 13152, 10313, 10246, 66, 554, 1941, 8070, 5726, 9988, 1997, 693, 2405,
    549, 350, 11750, 5111, 6237, 5873, 4531, 1289, 1709, 2238, 831, 202, 2893, 1726, 47, 23812,
    1553, 1053, 2927, 635, 5906, 4088, 1084, 7304, 7714, 1363, 3417, 2427, 3119, 7934, 4549, 2271,
    1330, 1483, 650, 223, 649, 7917, 3966, 7318, 411, 8992, 2455, 1135, 4953, 919, 2130, 610, 1844,
    137, 403, 3757, 9114, 1366, 646, 8467, 7277, 7171, 2442, 93, 857, 895, 2134, 38617, 11100,
    12622, 67, 3883, 5807, 743, 5005, 731, 10084, 69, 46255, 865, 6671, 1071, 3798, 1855, 1426,
    3361, 571, 2697, 161, 286, 497, 2207, 2038, 2015, 12745, 10440, 313, 1443, 8753, 11702, 267,
    7226, 27666, 398, 1309, 170, 10128, 1398, 1839, 8276, 3414, 55, 3107, 3154, 2405, 1814, 8903,
    4845, 1417, 1667, 2427, 549, 818, 475, 1499, 1169, 767, 537, 1697, 2386, 2075, 3673, 1871,
    9379, 1938, 3229, 1237, 222, 209, 5473, 5589, 4397, 9112, 5233, 12487, 1835, 149, 7860, 5289,
    10055, 4185, 53, 1394, 2265, 2610, 6079, 8316, 13263, 3475, 9011, 22, 159, 6061, 82, 2361, 349,
    25061, 10631, 2375, 10013, 9529, 10695, 17510, 231, 9334, 1798, 1263, 4233, 1565, 23669, 930,
    1958, 4258, 1674, 6817, 33844, 2653, 5899, 749, 1627, 639, 4551, 2575, 11826, 2063, 8731, 1695,
    514, 339, 233, 3975, 1795, 7759, 1851, 1463, 3431, 1310, 4105, 4382, 3243, 9196, 4133, 14732,
    1146, 4154, 3782, 1494, 1370, 2986, 5165, 12898, 2091, 143, 11820, 22253, 874, 3347, 14570,
    13432, 29171, 581, 12929, 770, 9338, 3913, 2067, 466, 1442, 1065, 10328, 3273, 197, 4811, 8897,
    5417, 1769, 206, 119, 3406, 6754, 309, 3909, 646, 3677, 3514, 8797, 1435, 301, 9215, 3730,
    7282, 165, 3381, 303, 6369, 2627, 115, 761, 3695, 1233, 1297, 1803, 1130, 122, 511, 58, 6426,
    1389, 1433, 9102, 943, 2795, 13162, 9905, 899, 9740, 8719, 517, 5115, 4977, 1486, 8259, 7163,
    13549, 7561, 305, 119, 973, 7698, 5619, 5326, 5772, 32611, 8846, 1709, 4641, 782, 2503, 3905,
    4051, 4923, 165, 16132, 16827, 1913, 6590, 322, 741, 522, 7666, 1655, 5087, 468, 8541, 1169,
    4455, 22993, 2801, 6623, 446, 1979, 10923, 3503, 32053, 523, 5721, 9608, 955, 11766, 49919,
    1982, 66, 3347, 9508, 439, 145, 5697, 1026, 1569,
];

//////////////////// Lookup tables for number theoretic functions

// table for riemann zeta function. ZETA_LOG_TABLE[i] = ln(ζ(i-2))
#[cfg(feature = "big-table")]
pub const ZETA_LOG_TABLE: [f64; 64] = [
    4.977003024707453704422733e-01,
    1.840341753914914169065753e-01,
    7.910987306733563428906564e-02,
    3.626225964922791950018066e-02,
    1.719438760265828980422498e-02,
    8.314614969275204922616318e-03,
    4.069066307412954990041509e-03,
    2.006378701528290387640086e-03,
    9.940808656690607904293788e-04,
    4.940665331469770151992527e-04,
    2.460562789788238511994034e-04,
    1.227058189115578395490813e-04,
    6.124625946826462580263595e-05,
    3.058776849645994703411761e-05,
    1.528214263611525392016002e-05,
    7.637168474654366096660528e-06,
    3.817285979154446217526640e-06,
    1.908210895918369244355067e-06,
    9.539615788513043827126850e-07,
    4.769328730553057084173103e-07,
    2.384504742984163710492051e-07,
    1.192199188586162536606321e-07,
    5.960818727469145156012688e-08,
    2.980350307052787786014558e-08,
    1.490155471733687385350177e-08,
    7.450711762078876717827681e-09,
    3.725334017849400424712395e-09,
    1.862659721778298369227706e-09,
    9.313274319859827471326702e-10,
    4.656629063949574247211838e-10,
    2.328311833405453758759852e-10,
    1.164155017202289036848590e-10,
    5.820772087733293568249733e-11,
    2.910385044454747711695271e-11,
    1.455192189093610577568052e-11,
    7.275959835031012016668332e-12,
    3.637979547372033417761987e-12,
    1.818989650305411709490059e-12,
    9.094947840259752504178820e-13,
    4.547473783041120242576761e-13,
    2.273736845824393946183985e-13,
    1.136868407680163290842639e-13,
    5.684341987627423858168184e-14,
    2.842170976889261614354352e-14,
    1.421085482803150734740459e-14,
    7.105427395210827461615018e-15,
    3.552713691337107622981434e-15,
    1.776356843579118836645811e-15,
    8.881784210930812248514214e-16,
    4.440892103143812146018826e-16,
    2.220446050798041906634718e-16,
    1.110223025141066149019450e-16,
    5.551115124845480986572498e-17,
    2.775557562136123907111332e-17,
    1.387778780972523194213624e-17,
    6.938893904544153439927978e-18,
    3.469446952165922542917276e-18,
    1.734723476047576545776400e-18,
    8.673617380119933001297893e-19,
    4.336808690020650566764225e-19,
    2.168404344997219811593044e-19,
    1.084202172494241415200165e-19,
    5.421010862456645841915045e-20,
    2.710505431223468975644559e-20,
];

// tables for moebius function with small input
pub const MOEBIUS_ODD: [u64; 4] = [
    0x4289108a05208102,
    0x19988004a8a12422,
    0x1a8245028906a062,
    0x229428012aa26a00,
];

//////////////////// Prime gaps for prime indexing ////////////////////

/* Code to generate wheel offsets:
```python
import numpy as np
wheel = [2,3,5]
wheel_size = np.product(wheel)
flags = np.ones(wheel_size, dtype=bool)
tester = np.arange(wheel_size)
for p in wheel:
    flags[tester % p == 0] = False
flags = np.concatenate([flags, flags])
next_table = np.zeros(wheel_size, dtype=int)
prev_table = np.zeros(wheel_size, dtype=int)
for i in range(wheel_size):
    next_table[i] = np.argmax(flags[i:])
    prev_table[i] = np.argmax(flags[:wheel_size+i+1][::-1])

next_zeros, = np.where(next_table == 0)
next_table[next_zeros] = next_table[(next_zeros+1) % wheel_size]+1
prev_zeros, = np.where(prev_table == 0)
prev_table[prev_zeros] = prev_table[(prev_zeros-1) % wheel_size]+1

print(next_table)
print(prev_table)
```
*/

// offset for moving to the prev/next prime candidate on wheel30 (2*3*5)
#[cfg(not(feature = "big-table"))]
pub const WHEEL_SIZE: u8 = 30;
#[cfg(not(feature = "big-table"))]
pub const WHEEL_NEXT: [u8; 30] = [
    1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2,
];
#[cfg(not(feature = "big-table"))]
pub const WHEEL_PREV: [u8; 30] = [
    1, 2, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6,
];

// offset for moving to the prev/next prime candidate on wheel2310 (2*3*5*7*11)
#[cfg(feature = "big-table")]
pub const WHEEL_SIZE: u16 = 2310;
#[cfg(feature = "big-table")]
pub const WHEEL_NEXT: [u8; 2310] = [
    1, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1,
    6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5,
    4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3,
    2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3,
    2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2,
    1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 10, 9, 8, 7, 6, 5,
    4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 10, 9, 8, 7, 6, 5,
    4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3,
    2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1,
    2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1,
    4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 10, 9, 8, 7, 6, 5, 4,
    3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2,
    1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2,
    1, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1,
    4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 12, 11, 10,
    9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2,
    1, 6, 5, 4, 3, 2, 1, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1,
    8, 7, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5,
    4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4,
    3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1, 10, 9, 8, 7, 6, 5, 4, 3,
    2, 1, 2, 1, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2,
    1, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2,
    1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 8, 7, 6,
    5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 6, 5, 4,
    3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2,
    1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1,
    4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 10, 9, 8, 7, 6, 5, 4,
    3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2,
    1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 8,
    7, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4,
    3, 2, 1, 2, 1, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2,
    1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2,
    1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1,
    10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4,
    3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2,
    1, 6, 5, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 10, 9, 8, 7,
    6, 5, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 8, 7,
    6, 5, 4, 3, 2, 1, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 8, 7, 6,
    5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2,
    1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1,
    2, 1, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2,
    1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6,
    5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 8, 7, 6, 5, 4,
    3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2,
    1, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 6,
    5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 4, 3, 2,
    1, 2, 1, 4, 3, 2, 1, 2, 1, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1,
    2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2,
    1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 6,
    5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2,
    1, 2, 1, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2,
    1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4,
    3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 10, 9,
    8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2,
    1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1,
    6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5,
    4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1, 12, 11, 10, 9, 8, 7, 6,
    5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2,
    1, 2, 1, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3,
    2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2,
    1, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1,
    6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5,
    4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 8, 7, 6, 5, 4,
    3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2,
    1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 6,
    5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2,
    1, 2, 1, 4, 3, 2, 1, 2, 1, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2,
    1, 2, 1, 4, 3, 2, 1, 2, 1, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1,
    2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2,
    1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2,
    1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4,
    3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2,
    1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1,
    12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2,
];
#[cfg(feature = "big-table")]
pub const WHEEL_PREV: [u8; 2310] = [
    1, 2, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1,
    2, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 1, 2, 1,
    2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5,
    6, 7, 8, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12,
    13, 14, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 1, 2, 1, 2, 3, 4, 5,
    6, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 1, 2, 3, 4, 5, 6, 7,
    8, 9, 10, 1, 2, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 1, 2, 3, 4, 5, 6, 7,
    8, 9, 10, 1, 2, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 1, 2, 3, 4, 5, 6, 1, 2,
    3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 1, 2, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4,
    1, 2, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 7, 8, 1, 2, 3, 4, 1, 2,
    1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 7, 8, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 7, 8,
    9, 10, 1, 2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 1, 2, 3,
    4, 1, 2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 5,
    6, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 1, 2, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 1, 2, 1, 2, 3, 4, 1, 2,
    1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 7, 8, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2, 3, 4,
    5, 6, 7, 8, 9, 10, 11, 12, 1, 2, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 5, 6, 1, 2, 3,
    4, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 1, 2, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4,
    1, 2, 3, 4, 5, 6, 7, 8, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 1, 2, 3, 4, 1, 2,
    3, 4, 5, 6, 1, 2, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 1, 2, 1, 2, 3, 4, 1, 2, 3,
    4, 5, 6, 1, 2, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 5, 6, 7, 8, 9,
    10, 1, 2, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 1, 2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6,
    1, 2, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 1, 2,
    1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4,
    5, 6, 7, 8, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 7, 8, 1, 2, 3, 4,
    5, 6, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 7, 8,
    1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 1, 2, 1, 2, 3, 4, 5, 6, 1,
    2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 1, 2, 1, 2, 3, 4, 5, 6, 7, 8,
    9, 10, 1, 2, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 7, 8, 1, 2, 3, 4, 5, 6, 1, 2, 3,
    4, 1, 2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 1, 2, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1,
    2, 3, 4, 5, 6, 7, 8, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 7, 8, 1, 2, 3, 4, 1, 2, 1,
    2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 7, 8, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 3, 4, 5,
    6, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 1, 2, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1,
    2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2, 1,
    2, 3, 4, 5, 6, 7, 8, 9, 10, 1, 2, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 1, 2, 1, 2, 3, 4, 5, 6, 1, 2,
    3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6,
    1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 7, 8, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6,
    7, 8, 9, 10, 1, 2, 3, 4, 5, 6, 7, 8, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2, 3,
    4, 5, 6, 7, 8, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 1, 2, 3, 4, 5, 6, 1, 2, 1, 2, 3, 4, 1, 2, 3, 4,
    5, 6, 7, 8, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6,
    1, 2, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 1,
    2, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 1, 2, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2,
    1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 1, 2, 1, 2,
    3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6,
    7, 8, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 7, 8, 1, 2, 3, 4, 5, 6,
    1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 7, 8, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 1, 2, 3, 4, 5, 6, 1, 2,
    3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 7, 8, 1, 2, 3, 4,
    1, 2, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 1, 2, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 1,
    2, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 1, 2,
    1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 7, 8, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 1, 2,
    3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 7, 8, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6,
    1, 2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 7, 8, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2,
    1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 3, 4,
    5, 6, 1, 2, 3, 4, 5, 6, 1, 2, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4,
    5, 6, 7, 8, 9, 10, 1, 2, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 1, 2, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 1,
    2, 3, 4, 5, 6, 1, 2, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 1, 2,
    3, 4, 5, 6, 1, 2, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 3, 4,
    5, 6, 1, 2, 3, 4, 5, 6, 7, 8, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 5, 6, 7, 8, 9,
    10, 11, 12, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6,
    1, 2, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 7,
    8, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 1, 2,
    1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 1, 2, 3, 4, 5, 6, 1, 2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 1,
    2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 1, 2, 1, 2, 3,
    4, 5, 6, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6,
    7, 8, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 7, 8, 1, 2, 3, 4, 5, 6,
    1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 1, 2, 3, 4, 5, 6, 1, 2,
    3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4,
    1, 2, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12,
    1, 2, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 1, 2, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1,
    2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 1, 2, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 1, 2,
    1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 1, 2, 3, 4, 1, 2,
    1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 7, 8, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 3, 4,
    5, 6, 1, 2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4,
    1, 2, 1, 2, 3, 4, 1, 2, 3, 4, 5, 6, 1, 2, 1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 1, 2, 1, 2, 3, 4, 1, 2,
    3, 4, 5, 6, 7, 8, 9, 10, 11, 12,
];

//////////////////// Modulo tables for fast perfect power check ////////////////////
/* Code to generate residuals:
```python
import math
import numpy as np

N = 8 # number of modulus to be selected
LIM = 1024 # limit of modulus
fn = lambda x: x*x # power function

# first select N modulus under LIM (assuming factor 2 is already considered)
frequency = [(len(np.unique([fn(i)%L for i in range(L)]))/L, L)
             for L in range(1, LIM, 2)]
selections = []
select_freq = 1
for (f, n) in sorted(frequency):
    for s in selections:
        if math.gcd(n, s) != 1:
            break
    else:
        selections.append(n)
        select_freq *= f
    if len(selections) >= N:
        break

print("Moduli:", selections)
print("Possibility", select_freq)

# Then generate bit masks for the numbers
for s in selections:
    bits = [0] * (s // 64 + 1)
    for i in range(s):
        ii = fn(i) % s
        bits[ii//64] |= 1 << (ii%64)
    print("Masks for", s, ":", ['0x%016x' % b for b in bits])
```
*/

#[cfg(not(feature = "big-table"))]
pub const QUAD_MODULI: [u8; 4] = [63, 65, 61, 59]; // note that 0 and 64 are both square mod 65
#[cfg(not(feature = "big-table"))]
pub const QUAD_RESIDUAL: [u64; 4] = [0x0402483012450293, 0x218a019866014613, 0x1713e6940a59f23b, 0x022b62183e7b92bb];

#[cfg(not(feature = "big-table"))]
pub const CUBIC_MODULI: [u8; 4] = [63, 61, 43, 37];
#[cfg(not(feature = "big-table"))]
pub const CUBIC_RESIDUAL: [u64; 4] = [0x4080001818000103, 0x1434026619900b0b, 0x0000068908610917, 0x00000010ac804d43];

#[cfg(feature = "big-table")]
pub const QUAD_MODULI: [u16; 8] = [819, 935, 989, 899, 1007, 1021, 1019, 1013];
#[cfg(feature = "big-table")]
pub const QUAD_RESIDUAL: [[u64; 16]; 8] = [
    [0x0002081002410213, 0xc200001009028001, 0x0000120000014024, 0x008048020208a050, 0x1800048200244029, 0x0420120180020014, 0x2600000080490408, 0x9100041060200800, 0x0904004800120100, 0x2402000400082201, 0x402100920c000040, 0x0820804002004880, 0x0000002403012000, 0, 0, 0],
    [0x1882001406018213, 0x0288c11002420065, 0x820846400c010180, 0x4841142230000010, 0x8004000e01100101, 0x00c0421801280810, 0x1304202104018060, 0xe400823014001088, 0x04228001100a4000, 0x00c23144008c0401, 0x0100011083204400, 0x100024108e001802, 0x2200c00208452040, 0x00310020080c0204, 0x0000000000140210, 0],
    [0x0c52821883812253, 0x9220403190824001, 0x1e0026b004455c20, 0xa040d1c209a58032, 0x298a68420480a049, 0x1003026612000074, 0x6b8048c400192281, 0x300102d8021420a0, 0x85211508223e0404, 0x220a810845800181, 0xcc50291070510060, 0x840513520a408200, 0xd0a6422048d38098, 0x00083c2112b00305, 0x990d8401510220b4, 0x0000000001400682],
    [0xc80a205a121102b3, 0x2201a89060874089, 0x4890e22001211412, 0x9120328e08001414, 0xcd400a825a001c43, 0x6330022830e08151, 0x169c90483ca12044, 0x0416201002620801, 0x4081490e008b8218, 0xc422404328122401, 0x711e881088340820, 0x094632241a034000, 0x020129c80021cb00, 0x204038620200e624, 0, 0],
    [0xc2429c5013030ad3, 0x0a890459a0062015, 0x18019308c361404c, 0x51002032052042b8, 0x00b2604604098505, 0x3614a6400cc04390, 0x022c03848101900d, 0x0220505212087021, 0x45804800ab0611a6, 0x110a08110156680b, 0x10a72415020038c8, 0xc06114642ac00401, 0x0180003498171808, 0x4c20010128908622, 0x0c58060200714052, 0x0000002500040682],
    [0x718bb6522e93da3b, 0x6a5e15181aebcef9, 0x5cb903b8072b20b4, 0xebb96e0b5e64f6b7, 0xf54a7312517f386d, 0x4c8fba9e7f589015, 0xe68bc3c5a26998c0, 0x0930ba16c101df98, 0xc67ee020da174324, 0x80c6659168f0f459, 0xea0246bf9e577c4c, 0xed873fa2923394ab, 0xbb5bc99eb41da775, 0x8b4135380770274e, 0xa7dcf5d6062a1e95, 0x1716f25d129b7463],
    [0x138b3c12ae9bda3b, 0xf65a353afaa7de71, 0x08311be00eab42d6, 0xa069e43b58ced0b3, 0xd589933aedfb2ceb, 0xfdbdfa54563a03dd, 0xe63f0f092c7538d8, 0x3b77e1b60b6ff8b2, 0x8b2e0092f9278112, 0x0e4e351cb6f0f039, 0x4443fa395d5a0424, 0xa28cb2048a3366e5, 0xf32f48ce523d869f, 0x094bd2a8ff82773e, 0x771841aa0a353a59, 0x023a4268ab7c32e3],
    [0x557a191c03a9ee53, 0xcee6c01b76bed685, 0x62e33aa92c63b5c4, 0x2d830be3875ffeba, 0x59a7c9d2ba4e854f, 0x49f183525a139bb6, 0x1762cc0a2801b21b, 0x73843dd8ff3029f2, 0xba13e5033fc6ef08, 0xe4b6136005140cd1, 0x669b76721692b063, 0x6d3ca85c9752e4f9, 0xd1975ffeb871f430, 0xdcc8eb718d255731, 0xaaa85adf5bb600d9, 0x00129de5700e2617],
];

#[cfg(feature = "big-table")]
pub const CUBIC_MODULI: [u16; 8] = [819, 817, 925, 961, 1021, 1009, 997, 991];
#[cfg(feature = "big-table")]
pub const CUBIC_RESIDUAL: [[u64; 16]; 8] = [
    [0x0000000008000103, 0x204000080c000001, 0x4020000200000000, 0x1000000001000000, 0x0008000080000010, 0x06000000c0800000, 0x0600000000000000, 0x0000000010300000, 0x0080800000100001, 0x4020000008000000, 0x2040000000040000, 0x0008000003010000, 0x0004080001000000, 0,                  0,                  0                 ],
    [0x0000208008000903, 0x2202084041900803, 0x00801c1000821020, 0x4080100003000000, 0x010282202004040c, 0x0800010000c0000c, 0x0040640800004098, 0x0200c0000c000200, 0x0408c08080101105, 0x0400000003000020, 0x01101021040020e0, 0x0003004026080841, 0x0001024000400410, 0,                  0,                  0                 ],
    [0x80098870ac804943, 0x26a180b201140e15, 0x850ac804d4384640, 0x2b001310e1590092, 0x800d43014402280c, 0x250a149009a870ac, 0x1856002621c2b201, 0x19001a860a880454, 0x024a1429201350e1, 0xa030ac004d438564, 0xc23200350c051008, 0xc804d428524026a1, 0x134061590098870a, 0x438464006a1c0a20, 0x0000000010a4804d, 0                 ],
    [0xb440c08b68818117, 0x6d103022da206045, 0x5b440c08b6881811, 0x16d103022da20604, 0x45b440c08b688181, 0x116d103022da2060, 0x045b440c08b68818, 0x8116d103022da206, 0x6045b440c08b6881, 0x18116d103022da20, 0x06045b440c08b688, 0x818116d103022da2, 0x206045b440c08b68, 0x8818116d103022da, 0xa206045b440c08b6, 0x0000000000000001],
    [0x815304911d2573df, 0x602b11250832c381, 0xa8400704788f4809, 0xb4082c4d09210411, 0x9540c075104a8243, 0xc4e01080a2bd8818, 0x226b040380122321, 0xed1208408cd21071, 0x238212cc4084122d, 0xe131120070083591, 0x46046f51404201c8, 0x709054822b80c0aa, 0x620821242c8d040b, 0xa404bc4788380085, 0x6070d30429223501, 0x1ef3a92e224832a0],
    [0x7a049c880998630b, 0x60b3438000062731, 0x87da2e012960281a, 0x40cc00a04918b811, 0x1920480419089410, 0x15e806b391a91404, 0x803292a20091761c, 0x484b0505c1a2b3cc, 0x3004cf35160e8283, 0x5ea0e1ba24011525, 0x126080a256273580, 0xcc0820a442608048, 0x6f86207462481400, 0x341960501a5201d1, 0x817a33918000070b, 0x00014318664044e4],
    [0x50452131cb15977f, 0x3182b013240b0d09, 0x0e1000c7085100e1, 0x2789402665020ba7, 0x80c0190908a05403, 0x02d401a22088fa95, 0x1c9180248065e415, 0x84bf486910022614, 0x00624e0a19100225, 0x600ad02a09e98049, 0x2600c06a57c44111, 0x00a479300a814424, 0xc0021c3974102999, 0x03506321c0228438, 0x212882a42c340932, 0x0000001fba6a34e3],
    [0x300cd4682d12430b, 0x704fc6a280c6e183, 0x009000c122c0a750, 0x684c67a807a32039, 0x486185ba21680264, 0x059194950edba400, 0x440030c0c94d1004, 0x80a425018280082e, 0x030c002274100141, 0xa92989a02008b293, 0x5da186120025db70, 0x15e6321626401684, 0x830009009c04c5e0, 0x4563f20e0ae50344, 0x162b300cc1876301, 0x0000000050c248b4],
];
