use crate::*;

pub struct MessageBuilder {
    id: u8,
    sender: String,
    recipient: Recipient,
}

impl MessageBuilder {
    pub fn new() -> Self {
        Self {
            id: 0,
            sender: String::new(),
            recipient: Recipient::Broadcast,
        }
    }

    pub fn reply(self, reply: Reply) -> Message {
        Message {
            id: self.id,
            recipient: self.recipient,
            sender: self.sender,
            message: MessageType::Reply(reply),
        }
    }
    pub fn request(self, request: Request) -> Message {
        Message {
            id: self.id,
            recipient: self.recipient,
            sender: self.sender,
            message: MessageType::Request(request),
        }
    }

    pub fn log(self, level: log::LogLevel, message: &str) -> Message {
        Message {
            id: self.id,
            recipient: self.recipient,
            sender: self.sender,
            message: MessageType::Log(log::LogMessage {
                level,
                message: message.to_string(),
            }),
        }
    }

    pub fn id(mut self, id: u8) -> Self {
        self.id = id;
        self
    }

    pub fn sender(mut self, sender: &str) -> Self {
        self.sender = sender.to_owned();
        self
    }

    pub fn recipient(mut self, recipient: &str) -> Self {
        self.recipient = recipient.into();
        self
    }
}

// pub struct RequestBuilder {
//     message_builder: MessageBuilder
// }
//
// impl RequestBuilder {
//     pub fn new() -> Self {
//         Self {
//             message_builder: MessageBuilder::new(),
//         }
//     }
// }
//
// impl Deref for RequestBuilder {
//     type Target = MessageBuilder;
//
//     #[inline]
//     fn deref(&self) -> &Self::Target {
//         &self.message_builder
//     }
// }
//
// impl DerefMut for RequestBuilder {
//     #[inline]
//     fn deref_mut(&mut self) -> &mut MessageBuilder {
//         &mut self.message_builder
//     }
// }
//
// pub struct ReplyBuilder {
//     message_builder: MessageBuilder,
//     reply: Option<Reply>,
// }
//
// // reply builds from the request
// impl ReplyBuilder {
//     pub fn new() -> Self {
//         ReplyBuilder {
//             message_builder: MessageBuilder::new(),
//             reply: None,
//         }
//     }
//
//     pub fn build(self) -> Message {
//         self.message_builder.reply(Reply::System(SystemReply::Error("Not implemented".to_string())))
//     }
// }
//
// impl Deref for ReplyBuilder {
//     type Target = MessageBuilder;
//
//     #[inline]
//     fn deref(&self) -> &Self::Target {
//         &self.message_builder
//     }
// }
//
// impl DerefMut for ReplyBuilder {
//     #[inline]
//     fn deref_mut(&mut self) -> &mut MessageBuilder {
//         &mut self.message_builder
//     }
// }

impl From<&str> for Recipient {
    fn from(recipient: &str) -> Self {
        Recipient::Unicast(recipient.to_owned())
    }
}
