use serde::{Deserialize, Serialize};
use std::convert::TryFrom;

pub mod log;
pub mod nucleon;
pub mod system;
pub mod util;

pub mod prelude {
    pub use super::log::*;
    pub use super::nucleon::*;
    pub use super::system::*;
    pub use super::{Message, MessageType, Recipient, Reply, Request};

    pub use super::util::*;

    pub use std::convert::TryFrom;

    pub use serde_json::Value;
}

#[derive(Clone, Debug, Serialize, Deserialize)]
pub enum Request {
    System(system::SystemRequest),
    Nucleon(nucleon::NucleonRequest),
}

#[derive(Clone, Debug, Serialize, Deserialize)]
pub enum Reply {
    System(system::SystemReply),
    Nucleon(nucleon::NucleonReply),
}

#[derive(Clone, Debug, Serialize, Deserialize)]
pub enum MessageType {
    Request(Request),
    Reply(Reply),
    Log(log::LogMessage),
}

impl ReplyTrait for MessageType {}

trait ReplyTrait {}

#[derive(Clone, Debug, Serialize, Deserialize)]
pub enum Recipient {
    Unicast(String),
    Broadcast,
}

#[derive(Clone, Debug, Serialize, Deserialize)]
pub struct Message {
    pub message: MessageType,
    pub id: u8,
    pub recipient: Recipient,
    pub sender: String,
}

impl TryFrom<&str> for Message {
    type Error = serde_json::Error;

    #[inline]
    fn try_from(msg: &str) -> Result<Self, Self::Error> {
        serde_json::from_str::<Message>(msg)
    }
}

impl TryFrom<&Message> for String {
    type Error = serde_json::Error;

    #[inline]
    fn try_from(msg: &Message) -> Result<Self, Self::Error> {
        serde_json::to_string(msg)
    }
}

#[cfg(test)]
mod tests {
    #[test]
    fn builder() {
        use crate::prelude::*;
        let _reply = Reply::System(SystemReply::Success);
        // let m = MessageBuilder::new().build::<Reply>(reply);
    }
}
