use libloading::{Error, Library, Symbol};
use nucleon::exports::*;

pub struct NucleonManager {
    library: std::sync::Mutex<Library>,
}

impl NucleonManager {
    pub fn new(path: &str) -> Result<Self, Error> {
        let library = unsafe { Library::new(path) }?;

        let init: Symbol<NucleonInit> = unsafe { library.get(b"init") }?;
        init();

        // make sure all symbols are present
        let _ = unsafe { library.get::<NucleonCreate>(b"new") }?;
        let _ = unsafe { library.get::<NucleonStaticInfo>(b"static_info") }?;

        Ok(Self {
            library: std::sync::Mutex::new(library),
        })
    }

    pub fn create_nucleon(&self, config: &NucleonConfig, tx: &Sender<Message>) -> Box<dyn Nucleon> {
        let library = self.library.lock().unwrap();
        let nucleon_create = unsafe { library.get::<NucleonCreate>(b"new").unwrap() };
        let raw = (nucleon_create)(config, tx);
        unsafe { Box::from_raw(raw) }
    }

    pub fn static_info(&self) -> NucleonInfo {
        let library = self.library.lock().unwrap();
        let nucleon_static_info =
            unsafe { library.get::<NucleonStaticInfo>(b"static_info").unwrap() };
        (nucleon_static_info)()
    }
}
