//! Key Encapsulation mechanism implementation of NTRU

use tiny_keccak::{Hasher, Sha3};

use crate::api::{
    CRYPTO_BYTES, CRYPTO_CIPHERTEXTBYTES, CRYPTO_PUBLICKEYBYTES, CRYPTO_SECRETKEYBYTES,
};
use crate::cmov::cmov;
use crate::owcpa::{owcpa_dec, owcpa_enc, owcpa_keypair};
use crate::pack3::poly_s3_tobytes;
use crate::params::{
    NTRU_CIPHERTEXTBYTES, NTRU_OWCPA_MSGBYTES, NTRU_OWCPA_SECRETKEYBYTES, NTRU_PACK_TRINARY_BYTES,
    NTRU_PRFKEYBYTES, NTRU_SAMPLE_FG_BYTES, NTRU_SAMPLE_RM_BYTES, NTRU_SHAREDKEYBYTES,
};
use crate::poly::{poly_z3_to_zq, Poly};
use crate::rng::RNGState;
use crate::sample::sample_rm;

use std::error;

type R = Result<(), Box<dyn error::Error>>;

/// Given an RNG instance, compute some public and secret key.
/// The public key is meant to be shared with any party,
/// but access to the secret key must be limited to the generating party.
pub fn crypto_kem_keypair(
    pk: &mut [u8; CRYPTO_PUBLICKEYBYTES],
    sk: &mut [u8; CRYPTO_SECRETKEYBYTES],
    rng: &mut impl RNGState,
) -> R {
    let mut seed = [0u8; NTRU_SAMPLE_FG_BYTES];
    rng.randombytes(&mut seed)?;
    owcpa_keypair(pk, sk, seed);

    let mut sk_copy = [0u8; NTRU_PRFKEYBYTES];
    sk_copy.copy_from_slice(&sk[NTRU_OWCPA_SECRETKEYBYTES..]);
    rng.randombytes(&mut sk_copy)?;
    sk[NTRU_OWCPA_SECRETKEYBYTES..].copy_from_slice(&sk_copy);

    Ok(())
}

/// Given an RNG instance and a public key, sample a shared key.
/// This shared key is returned through parameter `k` whereas
/// ciphertext is returned as `c`.
pub fn crypto_kem_enc(
    c: &mut [u8; CRYPTO_CIPHERTEXTBYTES],
    k: &mut [u8; CRYPTO_BYTES],
    pk: &[u8; CRYPTO_PUBLICKEYBYTES],
    rng: &mut impl RNGState,
) -> R {
    let r = &mut Poly::new();
    let m = &mut Poly::new();
    let mut rm = [0u8; NTRU_OWCPA_MSGBYTES];
    let rm_seed = &mut [0u8; NTRU_SAMPLE_RM_BYTES];

    rng.randombytes(rm_seed)?;

    sample_rm(r, m, *rm_seed);

    let rm1 = <&mut [u8; NTRU_PACK_TRINARY_BYTES]>::try_from(&mut rm[..NTRU_PACK_TRINARY_BYTES])?;
    poly_s3_tobytes(rm1, r);
    let rm2 = <&mut [u8; NTRU_PACK_TRINARY_BYTES]>::try_from(
        &mut rm[NTRU_PACK_TRINARY_BYTES..2 * NTRU_PACK_TRINARY_BYTES],
    )?;
    poly_s3_tobytes(rm2, m);
    sha3_256(k, &rm);

    poly_z3_to_zq(r);
    owcpa_enc(c, r, m, pk);

    Ok(())
}

fn sha3_256(output: &mut [u8; 32], input: &[u8]) {
    let mut sha3 = Sha3::v256();
    sha3.update(input);
    sha3.finalize(output);
}

/// Given a secret key and a ciphertext,
/// determine the shared text and return it is argument `k`.
pub fn crypto_kem_dec(
    k: &mut [u8; CRYPTO_BYTES],
    c: &[u8; CRYPTO_CIPHERTEXTBYTES],
    sk: &[u8; CRYPTO_SECRETKEYBYTES],
) -> R {
    let rm = &mut [0u8; NTRU_OWCPA_MSGBYTES];
    let mut buf = [0u8; NTRU_PRFKEYBYTES + NTRU_CIPHERTEXTBYTES];
    let fail = owcpa_dec(rm, c, sk);
    /* If fail = 0 then c = Enc(h, rm). There is no need to re-encapsulate. */
    /* See comment in owcpa_dec for details.                                */

    sha3_256(k, rm);

    /* shake(secret PRF key || input ciphertext) */
    buf[..NTRU_PRFKEYBYTES].clone_from_slice(
        &sk[NTRU_OWCPA_SECRETKEYBYTES..(NTRU_PRFKEYBYTES + NTRU_OWCPA_SECRETKEYBYTES)],
    );
    buf[NTRU_PRFKEYBYTES..(NTRU_CIPHERTEXTBYTES + NTRU_PRFKEYBYTES)]
        .clone_from_slice(&c[..NTRU_CIPHERTEXTBYTES]);

    sha3_256(<&mut [u8; 32]>::try_from(&mut rm[0..32])?, &buf);

    cmov(k, rm, NTRU_SHAREDKEYBYTES as isize, fail as u8);

    Ok(())
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::rng::AesState;

    #[test]
    fn test_keypair_zeroed_rng() -> R {
        #[cfg(feature = "ntruhps2048509")]
        {
            let mut pk = [0u8; CRYPTO_PUBLICKEYBYTES];
            let mut sk = [0u8; CRYPTO_SECRETKEYBYTES];
            let mut rng = AesState::new();

            crypto_kem_keypair(&mut pk, &mut sk, &mut rng)?;

            let pk_ref = [
                0x73, 0xD3, 0xFD, 0x5B, 0x9C, 0xA4, 0xB1, 0xD4, 0x14, 0x8F, 0x1A, 0xF0, 0x38, 0xFD,
                0xD7, 0xF2, 0x2F, 0xD3, 0x8D, 0x95, 0xAD, 0x88, 0x8E, 0xE1, 0xF8, 0x9A, 0xF7, 0x23,
                0xAD, 0x9F, 0x59, 0x16, 0x92, 0x5F, 0x7E, 0x21, 0xE7, 0x71, 0xD1, 0xBB, 0x39, 0x4D,
                0x5B, 0x10, 0x09, 0x33, 0xF5, 0x60, 0x16, 0xC1, 0x15, 0x30, 0x02, 0x5B, 0xB4, 0x2A,
                0xE9, 0x89, 0x19, 0x25, 0x6D, 0xF3, 0x9D, 0xAB, 0xC2, 0x4B, 0x93, 0x32, 0x3F, 0xBE,
                0x53, 0xD6, 0x73, 0xC4, 0xA8, 0xD8, 0x11, 0x2C, 0xF2, 0x55, 0xE3, 0xF4, 0xA5, 0xE1,
                0x26, 0xCD, 0xA8, 0xDD, 0x84, 0x07, 0x8C, 0x69, 0x51, 0x0A, 0x6F, 0x77, 0x99, 0x09,
                0xD4, 0x79, 0x7A, 0x90, 0xB1, 0x56, 0x17, 0x64, 0xC9, 0x2E, 0xEE, 0x0C, 0x5F, 0x66,
                0x04, 0x33, 0x72, 0x43, 0x42, 0xE8, 0x64, 0x82, 0xF1, 0x87, 0xA9, 0xC2, 0x97, 0xB4,
                0x54, 0x6B, 0x70, 0x08, 0x46, 0x58, 0x75, 0x00, 0x1D, 0x5A, 0x53, 0x82, 0xBE, 0xA8,
                0x39, 0x29, 0xF5, 0x51, 0x46, 0xD9, 0x23, 0x37, 0x74, 0xD5, 0xA5, 0x68, 0x02, 0x75,
                0xF5, 0x4B, 0x78, 0x86, 0x72, 0x5C, 0xDE, 0x11, 0x75, 0x7C, 0xF9, 0xD0, 0x19, 0xA9,
                0xE2, 0x8C, 0x4A, 0x22, 0xA3, 0xE9, 0xA1, 0xBB, 0x80, 0x23, 0x6C, 0xCC, 0x4B, 0x55,
                0xE2, 0x10, 0x03, 0xF7, 0x2C, 0x0A, 0xAF, 0x1B, 0x10, 0xB9, 0x90, 0xD1, 0xF3, 0xA6,
                0xE7, 0x82, 0x74, 0x66, 0x3E, 0xEB, 0xD4, 0xBE, 0x35, 0x07, 0x4C, 0xD3, 0x07, 0x0C,
                0x45, 0xAB, 0x4D, 0xF2, 0x1C, 0xBF, 0x6B, 0xEC, 0xEA, 0x75, 0x00, 0x8C, 0x19, 0xB7,
                0x36, 0x97, 0x00, 0x56, 0xD8, 0xFA, 0x15, 0x2E, 0x8E, 0x27, 0x7A, 0xF2, 0xF6, 0x07,
                0x75, 0xF5, 0xA6, 0x3D, 0x38, 0xCB, 0xD7, 0x50, 0x22, 0xE3, 0x1E, 0x6F, 0xDE, 0xA1,
                0xE8, 0x42, 0x8D, 0x6B, 0xF8, 0x7F, 0x99, 0xA1, 0x74, 0xA9, 0xC6, 0xA7, 0x8F, 0xD8,
                0xDE, 0x84, 0xA3, 0xA4, 0xDF, 0xE4, 0x15, 0xB6, 0x19, 0xBC, 0x23, 0x3E, 0xBC, 0x89,
                0xD8, 0x8C, 0x58, 0xDA, 0xE7, 0x8F, 0xE3, 0x44, 0xE7, 0x7D, 0x55, 0x10, 0x7F, 0xA2,
                0x77, 0x8B, 0x27, 0x35, 0xAC, 0x13, 0x6A, 0xCC, 0xEB, 0x68, 0xC3, 0x2E, 0x82, 0xF5,
                0xEA, 0x9B, 0xE3, 0x27, 0x3B, 0x6F, 0x70, 0x83, 0x55, 0xA3, 0x9B, 0x98, 0x32, 0x0D,
                0x1A, 0x83, 0x80, 0xB7, 0xF8, 0x96, 0x74, 0xC4, 0xE8, 0xAD, 0x98, 0x64, 0x8E, 0x86,
                0x87, 0x77, 0x99, 0x61, 0xE0, 0x13, 0xE4, 0x4C, 0x39, 0x83, 0x5E, 0x8A, 0xA8, 0x7A,
                0x54, 0x8B, 0xE2, 0x99, 0x3F, 0x25, 0x8A, 0x20, 0x0E, 0x4A, 0xCC, 0xCC, 0x51, 0xF0,
                0x7C, 0xD7, 0xEA, 0x4A, 0xAF, 0xD4, 0x26, 0x2C, 0xE0, 0xB5, 0x2C, 0xFC, 0x74, 0x21,
                0xFE, 0x91, 0xB2, 0x1C, 0x44, 0x00, 0x02, 0x07, 0x61, 0x0B, 0x7E, 0x58, 0x71, 0x20,
                0xBC, 0x79, 0x59, 0x98, 0xEC, 0xD2, 0x09, 0x1F, 0x40, 0x88, 0x36, 0x98, 0x40, 0xC4,
                0x69, 0x21, 0xE6, 0xE1, 0x1F, 0x86, 0x84, 0x64, 0xFA, 0xCF, 0xE3, 0x1F, 0xF8, 0x70,
                0xC7, 0xDE, 0x54, 0x34, 0x1E, 0xBE, 0x68, 0x27, 0x62, 0xC7, 0x63, 0x4D, 0x3F, 0x52,
                0xDA, 0xA5, 0x57, 0x24, 0x22, 0xEB, 0x09, 0x0E, 0x8B, 0xB1, 0xB7, 0xB1, 0x33, 0x40,
                0x1E, 0x66, 0xC8, 0xFB, 0x70, 0xDB, 0x39, 0xB4, 0xCF, 0x0E, 0x51, 0x91, 0x52, 0x71,
                0xE1, 0x03, 0x70, 0x9D, 0x42, 0x02, 0xC0, 0x8F, 0x94, 0x0D, 0x24, 0x8F, 0xE8, 0x6F,
                0x20, 0xA1, 0x3B, 0x60, 0xF5, 0xA8, 0x81, 0x36, 0x81, 0xB0, 0xC7, 0x52, 0x35, 0x7B,
                0x7B, 0x4C, 0x65, 0x30, 0xA1, 0xA4, 0x41, 0xD6, 0xDE, 0x5B, 0xBD, 0xD9, 0x09, 0xE1,
                0x6E, 0xBC, 0xB4, 0xAC, 0x72, 0xB8, 0xCF, 0x1D, 0x79, 0xD1, 0x3A, 0x0D, 0x00, 0xA1,
                0x81, 0xA8, 0x01, 0x8B, 0xB3, 0xB3, 0x84, 0x19, 0x10, 0xAB, 0xA3, 0xE2, 0xB7, 0x18,
                0x73, 0xCB, 0x54, 0xCC, 0x72, 0xA7, 0x1C, 0xFE, 0x16, 0xC5, 0xE2, 0xC9, 0xE8, 0x44,
                0xFA, 0x29, 0xBE, 0x4C, 0x37, 0x79, 0x0B, 0xC0, 0x92, 0x98, 0xA9, 0x91, 0x0F, 0xE3,
                0x94, 0xC4, 0xB5, 0x8C, 0x2B, 0xDE, 0xC2, 0x9C, 0xCD, 0x37, 0x83, 0x8C, 0x9F, 0x1D,
                0x44, 0xDA, 0x07, 0xD7, 0x72, 0x03, 0xB2, 0x6E, 0x4C, 0xA7, 0x81, 0x2F, 0x5E, 0xFC,
                0xE8, 0xA1, 0x81, 0x95, 0x65, 0xF7, 0x95, 0x38, 0x07, 0xAE, 0xD0, 0x97, 0xD6, 0x8E,
                0x1A, 0xD3, 0x32, 0xB2, 0x2B, 0x14, 0xC5, 0x95, 0xEA, 0xFB, 0x39, 0xAB, 0x86, 0xF2,
                0x0D, 0xB4, 0xD3, 0xFA, 0xF5, 0x31, 0xDE, 0xFD, 0x8F, 0xD8, 0x86, 0xD3, 0xC6, 0xCF,
                0xDD, 0x8C, 0x0F, 0x0A, 0xF4, 0x22, 0xAA, 0x4B, 0x47, 0x97, 0xC4, 0x34, 0x08, 0xA5,
                0xC9, 0x83, 0xF9, 0x87, 0x13, 0xBA, 0xD3, 0x52, 0x82, 0x43, 0x0C, 0x9F, 0xD2, 0x27,
                0x6E, 0xC3, 0x5B, 0xD0, 0xFB, 0x18, 0x01, 0x6E, 0x50, 0x50, 0xEF, 0xEB, 0x19, 0xAF,
                0x69, 0x05, 0x17, 0xA1, 0xCE, 0x78, 0x6B, 0x07, 0x87, 0xC3, 0xBF, 0xF1, 0x5F, 0x68,
                0xF5, 0x13, 0xDD, 0x9A, 0x86, 0xF1, 0xF6, 0x4A, 0x62, 0x63, 0x4A, 0x36, 0x0C,
            ];
            assert_eq!(pk_ref, pk);
            let sk_ref = [
                0x89, 0x2D, 0x93, 0x85, 0x9B, 0x0F, 0x01, 0xAF, 0x2F, 0xE7, 0x4B, 0x45, 0xBF, 0x59,
                0xC7, 0x6E, 0x31, 0xAB, 0x85, 0xCD, 0xB1, 0x56, 0xC5, 0x46, 0x8F, 0x65, 0xC6, 0xA2,
                0x57, 0x11, 0x5B, 0xC4, 0x59, 0x72, 0x3B, 0x1F, 0xBD, 0xEE, 0x8C, 0x61, 0xEE, 0x6D,
                0x51, 0x7B, 0xB7, 0xA5, 0x26, 0x81, 0x5C, 0x8B, 0x74, 0xC8, 0x31, 0x35, 0x66, 0xCC,
                0x51, 0xC2, 0x5D, 0xD5, 0x87, 0x69, 0x10, 0x62, 0x4F, 0xCF, 0xE0, 0x37, 0xDB, 0x7B,
                0x10, 0xC9, 0x94, 0x98, 0x2B, 0x95, 0xD0, 0xAA, 0x20, 0x89, 0x3E, 0x4F, 0x67, 0x39,
                0x74, 0xC5, 0x7E, 0xA9, 0x33, 0xD8, 0x8C, 0x8F, 0x3D, 0xBD, 0x9E, 0x98, 0x70, 0xD2,
                0x09, 0x1B, 0x32, 0x08, 0xE9, 0x51, 0x27, 0x15, 0x1E, 0x0D, 0x40, 0x29, 0x33, 0xC2,
                0xD8, 0xE5, 0xA4, 0x0E, 0x6C, 0x4F, 0x33, 0x0D, 0x60, 0xA8, 0x5F, 0x8C, 0x15, 0xCA,
                0x37, 0xD8, 0xC9, 0x69, 0x5D, 0xD2, 0x14, 0x9A, 0x49, 0xE9, 0xEF, 0x1B, 0x29, 0x65,
                0x2C, 0xEB, 0x46, 0xF2, 0x00, 0x92, 0xE3, 0x87, 0x10, 0x47, 0x82, 0xDA, 0x94, 0xB7,
                0xBA, 0x59, 0xD2, 0x96, 0x4A, 0xE8, 0x6F, 0x3A, 0x23, 0x24, 0x57, 0x0F, 0xE8, 0x03,
                0x49, 0x4D, 0x7A, 0xD0, 0x41, 0x8C, 0x6B, 0x31, 0xDB, 0x6A, 0x04, 0x4E, 0x59, 0x8E,
                0xB4, 0x54, 0x4A, 0xE6, 0x03, 0x49, 0x52, 0xD5, 0x48, 0xAC, 0x2D, 0x45, 0x71, 0x03,
                0x0E, 0xDC, 0x0D, 0x28, 0xB9, 0xB5, 0x71, 0x12, 0x97, 0xED, 0xCD, 0x21, 0x23, 0x42,
                0x89, 0xA2, 0x0E, 0x49, 0x85, 0x56, 0x2F, 0x9C, 0xF4, 0x04, 0x6E, 0x66, 0x12, 0xD1,
                0x0D, 0xA8, 0x96, 0xBC, 0x5C, 0x86, 0x87, 0xF6, 0xEB, 0xEE, 0xD4, 0x08, 0x93, 0x57,
                0x4D, 0xC0, 0x46, 0x3A, 0xA2, 0x3B, 0xCE, 0xC7, 0x2C, 0x8C, 0xDE, 0xF4, 0xBF, 0x2C,
                0x50, 0xC6, 0xA3, 0x4F, 0x05, 0xA1, 0x71, 0x52, 0x51, 0xF8, 0x74, 0x2F, 0x52, 0xD0,
                0x7D, 0x44, 0xA4, 0x55, 0x35, 0x73, 0xD9, 0x6E, 0x79, 0x1D, 0x9F, 0x79, 0xB7, 0x61,
                0x72, 0x26, 0x7C, 0x8C, 0xDF, 0x27, 0x5C, 0xDC, 0x01, 0x03, 0x59, 0xA4, 0x39, 0x7E,
                0xEB, 0x57, 0x37, 0xA5, 0xD0, 0x6F, 0xE1, 0xC0, 0x7C, 0x67, 0xFD, 0xD4, 0xCF, 0x05,
                0xF6, 0x5F, 0x21, 0x96, 0x18, 0xE9, 0xAA, 0x8D, 0x7E, 0xD1, 0x46, 0xA2, 0x9B, 0xCA,
                0xBB, 0x8D, 0x23, 0xE0, 0x26, 0x6B, 0x10, 0x09, 0x1F, 0x0C, 0xF0, 0xC5, 0x81, 0x89,
                0x4A, 0x02, 0x41, 0x3B, 0xF3, 0x5D, 0xD3, 0x69, 0x38, 0xC6, 0xF5, 0xB4, 0x58, 0x78,
                0x17, 0x25, 0x18, 0x33, 0x27, 0x71, 0x71, 0x06, 0x8A, 0x75, 0x5B, 0x3C, 0x77, 0xFC,
                0x8B, 0x8F, 0x1B, 0x8C, 0xF7, 0xE0, 0xE2, 0x93, 0xBD, 0x79, 0x75, 0x8F, 0x5B, 0x8F,
                0xFD, 0x4A, 0xC1, 0x42, 0xA6, 0x0C, 0x6A, 0xA4, 0x5F, 0xEC, 0x1D, 0x71, 0x85, 0x01,
                0x8F, 0x9F, 0x99, 0x26, 0x97, 0x85, 0x7C, 0x9C, 0x99, 0x09, 0x11, 0xA3, 0x04, 0x97,
                0x16, 0xB4, 0xE4, 0xEC, 0x96, 0x59, 0x7F, 0x71, 0x61, 0x7A, 0xDD, 0x60, 0xE4, 0xCC,
                0x96, 0xE8, 0xD9, 0x0A, 0x7C, 0x7D, 0x6C, 0xA1, 0x09, 0xB0, 0x4F, 0x4B, 0x68, 0x12,
                0xBD, 0xA2, 0x33, 0x01, 0xCB, 0xA6, 0xE0, 0xD0, 0xCF, 0xF3, 0x98, 0x41, 0x47, 0x35,
                0x60, 0xE8, 0xFE, 0x80, 0xE5, 0xB1, 0xF0, 0x45, 0x98, 0x12, 0x36, 0x98, 0xDD, 0x34,
                0xB9, 0x56, 0xC6, 0x69, 0x5D, 0x66, 0x5D, 0x61, 0x21, 0x0D, 0xCC, 0x57, 0xC7, 0xEB,
                0x3B, 0xEC, 0x31, 0x9E, 0x9D, 0xD7, 0x04, 0x99, 0x2A, 0x3C, 0x8A, 0xE2, 0x41, 0x78,
                0x5E, 0x02, 0xC5, 0xED, 0xE4, 0x54, 0x21, 0x7D, 0xA1, 0xE7, 0x7F, 0xD5, 0x0E, 0xDB,
                0x56, 0xFD, 0x28, 0x38, 0x0D, 0xA7, 0x0E, 0x05, 0xBE, 0x1E, 0x20, 0xBE, 0x09, 0x88,
                0xCB, 0x62, 0xA9, 0xC0, 0xAE, 0x78, 0xD0, 0xE2, 0xF0, 0x1A, 0x80, 0x8B, 0xF6, 0x59,
                0x03, 0xA4, 0x1C, 0x7C, 0x78, 0x04, 0x40, 0x21, 0x00, 0xE6, 0xB4, 0xC5, 0x8B, 0x5F,
                0x92, 0x93, 0x96, 0xA0, 0xC5, 0x87, 0x84, 0x7A, 0xF6, 0x68, 0xCC, 0xB3, 0x66, 0xC8,
                0x30, 0xC9, 0xD4, 0x40, 0xDF, 0x1E, 0xCB, 0xE9, 0x98, 0x96, 0x71, 0xC2, 0x60, 0xCB,
                0xCE, 0xEA, 0x94, 0x64, 0x00, 0x78, 0x13, 0x2F, 0xEB, 0x40, 0xAC, 0xE4, 0xA4, 0xF4,
                0x0A, 0x05, 0x0A, 0x11, 0x8C, 0x6D, 0x8F, 0x5E, 0xCF, 0x94, 0x4A, 0x56, 0x8C, 0x6D,
                0x9A, 0x74, 0x1A, 0x77, 0xC1, 0xAB, 0x0E, 0x42, 0x69, 0xC2, 0x8B, 0x52, 0xAF, 0xD1,
                0x10, 0x4B, 0x29, 0xF2, 0x0D, 0x6E, 0x65, 0x1F, 0x6B, 0xB3, 0xBD, 0x41, 0xFA, 0xE2,
                0xF6, 0x0C, 0x81, 0xD5, 0xF1, 0x73, 0xAE, 0xD8, 0xB9, 0x1C, 0xD7, 0x2E, 0x84, 0xC9,
                0x7B, 0x7A, 0x62, 0x01, 0xD9, 0xFA, 0x65, 0xCB, 0xD7, 0x1D, 0x56, 0x25, 0xEE, 0x22,
                0x6E, 0x03, 0x09, 0x77, 0xEB, 0x0B, 0x27, 0x36, 0x78, 0x48, 0xAE, 0x2B, 0x86, 0xA3,
                0x63, 0xCF, 0x3C, 0x78, 0xF4, 0x6E, 0x42, 0x91, 0x07, 0xDE, 0x9B, 0xF7, 0x79, 0x57,
                0xE0, 0x10, 0x07, 0x02, 0xA1, 0x66, 0xD1, 0x76, 0x97, 0x21, 0xD9, 0x0F, 0x7C, 0x8C,
                0x17, 0xEE, 0x4E, 0x7C, 0x3E, 0x95, 0x50, 0xEA, 0x3A, 0x35, 0x17, 0xA3, 0x43, 0x04,
                0x74, 0x4B, 0xFD, 0x94, 0xB4, 0x85, 0x98, 0xF6, 0xCA, 0x22, 0x2A, 0x9C, 0x62, 0xAF,
                0x53, 0xD2, 0xA4, 0x07, 0x32, 0x84, 0xCC, 0x01, 0x7A, 0xD4, 0xCC, 0x63, 0xFF, 0xE1,
                0x23, 0x9F, 0xEC, 0x0F, 0x56, 0x98, 0x86, 0x6C, 0x2B, 0xC1, 0x85, 0x3F, 0xE5, 0x7E,
                0x83, 0x84, 0x80, 0xAC, 0x5A, 0x03, 0x7C, 0x79, 0x98, 0x3E, 0x4A, 0x16, 0x71, 0x29,
                0xC5, 0xB2, 0x63, 0x69, 0x3D, 0x10, 0xDC, 0xF7, 0x7B, 0x94, 0x5D, 0x61, 0x06, 0xE0,
                0x25, 0x08, 0x7E, 0x1E, 0xE8, 0x7B, 0xED, 0x85, 0xE5, 0x0F, 0x2A, 0xC9, 0x93, 0xDF,
                0xBA, 0x36, 0xE7, 0xED, 0x32, 0xF9, 0x6B, 0xC3, 0xD3, 0x9B, 0xC2, 0x9E, 0x7E, 0x76,
                0x90, 0xE4, 0xBF, 0x5C, 0x19, 0x68, 0x9A, 0x32, 0x84, 0x96, 0x4C, 0x36, 0x02, 0xB4,
                0x16, 0x1D, 0xC5, 0x9F, 0x9E, 0xBD, 0x18, 0xB9, 0x6D, 0x9C, 0xCA, 0x88, 0x03, 0x9E,
                0x1F, 0xAA, 0x47, 0x26, 0x63, 0xB2, 0xB4, 0x4C, 0xB4, 0xB9, 0xA2, 0x44, 0xE4, 0x02,
                0x75, 0x8A, 0xB3, 0xB9, 0x1C, 0xEA, 0x02, 0xA3, 0xE4, 0xBF, 0x0C, 0x08, 0x4E, 0xD5,
                0x64, 0xB8, 0xB8, 0x44, 0x48, 0x5A, 0xCD, 0x7A, 0x0C, 0x62, 0x0A, 0x96, 0x05, 0x3E,
                0xD7, 0x40, 0x57, 0xF2, 0xF7, 0x63, 0xB1, 0x9A, 0x21, 0xE0, 0x9F, 0xD0, 0xA4, 0x12,
                0x78, 0xE1, 0x79, 0xCE, 0xAD, 0xBF, 0x03, 0xF0, 0xE7, 0x27, 0x63, 0x3A, 0xD1, 0x99,
                0xAC, 0x5B, 0x47, 0xFE, 0x8A, 0x70, 0xBD, 0x4D, 0xB4, 0x0E, 0x3F, 0x4C, 0xE5, 0xEF,
                0x10, 0x8D, 0xF8, 0x55, 0x58, 0xDC, 0xCA, 0x3B, 0xE8, 0x59, 0x84,
            ];
            assert_eq!(sk_ref, sk);
        }

        #[cfg(feature = "ntruhps2048677")]
        {
            let mut pk = [0u8; CRYPTO_PUBLICKEYBYTES];
            let mut sk = [0u8; CRYPTO_SECRETKEYBYTES];
            let mut rng = AesState::new();

            crypto_kem_keypair(&mut pk, &mut sk, &mut rng)?;

            let pk_ref = [
                0x2A, 0xAC, 0xFE, 0x8F, 0x2E, 0xCB, 0xB0, 0xB8, 0x73, 0x5B, 0x2C, 0x48, 0x97, 0xDD,
                0xE4, 0xB7, 0x9C, 0xD7, 0x82, 0x30, 0x3F, 0x66, 0xB1, 0xCB, 0xFF, 0xF6, 0xB6, 0x97,
                0x56, 0x04, 0x0C, 0x42, 0xB2, 0xB3, 0xA5, 0xAD, 0x82, 0xA6, 0x82, 0x0E, 0x2D, 0x91,
                0xBE, 0xFB, 0x5B, 0xD4, 0xA2, 0x8C, 0x0B, 0x66, 0x0A, 0x28, 0xA7, 0x1F, 0x66, 0xBD,
                0x99, 0x4A, 0x7A, 0x6E, 0xC2, 0x1D, 0xEE, 0xB1, 0x26, 0xA4, 0x0A, 0xBF, 0xE8, 0x30,
                0x1E, 0xA6, 0x3B, 0x90, 0x1F, 0xEB, 0x26, 0x99, 0xC5, 0xF6, 0x68, 0xF8, 0xCA, 0x97,
                0x4A, 0x71, 0x2B, 0x00, 0xCA, 0x0A, 0x13, 0xEA, 0x59, 0x92, 0xDC, 0xC0, 0xD1, 0xD5,
                0x0C, 0x78, 0x21, 0xE0, 0xEB, 0x92, 0xAE, 0x78, 0xB6, 0x30, 0x2B, 0x8F, 0x32, 0x37,
                0x83, 0xF0, 0x8D, 0xA3, 0xA3, 0x5C, 0xBD, 0x10, 0xBD, 0x5D, 0xEC, 0x50, 0xB5, 0x16,
                0x93, 0x8E, 0xAB, 0x9E, 0x6B, 0x55, 0x9D, 0x35, 0x6C, 0x3E, 0xA1, 0xBD, 0x6A, 0x61,
                0x18, 0x4B, 0x0D, 0xAB, 0x1F, 0xBD, 0x57, 0x3C, 0x65, 0x33, 0x3C, 0x50, 0x2E, 0xF0,
                0x09, 0xB0, 0x21, 0x66, 0xB0, 0x37, 0x88, 0x96, 0xB6, 0x32, 0x54, 0xBC, 0x18, 0x70,
                0x62, 0x52, 0xCF, 0x89, 0xC7, 0xC4, 0x7A, 0x7B, 0xB9, 0x0D, 0x5C, 0x01, 0x1E, 0x6C,
                0xFD, 0xE0, 0xD9, 0x03, 0xDE, 0xF5, 0x0A, 0xBA, 0x5C, 0x1F, 0xF7, 0x2F, 0x82, 0x38,
                0xDB, 0x4B, 0xFA, 0x42, 0x05, 0x88, 0x7E, 0xFC, 0xF9, 0x3C, 0xE4, 0xC7, 0xC6, 0x19,
                0xEB, 0x03, 0xE3, 0xA6, 0xC3, 0x06, 0x41, 0xDA, 0x85, 0x3C, 0x42, 0xD2, 0xDD, 0x63,
                0x34, 0x04, 0xC2, 0xE3, 0xB2, 0xAE, 0xCC, 0x3C, 0x25, 0xF1, 0x95, 0x7E, 0x70, 0x38,
                0x81, 0x3A, 0x9C, 0x4F, 0x00, 0x23, 0xCD, 0xCF, 0xD0, 0xDA, 0xF9, 0x78, 0xF6, 0x57,
                0x46, 0x2F, 0xDC, 0x8D, 0x8C, 0xDB, 0x6C, 0x09, 0x79, 0xBC, 0xA3, 0xFC, 0x7F, 0x41,
                0x39, 0xB3, 0x58, 0x4F, 0xF0, 0x39, 0x01, 0x67, 0x7E, 0x4B, 0x47, 0x49, 0x9B, 0xD5,
                0x8B, 0x0B, 0xBB, 0xBB, 0xEC, 0xB2, 0x95, 0xD8, 0xDF, 0x0D, 0xE0, 0x47, 0x14, 0x65,
                0xE0, 0x7F, 0xEA, 0x8A, 0x15, 0x61, 0x5A, 0xB8, 0xD4, 0x90, 0xD4, 0x4C, 0x7F, 0xF3,
                0x88, 0x47, 0xAF, 0x67, 0x51, 0x1D, 0x66, 0x90, 0xEC, 0x02, 0x12, 0xB2, 0xB4, 0x8C,
                0x04, 0xB3, 0x38, 0xF5, 0x78, 0x01, 0x5B, 0x1C, 0x5C, 0x4F, 0xF3, 0x7B, 0xC1, 0xFA,
                0x2C, 0x15, 0x6D, 0xED, 0x2D, 0x3B, 0xE6, 0x7E, 0xCF, 0xE7, 0xF8, 0x60, 0x13, 0x86,
                0x4D, 0x7F, 0xAE, 0x2C, 0x22, 0xBF, 0x42, 0xFC, 0x02, 0x6A, 0x6F, 0xCA, 0x15, 0x00,
                0x1A, 0x0B, 0x06, 0x0E, 0x85, 0xF2, 0x11, 0x51, 0x80, 0x75, 0x5F, 0x7A, 0x42, 0x4A,
                0x09, 0x2A, 0x95, 0x2F, 0xA0, 0x9F, 0x66, 0xBD, 0xFE, 0xE0, 0x2C, 0x8F, 0xA0, 0xCA,
                0x44, 0xFE, 0xFA, 0x6A, 0x1D, 0x03, 0xC0, 0x60, 0x75, 0x47, 0x9D, 0x54, 0xD9, 0xAE,
                0xF7, 0x9B, 0xFC, 0x10, 0xEB, 0xD5, 0xD0, 0x94, 0x7A, 0x54, 0x11, 0xE9, 0x62, 0xAA,
                0xE6, 0xB0, 0xAF, 0x62, 0x4C, 0x43, 0x36, 0x95, 0x52, 0xDA, 0x24, 0x54, 0xBA, 0x1E,
                0x37, 0x51, 0xCF, 0x88, 0xE6, 0x0E, 0xBC, 0x62, 0x38, 0x49, 0x79, 0xDD, 0xEE, 0x15,
                0x17, 0x2C, 0xB7, 0x5E, 0x27, 0xA0, 0x51, 0x08, 0xEC, 0x5D, 0xEB, 0xFE, 0xCF, 0x26,
                0xE9, 0x21, 0xDD, 0x06, 0xB0, 0x5A, 0xBD, 0x5B, 0x62, 0xFF, 0xB2, 0x6A, 0x22, 0xC7,
                0x49, 0xE5, 0x39, 0x65, 0xD5, 0xC9, 0x9F, 0x0D, 0x05, 0x72, 0x24, 0x8C, 0x27, 0x1B,
                0x40, 0xDC, 0x04, 0xF4, 0x85, 0xEA, 0x6F, 0x05, 0x4C, 0x85, 0x27, 0x5E, 0x9C, 0xFF,
                0xC0, 0xA5, 0x8F, 0x61, 0x3B, 0xA0, 0xFF, 0x18, 0x9B, 0x62, 0xD3, 0xED, 0x74, 0x9D,
                0xBD, 0xAA, 0x9D, 0xAD, 0xC4, 0x9A, 0xB4, 0x3B, 0xC5, 0xE4, 0x7C, 0x37, 0xB9, 0x9D,
                0x63, 0x4A, 0x88, 0xA7, 0x5D, 0x81, 0x73, 0x4E, 0x4A, 0x5D, 0x62, 0x71, 0x96, 0x63,
                0x75, 0xD5, 0x6A, 0x28, 0x3D, 0x79, 0x03, 0x81, 0x31, 0x00, 0xE0, 0xCD, 0x60, 0x82,
                0xC8, 0xEC, 0x0E, 0x29, 0x92, 0x67, 0xD6, 0x61, 0x3B, 0x10, 0xF4, 0x66, 0x56, 0xAE,
                0x86, 0xED, 0xE5, 0x1C, 0x5D, 0x88, 0x35, 0xE4, 0x6A, 0x7D, 0xBE, 0xEE, 0xED, 0xD0,
                0xC4, 0xB5, 0x02, 0xDF, 0x47, 0x6A, 0x8F, 0x5B, 0x03, 0x01, 0x79, 0x72, 0xA8, 0x1E,
                0x9E, 0x6F, 0x87, 0xB6, 0xA2, 0xDF, 0x58, 0x85, 0x9C, 0x11, 0xA8, 0x92, 0x5E, 0x87,
                0xC7, 0x51, 0x5D, 0xE3, 0x73, 0xA1, 0xF7, 0xA9, 0x82, 0x7D, 0xD6, 0xFB, 0xAE, 0xF0,
                0x0B, 0xBE, 0x68, 0xAF, 0xF8, 0x51, 0x51, 0xF6, 0xC8, 0x4E, 0xE0, 0x31, 0x77, 0x4B,
                0x98, 0xD7, 0x15, 0x1F, 0x3A, 0x94, 0x6C, 0xD0, 0x59, 0x34, 0x0E, 0x6E, 0x63, 0xC3,
                0x0F, 0x67, 0xBC, 0xCA, 0x84, 0x11, 0xEB, 0x8D, 0x2F, 0xDB, 0x68, 0xE0, 0xAF, 0x96,
                0x3D, 0xEA, 0x8C, 0xAF, 0xB6, 0x37, 0x69, 0x0C, 0xCE, 0xD3, 0xAA, 0x24, 0xFE, 0xAD,
                0x64, 0xF3, 0x4D, 0x99, 0xB0, 0x01, 0x50, 0xB9, 0xA2, 0xFD, 0x58, 0xEB, 0x8E, 0x6D,
                0xA2, 0x8B, 0x15, 0x37, 0xB4, 0xC0, 0x01, 0x44, 0x16, 0xB7, 0x34, 0x3B, 0x15, 0xDA,
                0x36, 0x81, 0xC2, 0xB2, 0x6A, 0x86, 0x72, 0x34, 0x06, 0x5A, 0xF4, 0x6C, 0x93, 0x6F,
                0x39, 0xA7, 0xEE, 0xC8, 0xF9, 0x84, 0x92, 0x5A, 0x59, 0xAF, 0xE1, 0xC7, 0x88, 0x6A,
                0x78, 0x5E, 0xB2, 0x50, 0x1F, 0x9A, 0xCC, 0xEF, 0x63, 0x99, 0xFD, 0x5B, 0xFE, 0xA6,
                0x1F, 0x37, 0x68, 0x69, 0xF4, 0xD1, 0x84, 0x7B, 0x8E, 0x4F, 0xAB, 0xFA, 0x86, 0x84,
                0xD2, 0x8D, 0x71, 0xAF, 0x57, 0x65, 0x25, 0xEC, 0xC9, 0x78, 0x33, 0x7D, 0xE6, 0xFE,
                0x88, 0x58, 0x07, 0xE5, 0x42, 0x46, 0xAE, 0x95, 0x89, 0x1F, 0x6F, 0x56, 0x5D, 0xA4,
                0x33, 0x50, 0x42, 0x1C, 0xBC, 0x08, 0xA7, 0x8B, 0xC4, 0x9B, 0x13, 0x9D, 0x5A, 0x6C,
                0x90, 0x5F, 0xBC, 0x7D, 0x83, 0x6E, 0x0E, 0xE6, 0xD6, 0x9B, 0x76, 0xE6, 0xC7, 0x8D,
                0xC3, 0xAB, 0xF8, 0xB8, 0xA0, 0xF6, 0x9A, 0x3E, 0x44, 0x75, 0xB0, 0xE7, 0x35, 0x7E,
                0x14, 0x6D, 0x7B, 0xEE, 0xAC, 0x0F, 0xE5, 0x63, 0x13, 0xB1, 0xA0, 0x1A, 0xB8, 0x56,
                0x81, 0xF7, 0x2C, 0x63, 0x56, 0x94, 0xAD, 0x0E, 0x97, 0xA2, 0xAF, 0x2F, 0xDE, 0x50,
                0x84, 0xD7, 0x7E, 0xEB, 0x60, 0x40, 0x71, 0x1E, 0x7B, 0xC8, 0x91, 0xC0, 0x87, 0xD1,
                0x3F, 0x70, 0x36, 0xCB, 0xB3, 0x30, 0x01, 0x8D, 0x0B, 0xB5, 0x20, 0x50, 0x76, 0x8B,
                0xA3, 0x99, 0x7B, 0x83, 0x31, 0x60, 0x23, 0xDF, 0x7C, 0x37, 0xD4, 0x9E, 0x7E, 0x7E,
                0xF2, 0x72, 0xA0, 0x1F, 0xB6, 0x85, 0x98, 0xE8, 0x26, 0x56, 0x0E, 0xEC, 0x66, 0xFA,
                0xF1, 0x3A, 0x14, 0x58, 0x05, 0x02,
            ];
            assert_eq!(pk_ref, pk);
            let sk_ref = [
                0x89, 0x2D, 0x93, 0x85, 0x9B, 0x0F, 0x01, 0xAF, 0x2F, 0xE7, 0x4B, 0x45, 0xBF, 0x59,
                0xC7, 0x6E, 0x31, 0xAB, 0x85, 0xCD, 0xB1, 0x56, 0xC5, 0x46, 0x8F, 0x65, 0xC6, 0xA2,
                0x57, 0x11, 0x5B, 0xC4, 0x59, 0x72, 0x3B, 0x1F, 0xBD, 0xEE, 0x8C, 0x61, 0xEE, 0x6D,
                0x51, 0x7B, 0xB7, 0xA5, 0x26, 0x81, 0x5C, 0x8B, 0x74, 0xC8, 0x31, 0x35, 0x66, 0xCC,
                0x51, 0xC2, 0x5D, 0xD5, 0x87, 0x69, 0x10, 0x62, 0x4F, 0xCF, 0xE0, 0x37, 0xDB, 0x7B,
                0x10, 0xC9, 0x94, 0x98, 0x2B, 0x95, 0xD0, 0xAA, 0x20, 0x89, 0x3E, 0x4F, 0x67, 0x39,
                0x74, 0xC5, 0x7E, 0xA9, 0x33, 0xD8, 0x8C, 0x8F, 0x3D, 0xBD, 0x9E, 0x98, 0x70, 0xD2,
                0x09, 0x1B, 0x32, 0x3E, 0xD1, 0x10, 0x3E, 0xDA, 0x4E, 0x3F, 0xA1, 0x96, 0xEE, 0x29,
                0x29, 0x06, 0xDC, 0xC3, 0x8D, 0x59, 0x42, 0x62, 0x1B, 0x74, 0xBF, 0xB3, 0x99, 0xB6,
                0x7C, 0x74, 0x74, 0x60, 0x83, 0x3F, 0xEF, 0x0F, 0xE5, 0x02, 0xD5, 0x91, 0x53, 0x8A,
                0xA4, 0x7C, 0x9E, 0xC7, 0x3B, 0x29, 0x94, 0x5E, 0xDA, 0x98, 0x54, 0xE0, 0xD3, 0xB8,
                0x9E, 0x96, 0x8B, 0x0E, 0x66, 0xB0, 0x00, 0x5C, 0xD2, 0xCA, 0x42, 0x19, 0x6F, 0x13,
                0x20, 0xE2, 0x55, 0x3B, 0x6A, 0xDD, 0x31, 0x43, 0xD5, 0x9C, 0x35, 0xC0, 0xEE, 0xDB,
                0x1D, 0x74, 0x4B, 0x4C, 0x5B, 0x58, 0xD0, 0xC3, 0xCE, 0x66, 0x7C, 0x30, 0x3E, 0xEE,
                0x3E, 0x42, 0x78, 0x2C, 0x97, 0x67, 0x7C, 0xD8, 0x6E, 0x09, 0x31, 0x00, 0x8D, 0xCD,
                0x2B, 0xB6, 0x0A, 0xB2, 0x61, 0xDD, 0x96, 0x89, 0x8F, 0x27, 0x3B, 0x5A, 0x91, 0x0C,
                0xE8, 0x02, 0x23, 0xBE, 0xBF, 0x23, 0xA5, 0xA3, 0xD6, 0x29, 0xE4, 0x03, 0xDE, 0x03,
                0xD7, 0x02, 0xD5, 0x4A, 0xD4, 0xB5, 0xC5, 0x2D, 0x5E, 0x97, 0xCF, 0x3C, 0xC1, 0x9F,
                0x1F, 0x3B, 0xCC, 0x66, 0x75, 0xBF, 0x63, 0x68, 0xA7, 0xDD, 0x5F, 0x2D, 0xD7, 0x53,
                0xEB, 0x84, 0xAF, 0x34, 0x12, 0x02, 0xFB, 0xCB, 0x49, 0x45, 0x47, 0x5B, 0x7B, 0xEA,
                0x46, 0xAD, 0xCF, 0x4F, 0x16, 0x2F, 0x6D, 0x8E, 0xF5, 0xF5, 0x46, 0x4E, 0xE9, 0x54,
                0x2D, 0x2A, 0x8A, 0xC1, 0x8B, 0x86, 0xCE, 0xFF, 0x35, 0x49, 0x55, 0x84, 0x6A, 0xD3,
                0xBA, 0xCE, 0xB6, 0xD7, 0x1E, 0xBF, 0x90, 0x29, 0xD5, 0x25, 0x67, 0xC9, 0x2E, 0x04,
                0xA9, 0x81, 0x19, 0x46, 0x52, 0x2E, 0xD3, 0x21, 0x3B, 0xC6, 0x3B, 0x6C, 0x12, 0xB0,
                0x55, 0x4F, 0x80, 0x65, 0xC9, 0xA8, 0xD9, 0x65, 0xF4, 0x29, 0x40, 0x42, 0xBD, 0xB8,
                0xC2, 0x8C, 0xE1, 0xCC, 0xDC, 0x95, 0xFB, 0xF6, 0x6C, 0x7D, 0x72, 0x0C, 0x48, 0x83,
                0xE4, 0x24, 0x9B, 0xFB, 0xDF, 0xD9, 0x01, 0x9E, 0x9B, 0x98, 0x78, 0xAE, 0xF3, 0x7C,
                0x6D, 0x49, 0xD7, 0x19, 0x29, 0xC4, 0xEF, 0x1B, 0x91, 0x8B, 0x6A, 0x0D, 0xF6, 0x7D,
                0x74, 0xB0, 0x3D, 0x43, 0x33, 0xEC, 0x96, 0x2F, 0xA8, 0x6E, 0xFB, 0xD5, 0x74, 0xE5,
                0x52, 0xB9, 0xD6, 0x5D, 0x6D, 0x44, 0xE3, 0xAF, 0xB1, 0xFC, 0x30, 0x79, 0xAE, 0x40,
                0x1D, 0x93, 0x13, 0xEE, 0xC6, 0xD2, 0x7D, 0x48, 0x87, 0x2F, 0xE4, 0xEA, 0x82, 0x7C,
                0xA9, 0xC5, 0x03, 0x1B, 0xDF, 0xC5, 0x50, 0xEB, 0x28, 0x4B, 0x7B, 0x0E, 0x76, 0xBF,
                0x36, 0x79, 0x98, 0xE6, 0x95, 0x20, 0x56, 0xC2, 0xFC, 0x7D, 0xB7, 0x83, 0x41, 0x44,
                0x8F, 0xF8, 0xE2, 0x0B, 0x96, 0x8C, 0xEE, 0x81, 0xE1, 0x95, 0xC5, 0xD6, 0x97, 0x90,
                0xF6, 0x2B, 0x7D, 0x12, 0x8A, 0x81, 0x22, 0x9E, 0x42, 0x51, 0x59, 0x89, 0x1C, 0x2B,
                0x24, 0x18, 0x96, 0x07, 0x1C, 0xD8, 0x4E, 0xB9, 0x12, 0x88, 0x8E, 0xCA, 0xAF, 0xA8,
                0x3B, 0xF2, 0x59, 0x0C, 0xFA, 0x77, 0xB8, 0x87, 0x73, 0x3A, 0x2A, 0x7C, 0xCD, 0xC0,
                0xBE, 0x2B, 0x24, 0x9A, 0x2D, 0x44, 0x32, 0x4D, 0xA9, 0x04, 0x55, 0xB0, 0xC5, 0xD7,
                0x5B, 0x6F, 0xB9, 0x2E, 0x79, 0x2C, 0xD2, 0x45, 0xA1, 0xF8, 0xCA, 0xD3, 0x6A, 0xAF,
                0xE0, 0x73, 0xFA, 0xD7, 0x77, 0x3A, 0xA8, 0x90, 0xAA, 0x9A, 0x70, 0xF5, 0x57, 0x21,
                0x49, 0x13, 0xC9, 0xEA, 0xEA, 0x26, 0x10, 0xD7, 0x9B, 0x1D, 0x38, 0xE9, 0x3A, 0x05,
                0x58, 0xB1, 0x2E, 0x3A, 0xC8, 0x05, 0x7B, 0x57, 0x37, 0xC3, 0x0F, 0x78, 0x7D, 0x58,
                0x0D, 0xF0, 0xDB, 0x05, 0x59, 0x3E, 0xEA, 0x88, 0x7C, 0x70, 0x50, 0xED, 0x44, 0xA7,
                0x97, 0xDD, 0x14, 0x9E, 0x3C, 0x78, 0xBC, 0xAA, 0x70, 0x20, 0xCD, 0x44, 0xA4, 0x9A,
                0x4F, 0xBD, 0x0F, 0x95, 0x88, 0xC0, 0xFA, 0xBB, 0x2C, 0xB8, 0xF6, 0xBD, 0x16, 0x57,
                0x7D, 0xD4, 0x1C, 0x6C, 0x90, 0xD5, 0x77, 0x7B, 0x3F, 0x33, 0x3A, 0xB0, 0x36, 0x1A,
                0x50, 0xF1, 0x97, 0xD0, 0x79, 0xCB, 0x9F, 0xF5, 0xC4, 0x53, 0x51, 0xFB, 0xFF, 0x64,
                0xBB, 0x84, 0x1A, 0x0B, 0xFD, 0x6E, 0xE5, 0x25, 0x78, 0xBF, 0xD4, 0x6E, 0xFA, 0x2C,
                0xA3, 0x49, 0xF7, 0x21, 0xF1, 0x1D, 0xB1, 0x52, 0xBE, 0x1B, 0x5B, 0x04, 0x09, 0x95,
                0x54, 0x28, 0x85, 0x12, 0xD0, 0xBE, 0xC8, 0x9E, 0xF3, 0x7B, 0x5E, 0xBB, 0x37, 0x16,
                0x15, 0xC8, 0xBC, 0x06, 0xE1, 0xD5, 0xBC, 0x8A, 0xDB, 0x26, 0xBC, 0xBB, 0x46, 0x7F,
                0xBB, 0x69, 0xFE, 0x85, 0x6E, 0x52, 0xD8, 0x68, 0x3D, 0x88, 0xE1, 0x5C, 0x74, 0x89,
                0x9C, 0xC2, 0x4E, 0xD8, 0x84, 0x1D, 0xAD, 0x2F, 0xAE, 0x68, 0x0E, 0x16, 0x0E, 0x42,
                0x7E, 0x33, 0xB7, 0xD6, 0x8A, 0x99, 0xFE, 0xC8, 0x76, 0xC0, 0x65, 0xDD, 0xDD, 0x20,
                0x30, 0x00, 0xF8, 0x5C, 0x10, 0x2C, 0x09, 0x17, 0xB4, 0x8D, 0x42, 0xEC, 0xAC, 0x13,
                0x28, 0x63, 0x66, 0x83, 0xA7, 0xC1, 0x7A, 0xED, 0x2B, 0x0A, 0x3A, 0x8A, 0x72, 0x7C,
                0x6D, 0x78, 0xED, 0x52, 0x8F, 0x02, 0x23, 0xD4, 0xC6, 0x9E, 0xB3, 0x26, 0xBB, 0x04,
                0x69, 0xF1, 0x98, 0x3A, 0x77, 0xAC, 0x15, 0xA9, 0x67, 0x7A, 0x41, 0x73, 0xE8, 0x40,
                0xCC, 0x61, 0x0F, 0xDB, 0x1A, 0x40, 0xC7, 0xA8, 0x2E, 0x10, 0xC3, 0x18, 0x62, 0x56,
                0x11, 0x7F, 0x48, 0xB5, 0x26, 0x65, 0x52, 0x48, 0xDF, 0xA3, 0x9F, 0xD2, 0xAE, 0x06,
                0xB2, 0x90, 0x2A, 0x25, 0xF6, 0xC4, 0x8D, 0xD5, 0x90, 0x4F, 0xD3, 0x33, 0xBC, 0xF0,
                0x92, 0xCB, 0x93, 0xFD, 0x61, 0x27, 0x74, 0xA3, 0xDD, 0x0E, 0x47, 0x21, 0x14, 0xE7,
                0xD1, 0x2B, 0x12, 0x7A, 0xE2, 0x4C, 0x49, 0x5E, 0x20, 0x56, 0x67, 0x91, 0x30, 0xC3,
                0x12, 0xB1, 0xBA, 0x66, 0xE1, 0x50, 0x0C, 0x70, 0x91, 0xF9, 0x5C, 0xA1, 0x56, 0x42,
                0x33, 0xE0, 0x8B, 0xB8, 0x87, 0x79, 0x68, 0x3C, 0x3C, 0x45, 0x38, 0x10, 0xA2, 0xE1,
                0x7C, 0xF9, 0x20, 0xB1, 0xE8, 0xC8, 0x29, 0x32, 0x1F, 0x06, 0xBA, 0xC0, 0xB7, 0xB8,
                0x95, 0xA5, 0xF3, 0x90, 0x91, 0xE5, 0x6F, 0x6F, 0x46, 0x80, 0xE0, 0xFE, 0x75, 0x4F,
                0xC6, 0x71, 0x82, 0xD6, 0x02, 0x84, 0x2B, 0xDB, 0x36, 0x24, 0x94, 0x60, 0x42, 0xC3,
                0x6C, 0x93, 0xE0, 0xDF, 0x50, 0x49, 0xEA, 0x68, 0xA7, 0xC4, 0xA3, 0x55, 0xB1, 0xA4,
                0x54, 0x95, 0x15, 0x1B, 0x88, 0xC0, 0x59, 0xFF, 0x57, 0xA7, 0xA7, 0x23, 0x97, 0x15,
                0x57, 0x61, 0x09, 0xF4, 0x4F, 0x2B, 0x4C, 0x66, 0xA7, 0x67, 0x8C, 0x0B, 0x37, 0x25,
                0x9B, 0xA6, 0x19, 0x60, 0x91, 0xA4, 0x2F, 0x12, 0x81, 0x85, 0xAA, 0x10, 0x3C, 0x1A,
                0x6B, 0x8A, 0x6C, 0x45, 0x28, 0x32, 0x15, 0x94, 0x4E, 0x0C, 0xA3, 0x46, 0xAC, 0x76,
                0x5D, 0xC0, 0x1F, 0x00, 0x6D, 0xC6, 0x0F, 0x64, 0x1E, 0xF5, 0x4E, 0x70, 0xC9, 0xC0,
                0x49, 0x1F, 0xB6, 0xC6, 0x70, 0xA5, 0x6B, 0x73, 0xF4, 0xDB, 0xF9, 0xCE, 0xE1, 0x2F,
                0xB8, 0x64, 0x6D, 0x43, 0xB6, 0xC2, 0x1E, 0x72, 0x7E, 0x34, 0xD4, 0xDE, 0xA3, 0x82,
                0xFC, 0x66, 0xEF, 0xB5, 0x59, 0xC1, 0xA9, 0x57, 0x4B, 0xBC, 0xFC, 0x8F, 0x35, 0xA0,
                0xA4, 0x10, 0x61, 0x02, 0xDC, 0x14, 0xD6, 0xC0, 0xBA, 0x22, 0x22, 0xF7, 0x8A, 0x14,
                0x35, 0xC4, 0x87, 0xB9, 0xE5, 0x89, 0x15, 0xA4, 0xC4, 0x42, 0x26, 0x8D, 0x3E, 0xE6,
                0x17, 0x21, 0x7D, 0x5D, 0x3F, 0x03, 0x98, 0x1F, 0x36, 0x92, 0x55, 0x6D, 0x5E, 0xAB,
                0x0E, 0xC5, 0xF4, 0x84, 0x35, 0xDF, 0xDC, 0x0D, 0x4F, 0x02, 0x28, 0x58, 0xC5, 0xE4,
                0x91, 0x6D, 0xAD, 0x34, 0x0C, 0x55, 0xE3, 0xBF, 0x29, 0xBA, 0x3B, 0x89, 0xEE, 0x47,
                0xCD, 0x1B, 0xB8, 0xBF, 0x79, 0x79, 0x74, 0xA3, 0x0D, 0xC2, 0x28, 0x84, 0x90, 0x55,
                0x46, 0xAD, 0x40, 0x79, 0x9A, 0x15, 0x04, 0xB2, 0x8A, 0x9E, 0x10, 0xFD, 0x7C, 0x3F,
                0xF3, 0xBD, 0xB2, 0x5E, 0xF6, 0xED, 0x3C, 0x5C, 0x80, 0x44, 0x0D, 0x5F, 0x47, 0x5A,
                0x55, 0x89, 0x34, 0xB7, 0x8B, 0x48, 0x8C, 0x87, 0x55, 0x25, 0xA5, 0x06, 0xC7, 0x05,
                0x81, 0xE8, 0xD3, 0x43, 0x78, 0x0C, 0xA1, 0xEC, 0x49, 0x8E, 0x4F, 0x93, 0x98, 0xE1,
                0x8A, 0xE9, 0x89, 0xA3, 0x00, 0xE5, 0xE0, 0xF9, 0x78, 0x87, 0xC1, 0x0B, 0xE9, 0x26,
                0xB1, 0xF6,
            ];
            assert_eq!(sk_ref, sk);
        }

        #[cfg(feature = "ntruhps4096821")]
        {
            let mut pk = [0u8; CRYPTO_PUBLICKEYBYTES];
            let mut sk = [0u8; CRYPTO_SECRETKEYBYTES];
            let mut rng = AesState::new();

            crypto_kem_keypair(&mut pk, &mut sk, &mut rng)?;

            let pk_ref = [
                0x41, 0xA1, 0xAC, 0xD5, 0x8C, 0xCA, 0x3D, 0xFF, 0x95, 0x03, 0x2A, 0xCD, 0x49, 0xE5,
                0x04, 0x9A, 0x94, 0x0C, 0xE3, 0xA3, 0xE4, 0x25, 0x5F, 0x69, 0x1D, 0x5C, 0xC6, 0x66,
                0xED, 0xE7, 0x4E, 0x14, 0xE3, 0xEA, 0x40, 0xF3, 0xBE, 0xBB, 0xDE, 0x2F, 0x7F, 0x4B,
                0x9D, 0xD9, 0xBB, 0x44, 0x76, 0x22, 0x8E, 0x6D, 0x42, 0xF5, 0xFC, 0xA3, 0xB5, 0x94,
                0x26, 0xAA, 0xE2, 0x19, 0x58, 0xF6, 0xDE, 0xD8, 0xC9, 0x64, 0xB5, 0xFD, 0x6C, 0xA6,
                0xF9, 0xE3, 0xE7, 0x60, 0xB4, 0x3C, 0x5C, 0x2E, 0x5A, 0x9A, 0xCF, 0x95, 0xF0, 0x7B,
                0xD7, 0xFF, 0x4D, 0x23, 0x50, 0xB6, 0x23, 0x16, 0x8F, 0x2D, 0x39, 0xCB, 0x48, 0xDF,
                0xA2, 0x98, 0xA9, 0x52, 0x20, 0x5D, 0x3A, 0xB1, 0x32, 0x94, 0xA6, 0x57, 0x68, 0x1E,
                0xB3, 0x9C, 0x11, 0x4A, 0x48, 0x5E, 0x42, 0x77, 0xC5, 0x4A, 0x83, 0x74, 0xB7, 0x7C,
                0xF2, 0x4E, 0x8E, 0xF1, 0x4C, 0xAD, 0x16, 0x1B, 0x91, 0x3A, 0x89, 0xEA, 0x85, 0x97,
                0xFD, 0xB5, 0xF1, 0x64, 0x5E, 0x95, 0x74, 0x70, 0x69, 0xE5, 0x24, 0x42, 0xD4, 0x00,
                0xB7, 0xEC, 0x88, 0xEE, 0x68, 0x08, 0x9E, 0xD3, 0x18, 0x61, 0x7D, 0xE5, 0x07, 0x31,
                0x20, 0x60, 0x01, 0xCE, 0x26, 0xF2, 0x33, 0x3A, 0xCB, 0xCA, 0x74, 0x70, 0x5C, 0x6D,
                0xDC, 0x18, 0xA8, 0x1E, 0xD4, 0x6C, 0x51, 0xF2, 0xA9, 0x1C, 0xC7, 0x23, 0x9D, 0x28,
                0x75, 0xF8, 0x58, 0xCE, 0x46, 0xCD, 0xEF, 0x8D, 0xDD, 0x96, 0x62, 0xEB, 0xE0, 0x36,
                0x09, 0x6E, 0x98, 0x5C, 0x82, 0x70, 0xF0, 0x26, 0xE5, 0x74, 0x53, 0xCA, 0x08, 0x6A,
                0x7E, 0xE8, 0x5D, 0x32, 0x24, 0x22, 0x61, 0xE1, 0xFB, 0x9D, 0x40, 0xD4, 0xE3, 0x46,
                0x2E, 0x7D, 0x90, 0x9E, 0x1E, 0xEF, 0xE8, 0x76, 0x3D, 0xBE, 0x37, 0x2D, 0xC0, 0x79,
                0x04, 0xD6, 0x40, 0xED, 0x20, 0x4F, 0xE0, 0x12, 0xCB, 0x29, 0xB5, 0xFD, 0x4F, 0x26,
                0x1A, 0x30, 0x61, 0xEF, 0x45, 0xCE, 0x57, 0xC1, 0x78, 0x6A, 0xAC, 0xB2, 0x8C, 0xC6,
                0x7C, 0x52, 0x04, 0xBC, 0x2D, 0xEC, 0x67, 0x04, 0xEC, 0xF2, 0x80, 0xDE, 0x29, 0xEC,
                0xB1, 0x68, 0x9F, 0xD7, 0x3C, 0x0F, 0x98, 0x31, 0x75, 0x81, 0x23, 0x55, 0xBA, 0xD4,
                0xCC, 0x3F, 0x85, 0x89, 0x0A, 0x67, 0x91, 0x48, 0x2A, 0x03, 0xF2, 0xB3, 0x1B, 0xF6,
                0x27, 0xFC, 0xEA, 0x5B, 0x7E, 0x4B, 0xD4, 0x02, 0x49, 0x34, 0x05, 0x30, 0xD7, 0x38,
                0x67, 0x39, 0xAA, 0xBD, 0x19, 0xEB, 0xD1, 0x5C, 0x21, 0x97, 0x7A, 0xEE, 0x86, 0x72,
                0x22, 0xC4, 0xF2, 0xAE, 0x54, 0xB5, 0xA9, 0xC7, 0xFC, 0xDD, 0xF4, 0xB1, 0x15, 0x5A,
                0xC0, 0x02, 0x40, 0x12, 0x06, 0x6B, 0xCB, 0xC7, 0xC5, 0x20, 0x46, 0x1E, 0x92, 0x1D,
                0x05, 0x0B, 0x3F, 0x5D, 0x66, 0x1C, 0x2B, 0x4D, 0xEF, 0x82, 0xB4, 0x02, 0x18, 0x45,
                0xA4, 0x67, 0x60, 0x7D, 0x98, 0xEB, 0x77, 0x6F, 0x20, 0xD6, 0x9E, 0xB7, 0x0C, 0xD2,
                0x1D, 0x04, 0xFF, 0x4E, 0x8F, 0xE3, 0x9F, 0xB9, 0x1F, 0xD3, 0xA3, 0xBC, 0x2E, 0x1D,
                0xB3, 0x47, 0xBA, 0x6D, 0xA3, 0xBD, 0xCD, 0xFD, 0x07, 0x14, 0xB0, 0x52, 0x2F, 0x05,
                0x02, 0xE6, 0x66, 0x79, 0x44, 0x61, 0x4C, 0xA8, 0xEA, 0x86, 0x6E, 0xCC, 0x4F, 0xE8,
                0x5F, 0x0A, 0xAD, 0x3E, 0xD3, 0x2C, 0x1D, 0xEB, 0x85, 0xB7, 0x64, 0x31, 0xD0, 0x55,
                0xF7, 0x6C, 0x41, 0x86, 0x98, 0x44, 0x23, 0xFD, 0x49, 0x1E, 0x81, 0xA8, 0xE5, 0xD2,
                0xA9, 0xB6, 0xEA, 0xEB, 0x3F, 0xF1, 0xF3, 0x74, 0x22, 0x02, 0x13, 0x1B, 0x84, 0xE2,
                0xDD, 0x03, 0x8E, 0x01, 0x13, 0x1D, 0x95, 0x91, 0xF3, 0x5A, 0x8D, 0x94, 0xEC, 0x72,
                0x78, 0xC2, 0xB5, 0x67, 0x4B, 0x23, 0x21, 0xA2, 0xD7, 0x50, 0x76, 0x0C, 0x5D, 0xCD,
                0xEA, 0xE2, 0x91, 0x03, 0xD9, 0x89, 0xD1, 0x0F, 0xCF, 0x83, 0xC4, 0x0B, 0x8A, 0xA7,
                0x3D, 0x1F, 0xF9, 0x31, 0xF5, 0x2F, 0xCA, 0xDF, 0xB1, 0xC4, 0x68, 0x24, 0x70, 0xC2,
                0x38, 0x8B, 0x59, 0x54, 0x05, 0x74, 0xE5, 0xA0, 0x93, 0xB1, 0x47, 0x7C, 0x4E, 0x19,
                0xD8, 0xD0, 0xF5, 0x86, 0xB4, 0x94, 0xEB, 0x2A, 0x77, 0x4E, 0x79, 0xFE, 0xA3, 0x0B,
                0x1C, 0x96, 0x52, 0x95, 0x84, 0x12, 0x7A, 0x5D, 0x14, 0xC8, 0x10, 0x0D, 0x6A, 0xCF,
                0xE0, 0xFE, 0x7B, 0xF0, 0xC2, 0xC7, 0xA5, 0xD3, 0x14, 0x96, 0xB7, 0x85, 0x73, 0x30,
                0xD4, 0xD3, 0x79, 0xFE, 0x08, 0x4B, 0xBE, 0x4E, 0x8E, 0x27, 0x7C, 0x32, 0x15, 0x2A,
                0xA3, 0x79, 0xF2, 0x08, 0x09, 0xED, 0x0A, 0xF1, 0xAE, 0x6A, 0x03, 0x2B, 0xF2, 0x44,
                0xBD, 0xD3, 0x73, 0xEA, 0x99, 0xF3, 0xCF, 0xA8, 0x44, 0xC1, 0xFE, 0x7A, 0x2A, 0xF4,
                0xA6, 0x6F, 0xFA, 0x49, 0xDB, 0x78, 0x3B, 0x83, 0xCA, 0x7D, 0x75, 0xD5, 0x01, 0xCF,
                0xA8, 0xB4, 0xAC, 0xB0, 0xF9, 0x3F, 0xF9, 0x22, 0x20, 0xBF, 0xD6, 0xBF, 0xA9, 0xED,
                0x4D, 0xFB, 0xCC, 0xE8, 0x59, 0x23, 0xD5, 0xAD, 0xD0, 0x8B, 0x4B, 0xA9, 0xC0, 0x6C,
                0x00, 0xBA, 0x4C, 0xCC, 0x3C, 0xF9, 0xC6, 0xE9, 0x96, 0x90, 0x4B, 0x29, 0xDC, 0xEA,
                0x36, 0xC0, 0xBF, 0x2F, 0x40, 0x9E, 0x8C, 0x1D, 0xD2, 0xF9, 0xD2, 0xFC, 0x31, 0x6C,
                0x83, 0xA9, 0x7B, 0x56, 0x25, 0x20, 0xF6, 0xAB, 0x13, 0x99, 0x45, 0xB2, 0x33, 0x3D,
                0x02, 0x56, 0x77, 0xDF, 0x5B, 0x20, 0x04, 0x5F, 0xFC, 0xD4, 0xB4, 0x1A, 0xA2, 0xAF,
                0x9B, 0x06, 0x0F, 0x35, 0xEF, 0x25, 0x68, 0x8C, 0x13, 0xBC, 0x39, 0x80, 0xE3, 0xAC,
                0xD9, 0xA0, 0xC4, 0x9A, 0x14, 0xA1, 0x4B, 0x8B, 0x29, 0xE6, 0x0B, 0x05, 0xC9, 0x51,
                0x8F, 0xAE, 0xEB, 0x38, 0xBE, 0x95, 0x82, 0x40, 0x2E, 0x00, 0x02, 0xC7, 0x97, 0x93,
                0x1A, 0xE4, 0xE2, 0xA3, 0xBF, 0xD6, 0x05, 0xA5, 0x40, 0x54, 0x1C, 0x95, 0x03, 0xB7,
                0x62, 0x35, 0x33, 0x50, 0x25, 0x4B, 0x58, 0xA8, 0xBC, 0x85, 0xB4, 0x56, 0x74, 0x26,
                0x50, 0x76, 0xC6, 0x85, 0x32, 0xDE, 0x73, 0x55, 0x7E, 0x80, 0xFD, 0x49, 0x88, 0x42,
                0x56, 0x34, 0x1B, 0x84, 0x9A, 0xE4, 0x16, 0xC1, 0x81, 0x5E, 0x4A, 0xEF, 0xCA, 0xB2,
                0x84, 0xB5, 0x56, 0xBA, 0x62, 0x57, 0x16, 0xA4, 0xCD, 0x95, 0x84, 0xC0, 0x29, 0x91,
                0xA5, 0x05, 0xB6, 0xDA, 0xC8, 0x65, 0xF9, 0xD7, 0xE3, 0x1A, 0x19, 0x1D, 0x82, 0xB5,
                0x2E, 0x8A, 0xE9, 0xD5, 0x2B, 0xB0, 0x1D, 0xAC, 0x48, 0xA9, 0x17, 0x02, 0x9D, 0xF7,
                0x80, 0x1A, 0x2A, 0xAE, 0x45, 0x8F, 0x86, 0xF8, 0x95, 0xCB, 0x2C, 0xFB, 0x4A, 0xC1,
                0xC4, 0xFC, 0x93, 0xF4, 0xE7, 0xDB, 0x60, 0x7B, 0x4D, 0x15, 0x79, 0x0E, 0xCE, 0x4F,
                0x6A, 0x08, 0xCB, 0x34, 0x37, 0x5D, 0xF7, 0x03, 0x9E, 0xB4, 0x39, 0xDC, 0xF6, 0x53,
                0x36, 0x9D, 0x9C, 0x0E, 0x0A, 0x15, 0x07, 0xAC, 0x48, 0x2C, 0x4E, 0x3C, 0x86, 0x75,
                0x7D, 0x38, 0xD2, 0xC3, 0x9A, 0x74, 0x9A, 0x47, 0xE7, 0x49, 0xC1, 0x71, 0xA2, 0xA4,
                0x6E, 0xC2, 0x82, 0x89, 0xC0, 0xFF, 0xA3, 0x31, 0x33, 0x7B, 0x65, 0x85, 0x2C, 0x3B,
                0x86, 0xFA, 0x13, 0x82, 0x3E, 0x54, 0x4D, 0x12, 0x15, 0x68, 0x86, 0x45, 0x0E, 0xE9,
                0xBD, 0x4D, 0xFC, 0xEF, 0x00, 0x4B, 0xCC, 0xB6, 0x9D, 0x8C, 0xF2, 0xB9, 0x1C, 0xA3,
                0xA5, 0x61, 0x3E, 0x1F, 0x3D, 0x0A, 0x4A, 0xEC, 0x53, 0x01, 0xEA, 0x3B, 0xD6, 0x05,
                0x3A, 0x7C, 0x61, 0x6B, 0xDC, 0xC1, 0x2E, 0x6D, 0xCD, 0xE8, 0x83, 0x70, 0x96, 0x13,
                0x61, 0x80, 0x9E, 0x50, 0x6E, 0x5F, 0x76, 0x35, 0x2D, 0x34, 0xE9, 0x83, 0xC1, 0xBD,
                0x23, 0x99, 0xBB, 0xA5, 0xED, 0xB1, 0x2D, 0x0C, 0x94, 0xC6, 0x36, 0x17, 0x75, 0x1C,
                0x7A, 0x3B, 0x6C, 0x1F, 0x25, 0x5F, 0x85, 0xAC, 0xD1, 0x38, 0x34, 0x4B, 0xA2, 0x8B,
                0x12, 0x3E, 0xDB, 0x24, 0x77, 0x43, 0x69, 0x4B, 0xF6, 0x6C, 0x60, 0xF5, 0xB4, 0xC5,
                0xAD, 0x70, 0x40, 0xE0, 0x7C, 0x8D, 0x64, 0x79, 0x88, 0x3B, 0xF3, 0x9D, 0xCC, 0x1E,
                0x6C, 0x0C, 0xF6, 0x7F, 0xE5, 0xFE, 0x6B, 0xC7, 0x64, 0x5C, 0xAF, 0xC3, 0xB8, 0xF1,
                0xA4, 0x04, 0xB2, 0xD3, 0xB3, 0x6C, 0x1A, 0xBC, 0x1B, 0x31, 0xE4, 0x16, 0xB3, 0x65,
                0x0D, 0x21, 0x2C, 0x1A, 0x3F, 0xAC, 0x81, 0x48, 0xD5, 0xAC, 0xD5, 0xF0, 0xCD, 0x8E,
                0xCA, 0x7A, 0xB9, 0x92, 0x80, 0xDC, 0x94, 0x63, 0xE3, 0x94, 0x89, 0xB4, 0x79, 0xBF,
                0x35, 0xF0, 0x10, 0x50, 0x88, 0x2D, 0xFB, 0x62, 0xCE, 0x09, 0x96, 0x61, 0x44, 0x87,
                0x6F, 0xDC, 0x00, 0x5A, 0xFC, 0xC6, 0xE4, 0xCD, 0x94, 0x5E, 0x1A, 0xBA, 0xEA, 0xFC,
                0x97, 0x6D, 0xB5, 0x12, 0xBB, 0x18, 0x29, 0x01, 0x17, 0xCD, 0xA5, 0xC9, 0x78, 0xA8,
                0x47, 0x94, 0x7D, 0xBD, 0xA1, 0x19, 0x50, 0xE4, 0x39, 0x2C, 0xBD, 0x66, 0x01, 0x83,
                0xB4, 0xE7, 0xF6, 0x1A, 0x99, 0xFF, 0xF9, 0x20, 0xDC, 0xC3, 0x39, 0x00, 0xE1, 0x79,
                0xA2, 0x6E, 0xFB, 0xC7, 0x5A, 0xBB, 0x2C, 0x0B, 0xB3, 0x87, 0x22, 0x45,
            ];
            assert_eq!(pk_ref, pk);
            let sk_ref = [
                0x89, 0x2D, 0x93, 0x85, 0x9B, 0x0F, 0x01, 0xAF, 0x2F, 0xE7, 0x4B, 0x45, 0xBF, 0x59,
                0xC7, 0x6E, 0x31, 0xAB, 0x85, 0xCD, 0xB1, 0x56, 0xC5, 0x46, 0x8F, 0x65, 0xC6, 0xA2,
                0x57, 0x11, 0x5B, 0xC4, 0x59, 0x72, 0x3B, 0x1F, 0xBD, 0xEE, 0x8C, 0x61, 0xEE, 0x6D,
                0x51, 0x7B, 0xB7, 0xA5, 0x26, 0x81, 0x5C, 0x8B, 0x74, 0xC8, 0x31, 0x35, 0x66, 0xCC,
                0x51, 0xC2, 0x5D, 0xD5, 0x87, 0x69, 0x10, 0x62, 0x4F, 0xCF, 0xE0, 0x37, 0xDB, 0x7B,
                0x10, 0xC9, 0x94, 0x98, 0x2B, 0x95, 0xD0, 0xAA, 0x20, 0x89, 0x3E, 0x4F, 0x67, 0x39,
                0x74, 0xC5, 0x7E, 0xA9, 0x33, 0xD8, 0x8C, 0x8F, 0x3D, 0xBD, 0x9E, 0x98, 0x70, 0xD2,
                0x09, 0x1B, 0x32, 0x3E, 0xD1, 0x10, 0x3E, 0xDA, 0x4E, 0x3F, 0xA1, 0x96, 0xEE, 0x29,
                0x29, 0x06, 0xDC, 0xC3, 0x8D, 0x59, 0x42, 0x62, 0x1B, 0x74, 0xBF, 0xB3, 0x99, 0xB6,
                0x7C, 0x74, 0x74, 0x60, 0x83, 0x3F, 0xEF, 0x0F, 0xE5, 0x23, 0x1B, 0xA1, 0x44, 0xC8,
                0x1D, 0x66, 0x94, 0x07, 0xDA, 0xA4, 0xAA, 0x18, 0x8D, 0x55, 0xD9, 0x9A, 0xD2, 0x87,
                0xC0, 0x89, 0xAC, 0xB0, 0x6E, 0x78, 0x0F, 0xD7, 0x1F, 0x8D, 0x9A, 0x27, 0xC0, 0x6D,
                0xB8, 0xD1, 0x98, 0xC8, 0xD6, 0xA3, 0x53, 0x69, 0x01, 0xE9, 0x81, 0x79, 0x67, 0x86,
                0x04, 0xE0, 0x82, 0xBC, 0xE5, 0x4E, 0x74, 0x18, 0xE2, 0x9F, 0xC0, 0x2B, 0x0D, 0xE6,
                0xE1, 0x73, 0xA5, 0x4C, 0x4C, 0x26, 0x87, 0x45, 0xA1, 0x0C, 0xA1, 0xD9, 0x0F, 0xE0,
                0x0C, 0x76, 0xD8, 0x20, 0x0D, 0x0B, 0xE5, 0xA7, 0x89, 0x80, 0xB5, 0xE4, 0x16, 0xF1,
                0xC0, 0xDE, 0xB9, 0xD9, 0x69, 0x76, 0x7B, 0x43, 0x3A, 0x9F, 0xA7, 0x9F, 0x30, 0x8D,
                0x88, 0x6C, 0xE2, 0x7A, 0x4A, 0x38, 0x36, 0x13, 0x01, 0x44, 0x02, 0xCA, 0xCA, 0xB5,
                0x68, 0x2C, 0x6A, 0x37, 0x62, 0x0F, 0x66, 0x6B, 0x4D, 0x1C, 0x64, 0x3A, 0x82, 0xC2,
                0x87, 0xC0, 0x40, 0x69, 0xE4, 0xC6, 0x7E, 0xD7, 0x5B, 0x11, 0xC7, 0x87, 0x7C, 0x97,
                0x4D, 0xB7, 0xD9, 0x5F, 0x6E, 0xD3, 0xD2, 0xA9, 0xED, 0x1B, 0x4E, 0x12, 0x06, 0x2C,
                0x35, 0xE9, 0x3F, 0xA7, 0x2C, 0x76, 0x8A, 0xC9, 0x89, 0x5D, 0x82, 0x3A, 0x38, 0x43,
                0x9C, 0x87, 0xC5, 0xEC, 0x3E, 0x77, 0xD5, 0x82, 0x21, 0x52, 0x75, 0xB5, 0xE1, 0x15,
                0xBF, 0xE5, 0x82, 0xD0, 0x17, 0x22, 0xA2, 0x97, 0x96, 0xBF, 0xB8, 0x7F, 0x0A, 0x95,
                0x3F, 0x6D, 0x54, 0xCE, 0x7B, 0xB3, 0xB6, 0x4D, 0x67, 0x75, 0xA7, 0xA8, 0xE8, 0xEE,
                0x46, 0x5B, 0x57, 0x45, 0x25, 0xC0, 0x54, 0x1D, 0xB9, 0x8C, 0xA2, 0xC1, 0x20, 0xE6,
                0x0B, 0x7A, 0xDB, 0x8F, 0xFD, 0xFE, 0xA2, 0xA6, 0x66, 0x25, 0xD5, 0xF5, 0x0F, 0xDA,
                0xE9, 0x03, 0xAE, 0x4A, 0x2D, 0xB5, 0xA8, 0x06, 0xB2, 0x1C, 0x20, 0x4D, 0x02, 0x25,
                0xB6, 0xBC, 0x26, 0x2A, 0xD3, 0xF6, 0x3C, 0x8B, 0x42, 0x57, 0x9A, 0x27, 0x7D, 0xC7,
                0x74, 0x89, 0x56, 0xAA, 0x24, 0xA6, 0xA2, 0x3C, 0x71, 0x46, 0xDC, 0x6C, 0xEC, 0xFC,
                0x12, 0x85, 0x07, 0xCA, 0xC4, 0xF0, 0xCE, 0xB4, 0x7A, 0x85, 0x08, 0x4B, 0x8C, 0xDE,
                0x8C, 0x1A, 0x02, 0xB7, 0x78, 0xA9, 0x6A, 0x2A, 0x52, 0x06, 0x59, 0xEA, 0x7E, 0x31,
                0x85, 0xC0, 0xF9, 0xBC, 0x60, 0x6A, 0xE0, 0x1D, 0xE5, 0x56, 0x88, 0x02, 0xF3, 0x09,
                0x1D, 0xAE, 0x0F, 0x90, 0xED, 0x65, 0x5F, 0x3D, 0x67, 0xDA, 0xAF, 0x54, 0x60, 0x55,
                0xD2, 0x2D, 0x23, 0x35, 0xD9, 0x5D, 0x66, 0xCD, 0xA0, 0xF2, 0xAF, 0x72, 0xD6, 0x35,
                0xF0, 0xB9, 0x64, 0xC8, 0x05, 0xC6, 0xC9, 0xDB, 0x29, 0xC9, 0x95, 0xAD, 0x60, 0xD9,
                0x73, 0x59, 0x11, 0x55, 0xAA, 0x67, 0x07, 0x3E, 0x10, 0xC7, 0x00, 0x48, 0xDC, 0xE3,
                0x0B, 0xA9, 0xAD, 0x21, 0x56, 0x25, 0x26, 0xF4, 0x93, 0x99, 0xBA, 0x2D, 0x28, 0x27,
                0xFC, 0x92, 0x29, 0xDB, 0x2A, 0xA4, 0x89, 0xE8, 0xD9, 0x19, 0xC4, 0xF7, 0xFB, 0xD7,
                0x01, 0x08, 0x8F, 0x5F, 0x3E, 0x0E, 0x5C, 0x6F, 0xF2, 0xD0, 0x2D, 0x9A, 0x31, 0x37,
                0x10, 0x5B, 0x4D, 0x86, 0x0B, 0xCF, 0x41, 0x89, 0x0D, 0x18, 0x24, 0x4D, 0xD7, 0xD3,
                0x7E, 0x58, 0x58, 0x37, 0x5D, 0x60, 0xD1, 0xD4, 0x47, 0x98, 0x8C, 0x18, 0x39, 0x79,
                0x39, 0x82, 0x43, 0x7C, 0x8D, 0xE4, 0x3B, 0x31, 0x38, 0x1C, 0xAC, 0xD0, 0xA3, 0xE6,
                0x94, 0xDB, 0x72, 0x57, 0xB9, 0x7A, 0xD6, 0x3E, 0x69, 0x5D, 0xDA, 0x74, 0x4F, 0xFB,
                0xB0, 0x79, 0xF8, 0xD3, 0x01, 0x8C, 0x06, 0xA6, 0x91, 0xC0, 0x54, 0xA4, 0x1F, 0xC3,
                0x3C, 0x74, 0x28, 0x9A, 0xBD, 0x33, 0xAD, 0x52, 0xBE, 0x38, 0xA4, 0xB5, 0x0F, 0x49,
                0x35, 0x65, 0x85, 0x4C, 0xF6, 0x0B, 0x58, 0xD5, 0x34, 0xFC, 0x8E, 0x92, 0x5D, 0xBF,
                0xCC, 0x00, 0x38, 0x84, 0x18, 0x48, 0xF4, 0xD6, 0x9F, 0x34, 0xD3, 0xB3, 0xBA, 0xC3,
                0xC1, 0x82, 0x0D, 0xB6, 0xEB, 0xC1, 0xCE, 0x67, 0x1B, 0x9C, 0xB0, 0x3C, 0xB7, 0x8E,
                0x61, 0x6C, 0x96, 0xB8, 0x23, 0x27, 0x57, 0xDB, 0x2B, 0x2C, 0xEC, 0xFC, 0xBF, 0x35,
                0x5E, 0x97, 0x7F, 0xC9, 0x07, 0x8F, 0x67, 0x8C, 0x21, 0x2B, 0xA6, 0xD8, 0xB5, 0xA3,
                0x69, 0x32, 0x63, 0x1D, 0x70, 0x14, 0x04, 0x79, 0x73, 0x3E, 0x5B, 0xE8, 0xC3, 0xEB,
                0xD1, 0x01, 0x49, 0x00, 0x29, 0x6E, 0x7D, 0xB1, 0x87, 0xE9, 0xB0, 0x28, 0x15, 0x73,
                0x35, 0xDE, 0xAE, 0x0E, 0x39, 0xFF, 0x30, 0xF4, 0x39, 0x3D, 0x0E, 0x1A, 0xF8, 0x35,
                0x06, 0x1D, 0x3D, 0x95, 0xDA, 0xAA, 0x11, 0x64, 0x9B, 0xCC, 0x9B, 0x2A, 0xED, 0xC0,
                0xE6, 0xE8, 0x7F, 0x7A, 0x55, 0x20, 0xDB, 0xCE, 0xC6, 0xE6, 0x28, 0x7E, 0xFD, 0x64,
                0xC7, 0x6A, 0x16, 0xB5, 0x44, 0xAF, 0x77, 0x48, 0x51, 0x44, 0x69, 0x7B, 0xC0, 0xB4,
                0x19, 0x05, 0x09, 0x83, 0x53, 0x0D, 0x50, 0xB8, 0x57, 0x81, 0xBE, 0x53, 0x46, 0xC4,
                0x1A, 0x69, 0x81, 0x79, 0x17, 0xB0, 0xF5, 0x03, 0x4D, 0x6D, 0x12, 0xE4, 0xEB, 0xA4,
                0x7C, 0x48, 0xDE, 0x06, 0x3B, 0x78, 0xB2, 0xD4, 0x4C, 0xE1, 0xE3, 0x7F, 0xE8, 0x7F,
                0x38, 0x4F, 0xE1, 0x97, 0x25, 0x23, 0xE2, 0x4B, 0xF6, 0xD2, 0x5F, 0xAA, 0xAD, 0x2F,
                0xF7, 0x78, 0x82, 0xB8, 0xE2, 0x21, 0x67, 0xC3, 0xF6, 0xC2, 0x95, 0xD3, 0xCD, 0x8D,
                0x62, 0x07, 0x61, 0x71, 0xC6, 0x2F, 0xDF, 0x86, 0xD7, 0x78, 0x0B, 0x15, 0x64, 0x9B,
                0xE3, 0x97, 0x51, 0x3C, 0x47, 0x89, 0x15, 0x53, 0xF6, 0xFF, 0xC9, 0x85, 0x7C, 0xFA,
                0x53, 0xD1, 0xAD, 0x86, 0x3B, 0x8A, 0x36, 0xED, 0xA0, 0x49, 0x33, 0x1B, 0xEA, 0x05,
                0x6D, 0xE1, 0xC3, 0x10, 0xA1, 0x75, 0xA8, 0x8E, 0xD0, 0x46, 0x39, 0xAC, 0x18, 0xD1,
                0x70, 0x3E, 0xD5, 0x81, 0xD1, 0x38, 0x53, 0x8D, 0x63, 0x1C, 0xBB, 0x99, 0x5D, 0x3C,
                0xDC, 0x88, 0x9E, 0xA6, 0xB3, 0xBE, 0xF3, 0x12, 0x7D, 0xD1, 0xC7, 0x28, 0xFA, 0x20,
                0xF6, 0x59, 0xD3, 0xCD, 0xAF, 0x49, 0xF0, 0x2F, 0xB2, 0x16, 0xFC, 0xBC, 0xC5, 0x1B,
                0x8D, 0xDA, 0xC5, 0x09, 0x5D, 0xD5, 0xBD, 0xF6, 0xD1, 0x34, 0xF3, 0x00, 0xBB, 0x5C,
                0xE5, 0xD4, 0xEB, 0x1C, 0xA5, 0xBC, 0x79, 0x92, 0x6E, 0x51, 0xE1, 0x8B, 0xED, 0xC1,
                0x5A, 0x07, 0x71, 0x17, 0x05, 0x6C, 0x6F, 0x51, 0xB3, 0x59, 0x68, 0x7B, 0x95, 0xEB,
                0xA2, 0xD7, 0xAB, 0xF0, 0xE9, 0x7B, 0x3B, 0xAA, 0x7D, 0x74, 0x82, 0x45, 0xEA, 0x03,
                0xF7, 0x7E, 0x40, 0x4A, 0xFE, 0xFB, 0x52, 0xA2, 0x91, 0xE4, 0xC1, 0x03, 0xAA, 0x51,
                0x5B, 0x5D, 0xB3, 0xD7, 0x53, 0xC9, 0x30, 0xB0, 0x57, 0x74, 0xE6, 0x9F, 0x3F, 0x5B,
                0x69, 0xC7, 0xF3, 0x0E, 0xA9, 0x44, 0xCB, 0xE7, 0x80, 0x6A, 0xCB, 0x4D, 0xF0, 0x69,
                0x95, 0xB2, 0x3A, 0x14, 0x5B, 0x3F, 0x8D, 0x52, 0xD5, 0x77, 0xCC, 0xD9, 0x36, 0x06,
                0x62, 0x5C, 0x07, 0x84, 0xD9, 0xD2, 0x7B, 0x91, 0x20, 0xA2, 0xC3, 0xE6, 0x30, 0x69,
                0x30, 0x6A, 0x71, 0x5A, 0x44, 0xC2, 0xDF, 0xB7, 0x42, 0xD5, 0x7E, 0x7C, 0x9B, 0xA0,
                0xFB, 0x82, 0xB3, 0xC6, 0x50, 0xD9, 0xFB, 0xA0, 0x9C, 0xED, 0xB1, 0x25, 0x50, 0xC2,
                0x6C, 0x51, 0x71, 0xC7, 0xE2, 0x96, 0x0E, 0x27, 0xF1, 0x91, 0x0E, 0xD0, 0x21, 0x2B,
                0xA4, 0x63, 0xAD, 0xF7, 0xFF, 0x66, 0x04, 0x44, 0x55, 0x8B, 0xBF, 0xCF, 0x7B, 0x72,
                0xCB, 0x4D, 0x80, 0xE2, 0x50, 0x9E, 0x1D, 0x97, 0x8A, 0x60, 0x13, 0xF8, 0x56, 0xA8,
                0x65, 0xF6, 0xDB, 0x7A, 0x89, 0x17, 0x74, 0xEB, 0x41, 0x07, 0x5A, 0xBD, 0xFB, 0x52,
                0xF1, 0xB9, 0xF3, 0x73, 0xC1, 0x3B, 0x68, 0xAB, 0x11, 0x49, 0xFF, 0xC5, 0x19, 0x93,
                0x52, 0xAC, 0xB0, 0x0E, 0x2B, 0xD6, 0xE9, 0x34, 0x9D, 0xAA, 0x3E, 0x04, 0x20, 0xF0,
                0x36, 0xB3, 0x00, 0x41, 0x93, 0x7E, 0xA2, 0x97, 0x37, 0x70, 0xE0, 0x09, 0xE0, 0x49,
                0x39, 0x76, 0x96, 0x44, 0x8C, 0x9F, 0x47, 0x24, 0x24, 0x37, 0x08, 0x10, 0xC0, 0x45,
                0xDE, 0x3B, 0x28, 0xEE, 0x7A, 0x9D, 0x01, 0xC2, 0x6D, 0x51, 0x69, 0xE2, 0x0C, 0x03,
                0xC2, 0x5F, 0x8E, 0xE4, 0xF0, 0xC0, 0x7F, 0x58, 0x51, 0xC3, 0x00, 0x4C, 0xE3, 0xE2,
                0x89, 0xE4, 0xA7, 0xF7, 0xFC, 0xAB, 0xDF, 0x15, 0xCD, 0x68, 0x6D, 0xBA, 0x98, 0x8E,
                0xF8, 0x21, 0x10, 0x43, 0x45, 0x9B, 0xD4, 0xF0, 0x4D, 0x60, 0xD7, 0x23, 0xD8, 0x38,
                0xFB, 0x27, 0xB0, 0x1D, 0x7D, 0xE6, 0xC7, 0x75, 0xF0, 0x5B, 0xB0, 0x0D, 0xC0, 0x3B,
                0xBC, 0x62, 0x04, 0x70, 0xDF, 0xBA, 0x59, 0xFC, 0x97, 0x3C, 0xB6, 0x4F, 0x79, 0xF0,
                0x37, 0x2C, 0x35, 0x09, 0x8D, 0x32, 0x02, 0xAF, 0x65, 0x33, 0x01, 0x7A, 0xCB, 0xF7,
                0x3F, 0x1E, 0x98, 0x3B, 0x6F, 0xEF, 0xC1, 0x85, 0x1F, 0x20, 0xDF, 0x25, 0x14, 0x49,
                0x88, 0x5D, 0xEC, 0x97, 0x32, 0x87, 0xBA, 0xD2, 0xFD, 0x01, 0xC0, 0xFC, 0x04, 0x93,
                0xE2, 0x90, 0xF3, 0x3F, 0x22, 0x51, 0x6D, 0xA8, 0xA9, 0xB7, 0xD9, 0xB0, 0x67, 0x6F,
                0xD8, 0xB3, 0x83, 0x46, 0x0A, 0xDA, 0xD9, 0x68, 0x1D, 0x0C, 0xC2, 0x51, 0x8C, 0xDE,
                0xDD, 0x6D, 0xF5, 0x7F, 0xD1, 0xF1, 0x0C, 0x67, 0xE3, 0xE2, 0xE8, 0x0A, 0xF2, 0x25,
                0xF7, 0x09, 0xC5, 0xED, 0x91, 0x59, 0x85, 0xEA, 0x19, 0xE1, 0xF1, 0x58, 0x09, 0x93,
                0x82, 0x6B, 0xDA, 0xB6, 0x7B, 0x1C, 0xDA, 0x89, 0x53, 0x44, 0x1A, 0x9E, 0xC8, 0x5F,
                0xDE, 0x71, 0x18, 0x9F, 0x8E, 0x7B, 0x8E, 0x43, 0x8D, 0xA7, 0xA8, 0x04, 0xD0, 0x8C,
                0xB2, 0xDC, 0xF7, 0xA0, 0xCC, 0x9C, 0x7B, 0xC3, 0x4A, 0x6C, 0xC5, 0xE6, 0x42, 0x64,
                0xAA, 0x7E, 0xA0, 0x8B, 0x65, 0x3F, 0x80, 0xF1, 0x2F, 0x4B, 0x85, 0xC5, 0x5E, 0x50,
                0xB4, 0x09, 0x6B, 0x0D, 0x0A, 0x7E, 0xFB, 0xFE, 0xA5, 0x81, 0x79, 0x28, 0x00, 0xEE,
                0xA9, 0xD4, 0x5D, 0x97, 0xE1, 0xEC, 0x19, 0xC3, 0xCB, 0x9D, 0xBC, 0x56, 0xBF, 0xCE,
                0x03, 0xFC, 0xC4, 0xA3, 0x59, 0xBB, 0x21, 0x7A, 0xE1, 0xBF, 0x79, 0x0F, 0xE0, 0x59,
                0xD1, 0x35, 0xD7, 0x80, 0x54, 0xEB, 0x75, 0xCA, 0x41, 0x19, 0xD8, 0x09, 0xF6, 0xD8,
                0x64, 0xCB, 0x08, 0x5D, 0x74, 0x0A, 0x45, 0x6B, 0x31, 0x06, 0x88, 0xBE, 0x8E, 0xE3,
                0xE0, 0x3D, 0x2F, 0x61, 0x44, 0xDD, 0x68, 0x1C, 0x36, 0x7A, 0xDA, 0xE3, 0xFC, 0xC9,
                0xF5, 0xA7, 0x97, 0x8E, 0x06, 0x7F, 0xF0, 0x5B, 0xF1, 0x71, 0x17, 0xF1, 0x52, 0x5F,
                0x96, 0xFA, 0x0D, 0x80, 0x11, 0x8B, 0x8D, 0x57, 0x8A, 0x47, 0x78, 0x67, 0x58, 0x55,
                0x6E, 0xD6, 0xA0, 0xD5, 0xE2, 0x81, 0x31, 0xC0,
            ];
            assert_eq!(sk_ref, sk);
        }

        #[cfg(feature = "ntruhrss701")]
        {
            let mut pk = [0u8; CRYPTO_PUBLICKEYBYTES];
            let mut sk = [0u8; CRYPTO_SECRETKEYBYTES];
            let mut rng = AesState::new();

            crypto_kem_keypair(&mut pk, &mut sk, &mut rng)?;

            let pk_ref = [
                0x85, 0xE3, 0xD2, 0x88, 0x23, 0xE9, 0xE1, 0xA9, 0x31, 0xB6, 0xE5, 0x83, 0xAD, 0xF1,
                0xB0, 0x16, 0x1E, 0xD1, 0x75, 0x78, 0x32, 0x4E, 0xD2, 0x89, 0x19, 0x62, 0x48, 0x34,
                0x31, 0x41, 0x04, 0xDC, 0x82, 0x88, 0x6E, 0x81, 0x1A, 0x6D, 0xF7, 0x67, 0x99, 0xC5,
                0x03, 0x8F, 0x04, 0xF4, 0xB2, 0x30, 0x91, 0x5F, 0xC8, 0x71, 0xE5, 0x95, 0xBF, 0x60,
                0x46, 0x91, 0x20, 0x8C, 0xE6, 0x00, 0x9B, 0xEE, 0x21, 0xB3, 0x94, 0x69, 0xCC, 0x74,
                0xBC, 0xAE, 0xF7, 0x8D, 0xEC, 0x29, 0xC7, 0x21, 0x02, 0xB8, 0x8D, 0xDA, 0xAB, 0xF9,
                0xA6, 0x97, 0x4D, 0x57, 0xC2, 0x93, 0x98, 0xCD, 0xE6, 0x84, 0xF5, 0x94, 0x58, 0xBD,
                0x38, 0x5A, 0xA0, 0x72, 0xBE, 0x21, 0x50, 0xDC, 0x66, 0x04, 0xF1, 0x0A, 0x6A, 0x73,
                0x0F, 0xFB, 0x97, 0x81, 0xD0, 0xB2, 0xDB, 0x87, 0x31, 0xB7, 0xEB, 0xDC, 0x87, 0x6E,
                0x99, 0x1E, 0xAC, 0x59, 0x0B, 0x64, 0x08, 0xD5, 0x35, 0x03, 0x82, 0x46, 0x0F, 0xA6,
                0x92, 0xBD, 0xA8, 0x94, 0x56, 0x2D, 0xEF, 0x48, 0x05, 0x4D, 0x7A, 0x68, 0x2C, 0xBE,
                0x9D, 0x54, 0x10, 0x02, 0x71, 0x83, 0xDB, 0x62, 0x07, 0xB6, 0x37, 0x96, 0xCD, 0x3B,
                0xFC, 0x2A, 0x42, 0x66, 0x6D, 0x11, 0x30, 0x69, 0x28, 0x46, 0xAA, 0x34, 0xAA, 0x00,
                0xAE, 0x9A, 0x72, 0xDF, 0x10, 0x84, 0x3D, 0x98, 0x56, 0x14, 0x41, 0x1A, 0xB4, 0x77,
                0x6A, 0x80, 0x28, 0x66, 0xE2, 0x41, 0xDB, 0xDC, 0x00, 0x78, 0xE6, 0xE0, 0x6A, 0x5C,
                0x8A, 0xA5, 0xF1, 0x0E, 0x6E, 0x9D, 0x4F, 0x55, 0x70, 0x82, 0xF7, 0x9F, 0x0E, 0x89,
                0x86, 0xC7, 0xE1, 0xC9, 0xA7, 0x75, 0x44, 0x5E, 0xB4, 0x10, 0x28, 0xB0, 0x26, 0x53,
                0x0E, 0x64, 0x18, 0x49, 0x06, 0xB0, 0x5C, 0x27, 0xC7, 0x09, 0x07, 0x9E, 0x43, 0x0F,
                0xDD, 0x8E, 0x8E, 0xC4, 0x83, 0x1F, 0x86, 0xD7, 0x83, 0xCD, 0xFD, 0x6C, 0x17, 0xD7,
                0xDC, 0x36, 0x28, 0xDB, 0xCF, 0xD9, 0x75, 0x65, 0xC5, 0x7E, 0x43, 0x00, 0xD0, 0x22,
                0xD8, 0x0A, 0x5D, 0x8F, 0x09, 0x29, 0x9C, 0xE2, 0x7C, 0xB7, 0x07, 0x9E, 0xAA, 0xF3,
                0x6B, 0x9C, 0xF6, 0x4D, 0x90, 0x8A, 0x65, 0x08, 0x1C, 0x48, 0x01, 0x64, 0x80, 0x56,
                0x81, 0xE9, 0xAA, 0x54, 0xD9, 0x60, 0x34, 0x63, 0xC8, 0x37, 0xE3, 0x46, 0xAB, 0x76,
                0xDC, 0xA9, 0x80, 0x42, 0x7E, 0x78, 0xCF, 0xB4, 0x49, 0xA3, 0x1E, 0xE8, 0x65, 0x32,
                0x90, 0xF8, 0xE3, 0xEB, 0xF2, 0x2E, 0xC5, 0x8E, 0x53, 0x15, 0xD5, 0xC6, 0xFF, 0x6C,
                0xA6, 0xFC, 0xE2, 0x19, 0x1B, 0xAC, 0x9F, 0xCF, 0xD0, 0xE2, 0x76, 0x05, 0xFA, 0x42,
                0x19, 0x61, 0x62, 0x72, 0x12, 0xB4, 0x6F, 0x2F, 0xD5, 0xDD, 0xE5, 0x2F, 0x26, 0x08,
                0xB9, 0xBD, 0x0C, 0x09, 0x2C, 0x77, 0x6D, 0xAF, 0x4F, 0x38, 0x0F, 0x4D, 0x29, 0xC6,
                0xD3, 0xE2, 0x1A, 0xD8, 0x93, 0x36, 0xE1, 0x17, 0xB4, 0x3B, 0xAF, 0xA5, 0x79, 0x84,
                0x92, 0xA0, 0xA2, 0x24, 0x0C, 0x71, 0x27, 0x62, 0x57, 0x91, 0xDE, 0x51, 0xA8, 0xD0,
                0xC1, 0xBA, 0x60, 0x9B, 0xEA, 0x57, 0xBB, 0x1E, 0x6A, 0xB0, 0x45, 0x5A, 0x56, 0xD5,
                0x77, 0xC1, 0x2F, 0xF7, 0x25, 0x6C, 0x1B, 0xCB, 0xD8, 0xF5, 0x2E, 0x5B, 0x04, 0x83,
                0x56, 0x56, 0xAD, 0xC1, 0xFC, 0x49, 0xE6, 0x43, 0x61, 0xF5, 0x3C, 0xDC, 0x30, 0x15,
                0xAA, 0x25, 0x31, 0x3B, 0x09, 0x20, 0x13, 0x3E, 0xF9, 0xD8, 0xCB, 0xD2, 0x82, 0xC7,
                0x1B, 0xEF, 0xD1, 0x14, 0xDA, 0x89, 0xF0, 0x89, 0x91, 0x3C, 0x3C, 0x6D, 0xEB, 0xBE,
                0xBD, 0xE8, 0x33, 0x91, 0xC1, 0xD4, 0x20, 0x64, 0x8C, 0x29, 0xC1, 0xCC, 0xCD, 0x31,
                0x7F, 0x57, 0x52, 0xC4, 0x55, 0xA3, 0xA2, 0x67, 0x1C, 0x9C, 0x45, 0x0F, 0x5B, 0x0B,
                0x41, 0x39, 0xCF, 0xD6, 0x9B, 0x1B, 0xBF, 0x75, 0x4E, 0xC1, 0x65, 0xD9, 0x15, 0xA1,
                0x10, 0x6A, 0x9A, 0x1B, 0x6F, 0x92, 0x52, 0x24, 0xE5, 0x38, 0xD3, 0x43, 0x9B, 0xDC,
                0x93, 0x8B, 0x2D, 0x16, 0xCF, 0x64, 0xF4, 0x06, 0xD1, 0x74, 0xCB, 0xF3, 0x0D, 0x41,
                0x77, 0x01, 0xF5, 0xB6, 0xF6, 0x68, 0xB2, 0xCC, 0x51, 0x99, 0x40, 0xED, 0xAB, 0x76,
                0x34, 0xD0, 0xF4, 0x71, 0x70, 0x28, 0x9B, 0x45, 0x3F, 0xC1, 0xA9, 0x58, 0xB7, 0x44,
                0x06, 0xC7, 0x27, 0x17, 0x9E, 0x1B, 0x8D, 0xD6, 0x47, 0x44, 0x9B, 0xC7, 0x02, 0xC0,
                0x8A, 0xA2, 0xA7, 0x32, 0x96, 0x68, 0xB8, 0x07, 0x76, 0xB6, 0xD0, 0xDA, 0x2A, 0x8F,
                0xC8, 0x97, 0xC7, 0xE0, 0x4A, 0x8E, 0xBE, 0xB1, 0x7C, 0xE8, 0xF3, 0x38, 0x46, 0x46,
                0xE1, 0xF2, 0x75, 0xAC, 0x8C, 0x07, 0x7D, 0xC3, 0x53, 0x14, 0x2D, 0x96, 0xA8, 0xC3,
                0x9C, 0x45, 0xD2, 0xBD, 0x35, 0x35, 0x30, 0x11, 0x52, 0x4C, 0x7A, 0xFF, 0xE3, 0xAB,
                0x60, 0x51, 0x80, 0x93, 0xB7, 0xC7, 0x64, 0x7B, 0x1F, 0x5A, 0x71, 0xBF, 0x0A, 0x46,
                0x9B, 0xC9, 0x31, 0x0E, 0x49, 0xEB, 0xB3, 0x01, 0x65, 0xB5, 0x3A, 0x99, 0x20, 0x6A,
                0x74, 0x1B, 0xED, 0x0F, 0xC7, 0xC3, 0xBC, 0x38, 0xC8, 0xF8, 0x3D, 0x90, 0x72, 0x81,
                0xED, 0x52, 0x6C, 0xA0, 0x9C, 0x8D, 0x7A, 0x17, 0x33, 0xD0, 0x60, 0x74, 0x20, 0xD7,
                0x34, 0xBD, 0x5F, 0x30, 0x4F, 0x51, 0xFA, 0xDC, 0xC0, 0xC1, 0xAF, 0x58, 0x0D, 0x01,
                0x2C, 0xDD, 0xB6, 0x01, 0x29, 0x6B, 0x11, 0x01, 0x69, 0x9E, 0xB5, 0x35, 0xAC, 0xA5,
                0x5B, 0x51, 0x83, 0x38, 0xA1, 0x20, 0x49, 0xBD, 0xF9, 0x14, 0xF6, 0x31, 0xEA, 0x52,
                0x9F, 0x62, 0x6D, 0xFF, 0x58, 0x21, 0x5D, 0x40, 0xC4, 0xCF, 0x39, 0xF0, 0xE4, 0xCB,
                0x99, 0x46, 0x34, 0xFD, 0x20, 0x13, 0xD8, 0xD7, 0x7E, 0x72, 0x41, 0x5A, 0x03, 0x5E,
                0x67, 0x2E, 0xD3, 0x86, 0x93, 0x87, 0x80, 0x1E, 0x11, 0xB9, 0x89, 0xF9, 0x41, 0x6E,
                0xAB, 0x31, 0x56, 0x5C, 0x25, 0x94, 0xF2, 0xD8, 0x27, 0x21, 0xE9, 0xEB, 0xCB, 0x1B,
                0xBE, 0xCC, 0xDE, 0x41, 0xBE, 0x59, 0x06, 0x31, 0x07, 0x32, 0x43, 0x1A, 0xD2, 0xF4,
                0x35, 0x1F, 0xE2, 0x3D, 0x46, 0x8C, 0x8B, 0x46, 0xE9, 0x02, 0x78, 0xAC, 0x70, 0x90,
                0x6C, 0xD4, 0x89, 0xB7, 0x7C, 0x18, 0x62, 0xA9, 0x0A, 0x87, 0xC5, 0x47, 0x32, 0xBB,
                0xAE, 0x7E, 0x01, 0x19, 0x7E, 0x76, 0x6B, 0xC6, 0x4D, 0xC2, 0x3D, 0xED, 0x62, 0x3B,
                0x3B, 0x67, 0x0C, 0x3F, 0xC1, 0x94, 0x3A, 0xA7, 0x23, 0x35, 0xFB, 0x8A, 0x07, 0x79,
                0xEC, 0x33, 0x85, 0x26, 0xB1, 0xAD, 0xE9, 0xC0, 0x84, 0x63, 0xFE, 0xE7, 0x34, 0x2A,
                0x50, 0x8D, 0xC0, 0x43, 0x8F, 0xEE, 0x27, 0x3B, 0xF5, 0x84, 0x8D, 0x11, 0xB5, 0x0D,
                0x87, 0xF2, 0x56, 0x0C, 0x79, 0xA3, 0x79, 0x48, 0xA9, 0xBF, 0x0D, 0x1C, 0xCA, 0xD2,
                0x54, 0x43, 0x7A, 0x63, 0x35, 0xFD, 0x56, 0xF1, 0x5C, 0x80, 0xAE, 0x78, 0x53, 0xB6,
                0x00, 0x8D, 0xBA, 0x48, 0xF5, 0xB9, 0xDC, 0x38, 0xAC, 0xC5, 0x2C, 0x1D, 0xD2, 0xE0,
                0x3E, 0x33, 0x59, 0xFE, 0x77, 0x4A, 0x6E, 0x03, 0xC5, 0x3B, 0x85, 0xC2, 0x70, 0x79,
                0x65, 0xBF, 0xCB, 0x50, 0x6E, 0xD5, 0xE5, 0xEE, 0xDD, 0xCB, 0x34, 0xD7, 0x0F, 0x7F,
                0xF2, 0xA5, 0x4F, 0x9B, 0x8F, 0xCB, 0x21, 0xF6, 0xAB, 0x0F, 0xEA, 0x1A, 0x3A, 0xE4,
                0xDD, 0x8D, 0xA4, 0x53, 0xC7, 0x99, 0x04, 0x79, 0x22, 0x4B, 0x68, 0x1F, 0x5B, 0x3C,
                0x06, 0x80, 0x64, 0x57, 0x34, 0xE8, 0x2E, 0x90, 0x25, 0x3F, 0x36, 0x53, 0x7C, 0xA5,
                0x8A, 0xB2, 0x78, 0x15, 0x8E, 0x0D, 0xA0, 0x33, 0x9F, 0xFD, 0xF1, 0xD7, 0xDF, 0x9C,
                0x23, 0x22, 0x5E, 0x76, 0x31, 0xD6, 0xD4, 0x66, 0xEE, 0xB0, 0x68, 0x26, 0x48, 0x73,
                0x45, 0x79, 0x1B, 0xBD, 0xAF, 0xAB, 0x9E, 0x99, 0xEF, 0x69, 0x0E, 0xB1, 0x31, 0xCC,
                0xCD, 0xE2, 0x5B, 0x23, 0x56, 0xBA, 0x61, 0x88, 0xE6, 0x68, 0x5E, 0x59, 0x8D, 0x03,
                0x89, 0xE8, 0xE4, 0x43, 0x73, 0xFE, 0x7E, 0x79, 0x89, 0x66, 0xA4, 0x3C, 0xBC, 0xE9,
                0xAF, 0x1C, 0x4D, 0x34, 0xDF, 0x2F, 0x7C, 0x89, 0xEC, 0x90, 0x73, 0xD1, 0xBC, 0x2E,
                0xD9, 0x4A, 0xF9, 0xA5, 0x9A, 0x81, 0x5C, 0x6B, 0xA8, 0x35, 0x7B, 0xA2, 0x30, 0x32,
                0x5D, 0x36, 0xE7, 0x8D, 0x9B, 0x60, 0x3D, 0x8B, 0xD9, 0x0F, 0xBF, 0x0F, 0x77, 0x4B,
                0x0B, 0x27, 0x11, 0x0F,
            ];
            assert_eq!(pk_ref, pk);
            let sk_ref = [
                0xD9, 0x48, 0xED, 0x9D, 0xE2, 0x0C, 0x02, 0xB2, 0x25, 0xC9, 0x42, 0x27, 0x6D, 0x56,
                0x80, 0x89, 0x29, 0xAB, 0xCE, 0xE5, 0x69, 0x59, 0x73, 0x28, 0xDF, 0x65, 0x7E, 0xA2,
                0xA8, 0x0E, 0xB6, 0xDC, 0xA9, 0x8A, 0x3A, 0x3D, 0x6C, 0xD6, 0xDC, 0x5E, 0x95, 0x88,
                0xA2, 0x93, 0x5D, 0xA8, 0x2E, 0x9F, 0xB5, 0x73, 0xC4, 0xE3, 0x29, 0x4D, 0xAE, 0xE4,
                0xA2, 0xE0, 0xB7, 0xED, 0xD8, 0x66, 0x1A, 0x5F, 0x47, 0xEA, 0x8E, 0x1C, 0x8A, 0x93,
                0x1A, 0xE7, 0xEF, 0x7A, 0x35, 0x7D, 0x77, 0xA7, 0x1F, 0x6E, 0x3D, 0x31, 0xB0, 0x21,
                0xC4, 0xDD, 0xC6, 0xA6, 0x2A, 0xBD, 0xDC, 0x71, 0x3E, 0xD8, 0xE5, 0x7A, 0xC2, 0xF0,
                0x12, 0x36, 0x28, 0x20, 0x76, 0x0D, 0x3D, 0xBF, 0x45, 0x24, 0xE8, 0x78, 0x95, 0x47,
                0x31, 0x03, 0x8C, 0xDB, 0xDE, 0x56, 0x4B, 0x5F, 0x1B, 0x8C, 0x6D, 0xB0, 0xE1, 0xB6,
                0xD7, 0x8C, 0xC4, 0x5D, 0xCA, 0x24, 0x94, 0x0C, 0x9E, 0x3B, 0x1B, 0x83, 0x4C, 0xE3,
                0x96, 0xD3, 0x2C, 0x7F, 0x35, 0x81, 0x48, 0xD6, 0xDA, 0x35, 0x69, 0xAC, 0x4C, 0x00,
                0x9E, 0x11, 0x78, 0xD3, 0x56, 0x76, 0x46, 0x5E, 0xC0, 0xD8, 0xE0, 0x75, 0xDB, 0x9B,
                0x71, 0xB6, 0x6E, 0x72, 0xA0, 0x0F, 0x28, 0x5F, 0xD3, 0x76, 0xBD, 0xD6, 0x95, 0x43,
                0x1C, 0x6D, 0x09, 0x00, 0x26, 0x8F, 0xDE, 0x9D, 0x97, 0xEF, 0x24, 0x0E, 0xCD, 0x95,
                0x3E, 0xD3, 0x50, 0xD6, 0xAC, 0x8F, 0x2C, 0x99, 0x6D, 0x81, 0xCE, 0x45, 0x42, 0x39,
                0xC4, 0xC0, 0xE3, 0x0A, 0x7D, 0x17, 0x62, 0xB2, 0x87, 0xA0, 0x68, 0x1C, 0x01, 0x5C,
                0x07, 0xB4, 0x1D, 0x96, 0x23, 0x1E, 0x2E, 0x05, 0xEA, 0xAA, 0x15, 0x9E, 0x3A, 0xBF,
                0xD2, 0xF2, 0x51, 0x15, 0xDE, 0x7D, 0x97, 0x61, 0x2C, 0xD6, 0xDE, 0xD0, 0x4A, 0x92,
                0x9F, 0x15, 0x7E, 0x88, 0xAB, 0xBF, 0xBE, 0xC7, 0x76, 0x22, 0xD9, 0x9B, 0xA9, 0x01,
                0x14, 0xB4, 0x9B, 0xEC, 0x9E, 0x1A, 0x39, 0x0E, 0x78, 0xF0, 0x04, 0xD4, 0x5F, 0x75,
                0xCF, 0xE0, 0x40, 0xE8, 0x53, 0x97, 0x75, 0xEF, 0x93, 0xA1, 0x6D, 0x90, 0x5A, 0x28,
                0x44, 0x03, 0xF7, 0x2D, 0x01, 0xFD, 0xFA, 0x19, 0xB6, 0x5C, 0x2C, 0x72, 0x6D, 0xD0,
                0x79, 0x0E, 0xBE, 0x7F, 0xF7, 0x7A, 0xD8, 0x0A, 0x43, 0x1D, 0x2D, 0x5D, 0x2F, 0xDC,
                0xF2, 0xC1, 0x07, 0x7F, 0x77, 0x1D, 0x09, 0x8F, 0x8D, 0xC0, 0xB2, 0x54, 0x91, 0xCA,
                0xF0, 0x9B, 0x3E, 0x60, 0x6B, 0x1E, 0x50, 0xB6, 0x60, 0xFE, 0x83, 0x93, 0xC1, 0xD1,
                0x1B, 0x90, 0x6D, 0xE0, 0x45, 0xC1, 0xE9, 0x0B, 0xCB, 0x2B, 0xA1, 0xF3, 0x9E, 0x84,
                0x9F, 0x0B, 0x6F, 0xFE, 0xF0, 0xF2, 0xEF, 0x1F, 0xA2, 0xD1, 0x6E, 0xA8, 0xA4, 0x54,
                0xB7, 0xC2, 0x6B, 0x36, 0x76, 0xD7, 0x06, 0xA9, 0xEE, 0xC1, 0x36, 0x68, 0xDA, 0x3A,
                0xE3, 0xE7, 0x20, 0x78, 0xCA, 0x4B, 0x05, 0x49, 0x9B, 0x01, 0xAA, 0x6E, 0xE6, 0x88,
                0x34, 0xD9, 0xE1, 0x1E, 0x26, 0x02, 0xCE, 0x7C, 0xD3, 0xEE, 0xE9, 0xC5, 0x63, 0x35,
                0x6F, 0xD9, 0x6E, 0x9E, 0x93, 0xC8, 0x5E, 0x2C, 0x65, 0xB5, 0x95, 0xFB, 0x4F, 0xFB,
                0xE0, 0x12, 0xB2, 0x80, 0x8A, 0xB9, 0xC1, 0x87, 0x5E, 0xE6, 0x5B, 0x33, 0x23, 0x51,
                0x62, 0xF2, 0x57, 0x25, 0x8F, 0x44, 0xAC, 0x68, 0x1F, 0x1E, 0x0F, 0xF8, 0x3F, 0xB6,
                0xBA, 0xBF, 0xE4, 0xE0, 0xDC, 0x1A, 0xDF, 0xD5, 0xE7, 0x6B, 0x0C, 0x30, 0x80, 0x4C,
                0xD5, 0xC7, 0x6F, 0x68, 0x13, 0xD1, 0xB3, 0xF2, 0xCE, 0xAA, 0x5A, 0xE0, 0x87, 0x10,
                0x46, 0x65, 0x71, 0x1A, 0x81, 0x6B, 0x0D, 0xAA, 0x70, 0x7D, 0x18, 0x86, 0x8A, 0x5D,
                0x6B, 0x6A, 0xD7, 0x84, 0x18, 0x26, 0xC9, 0xCC, 0x39, 0x90, 0xD7, 0xFF, 0x96, 0x5F,
                0xF2, 0x2D, 0x0F, 0x9C, 0x18, 0xCC, 0xF8, 0xE2, 0xC1, 0xB9, 0x83, 0x40, 0x78, 0xDD,
                0x6F, 0x32, 0xAE, 0xC0, 0xE6, 0xDD, 0x8E, 0x1B, 0x21, 0xED, 0x4E, 0x5D, 0xCA, 0x31,
                0x0F, 0xE3, 0x12, 0x57, 0xBB, 0x9B, 0xB7, 0xD4, 0xA4, 0xBC, 0xF2, 0x2A, 0x95, 0xEB,
                0x7F, 0x46, 0x8A, 0xFF, 0xA4, 0xD3, 0xCF, 0x0F, 0x09, 0xA3, 0x6E, 0x8B, 0x47, 0x66,
                0xD8, 0x52, 0xFC, 0x23, 0x39, 0x88, 0xB6, 0x2F, 0x4A, 0xE3, 0xBD, 0x9E, 0x16, 0xBC,
                0x8E, 0xBD, 0x54, 0x33, 0xB6, 0x7B, 0xEC, 0x60, 0x88, 0xA3, 0xBE, 0x04, 0xD8, 0xA4,
                0xF3, 0x91, 0x88, 0xF6, 0x9B, 0xF8, 0x2A, 0xDA, 0xFF, 0xC5, 0x75, 0xF4, 0xA7, 0x91,
                0xAD, 0xB7, 0xB5, 0xD8, 0xE2, 0x49, 0x1E, 0xB7, 0xE6, 0x9F, 0x50, 0xCD, 0x3F, 0xA6,
                0x14, 0x69, 0xB1, 0x38, 0xC9, 0xF1, 0x64, 0x7D, 0x24, 0x6D, 0x48, 0xC0, 0x2C, 0x4A,
                0x97, 0x27, 0xF4, 0x49, 0x1E, 0x51, 0xED, 0x98, 0x69, 0x41, 0x10, 0xB0, 0xB9, 0x15,
                0xCF, 0x08, 0xE8, 0xD2, 0x15, 0x30, 0xC5, 0x82, 0x24, 0x4D, 0x0F, 0x6B, 0x67, 0xB1,
                0x72, 0xFF, 0x76, 0x27, 0x39, 0xE9, 0xCB, 0x2B, 0x62, 0x11, 0x4A, 0x57, 0x51, 0x89,
                0xDB, 0x36, 0x50, 0x7A, 0x42, 0x03, 0xEE, 0x03, 0x00, 0x8C, 0x25, 0x6A, 0x99, 0x7D,
                0xDF, 0xB7, 0xE4, 0x13, 0xF9, 0x42, 0x6A, 0x4D, 0x1C, 0x90, 0x94, 0x26, 0x19, 0x0F,
                0xD5, 0xE9, 0x76, 0x5E, 0xC1, 0xC8, 0xBB, 0x91, 0xF2, 0xC7, 0xE8, 0x78, 0xD3, 0xD6,
                0x6E, 0x4F, 0xF9, 0x16, 0xAA, 0xEC, 0x05, 0xD8, 0xFB, 0xB5, 0xAB, 0x1F, 0x20, 0x77,
                0x54, 0x48, 0x5E, 0x67, 0xE1, 0xFC, 0xB7, 0x0E, 0x5E, 0x45, 0xAD, 0x5A, 0x28, 0xA3,
                0x48, 0xEF, 0xD2, 0x8F, 0x9E, 0xBF, 0xFE, 0x47, 0x45, 0xEE, 0xA9, 0xD6, 0x66, 0xA5,
                0xBE, 0x53, 0x25, 0x5F, 0x9A, 0x97, 0xF4, 0xEC, 0x6C, 0xEB, 0xD1, 0xC8, 0x4F, 0xA3,
                0xD5, 0xD7, 0xFC, 0x08, 0xE5, 0x92, 0xAD, 0x57, 0xC1, 0x14, 0xB8, 0xF0, 0xA6, 0x7D,
                0x4A, 0x6B, 0x6C, 0xF5, 0x64, 0x12, 0x47, 0xF8, 0x69, 0x16, 0xE6, 0x64, 0x9F, 0x01,
                0xF1, 0x33, 0xC7, 0x51, 0xF0, 0x75, 0x7B, 0x4D, 0x37, 0xB9, 0x57, 0xA3, 0x17, 0xBA,
                0xBE, 0xEA, 0x4D, 0xC1, 0x5A, 0x74, 0x72, 0xEF, 0xEA, 0x98, 0xAC, 0xF7, 0xE7, 0x9A,
                0xDE, 0xE0, 0x2F, 0x99, 0x2D, 0x1E, 0x3D, 0x84, 0xFE, 0xC3, 0x2D, 0xE0, 0x57, 0x35,
                0x82, 0xE5, 0x2E, 0x76, 0x44, 0x1D, 0xE3, 0xC6, 0x99, 0x40, 0x76, 0xA9, 0xE0, 0x9B,
                0x0B, 0x73, 0x17, 0x12, 0xFA, 0x04, 0xC4, 0x4F, 0xC2, 0xDC, 0xB3, 0xE3, 0x88, 0x7C,
                0x1B, 0xF9, 0xCA, 0xD4, 0x9F, 0xD8, 0x3C, 0xB4, 0xE4, 0x4E, 0xEE, 0x46, 0x16, 0x66,
                0xA9, 0xE6, 0x09, 0x2F, 0x3B, 0x20, 0xBF, 0xF0, 0x0B, 0x7F, 0x09, 0x90, 0xF2, 0x9E,
                0x21, 0x81, 0x01, 0x26, 0xBF, 0xB5, 0x94, 0x97, 0xF6, 0xA1, 0x19, 0x69, 0x50, 0x77,
                0x08, 0x8D, 0x28, 0xDD, 0xE1, 0x08, 0xCB, 0xC0, 0x31, 0x26, 0x14, 0xF9, 0x9A, 0x00,
                0x3E, 0x3B, 0x6C, 0x24, 0x44, 0x55, 0xBF, 0x53, 0xCD, 0x54, 0xBF, 0xE8, 0x61, 0xC6,
                0x72, 0xF0, 0x67, 0xA7, 0xB6, 0x5B, 0x35, 0xC0, 0x03, 0x93, 0x8B, 0x15, 0xC2, 0x6C,
                0x3F, 0x5C, 0x7A, 0x93, 0xB5, 0x96, 0x77, 0xF4, 0x84, 0x7A, 0xCC, 0x82, 0x3D, 0xA5,
                0x36, 0xA0, 0x1B, 0xB3, 0x1A, 0x65, 0xBD, 0xAC, 0xD6, 0x25, 0x2D, 0xFD, 0xC5, 0x7C,
                0x33, 0xC0, 0xB0, 0xE7, 0x75, 0xF9, 0x58, 0xE1, 0x76, 0x6E, 0xFB, 0x06, 0x6E, 0xD3,
                0x98, 0x8C, 0x76, 0xD7, 0x3A, 0xFE, 0x55, 0xFE, 0x9F, 0x62, 0xE7, 0xE9, 0x83, 0x28,
                0x4F, 0x9B, 0xDE, 0x4F, 0xA9, 0x8D, 0x9B, 0xAC, 0x8C, 0x1B, 0xCD, 0xF5, 0xE6, 0xF7,
                0x18, 0xC6, 0xAA, 0xBB, 0xDB, 0x4D, 0xEE, 0xB1, 0x23, 0x4F, 0x18, 0xCA, 0x4E, 0x2B,
                0xC9, 0x57, 0xB7, 0xC5, 0x46, 0x9E, 0xF9, 0x13, 0x71, 0x45, 0x42, 0x96, 0xE8, 0x77,
                0x19, 0x8A, 0x7B, 0x7F, 0x90, 0x11, 0x53, 0x43, 0x55, 0x97, 0x89, 0xCD, 0xF0, 0xE0,
                0x40, 0x08, 0xB7, 0xD6, 0xF2, 0x28, 0xD2, 0xD3, 0x19, 0x19, 0xB8, 0x79, 0x22, 0x67,
                0x52, 0xE6, 0xF2, 0x4F, 0xCB, 0xDE, 0x31, 0xD4, 0x00, 0xFC, 0x3E, 0xDE, 0x55, 0x34,
                0xFB, 0x62, 0x67, 0x47, 0x44, 0x75, 0xF1, 0x7E, 0xDC, 0xB4, 0xF4, 0x8A, 0xB5, 0x33,
                0x1A, 0x5B, 0x95, 0xAC, 0x66, 0xFC, 0x56, 0xAE, 0xEE, 0x90, 0x03, 0xC2, 0x4C, 0x30,
                0xF4, 0x3B, 0x8D, 0x6D, 0xBF, 0xC7, 0xF1, 0xB6, 0x41, 0x37, 0xEA, 0x6F, 0xB7, 0xD9,
                0xF3, 0xA5, 0x0D, 0x43, 0x48, 0x63, 0x79, 0x2E, 0xEE, 0xBB, 0xAE, 0xE2, 0xE7, 0x29,
                0x0E, 0xFA, 0xA7, 0x79, 0xC9, 0x3C, 0x86, 0xBD, 0xC2, 0x3C, 0x57, 0x69, 0x89, 0x87,
                0x6D, 0x8D, 0xE2, 0xA5, 0xBD, 0x0A, 0x7D, 0x79, 0x69, 0x4F, 0x0E, 0x88, 0xEA, 0x5B,
                0xF7, 0xA5, 0xE9, 0x1D, 0x52, 0x87, 0x72, 0x37, 0xAA, 0x04, 0xF0, 0x13, 0x67, 0x88,
                0x2C, 0x3E, 0x2F, 0x37, 0x53, 0x72, 0xE1, 0x2C, 0x60, 0x09, 0x7E, 0xBE, 0x4C, 0xA6,
                0x76, 0x06, 0x3C, 0x72, 0x51, 0x58, 0xEC, 0x85, 0x41, 0x5B, 0x0F, 0x52, 0xBC, 0x5C,
                0x2E, 0xB7, 0x1D, 0xA8, 0xE9, 0x31, 0xD6, 0x44, 0xBE, 0x92, 0xE4, 0xF2, 0xE8, 0x39,
                0x22, 0x74, 0xD1, 0xAE, 0x52, 0x9C, 0xE7, 0x0E, 0xF1, 0x01, 0x71, 0xB2, 0xAC, 0x82,
                0x3A, 0x45, 0xB2, 0x2B, 0xD9, 0xAA, 0x4D, 0xAA, 0xD4, 0xB5, 0x06, 0xEF, 0x0F, 0xD7,
                0x0E, 0x3B, 0xD9, 0x20, 0xAE, 0xF1, 0x83, 0x68, 0x82, 0x64, 0x03, 0x23, 0x1E, 0xF2,
                0xC2, 0xAC, 0xF3, 0xE7, 0x93, 0x8A, 0xB2, 0xCA, 0xA7, 0xE0, 0x2A, 0xFC, 0x42, 0x67,
                0x5F, 0x6C, 0xA4, 0xC7, 0xBF, 0x3A, 0x7B, 0x6B, 0x14, 0xA7, 0x66, 0x0E, 0xA6, 0xBF,
                0x27, 0x69, 0xF4, 0xA5, 0xB0, 0x3F, 0xCF, 0x6A, 0x9B, 0x12, 0xBC, 0xE8, 0x0F, 0xC6,
                0x2E, 0x05, 0x55, 0xDC, 0xBE, 0x87, 0x3C, 0x06, 0x05, 0xEB, 0x4F, 0x15, 0x4B, 0x98,
                0x01, 0xB5, 0x61, 0x86, 0x00, 0x89, 0xDC, 0x2D, 0x96, 0xF5, 0xAD, 0x31, 0x39, 0xDB,
                0x2C, 0xCB, 0xC2, 0xF6, 0xF0, 0xDC, 0x10, 0x98, 0x1B, 0x41, 0xF2, 0xCB, 0x2F, 0x15,
                0xBA, 0x6C, 0x44, 0x36, 0x23, 0x81, 0x7C, 0x8C, 0x57, 0xB7, 0xF4, 0x39, 0xF5, 0x4C,
                0x9C, 0x90, 0x80, 0x07, 0xA1, 0xEE, 0xBD, 0xE7, 0xE1, 0xDF, 0x82, 0x13, 0x42, 0xB0,
                0xF6, 0x29, 0x2D, 0xA0, 0x87, 0xB6, 0xE8, 0x44, 0xBF, 0xD5, 0xB9, 0x76, 0xE4, 0x92,
                0x81, 0xDE, 0x8E, 0x0D, 0x8A, 0xD5, 0x15, 0x47, 0x40, 0x46, 0x01, 0x53, 0x54, 0x2B,
                0xE5, 0x0A, 0x3F, 0x70, 0x83, 0x48, 0x06, 0x71, 0x96, 0xF5, 0x71, 0x17, 0x4D, 0xA6,
                0xC3, 0xC0, 0x59, 0x5E, 0xF6, 0x07, 0x10, 0x36,
            ];
            assert_eq!(sk_ref, sk);
        }

        Ok(())
    }

    #[test]
    fn test_enc_kat1() -> R {
        #[cfg(feature = "ntruhps2048509")]
        {
            let pk = [0u8; CRYPTO_PUBLICKEYBYTES];
            let mut rng = AesState::new();
            let mut ct = [0u8; CRYPTO_CIPHERTEXTBYTES];
            let mut ss = [0u8; CRYPTO_BYTES];

            crypto_kem_enc(&mut ct, &mut ss, &pk, &mut rng)?;

            let ct_ref = [
                0x00, 0xF8, 0x3F, 0x00, 0xFE, 0xFF, 0xFF, 0x00, 0x00, 0x20, 0x00, 0x01, 0x08, 0x00,
                0x00, 0x02, 0x00, 0x80, 0xFF, 0x03, 0x20, 0x00, 0xFF, 0xFF, 0x3F, 0x00, 0xFE, 0x0F,
                0x80, 0xFF, 0x03, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x10, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x01, 0x00, 0x40, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0xFF,
                0x0F, 0x40, 0x00, 0xFE, 0x0F, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0x0F, 0x40, 0x00,
                0xFE, 0x0F, 0x80, 0x00, 0xFC, 0x1F, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x01, 0xF8, 0x7F, 0x00, 0x00, 0x10, 0x80, 0xFF, 0x03, 0x00,
                0x00, 0x00, 0xF8, 0x3F, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0xFF, 0xFF,
                0x3F, 0x00, 0x00, 0x00, 0x80, 0xFF, 0xFF, 0x3F, 0x00, 0x00, 0x08, 0xC0, 0xFF, 0x01,
                0xF0, 0x7F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0xC0, 0xFF, 0x01, 0x00, 0x00, 0x00,
                0x00, 0xE0, 0xFF, 0x00, 0xF8, 0x3F, 0x00, 0x00, 0xF0, 0x7F, 0x00, 0x04, 0x20, 0x00,
                0x00, 0x00, 0xC0, 0xFF, 0x01, 0xF0, 0xFF, 0x00, 0x04, 0x00, 0x00, 0x01, 0x00, 0x00,
                0x00, 0xFE, 0x1F, 0x00, 0x00, 0xFC, 0xFF, 0xFF, 0x00, 0xF8, 0x7F, 0x00, 0x00, 0x10,
                0x80, 0xFF, 0x03, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x10, 0x80, 0xFF, 0x03,
                0x00, 0x00, 0x01, 0x08, 0x40, 0x00, 0x00, 0x10, 0x00, 0x00, 0xFC, 0xFF, 0xFF, 0x01,
                0xF8, 0xFF, 0xFF, 0xFF, 0x0F, 0x80, 0xFF, 0xFF, 0x1F, 0x00, 0x00, 0xF8, 0xFF, 0xFF,
                0x01, 0x00, 0x00, 0x00, 0x04, 0x20, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0xFF, 0xFF, 0x1F,
                0x00, 0x01, 0x00, 0xC0, 0xFF, 0x01, 0x00, 0x80, 0xFF, 0xFF, 0x1F, 0x00, 0x00, 0xF8,
                0xFF, 0xFF, 0x01, 0x10, 0x00, 0x00, 0x00, 0x20, 0x00, 0x01, 0xF8, 0xFF, 0xFF, 0x01,
                0x10, 0x80, 0xFF, 0x03, 0x20, 0x00, 0xFF, 0xFF, 0x3F, 0x00, 0xFE, 0x0F, 0x00, 0x00,
                0xFC, 0xFF, 0xFF, 0x01, 0xF8, 0x3F, 0x00, 0x00, 0x10, 0x00, 0x00, 0x04, 0x20, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x02, 0xF0, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xFF, 0x07, 0xC0,
                0xFF, 0xFF, 0x0F, 0x00, 0x00, 0x04, 0x00, 0x00, 0x01, 0x08, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF0, 0xFF, 0xFF, 0x03,
                0x20, 0x00, 0xFF, 0x0F, 0xC0, 0xFF, 0xFF, 0x0F, 0x80, 0xFF, 0x07, 0x00, 0x00, 0x01,
                0x00, 0x40, 0x00, 0xFE, 0x0F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF8, 0x7F, 0x00,
                0x00, 0xF0, 0x7F, 0x00, 0x00, 0x20, 0x00, 0x00, 0xF8, 0xFF, 0xFF, 0x01, 0x10, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFE, 0x0F, 0x00, 0x00, 0xFC, 0x1F,
                0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0xF0, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x01, 0xF8,
                0x7F, 0x00, 0x00, 0x00, 0x80, 0xFF, 0x03, 0x20, 0x00, 0x00, 0x08, 0x00, 0x00, 0x02,
                0x10, 0x80, 0xFF, 0x03, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF0, 0x7F, 0x00,
                0xFC, 0x3F, 0x00, 0xFF, 0xFF, 0x3F, 0x00, 0xFE, 0x0F, 0x80, 0xFF, 0x03, 0x20, 0x00,
                0x00, 0x08, 0xC0, 0xFF, 0x03, 0x00, 0x00, 0x00, 0x00, 0xE0, 0xFF, 0x01, 0xF8, 0x3F,
                0x00, 0x02, 0x00, 0x80, 0xFF, 0x03, 0xE0, 0xFF, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0x07, 0x40, 0x00, 0xFE, 0x0F, 0x00, 0x00, 0x00,
                0x00, 0x00, 0xFF, 0x07, 0x00, 0x00, 0x00, 0x00, 0x80, 0xFF, 0xFF, 0xFF, 0xFF, 0x01,
                0x00, 0xC0, 0xFF, 0x03, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0, 0xFF,
                0x03, 0x00, 0x80, 0xFF, 0x03, 0x00, 0x00, 0xFF, 0xFF, 0x7F, 0x00, 0xFE, 0x0F, 0x00,
                0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0xC0, 0xFF, 0x03, 0xF0, 0xFF, 0x00, 0x04, 0xE0,
                0xFF, 0x00, 0x08, 0x40, 0x00, 0xFE, 0x0F, 0x00, 0x00, 0x00, 0x20, 0x00, 0x01, 0x08,
                0xC0, 0xFF, 0x03, 0x00, 0x80, 0xFF, 0x07, 0x00, 0x00, 0x00, 0xF8, 0x7F, 0x00, 0x02,
                0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x80, 0x00,
                0x04, 0xE0, 0xFF, 0x00, 0x00, 0x40, 0x00, 0xFE, 0x0F, 0x00, 0x00, 0x00, 0x20, 0x00,
                0x00, 0x00, 0xC0, 0xFF, 0x01, 0x10, 0x00, 0x00, 0xFC, 0xFF, 0xFF, 0x00, 0x00, 0x40,
                0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00,
                0x80, 0xFF, 0x03, 0x20, 0x00, 0xFF, 0xFF, 0x3F, 0x00, 0x02, 0x00, 0x80, 0xFF, 0x07,
                0xE0, 0xFF, 0xFF, 0x0F, 0x00, 0x00, 0xFE, 0xFF, 0xFF, 0xFF, 0x03, 0x20, 0x00, 0x00,
                0x08, 0x40, 0x00, 0xFE, 0x0F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00,
                0x02, 0xF0, 0x7F, 0x00, 0x04, 0xE0, 0xFF, 0x01, 0x08, 0x00, 0x00, 0x00, 0x00,
            ];
            assert_eq!(ct_ref, ct);
            let ss_ref = [
                0x81, 0x67, 0xD3, 0x46, 0x01, 0x94, 0xC1, 0x16, 0xD9, 0xD5, 0xBF, 0x74, 0x9C, 0x34,
                0x35, 0x4F, 0x57, 0x98, 0xA6, 0xEA, 0x15, 0xD1, 0xCD, 0x6C, 0x4B, 0x67, 0xC0, 0x81,
                0x4A, 0xE8, 0xB4, 0x1C,
            ];
            assert_eq!(ss_ref, ss);
        }

        #[cfg(feature = "ntruhps2048677")]
        {
            let pk = [0u8; CRYPTO_PUBLICKEYBYTES];
            let mut rng = AesState::new();
            let mut ct = [0u8; CRYPTO_CIPHERTEXTBYTES];
            let mut ss = [0u8; CRYPTO_BYTES];

            crypto_kem_enc(&mut ct, &mut ss, &pk, &mut rng)?;

            let ct_ref = [
                0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x80, 0xFF, 0x03, 0x20, 0x00, 0x01, 0x00, 0x00,
                0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0xFF, 0x07, 0x00, 0x00, 0x00, 0x10,
                0x80, 0xFF, 0x03, 0xE0, 0xFF, 0x01, 0xF8, 0x3F, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFC,
                0x1F, 0x00, 0x00, 0x00, 0xC0, 0xFF, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0xFF,
                0x07, 0x00, 0x00, 0x00, 0xF0, 0x7F, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00,
                0xFE, 0x0F, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0x07, 0x00, 0x00, 0x00, 0x00, 0x80,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0xFF, 0xFF, 0x3F,
                0x00, 0x01, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFC, 0x1F, 0x00, 0x01, 0x08,
                0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFC, 0x1F, 0x00, 0x00, 0xF8, 0xFF, 0xFF, 0x03,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0x07, 0x00, 0x00, 0x00, 0x10, 0x80, 0x00,
                0x00, 0x20, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00,
                0x01, 0x00, 0x00, 0x00, 0x00, 0x10, 0x80, 0xFF, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x80, 0x00, 0xFC, 0x1F, 0x00, 0x00, 0xF8, 0x3F, 0x00, 0x00, 0x00,
                0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04,
                0x00, 0x00, 0x01, 0x08, 0xC0, 0xFF, 0x01, 0x00, 0x00, 0x00, 0xFC, 0x1F, 0x00, 0x00,
                0xF8, 0x3F, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00,
                0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0, 0xFF, 0xFF, 0xFF, 0xFF,
                0xFF, 0x03, 0x20, 0x00, 0xFF, 0xFF, 0x3F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x08, 0xC0, 0xFF, 0x03, 0x00, 0x00, 0x00, 0x00, 0xE0, 0xFF, 0x00, 0xF8,
                0x3F, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFC, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x02,
                0x00, 0x80, 0xFF, 0x07, 0xE0, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0xF0, 0xFF, 0xFF, 0x03, 0xE0, 0xFF,
                0x00, 0xF8, 0x3F, 0x00, 0x00, 0x10, 0x80, 0xFF, 0x03, 0xE0, 0xFF, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0xF8, 0x3F, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0xE0, 0xFF, 0xFF, 0x07, 0xC0, 0xFF, 0x01, 0x00, 0x00, 0x00, 0xFC,
                0x1F, 0x00, 0xFF, 0xFF, 0x3F, 0x00, 0x00, 0x00, 0x80, 0xFF, 0x03, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0xFE, 0x0F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xF8, 0x3F, 0x00,
                0x00, 0xF0, 0x7F, 0x00, 0xFC, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x01, 0x00, 0x00,
                0x00, 0xFC, 0x1F, 0x00, 0xFF, 0x0F, 0x00, 0x00, 0xFE, 0xFF, 0xFF, 0x00, 0x04, 0x00,
                0x00, 0x00, 0x08, 0x40, 0x00, 0x00, 0xF0, 0xFF, 0xFF, 0x03, 0x20, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x01, 0x08, 0xC0, 0xFF, 0x01,
                0x00, 0x80, 0xFF, 0x03, 0x20, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
                0xFC, 0x1F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x20, 0x00,
                0x00, 0x00, 0x00, 0x00, 0xFE, 0x0F, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0x07, 0x40,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x08, 0x40, 0x00, 0x00, 0xF0,
                0x7F, 0x00, 0x00, 0xE0, 0xFF, 0x00, 0x00, 0x40, 0x00, 0x00, 0xF0, 0x7F, 0x00, 0xFC,
                0x3F, 0x00, 0xFF, 0x07, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0xF8, 0x3F, 0x00, 0x00, 0x00, 0x80, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0xFC, 0xFF, 0xFF, 0x00, 0x00, 0x40, 0x00, 0xFE, 0x0F, 0x80,
                0x00, 0x00, 0xE0, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFC, 0x1F, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x02, 0x10, 0x00, 0x00, 0x00, 0xE0, 0xFF, 0xFF, 0x0F, 0x40, 0x00, 0x00,
                0xF0, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF8, 0x3F, 0x00, 0x00, 0x00, 0x80, 0xFF,
                0xFF, 0x1F, 0x00, 0x01, 0x00, 0xC0, 0xFF, 0x01, 0xF0, 0x7F, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x08, 0x40, 0x00, 0x00, 0xF0, 0x7F, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x40,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0x07, 0xC0, 0xFF, 0xFF, 0x0F,
                0x00, 0x00, 0xFC, 0xFF, 0xFF, 0x00, 0x08, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x02, 0x00, 0x80, 0xFF, 0x03, 0x00, 0x00, 0xFF, 0x07, 0x00, 0x00,
                0xFE, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x10, 0x80,
                0x00, 0xFC, 0x3F, 0x00, 0x00, 0xF8, 0x3F, 0x00, 0x02, 0xF0, 0x7F, 0x00, 0x00, 0x00,
                0x00, 0xFF, 0x0F, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x01, 0x08,
                0x00, 0x00, 0x00, 0xF0, 0x7F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02,
                0x10, 0x00, 0x00, 0x00, 0xE0, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xFE, 0xFF, 0x7F, 0x00,
                0x00, 0x20, 0x00, 0x00, 0x08, 0xC0, 0xFF, 0x01, 0x00, 0x00, 0x00, 0x00, 0xE0, 0xFF,
                0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0xF8, 0x3F,
                0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00,
                0x80, 0xFF, 0x03, 0xE0, 0xFF, 0x00, 0x00, 0x40, 0x00, 0x02, 0xF0, 0xFF, 0x00, 0x04,
                0xE0, 0xFF, 0x01, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x04, 0x00, 0x00, 0x01,
                0x08, 0x00, 0x00, 0x00, 0xF0, 0x7F, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0xF0, 0x7F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x40, 0x00, 0x00, 0x00, 0x00,
                0x00, 0xFC, 0x1F, 0x00, 0xFF, 0x07, 0x00, 0x00, 0x02, 0x00, 0x80, 0xFF, 0xFF, 0x3F,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x04, 0x00, 0x00, 0xFF, 0x07,
                0x40, 0x00, 0xFE, 0x0F, 0x80, 0xFF, 0x07, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
                0xF0, 0x7F, 0x00, 0x04, 0xE0, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF0, 0x7F, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0xE0, 0xFF,
                0x00, 0x00, 0x00, 0x00, 0x02, 0x00,
            ];
            assert_eq!(ct_ref, ct);
            let ss_ref = [
                0xEA, 0x5A, 0x85, 0x2E, 0x7A, 0x1B, 0x6F, 0x6C, 0x9F, 0xE8, 0x59, 0x97, 0xD4, 0x6D,
                0x51, 0x82, 0x27, 0xCD, 0x7A, 0x1E, 0x0C, 0x81, 0xA4, 0x72, 0x38, 0x55, 0x19, 0x33,
                0x03, 0x98, 0x19, 0x0E,
            ];
            assert_eq!(ss_ref, ss);
        }

        #[cfg(feature = "ntruhps4096821")]
        {
            let pk = [0u8; CRYPTO_PUBLICKEYBYTES];
            let mut rng = AesState::new();
            let mut ct = [0u8; CRYPTO_CIPHERTEXTBYTES];
            let mut ss = [0u8; CRYPTO_BYTES];

            crypto_kem_enc(&mut ct, &mut ss, &pk, &mut rng)?;

            let ct_ref = [
                0x01, 0x00, 0x00, 0xFF, 0x1F, 0x00, 0xFF, 0x1F, 0x00, 0xFF, 0x0F, 0x00, 0x00, 0x00,
                0x00, 0x00, 0xF0, 0xFF, 0xFF, 0x1F, 0x00, 0x00, 0xF0, 0xFF, 0x00, 0xF0, 0xFF, 0x00,
                0xF0, 0xFF, 0x01, 0xF0, 0xFF, 0x00, 0x00, 0x00, 0x01, 0xF0, 0xFF, 0x00, 0x10, 0x00,
                0x01, 0x10, 0x00, 0x01, 0x00, 0x00, 0xFF, 0x1F, 0x00, 0xFF, 0xFF, 0xFF, 0x00, 0x00,
                0x00, 0x01, 0xF0, 0xFF, 0x00, 0x10, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x1F, 0x00, 0x01,
                0xF0, 0xFF, 0x00, 0x10, 0x00, 0x00, 0x10, 0x00, 0xFF, 0x0F, 0x00, 0x00, 0xF0, 0xFF,
                0xFF, 0x1F, 0x00, 0x01, 0xF0, 0xFF, 0x01, 0x10, 0x00, 0x01, 0xF0, 0xFF, 0x01, 0xF0,
                0xFF, 0xFF, 0x0F, 0x00, 0x00, 0xF0, 0xFF, 0x00, 0x10, 0x00, 0xFF, 0x1F, 0x00, 0xFF,
                0x0F, 0x00, 0xFF, 0x1F, 0x00, 0x01, 0x10, 0x00, 0xFF, 0x1F, 0x00, 0x00, 0x00, 0x00,
                0x00, 0xF0, 0xFF, 0x00, 0x00, 0x00, 0x01, 0x10, 0x00, 0x01, 0x10, 0x00, 0x00, 0x00,
                0x00, 0xFF, 0x0F, 0x00, 0xFF, 0x0F, 0x00, 0x00, 0xF0, 0xFF, 0xFF, 0x1F, 0x00, 0x01,
                0x10, 0x00, 0xFF, 0x0F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0x1F, 0x00,
                0x01, 0x00, 0x00, 0xFF, 0x0F, 0x00, 0x01, 0xF0, 0xFF, 0x00, 0x10, 0x00, 0x00, 0x00,
                0x00, 0x01, 0x00, 0x00, 0xFF, 0x0F, 0x00, 0x00, 0x00, 0x00, 0xFF, 0x0F, 0x00, 0x00,
                0x10, 0x00, 0x01, 0x00, 0x00, 0x01, 0xF0, 0xFF, 0x00, 0x10, 0x00, 0x01, 0xF0, 0xFF,
                0xFF, 0x0F, 0x00, 0xFF, 0x1F, 0x00, 0xFF, 0x0F, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x0F,
                0x00, 0x01, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF, 0x01, 0xF0, 0xFF, 0x00, 0x10, 0x00, 0x00,
                0xF0, 0xFF, 0x00, 0x10, 0x00, 0xFF, 0xFF, 0xFF, 0x01, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF,
                0xFF, 0x0F, 0x00, 0x00, 0xF0, 0xFF, 0x00, 0x10, 0x00, 0xFF, 0x1F, 0x00, 0x00, 0x00,
                0x00, 0x01, 0xF0, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xF0, 0xFF, 0x01, 0x10, 0x00, 0xFF,
                0x0F, 0x00, 0xFF, 0x0F, 0x00, 0xFF, 0xFF, 0xFF, 0x01, 0xF0, 0xFF, 0x00, 0xF0, 0xFF,
                0xFF, 0x0F, 0x00, 0x01, 0x10, 0x00, 0xFF, 0x0F, 0x00, 0xFF, 0xFF, 0xFF, 0x00, 0x10,
                0x00, 0x01, 0x10, 0x00, 0xFF, 0x1F, 0x00, 0xFF, 0xFF, 0xFF, 0x01, 0x00, 0x00, 0x01,
                0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF0, 0xFF, 0x01, 0x00, 0x00, 0x01, 0x10, 0x00,
                0x01, 0xF0, 0xFF, 0xFF, 0x1F, 0x00, 0xFF, 0x1F, 0x00, 0x00, 0xF0, 0xFF, 0x00, 0x10,
                0x00, 0x01, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x10, 0x00, 0x01, 0x10, 0x00, 0xFF, 0x1F, 0x00, 0x00, 0x10, 0x00,
                0x00, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0xF0, 0xFF, 0xFF, 0x1F, 0x00, 0xFF, 0xFF,
                0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0x0F, 0x00, 0x00,
                0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x01, 0x10, 0x00, 0x00, 0x00, 0x00,
                0xFF, 0x0F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x01, 0x10, 0x00, 0xFF, 0x0F,
                0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0xFF, 0x1F, 0x00, 0x00, 0x00, 0x00, 0x01,
                0x00, 0x00, 0xFF, 0x0F, 0x00, 0x01, 0x00, 0x00, 0xFF, 0x1F, 0x00, 0x00, 0x10, 0x00,
                0x00, 0x00, 0x00, 0xFF, 0x1F, 0x00, 0x00, 0x00, 0x00, 0x01, 0xF0, 0xFF, 0x01, 0xF0,
                0xFF, 0x01, 0x10, 0x00, 0x00, 0x10, 0x00, 0x00, 0xF0, 0xFF, 0x01, 0xF0, 0xFF, 0x01,
                0x00, 0x00, 0x00, 0xF0, 0xFF, 0x00, 0xF0, 0xFF, 0xFF, 0x1F, 0x00, 0xFF, 0xFF, 0xFF,
                0xFF, 0xFF, 0xFF, 0x01, 0x00, 0x00, 0xFF, 0x0F, 0x00, 0xFF, 0x1F, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0xF0, 0xFF, 0xFF, 0x1F, 0x00, 0x00, 0xF0, 0xFF, 0x01,
                0x10, 0x00, 0xFF, 0x1F, 0x00, 0x01, 0xF0, 0xFF, 0x00, 0xF0, 0xFF, 0x01, 0x00, 0x00,
                0xFF, 0x0F, 0x00, 0x00, 0x00, 0x00, 0x01, 0x10, 0x00, 0xFF, 0x0F, 0x00, 0x01, 0x00,
                0x00, 0xFF, 0x1F, 0x00, 0x00, 0xF0, 0xFF, 0xFF, 0x0F, 0x00, 0xFF, 0x1F, 0x00, 0x00,
                0x10, 0x00, 0xFF, 0x0F, 0x00, 0xFF, 0x0F, 0x00, 0x01, 0x10, 0x00, 0xFF, 0x0F, 0x00,
                0xFF, 0x1F, 0x00, 0x01, 0x00, 0x00, 0xFF, 0x1F, 0x00, 0xFF, 0x1F, 0x00, 0xFF, 0x0F,
                0x00, 0x01, 0xF0, 0xFF, 0xFF, 0x0F, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x0F, 0x00, 0x01,
                0x00, 0x00, 0xFF, 0x1F, 0x00, 0xFF, 0x0F, 0x00, 0xFF, 0x1F, 0x00, 0xFF, 0x0F, 0x00,
                0xFF, 0xFF, 0xFF, 0x00, 0x10, 0x00, 0xFF, 0x1F, 0x00, 0x01, 0x10, 0x00, 0xFF, 0xFF,
                0xFF, 0x01, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF0, 0xFF, 0x01, 0x00, 0x00, 0x00,
                0x10, 0x00, 0xFF, 0xFF, 0xFF, 0x00, 0x10, 0x00, 0xFF, 0x1F, 0x00, 0x00, 0x10, 0x00,
                0x00, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x00, 0x10,
                0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01, 0x10, 0x00, 0x01, 0x00, 0x00, 0x01,
                0xF0, 0xFF, 0xFF, 0x0F, 0x00, 0x01, 0x10, 0x00, 0x00, 0xF0, 0xFF, 0xFF, 0x1F, 0x00,
                0xFF, 0x1F, 0x00, 0x01, 0x00, 0x00, 0xFF, 0x1F, 0x00, 0x01, 0x10, 0x00, 0xFF, 0x0F,
                0x00, 0x00, 0x00, 0x00, 0xFF, 0x1F, 0x00, 0xFF, 0x0F, 0x00, 0x01, 0xF0, 0xFF, 0x00,
                0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x01, 0x10, 0x00, 0xFF, 0x1F, 0x00, 0x01, 0xF0, 0xFF,
                0x01, 0x00, 0x00, 0x01, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10,
                0x00, 0xFF, 0x1F, 0x00, 0x01, 0xF0, 0xFF, 0x01, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x01,
                0x00, 0x00, 0x00, 0x10, 0x00, 0x01, 0x10, 0x00, 0xFF, 0x1F, 0x00, 0x01, 0x10, 0x00,
                0x00, 0x00, 0x00, 0xFF, 0x0F, 0x00, 0x00, 0xF0, 0xFF, 0xFF, 0x0F, 0x00, 0x00, 0xF0,
                0xFF, 0x01, 0x10, 0x00, 0x01, 0xF0, 0xFF, 0x01, 0x10, 0x00, 0x01, 0x00, 0x00, 0x01,
                0xF0, 0xFF, 0xFF, 0x0F, 0x00, 0xFF, 0x0F, 0x00, 0x01, 0xF0, 0xFF, 0x00, 0xF0, 0xFF,
                0x01, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x1F, 0x00, 0xFF, 0xFF, 0xFF, 0x00, 0xF0,
                0xFF, 0x00, 0x00, 0x00, 0x01, 0x10, 0x00, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x01,
                0xF0, 0xFF, 0xFF, 0x0F, 0x00, 0x00, 0x10, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00,
                0x00, 0xF0, 0xFF, 0xFF, 0x1F, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xF0,
                0xFF, 0x00, 0x00, 0x00, 0xFF, 0x0F, 0x00, 0xFF, 0x0F, 0x00, 0x01, 0x10, 0x00, 0xFF,
                0x0F, 0x00, 0x01, 0x10, 0x00, 0x00, 0x10, 0x00, 0xFF, 0x1F, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x01, 0xF0, 0xFF, 0xFF, 0x0F, 0x00, 0x01, 0x10, 0x00, 0xFF, 0xFF,
                0xFF, 0x00, 0xF0, 0xFF, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x00, 0x10, 0x00, 0xFF,
                0x1F, 0x00, 0x00, 0x10, 0x00, 0x00, 0xF0, 0xFF, 0xFF, 0x1F, 0x00, 0x00, 0xF0, 0xFF,
                0x00, 0x10, 0x00, 0x01, 0xF0, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xF0, 0xFF, 0xFF, 0x1F,
                0x00, 0xFF, 0x1F, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
                0xFF, 0xFF, 0x01, 0xF0, 0xFF, 0xFF, 0x1F, 0x00, 0xFF, 0xFF, 0xFF, 0x01, 0xF0, 0xFF,
                0x00, 0x10, 0x00, 0x01, 0x10, 0x00, 0x00, 0xF0, 0xFF, 0x00, 0x10, 0x00, 0x00, 0x00,
                0x00, 0x01, 0x00, 0x00, 0x01, 0xF0, 0xFF, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
                0xF0, 0xFF, 0x00, 0x10, 0x00, 0x00, 0xF0, 0xFF, 0x00, 0x10, 0x00, 0xFF, 0x0F, 0x00,
                0x00, 0xF0, 0xFF, 0x00, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xF0,
                0xFF, 0x01, 0x10, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0xFF, 0x1F, 0x00, 0xFF,
                0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00,
                0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
                0x00, 0x01, 0x00, 0x00, 0xFF, 0x1F, 0x00, 0x01, 0x00, 0x00, 0xFF, 0x0F, 0x00, 0x01,
                0xF0, 0xFF, 0x00, 0xF0, 0xFF, 0xFF, 0x1F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0xFF, 0x0F, 0x00, 0x00, 0x10, 0x00, 0x00, 0xF0, 0xFF, 0x01, 0x10, 0x00, 0x00, 0x00,
                0x00, 0xFF, 0xFF, 0xFF, 0x00, 0xF0, 0xFF, 0x00, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00,
                0xF0, 0xFF, 0x00, 0xF0, 0xFF, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x00, 0x10, 0x00,
                0xFF, 0x0F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0xFF, 0xFF, 0xFF, 0x01, 0x00,
                0x00, 0x00, 0x00, 0x00, 0xFF, 0x1F, 0x00, 0xFF, 0xFF, 0xFF, 0x00, 0x10, 0x00, 0x00,
                0x10, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF0, 0xFF,
                0x00, 0xF0, 0xFF, 0xFF, 0x0F, 0x00, 0x00, 0x10, 0x00, 0xFF, 0xFF, 0xFF,
            ];
            assert_eq!(ct_ref, ct);
            let ss_ref = [
                0xC1, 0x26, 0x61, 0x87, 0xFE, 0x6A, 0x8C, 0x5F, 0xF4, 0x78, 0x60, 0x25, 0xCE, 0xD3,
                0x16, 0xAE, 0xE7, 0x39, 0xFB, 0x3F, 0xEC, 0x5D, 0x59, 0x9F, 0xB3, 0xFD, 0x0E, 0xCD,
                0xA3, 0xE9, 0xD9, 0x1B,
            ];
            assert_eq!(ss_ref, ss);
        }

        #[cfg(feature = "ntruhrss701")]
        {
            let pk = [0u8; CRYPTO_PUBLICKEYBYTES];
            let mut rng = AesState::new();
            let mut ct = [0u8; CRYPTO_CIPHERTEXTBYTES];
            let mut ss = [0u8; CRYPTO_BYTES];

            crypto_kem_enc(&mut ct, &mut ss, &pk, &mut rng)?;

            let ct_ref = [
                0x01, 0xE0, 0xFF, 0xFF, 0x7F, 0x00, 0x20, 0x00, 0xFE, 0x3F, 0x00, 0x08, 0x00, 0xFF,
                0x1F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0xFE, 0x5F,
                0x00, 0xFC, 0xFF, 0xFF, 0x1F, 0x00, 0xFE, 0xBF, 0x00, 0xF0, 0xFF, 0x01, 0x20, 0x00,
                0xFC, 0xFF, 0xFF, 0x2F, 0x00, 0xFC, 0x7F, 0x00, 0x08, 0x00, 0xFF, 0xFF, 0xFF, 0x07,
                0x80, 0xFF, 0x1F, 0x00, 0x02, 0xC0, 0xFF, 0xFF, 0xFF, 0x02, 0xC0, 0xFF, 0x0B, 0x00,
                0xFF, 0x0F, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0xFF, 0x1F, 0x00, 0x00, 0x80, 0xFF,
                0x2F, 0x00, 0xFC, 0x7F, 0x00, 0x00, 0x00, 0xFF, 0x3F, 0x00, 0x04, 0x00, 0x00, 0xF0,
                0xFF, 0x01, 0x40, 0x00, 0x00, 0x00, 0xFE, 0x5F, 0x00, 0xFC, 0xFF, 0xFF, 0x1F, 0x00,
                0x00, 0xC0, 0xFF, 0x07, 0x00, 0x02, 0x00, 0x00, 0xF8, 0xFF, 0x00, 0xF0, 0xFF, 0x05,
                0x80, 0xFF, 0x07, 0x00, 0x00, 0x40, 0x00, 0x00, 0x80, 0xFF, 0x1F, 0x00, 0xFC, 0x3F,
                0x00, 0x00, 0x00, 0x02, 0xC0, 0xFF, 0x07, 0x80, 0xFF, 0x2F, 0x00, 0x00, 0x00, 0x00,
                0xF8, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF0, 0xFF, 0x03, 0x00, 0x00, 0xF8,
                0xFF, 0x02, 0xC0, 0xFF, 0x07, 0x80, 0x00, 0x00, 0x00, 0xFC, 0x3F, 0x00, 0x00, 0x00,
                0x02, 0x00, 0x00, 0xFC, 0xFF, 0xFF, 0x1F, 0x00, 0xFE, 0x7F, 0x00, 0x00, 0x00, 0x00,
                0xE0, 0xFF, 0x07, 0x80, 0xFF, 0x2F, 0x00, 0x00, 0xC0, 0xFF, 0x07, 0x00, 0x01, 0xE0,
                0xFF, 0x07, 0x00, 0x00, 0x00, 0x00, 0xFC, 0xBF, 0x00, 0x00, 0x00, 0x00, 0xC0, 0xFF,
                0x07, 0x80, 0xFF, 0x2F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFE, 0x1F, 0x00, 0x00,
                0x80, 0x00, 0x10, 0x00, 0xFE, 0xFF, 0xFF, 0x17, 0x00, 0xFE, 0x1F, 0x00, 0x04, 0x80,
                0xFF, 0x0F, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0xFE, 0x1F, 0x00, 0x00, 0x80, 0x00,
                0x10, 0x00, 0xFE, 0xFF, 0xFF, 0x0F, 0x00, 0x01, 0xC0, 0xFF, 0x0B, 0x80, 0xFF, 0x0F,
                0x00, 0x00, 0xC0, 0xFF, 0x0F, 0x00, 0xFF, 0x3F, 0x00, 0xFC, 0x7F, 0x01, 0xE0, 0xFF,
                0x03, 0xC0, 0xFF, 0x17, 0x00, 0xFE, 0x3F, 0x00, 0xFC, 0x7F, 0x01, 0xF0, 0xFF, 0x01,
                0x40, 0x00, 0xF0, 0xFF, 0x01, 0xE0, 0xFF, 0x0B, 0x00, 0xFF, 0x1F, 0x00, 0x02, 0x80,
                0xFF, 0x17, 0x00, 0xFF, 0xFF, 0xFF, 0x07, 0x80, 0xFF, 0x1F, 0x00, 0x02, 0x00, 0x00,
                0x00, 0x00, 0xFE, 0x3F, 0x00, 0xFC, 0xFF, 0x00, 0x10, 0x00, 0xFE, 0xFF, 0xFF, 0x07,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x10, 0x00, 0x00, 0xC0, 0xFF, 0x07, 0x00,
                0xFF, 0x1F, 0x00, 0x08, 0x00, 0xFF, 0x0F, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x01,
                0x00, 0x00, 0x00, 0x80, 0xFF, 0x1F, 0x00, 0xFC, 0x7F, 0x00, 0x08, 0x00, 0xFE, 0x3F,
                0x00, 0x00, 0x80, 0xFF, 0x0F, 0x00, 0x04, 0xC0, 0xFF, 0x07, 0x00, 0x01, 0xE0, 0xFF,
                0xFF, 0x7F, 0x01, 0xE0, 0xFF, 0x01, 0x80, 0x00, 0xF0, 0xFF, 0x01, 0x00, 0x00, 0xFC,
                0x7F, 0x01, 0xF0, 0xFF, 0x01, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0xFC, 0xFF,
                0xFF, 0x0F, 0x00, 0x02, 0x40, 0x00, 0xF0, 0xFF, 0x00, 0x40, 0x00, 0xF8, 0x7F, 0x01,
                0x00, 0x00, 0x00, 0x80, 0xFF, 0x07, 0x00, 0x02, 0x00, 0x00, 0xF8, 0x7F, 0x01, 0xF0,
                0xFF, 0x01, 0x00, 0x00, 0x08, 0x00, 0xFF, 0x3F, 0x00, 0xFC, 0x7F, 0x00, 0x10, 0x00,
                0x00, 0x80, 0xFF, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xE0, 0xFF, 0x03,
                0x00, 0x00, 0xF8, 0xFF, 0x01, 0x00, 0x00, 0xFC, 0x7F, 0x01, 0xF0, 0xFF, 0xFF, 0x3F,
                0x00, 0x10, 0x00, 0xFF, 0xFF, 0xFF, 0x0B, 0x80, 0xFF, 0x1F, 0x00, 0xFE, 0xFF, 0xFF,
                0x07, 0x00, 0x01, 0x00, 0x00, 0x00, 0x80, 0xFF, 0x2F, 0x00, 0x00, 0xC0, 0xFF, 0x07,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0xFC, 0x3F, 0x00, 0x00, 0x00,
                0x02, 0xC0, 0xFF, 0x03, 0x00, 0x01, 0xF0, 0xFF, 0x01, 0x00, 0x00, 0x00, 0x00, 0x01,
                0xC0, 0xFF, 0x07, 0x80, 0x00, 0xE0, 0xFF, 0x03, 0x00, 0x00, 0x00, 0x00, 0xFF, 0x1F,
                0x00, 0x04, 0x00, 0x00, 0x10, 0x00, 0xFE, 0xFF, 0xFF, 0x07, 0x00, 0x00, 0x40, 0x00,
                0x00, 0x80, 0xFF, 0x1F, 0x00, 0x00, 0xC0, 0xFF, 0xFF, 0xFF, 0x00, 0x20, 0x00, 0x00,
                0x00, 0x00, 0x10, 0x00, 0x00, 0xC0, 0xFF, 0x0F, 0x00, 0xFF, 0xFF, 0xFF, 0x0B, 0x00,
                0x00, 0x00, 0x00, 0xFC, 0xBF, 0x00, 0xF0, 0xFF, 0x01, 0xE0, 0xFF, 0x0B, 0x80, 0xFF,
                0x0F, 0x00, 0x02, 0x00, 0x00, 0xF0, 0xFF, 0x02, 0xE0, 0xFF, 0xFF, 0x7F, 0x00, 0x20,
                0x00, 0x00, 0x00, 0x00, 0xF0, 0xFF, 0x00, 0x40, 0x00, 0x00, 0x00, 0xFF, 0x1F, 0x00,
                0x02, 0x80, 0xFF, 0x07, 0x00, 0x01, 0x20, 0x00, 0xF8, 0x7F, 0x01, 0xE0, 0xFF, 0x05,
                0x00, 0x00, 0xF8, 0xFF, 0xFF, 0x5F, 0x00, 0xFC, 0xFF, 0x00, 0x00, 0x00, 0xFE, 0x3F,
                0x00, 0x00, 0x00, 0x01, 0xC0, 0xFF, 0x03, 0x00, 0x01, 0x00, 0x00, 0xFE, 0x3F, 0x00,
                0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0xFF, 0x2F, 0x00, 0x00, 0x80, 0xFF, 0x07,
                0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0xFF, 0x17, 0x00,
                0x00, 0xC0, 0xFF, 0x07, 0x80, 0xFF, 0x1F, 0x00, 0x00, 0xC0, 0xFF, 0x17, 0x00, 0xFF,
                0xFF, 0xFF, 0x0B, 0x80, 0xFF, 0xFF, 0xFF, 0x01, 0x40, 0x00, 0xF8, 0xFF, 0x01, 0x20,
                0x00, 0xF8, 0x7F, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0xF8, 0xFF, 0x01, 0xE0, 0xFF,
                0x0B, 0x80, 0xFF, 0x1F, 0x00, 0xFC, 0x3F, 0x00, 0x08, 0x00, 0xFF, 0x1F, 0x00, 0x04,
                0x80, 0x00, 0xF0, 0xFF, 0x03, 0x00, 0x00, 0x00, 0x00, 0xFE, 0x1F, 0x00, 0x00, 0x00,
                0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0xFE, 0x1F, 0x00, 0x04, 0x00, 0x00,
                0x10, 0x00, 0xFE, 0x3F, 0x00, 0xF8, 0xFF, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE0,
                0xFF, 0x05, 0xC0, 0xFF, 0xFF, 0xFF, 0x02, 0xC0, 0xFF, 0x07, 0x80, 0xFF, 0x2F, 0x00,
                0xFC, 0x7F, 0x00, 0x00, 0x00, 0xFF, 0x3F, 0x00, 0x00, 0x80, 0xFF, 0x2F, 0x00, 0xFE,
                0xFF, 0xFF, 0x07, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0xFF, 0x0F, 0x00, 0x04, 0x00,
                0x00, 0xF0, 0xFF, 0x00, 0x00, 0x00, 0x08, 0x80, 0xFF, 0xFF, 0xFF, 0x01, 0x40, 0x00,
                0xF8, 0xFF, 0x00, 0x20, 0x00, 0x04, 0x00, 0xFF, 0x0F, 0x00, 0x04, 0x80, 0xFF, 0x0F,
                0x00, 0x00, 0x20, 0x00, 0xFC, 0xFF, 0xFF, 0x1F, 0x00, 0x00, 0x40, 0x00, 0xF0, 0xFF,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xE0, 0xFF, 0x01, 0x80, 0x00, 0x00, 0x00, 0xFE,
                0x1F, 0x00, 0x04, 0x00, 0x00, 0x10, 0x00, 0x00, 0x80, 0xFF, 0x0F, 0x00, 0xFF, 0x3F,
                0x00, 0x00, 0x00, 0x00, 0xF0, 0xFF, 0x05, 0xC0, 0xFF, 0xFF, 0xFF, 0x01, 0xE0, 0xFF,
                0x03, 0x00, 0x00, 0x00, 0x00, 0x04, 0x80, 0xFF, 0x17, 0x00, 0xFF, 0x3F, 0x00, 0x00,
                0x80, 0xFF, 0x1F, 0x00, 0xFC, 0xBF, 0x00, 0xF8, 0xFF, 0xFF, 0x1F, 0x00, 0x04, 0x80,
                0xFF, 0x2F, 0x00, 0xFC, 0x7F, 0x00, 0xF8, 0xFF, 0x02, 0xC0, 0xFF, 0x0B, 0x00, 0xFF,
                0x1F, 0x00, 0x00, 0xC0, 0xFF, 0x17, 0x00, 0x00, 0x00, 0x00, 0xF8, 0x7F, 0x00, 0x10,
                0x00, 0x02, 0x80, 0xFF, 0x0F, 0x00, 0x00, 0xE0, 0xFF, 0x07, 0x00, 0x00, 0xF0, 0xFF,
                0x05, 0x00, 0x00, 0xF8, 0xFF, 0x00, 0x00, 0x00, 0xFC, 0x7F, 0x01, 0xF0, 0xFF, 0xFF,
                0x3F, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0xFC, 0xFF, 0xFF, 0x2F, 0x00, 0xFE, 0x3F,
                0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0xFC, 0xFF, 0xFF, 0x1F, 0x00, 0x00, 0xC0, 0xFF,
                0x0F, 0x00, 0x00, 0x20, 0x00, 0xFC, 0xFF, 0xFF, 0x0F, 0x00, 0x04, 0x00, 0x00, 0xF0,
                0xFF, 0x02, 0xE0, 0xFF, 0xFF, 0x7F, 0x01, 0x00, 0x00, 0xFC, 0x7F, 0x00, 0x00, 0x00,
                0x01, 0xE0, 0xFF, 0x07, 0x80, 0xFF, 0x0F, 0x00, 0x00, 0x40, 0x00, 0xF8, 0xFF, 0x00,
                0xE0, 0xFF, 0x07, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF8, 0xFF, 0xFF, 0x1F,
                0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF0, 0xFF, 0x02, 0xE0, 0xFF,
                0xFF, 0x7F, 0x01, 0x00,
            ];
            assert_eq!(ct_ref, ct);
            let ss_ref = [
                0x56, 0x59, 0x33, 0x56, 0x9C, 0x40, 0xC0, 0xA5, 0x30, 0x15, 0x84, 0x1D, 0x5A, 0x9B,
                0x9F, 0x35, 0x7A, 0xC9, 0xE0, 0xD2, 0xBC, 0x81, 0xC4, 0x7B, 0x59, 0x97, 0xE1, 0x97,
                0xFD, 0x40, 0x99, 0x8A,
            ];
            assert_eq!(ss_ref, ss);
        }

        Ok(())
    }

    #[test]
    fn test_dec_kat1() -> R {
        #[cfg(feature = "ntruhps2048509")]
        {
            let ct = [
                0xB9, 0x34, 0x87, 0x2B, 0x04, 0x69, 0x34, 0x9C, 0xC3, 0xC8, 0x8A, 0xE5, 0x7A, 0xF9,
                0x8D, 0x6E, 0x23, 0x30, 0xE6, 0x66, 0xCE, 0x88, 0x9E, 0x37, 0x07, 0x52, 0x32, 0x24,
                0x89, 0x1C, 0x06, 0x85, 0x93, 0x3F, 0x03, 0x14, 0xB2, 0x8E, 0x1D, 0x6E, 0x1C, 0x35,
                0x59, 0xC8, 0x69, 0x5E, 0xF2, 0x6A, 0xB6, 0xBA, 0x3A, 0xF7, 0x01, 0xF6, 0x3F, 0xE2,
                0xB0, 0x64, 0x2E, 0x89, 0x6D, 0x17, 0xF7, 0x1E, 0xEA, 0x86, 0xF8, 0xC1, 0xAD, 0x18,
                0xC2, 0xB4, 0xB5, 0xF7, 0xF4, 0xD2, 0xF4, 0xEC, 0xA4, 0x91, 0x55, 0x0D, 0x09, 0x97,
                0x17, 0x78, 0x35, 0x60, 0xDF, 0xC4, 0xCA, 0xE9, 0xB7, 0x43, 0x51, 0xE7, 0x1A, 0x4B,
                0x1A, 0x2B, 0xE0, 0x01, 0xB8, 0xC5, 0xFF, 0x69, 0xAC, 0xF2, 0x5E, 0x8A, 0xEB, 0x01,
                0x03, 0x9C, 0x15, 0xC7, 0xA2, 0x9B, 0xAE, 0x63, 0xBA, 0x67, 0x49, 0xDD, 0x6C, 0xB8,
                0x3B, 0x3C, 0x26, 0x25, 0x8F, 0x55, 0xF1, 0x14, 0x29, 0xD9, 0x1F, 0xE9, 0x0C, 0x43,
                0x88, 0xB5, 0x23, 0x7B, 0x0A, 0x2E, 0xB7, 0x14, 0x3A, 0x31, 0xA9, 0xE5, 0x0C, 0xA1,
                0x60, 0x82, 0xC6, 0x1D, 0x1B, 0x53, 0x1A, 0xEA, 0xC9, 0x06, 0x0D, 0x5D, 0x7D, 0xB7,
                0x5B, 0x0D, 0xFB, 0x7A, 0x58, 0x69, 0x67, 0xB0, 0x47, 0x4F, 0x9E, 0x54, 0xD6, 0x91,
                0x67, 0xF9, 0xF1, 0xC1, 0x46, 0x11, 0x89, 0x94, 0x1C, 0x6F, 0xBC, 0xB0, 0xFB, 0xEA,
                0x0F, 0xC9, 0x75, 0x5E, 0x3D, 0xE6, 0x09, 0xD7, 0x85, 0xC9, 0xB8, 0xD0, 0x71, 0xFD,
                0x6A, 0x90, 0xC8, 0x27, 0xED, 0x0D, 0x83, 0x50, 0xB5, 0x07, 0x05, 0x2C, 0x8C, 0x2C,
                0x43, 0x66, 0x3F, 0x81, 0xCF, 0x36, 0x84, 0xD4, 0x28, 0x41, 0xB5, 0x1D, 0xC7, 0x07,
                0x85, 0x98, 0x18, 0xBC, 0xBB, 0x53, 0x46, 0x2D, 0x1F, 0x13, 0xEC, 0xA5, 0x41, 0xD1,
                0x06, 0xCA, 0x5B, 0x75, 0x6A, 0x6F, 0xF4, 0x82, 0xE6, 0xE0, 0xF6, 0x55, 0xA0, 0xDF,
                0x7F, 0xBB, 0x55, 0x84, 0x30, 0xE9, 0x5F, 0xBB, 0xB6, 0xD5, 0x8E, 0xF0, 0x7A, 0xE6,
                0xBE, 0x2A, 0x24, 0xB0, 0x77, 0x19, 0x43, 0xB9, 0xE6, 0x81, 0x39, 0x94, 0xA0, 0xA0,
                0x3D, 0x4C, 0xF7, 0x60, 0x29, 0xAB, 0xA1, 0xCB, 0x1E, 0xA3, 0x90, 0x46, 0x9F, 0x94,
                0x4D, 0x7A, 0x23, 0xB5, 0x0B, 0x81, 0x63, 0xE0, 0xB4, 0x28, 0x1E, 0x2B, 0x76, 0x9E,
                0x8E, 0x36, 0xA3, 0xC1, 0x63, 0x85, 0xCE, 0xAF, 0x94, 0x81, 0xB6, 0xC1, 0x5C, 0xF6,
                0x02, 0xC7, 0x9A, 0xED, 0xA0, 0xCB, 0xC2, 0x62, 0x78, 0xCC, 0x90, 0x44, 0x48, 0xB3,
                0x60, 0xB1, 0xB3, 0x2C, 0x59, 0x5E, 0xA7, 0xE1, 0x3E, 0xA7, 0x2D, 0x5D, 0xEE, 0x3A,
                0x0B, 0xBF, 0xE8, 0x01, 0x48, 0x1E, 0x27, 0x61, 0x39, 0xB1, 0x66, 0x3A, 0xC1, 0xF7,
                0xE7, 0xA2, 0xAD, 0x87, 0xF6, 0x17, 0xF9, 0x03, 0xDD, 0x4C, 0xA3, 0xD2, 0x4D, 0x88,
                0x77, 0x02, 0x43, 0x5F, 0xBD, 0xEB, 0x14, 0x9F, 0x55, 0x95, 0xF2, 0x76, 0xA5, 0xF1,
                0x6C, 0x6E, 0x3B, 0x5B, 0xFD, 0x63, 0xF1, 0xFC, 0xE5, 0xF8, 0x32, 0x19, 0x77, 0x9C,
                0xCB, 0x14, 0xB3, 0xA3, 0x4C, 0xEE, 0x18, 0xC5, 0xDA, 0x18, 0x63, 0x59, 0x6B, 0x40,
                0x54, 0xFA, 0x26, 0x3A, 0x6C, 0x94, 0x64, 0xEC, 0xD2, 0x46, 0xE8, 0xEC, 0xEE, 0x8A,
                0x9D, 0x21, 0x42, 0x00, 0x42, 0x55, 0xBA, 0xFB, 0x05, 0x9D, 0x68, 0x7E, 0x82, 0xC7,
                0xF5, 0xB2, 0x50, 0xE0, 0xAF, 0x38, 0x8C, 0xF6, 0x5F, 0x35, 0x8E, 0xC5, 0x38, 0x33,
                0xB8, 0xDF, 0xF6, 0xA0, 0x95, 0x2D, 0x44, 0x3C, 0x18, 0x92, 0x9A, 0x58, 0x89, 0xF4,
                0xA0, 0x35, 0x2F, 0x3F, 0x01, 0xBF, 0x51, 0x2F, 0xE3, 0x73, 0x60, 0xB5, 0xEC, 0xD7,
                0xCA, 0xF4, 0x05, 0x53, 0x01, 0xB6, 0x6B, 0x35, 0x2A, 0x58, 0xA1, 0x99, 0xE7, 0x93,
                0x64, 0xB6, 0x95, 0x71, 0xAC, 0xD6, 0x44, 0x02, 0xDC, 0x82, 0xF0, 0xDB, 0x00, 0x2B,
                0xC5, 0x67, 0x59, 0x57, 0x87, 0xE9, 0x66, 0xAC, 0xBD, 0x8E, 0xB0, 0xD7, 0x3A, 0x49,
                0x4A, 0x66, 0x21, 0x79, 0x67, 0xEC, 0xA7, 0xC1, 0x1F, 0xE4, 0x80, 0xFB, 0xB3, 0x42,
                0xD2, 0xFD, 0xA7, 0x3C, 0xCC, 0x2C, 0xBD, 0xB9, 0x15, 0x81, 0x5A, 0x4F, 0x81, 0x5F,
                0x61, 0x74, 0x9D, 0x4F, 0x78, 0x12, 0xE8, 0x92, 0x48, 0xAA, 0x84, 0xE9, 0xFC, 0x23,
                0x30, 0x11, 0x33, 0x81, 0xC0, 0x5D, 0x05, 0xD7, 0x81, 0x8E, 0xB8, 0x88, 0xCD, 0x60,
                0x31, 0x15, 0xBF, 0x60, 0x9A, 0xA0, 0xDF, 0x47, 0xBD, 0x13, 0x62, 0x45, 0x64, 0x93,
                0x27, 0x2F, 0xAF, 0xB2, 0xDF, 0xF0, 0x5F, 0x3B, 0xFA, 0x9B, 0xCC, 0x59, 0x1D, 0x2A,
                0x98, 0x93, 0x54, 0xE4, 0x02, 0x2F, 0x61, 0x1D, 0x03, 0xFB, 0x00, 0x1F, 0xA2, 0x38,
                0x78, 0xC4, 0x23, 0xE2, 0x79, 0x4A, 0x97, 0x4D, 0xDC, 0x34, 0x88, 0xCD, 0x97, 0x93,
                0x80, 0x52, 0xFE, 0x71, 0xCF, 0xC5, 0xF9, 0x3E, 0xAF, 0x96, 0xF6, 0xB6, 0x70, 0x09,
                0x4C, 0x0D, 0x3B, 0x29, 0x43, 0x40, 0xE5, 0xC7, 0x3A, 0x3C, 0xBA, 0xB8, 0x06, 0xA9,
                0x80, 0x3A, 0x8A, 0x40, 0xDA, 0x78, 0x8C, 0x7E, 0x99, 0x7B, 0xD2, 0x1B, 0x02,
            ];
            let sk = [
                0xD0, 0x67, 0xD9, 0x8F, 0x00, 0x55, 0xE2, 0xC3, 0xDE, 0xEF, 0x10, 0x76, 0xBB, 0xB7,
                0x55, 0xAF, 0xD0, 0x65, 0x11, 0x2C, 0x85, 0xC4, 0x6E, 0x6C, 0x35, 0x06, 0x55, 0xD9,
                0x62, 0x50, 0x73, 0xE9, 0x4E, 0x45, 0x28, 0xC0, 0x53, 0xE7, 0x20, 0x20, 0x6C, 0xDB,
                0x78, 0x0A, 0x61, 0xAD, 0x51, 0x20, 0xC4, 0x98, 0xA4, 0xCD, 0x3E, 0x60, 0xC3, 0x34,
                0xD8, 0x70, 0x2F, 0x0F, 0x79, 0xD8, 0x14, 0x18, 0xAF, 0x95, 0x9C, 0xAB, 0x77, 0x22,
                0xEB, 0xDD, 0x30, 0xDF, 0x0A, 0x47, 0x9E, 0x14, 0x03, 0xD3, 0x37, 0xAD, 0x92, 0x3E,
                0x5D, 0xBC, 0xED, 0x4E, 0xE0, 0x48, 0x39, 0x02, 0x0F, 0x77, 0x51, 0xC0, 0x75, 0xA4,
                0x89, 0x2A, 0xAB, 0x19, 0xDC, 0x7B, 0xD2, 0xAE, 0x91, 0x27, 0xB2, 0x41, 0x83, 0x1D,
                0x18, 0x37, 0xB9, 0xE2, 0x24, 0x30, 0xCE, 0x19, 0x25, 0x9F, 0x78, 0xA5, 0xC5, 0xBF,
                0xD0, 0xE5, 0x36, 0x5E, 0x5B, 0xAE, 0xE5, 0x29, 0x18, 0xCC, 0xAB, 0xE5, 0x83, 0xB5,
                0x0F, 0x00, 0x12, 0x6D, 0x04, 0x2C, 0xBC, 0xDC, 0x38, 0x62, 0x99, 0xEE, 0x68, 0xEE,
                0x4E, 0xEC, 0xD8, 0x60, 0xE6, 0x13, 0x14, 0x22, 0xAE, 0xE6, 0xB2, 0x00, 0x73, 0x37,
                0x58, 0xC8, 0x2D, 0xC4, 0xBE, 0x23, 0x4F, 0x6C, 0x46, 0x35, 0x6F, 0x01, 0x7E, 0x7D,
                0x07, 0xE4, 0x5B, 0xDC, 0xC9, 0xAC, 0xD1, 0xAF, 0xDD, 0xDD, 0x5C, 0x91, 0x38, 0x80,
                0x52, 0xCE, 0x0A, 0x53, 0x6B, 0x92, 0x16, 0x06, 0xB5, 0xEA, 0xA6, 0x90, 0x2C, 0xA4,
                0xEF, 0x39, 0x14, 0x26, 0x11, 0xF9, 0xBB, 0x56, 0x6D, 0x4F, 0xD7, 0x54, 0x14, 0x43,
                0x55, 0x0D, 0x8C, 0x60, 0xB2, 0xF6, 0x99, 0x10, 0x73, 0xC9, 0xA1, 0xF8, 0x04, 0x3A,
                0xA4, 0x69, 0xE9, 0x12, 0x7B, 0x6A, 0x2F, 0x5E, 0xBD, 0x96, 0x80, 0x0D, 0x24, 0xB8,
                0xB9, 0xD5, 0x86, 0x1F, 0x11, 0x96, 0xB1, 0xC8, 0x05, 0x67, 0x6E, 0xA4, 0x67, 0x8B,
                0x12, 0x43, 0x73, 0x8F, 0x9A, 0xE9, 0x7A, 0xC5, 0x68, 0xBC, 0x5B, 0x6A, 0xE0, 0x78,
                0xC7, 0xBD, 0x4F, 0x61, 0x56, 0x81, 0x72, 0x08, 0x61, 0x3C, 0x8B, 0x59, 0xCF, 0x7B,
                0x7A, 0xF2, 0xA8, 0xFB, 0xE9, 0x63, 0x7C, 0x5B, 0x76, 0x43, 0x4E, 0x65, 0xF1, 0xB3,
                0x61, 0xA4, 0xF6, 0xBB, 0xF7, 0xE1, 0xFD, 0xE4, 0x7D, 0xD6, 0xDD, 0x2A, 0x53, 0x20,
                0x1D, 0x91, 0x0E, 0x62, 0xC3, 0xE1, 0x1A, 0xB0, 0xC2, 0x3D, 0x88, 0x77, 0xDE, 0xC7,
                0x1D, 0x27, 0xF7, 0x81, 0x46, 0xEE, 0xD3, 0x99, 0xA4, 0xE0, 0x63, 0xBE, 0xC0, 0xC1,
                0x75, 0xE3, 0xAB, 0x41, 0x8E, 0x08, 0xED, 0x21, 0x87, 0x5D, 0xA2, 0x5F, 0xA6, 0x7B,
                0xC0, 0x36, 0x43, 0xD3, 0xB7, 0x0B, 0x7C, 0x1A, 0x11, 0x8C, 0x15, 0x3B, 0x3A, 0xF5,
                0x8E, 0xD5, 0xC4, 0xCD, 0x1A, 0x7F, 0x32, 0x36, 0x4D, 0x13, 0x34, 0xA4, 0x4D, 0x34,
                0x99, 0x10, 0x4F, 0xAA, 0xDA, 0x55, 0x55, 0xB8, 0x66, 0x94, 0x77, 0x2E, 0xF0, 0x06,
                0x5F, 0xC6, 0x73, 0xB0, 0x75, 0x76, 0xB5, 0x30, 0x85, 0xC2, 0x71, 0xAD, 0xB3, 0x06,
                0x5A, 0x6C, 0x2A, 0x5D, 0x9E, 0x77, 0x4A, 0xBE, 0x94, 0xAF, 0xBE, 0xB8, 0xB6, 0x01,
                0xAB, 0x6F, 0x0B, 0x61, 0x9A, 0x7F, 0x85, 0x58, 0xD1, 0xA8, 0x24, 0xFF, 0x4F, 0xB9,
                0x46, 0xE8, 0x5C, 0x6D, 0x1B, 0x9C, 0x8F, 0x55, 0x64, 0x4F, 0xD1, 0xB3, 0x2F, 0x8E,
                0x8C, 0x8F, 0x9C, 0xE5, 0x47, 0x5D, 0x90, 0x4F, 0x1B, 0x1E, 0x01, 0x76, 0x31, 0x11,
                0xF4, 0xD2, 0x0D, 0x22, 0x92, 0x79, 0x55, 0xCE, 0x6C, 0xF2, 0xAD, 0xC2, 0x93, 0xCA,
                0x4B, 0x86, 0x9F, 0x95, 0x10, 0x7E, 0x95, 0x0E, 0xED, 0x95, 0x08, 0x8A, 0xF8, 0x62,
                0xA6, 0x1E, 0xC9, 0xCD, 0xD2, 0xFA, 0x33, 0xE2, 0xC3, 0x27, 0x0C, 0xB8, 0xC8, 0x9C,
                0x3A, 0x42, 0xD7, 0xFB, 0x2D, 0x9C, 0xE3, 0xDA, 0xFD, 0x7B, 0x6B, 0xE9, 0xAC, 0x16,
                0xCB, 0xFD, 0xDA, 0x05, 0x0B, 0x25, 0x5B, 0xC5, 0x06, 0x4D, 0x4F, 0xEA, 0x8E, 0x80,
                0x22, 0x8A, 0x53, 0x3C, 0xE5, 0x3D, 0x6C, 0x49, 0xFF, 0x8F, 0xDA, 0xCB, 0xE1, 0x1D,
                0xB8, 0xE1, 0x62, 0xC0, 0xA1, 0xFA, 0xB9, 0x24, 0xFB, 0xBA, 0x41, 0xBA, 0xC4, 0xF8,
                0x7E, 0x59, 0xCB, 0xF9, 0xEC, 0x28, 0xB2, 0xCF, 0xE3, 0xC8, 0x5E, 0xE1, 0x4C, 0x1E,
                0xED, 0x15, 0xAE, 0x08, 0x24, 0xFD, 0x6B, 0xC3, 0x9F, 0xB4, 0x58, 0xF3, 0x9A, 0x8B,
                0xD0, 0x6B, 0xB0, 0x52, 0xF7, 0xA0, 0xE3, 0x7A, 0xA7, 0x3A, 0x50, 0xF1, 0x63, 0x0F,
                0xEA, 0x41, 0x8A, 0x9D, 0x70, 0x91, 0x6C, 0xF3, 0x68, 0xA5, 0x51, 0xC2, 0xFB, 0x47,
                0x41, 0x50, 0x55, 0x85, 0xA8, 0x6C, 0x31, 0x2A, 0xE7, 0x89, 0xC7, 0xDD, 0x90, 0x23,
                0x12, 0x48, 0x16, 0xF9, 0x20, 0x93, 0x17, 0xF3, 0xBC, 0xDE, 0x00, 0x8A, 0x9F, 0x30,
                0xC7, 0xDC, 0x9B, 0xBD, 0x04, 0x7E, 0x53, 0x58, 0xF7, 0xBB, 0xB6, 0x87, 0xD6, 0xD6,
                0x1C, 0x49, 0xE9, 0x56, 0xD6, 0xF7, 0x16, 0xDB, 0xB8, 0xDC, 0xF8, 0x44, 0x59, 0x76,
                0x56, 0x6F, 0x3A, 0x30, 0x80, 0x8F, 0xBA, 0x5F, 0x58, 0x40, 0x64, 0x19, 0x8A, 0x9D,
                0x8A, 0xCC, 0xB0, 0x42, 0x0D, 0x5B, 0x0E, 0x51, 0xA2, 0x1A, 0x1A, 0xED, 0x64, 0xDF,
                0xB3, 0x76, 0x62, 0xC6, 0x37, 0xF8, 0x3E, 0xFC, 0x54, 0x75, 0x29, 0x58, 0x8F, 0x5D,
                0x34, 0x6B, 0xA5, 0x98, 0xFD, 0xF3, 0x70, 0xA4, 0xD5, 0xFF, 0x8C, 0xA9, 0xBC, 0x2A,
                0x9E, 0x34, 0x05, 0xEF, 0x5A, 0xDC, 0xE1, 0x4E, 0xDC, 0x2A, 0x9A, 0x5B, 0x9F, 0x50,
                0x21, 0x6B, 0x37, 0x70, 0x6D, 0x56, 0x1D, 0x7E, 0x4C, 0xF6, 0x29, 0xEB, 0xAD, 0x71,
                0x2B, 0xE2, 0xCA, 0x57, 0x5F, 0x0C, 0x15, 0xB6, 0x5B, 0x46, 0xA7, 0x43, 0xD6, 0xDB,
                0xC5, 0x56, 0x7F, 0x54, 0x5B, 0x48, 0x5E, 0x2F, 0x09, 0x06, 0xA2, 0xD0, 0x5A, 0xA2,
                0xEA, 0x56, 0xFD, 0x3D, 0x97, 0x65, 0x96, 0x0B, 0xDA, 0x0D, 0x3A, 0x5B, 0x1A, 0x05,
                0x38, 0x26, 0x70, 0xD9, 0x01, 0xB5, 0x3B, 0x6F, 0x44, 0xAB, 0x92, 0x40, 0xDA, 0x7F,
                0x1B, 0xEA, 0xE8, 0x1C, 0xA8, 0xC2, 0x79, 0x0D, 0x1D, 0xF2, 0xA8, 0xB1, 0x88, 0x12,
                0xF1, 0xB2, 0xF7, 0x7C, 0x83, 0x8E, 0xC1, 0x78, 0x97, 0x58, 0xA5, 0xB6, 0xFF, 0x68,
                0x16, 0x21, 0xFB, 0x73, 0x58, 0x9A, 0xED, 0xEF, 0x43, 0x06, 0x2C, 0x0F, 0x99, 0xC1,
                0x5F, 0xB7, 0x79, 0xF6, 0x9E, 0xBC, 0xA3, 0x94, 0x44, 0xEB, 0x97, 0xBC, 0x88, 0x02,
                0xF6, 0x26, 0xC8, 0xB1, 0x5D, 0xE6, 0x66, 0xCC, 0xB5, 0xCA, 0x52, 0x61, 0x2E, 0xA1,
                0xD5, 0x94, 0xE8, 0xAE, 0xD6, 0x2C, 0x04, 0x1D, 0xAB, 0x03, 0x17, 0xF0, 0xCE, 0x5A,
                0x41, 0xCE, 0x76, 0x72, 0x95, 0x3D, 0x30, 0x1C, 0xFF, 0xD7, 0x10, 0xF8, 0x0B, 0xEA,
                0xC3, 0xF1, 0x9C, 0x0E, 0x96, 0xE6, 0x8C, 0xF8, 0xFD, 0xAB, 0x81,
            ];
            let mut ss1 = [0u8; CRYPTO_BYTES];

            crypto_kem_dec(&mut ss1, &ct, &sk)?;

            let ss_ref = [
                0x17, 0x6F, 0xDB, 0xB0, 0x09, 0xDD, 0x3F, 0x84, 0x8B, 0x36, 0x5A, 0xB7, 0xF1, 0x8D,
                0x9C, 0x0C, 0x91, 0x72, 0x19, 0x31, 0xC8, 0x59, 0x4C, 0x2C, 0x6F, 0x04, 0x3C, 0x86,
                0x00, 0x79, 0x1A, 0x6C,
            ];
            assert_eq!(ss_ref, ss1);
        }

        #[cfg(feature = "ntruhps2048677")]
        {
            let ct = [
                0xA1, 0xD9, 0xCE, 0x59, 0x58, 0xDA, 0xCA, 0x0F, 0x97, 0x99, 0xC9, 0xAE, 0x5D, 0x43,
                0x95, 0xCB, 0x63, 0x68, 0x37, 0x1B, 0xB9, 0xF9, 0x3F, 0x90, 0x6D, 0xAE, 0x55, 0x2F,
                0x52, 0x9B, 0x1C, 0xA5, 0xDF, 0xEF, 0x9B, 0xFC, 0xBC, 0xED, 0x6C, 0x74, 0xF3, 0xF1,
                0x89, 0xF1, 0xA1, 0x65, 0xEE, 0x30, 0xA9, 0x0B, 0x24, 0x6D, 0xC7, 0x29, 0xFA, 0x08,
                0xD1, 0x4D, 0xC8, 0x26, 0x47, 0xE2, 0xED, 0xDC, 0xFB, 0x9B, 0xB2, 0x42, 0xDB, 0x2E,
                0xDF, 0xBD, 0x70, 0xF5, 0x55, 0x24, 0x10, 0x28, 0xA8, 0x0A, 0x66, 0x6A, 0xDF, 0xE7,
                0xE5, 0x97, 0xE7, 0x1D, 0x4B, 0x43, 0xC0, 0x72, 0xED, 0xB5, 0x43, 0x90, 0xE9, 0xD8,
                0xF5, 0xA7, 0x6E, 0xB8, 0x5F, 0xB1, 0xCF, 0xE5, 0x62, 0xDC, 0xCF, 0xE3, 0x33, 0xC3,
                0xFA, 0x8E, 0xD2, 0x22, 0x29, 0x98, 0xA5, 0x83, 0x28, 0x4E, 0x75, 0xA6, 0x5E, 0x5F,
                0xFF, 0x9D, 0xF5, 0x1F, 0x05, 0x49, 0xB5, 0xC8, 0xC0, 0xC1, 0x2D, 0x40, 0x89, 0xA2,
                0x12, 0x1D, 0x9D, 0xA1, 0xDA, 0xCF, 0xEB, 0xA4, 0xE6, 0x91, 0x9A, 0xBB, 0xC5, 0x80,
                0xAA, 0xB4, 0xEF, 0xB9, 0xFA, 0x34, 0x1A, 0x07, 0x3C, 0xFF, 0xA4, 0x0E, 0x5E, 0xBC,
                0xCB, 0x7E, 0xD8, 0x4C, 0x85, 0x97, 0x14, 0x7B, 0xB8, 0x88, 0xB5, 0x0E, 0xBC, 0xB4,
                0xA0, 0x61, 0x8E, 0x81, 0xD7, 0x3A, 0x64, 0xBB, 0xC5, 0x46, 0x5D, 0xA6, 0x4A, 0x52,
                0x50, 0x98, 0xE2, 0x76, 0x47, 0x62, 0x7B, 0xFD, 0x0F, 0x91, 0xD5, 0xD6, 0x36, 0x75,
                0x4A, 0x43, 0x10, 0xF5, 0xD3, 0x4E, 0x3B, 0xBE, 0x35, 0x02, 0xBA, 0xDA, 0xAF, 0x0A,
                0x58, 0x98, 0x55, 0xF3, 0xAB, 0x64, 0xE1, 0xF2, 0xDB, 0xF7, 0xBE, 0x8F, 0x7A, 0x25,
                0x7B, 0x55, 0x3F, 0x4A, 0x1B, 0xC1, 0x2D, 0xAB, 0xA1, 0x63, 0x42, 0x0D, 0x17, 0x82,
                0xFE, 0x48, 0x30, 0xE0, 0x13, 0xE1, 0x73, 0x97, 0xFF, 0x01, 0x2F, 0x5F, 0x69, 0x88,
                0xBA, 0xF8, 0x9F, 0xDB, 0xE6, 0x1E, 0xB1, 0x15, 0x03, 0x00, 0x30, 0x9C, 0x00, 0xDE,
                0xF1, 0xF0, 0x55, 0xB9, 0x0D, 0x1F, 0x59, 0xAC, 0x34, 0x95, 0x28, 0xFE, 0x94, 0xEB,
                0xFA, 0x2D, 0xF6, 0xDB, 0xDF, 0xB8, 0x3A, 0x61, 0x78, 0xF5, 0xF2, 0x2E, 0x7C, 0xAD,
                0x8F, 0xC4, 0x4E, 0xD4, 0x6A, 0x9C, 0x58, 0x22, 0x35, 0xD6, 0x2A, 0xB5, 0xD7, 0x50,
                0xF8, 0x2C, 0xD7, 0x31, 0xAC, 0x5E, 0x30, 0xED, 0x69, 0x1D, 0xAF, 0xD0, 0xBF, 0x05,
                0x12, 0x57, 0x1C, 0x5A, 0x39, 0xC1, 0xF6, 0x27, 0x20, 0x2F, 0x46, 0xCB, 0x5E, 0xE7,
                0xF6, 0xCD, 0x8C, 0xD4, 0xD4, 0xFD, 0x3A, 0x68, 0xE6, 0x99, 0x6B, 0x9A, 0xB7, 0x7F,
                0xA6, 0x86, 0xAB, 0x59, 0x65, 0x70, 0xAE, 0xE2, 0x3A, 0xBB, 0x05, 0x21, 0xCD, 0x04,
                0x18, 0x57, 0x31, 0x14, 0x1A, 0xE1, 0xFC, 0xFB, 0xE6, 0x18, 0x92, 0x4A, 0xC7, 0x62,
                0xBF, 0xD0, 0x52, 0x4A, 0x51, 0x08, 0xD1, 0x44, 0x7A, 0xA4, 0x8A, 0xCC, 0x95, 0x6F,
                0xD0, 0x69, 0x70, 0xF8, 0xD7, 0xBF, 0x90, 0xC4, 0x24, 0xE6, 0xF1, 0x43, 0x79, 0x03,
                0x23, 0x3A, 0x91, 0xFE, 0xC3, 0xBE, 0x10, 0x64, 0xF8, 0x53, 0xC4, 0x7D, 0xF5, 0xE5,
                0x62, 0x2F, 0x68, 0xA3, 0x17, 0xD8, 0x4B, 0x77, 0x0A, 0x11, 0x49, 0x42, 0xDC, 0xF7,
                0x7C, 0x62, 0x00, 0xCE, 0x8D, 0x75, 0x57, 0x63, 0x3D, 0xD9, 0x60, 0x55, 0x81, 0xCC,
                0x3B, 0xFC, 0x79, 0x17, 0x00, 0x44, 0xF7, 0xE2, 0x12, 0xFF, 0xA6, 0x4A, 0xD8, 0x30,
                0x7D, 0x73, 0xF6, 0xD8, 0x92, 0x9A, 0x38, 0x54, 0xFA, 0xA3, 0x21, 0x29, 0x96, 0x2C,
                0x9F, 0xCF, 0xAF, 0x3E, 0xB1, 0xF2, 0xF3, 0xB4, 0xB8, 0x3D, 0xD7, 0xEB, 0x28, 0x25,
                0x92, 0x0D, 0x0B, 0xC9, 0xC9, 0xC9, 0xE6, 0xC8, 0xE1, 0x29, 0xCC, 0x09, 0x73, 0x0D,
                0x42, 0x1E, 0x40, 0x3A, 0xC5, 0xA9, 0x84, 0x3B, 0xE7, 0xF4, 0x0E, 0xEC, 0x25, 0xDE,
                0x14, 0x4B, 0xAF, 0x75, 0x69, 0x04, 0xFD, 0xF8, 0x6C, 0x86, 0x74, 0x20, 0xB9, 0xD6,
                0xF0, 0x68, 0x72, 0x03, 0x69, 0x52, 0x4F, 0x4C, 0xAA, 0x53, 0xCE, 0xA8, 0x5A, 0xF1,
                0x7F, 0x50, 0xCF, 0xCC, 0x55, 0x4D, 0xB2, 0x87, 0xB1, 0x5F, 0x13, 0x4E, 0x64, 0x2F,
                0xB1, 0x4B, 0x43, 0x1A, 0x00, 0x89, 0x08, 0x7C, 0xFA, 0x47, 0xAC, 0xD8, 0x98, 0x5B,
                0x56, 0x34, 0x6E, 0xB8, 0x59, 0xCD, 0x3D, 0x2C, 0x9F, 0x9B, 0x14, 0xE0, 0xF9, 0xE5,
                0xC8, 0x24, 0x8D, 0x75, 0x34, 0xD8, 0xDC, 0x50, 0x29, 0xFE, 0xD0, 0x1A, 0xB2, 0x01,
                0xB2, 0x7B, 0xF1, 0xB9, 0x03, 0xEA, 0x19, 0x85, 0x92, 0xDA, 0xA2, 0x7A, 0xCA, 0x9C,
                0xE9, 0xBD, 0xF2, 0x8B, 0x48, 0x31, 0xB2, 0x2C, 0x9E, 0x22, 0x0E, 0xB2, 0x5D, 0x55,
                0x62, 0x15, 0xC2, 0x34, 0x76, 0x1F, 0x0A, 0x96, 0xB0, 0x85, 0x6A, 0xE2, 0x6F, 0xD9,
                0x2E, 0x4B, 0x40, 0xF7, 0xE1, 0x3C, 0x3C, 0x35, 0x87, 0xC6, 0x0A, 0xF3, 0xD5, 0xC4,
                0x94, 0x30, 0xF7, 0x68, 0x94, 0x9D, 0x6E, 0x0C, 0x24, 0x39, 0xED, 0xE3, 0x40, 0x98,
                0x58, 0xCA, 0x40, 0x55, 0xE5, 0xCA, 0xC3, 0x25, 0xE2, 0x1C, 0xE1, 0xBF, 0xBB, 0x2B,
                0xBD, 0x3B, 0xBA, 0x13, 0x6E, 0x59, 0xC7, 0xE1, 0xDA, 0x25, 0x98, 0x97, 0x51, 0x89,
                0x11, 0x1F, 0x07, 0x30, 0xB6, 0x9E, 0x7E, 0x1B, 0x54, 0x77, 0xD1, 0x38, 0x84, 0xAD,
                0xD7, 0x0B, 0x3C, 0xA8, 0xA9, 0x55, 0x75, 0x62, 0x03, 0x47, 0xCF, 0x50, 0xAF, 0xA7,
                0x2E, 0xDE, 0x63, 0x5E, 0xE7, 0x0C, 0x62, 0xB5, 0x66, 0xF5, 0xB6, 0xD2, 0x03, 0x1F,
                0x3C, 0xDA, 0xBB, 0xE2, 0xDC, 0x95, 0x00, 0xDD, 0xEB, 0xE7, 0x60, 0xB4, 0xD8, 0x01,
                0x58, 0x83, 0x5C, 0x2B, 0x00, 0xE6, 0xBC, 0x51, 0x4F, 0x2A, 0x76, 0x2B, 0xD6, 0x99,
                0xEA, 0x97, 0xE7, 0x32, 0x4C, 0xFA, 0x5A, 0x9E, 0xEC, 0x11, 0x42, 0x63, 0x96, 0xDC,
                0x2E, 0x93, 0xC9, 0x43, 0xC2, 0x09, 0xAF, 0x15, 0x25, 0x49, 0x0E, 0xF3, 0xF1, 0xF0,
                0x5E, 0xFE, 0x79, 0x77, 0x31, 0xC9, 0xAF, 0xF6, 0x18, 0x58, 0xFB, 0x92, 0xF4, 0xD0,
                0xC9, 0xB3, 0x4E, 0xA6, 0xAA, 0x18, 0x87, 0x50, 0x1D, 0x7D, 0x96, 0x1F, 0x1E, 0x7D,
                0x12, 0x56, 0xF7, 0x7E, 0x84, 0xBB, 0xBF, 0x53, 0x81, 0xEF, 0xD5, 0x76, 0x10, 0xEC,
                0x4E, 0xCC, 0x39, 0x00, 0xB3, 0xA3, 0x77, 0xBE, 0x30, 0x2F, 0x0F, 0x63, 0x3F, 0x1C,
                0x6C, 0x0E, 0x38, 0xED, 0x93, 0x27, 0xC7, 0x43, 0xD2, 0xEC, 0x4D, 0x7A, 0x2B, 0x06,
                0x44, 0x9F, 0xAA, 0x5F, 0x9D, 0x5D, 0x15, 0x29, 0x08, 0x32, 0x5D, 0xC7, 0x7F, 0x7C,
                0x26, 0x74, 0x97, 0x5B, 0x66, 0x2D, 0x8A, 0xC7, 0xF6, 0x7C, 0x78, 0xD5, 0x8F, 0x74,
                0xC8, 0xC0, 0x18, 0xD7, 0x7B, 0xFA, 0xF4, 0x47, 0xCF, 0x55, 0xA4, 0x6F, 0x79, 0x05,
                0x99, 0xE9, 0x9C, 0xCA, 0xF4, 0x0E,
            ];
            let sk = [
                0xD0, 0x67, 0xD9, 0x8F, 0x00, 0x55, 0xE2, 0xC3, 0xDE, 0xEF, 0x10, 0x76, 0xBB, 0xB7,
                0x55, 0xAF, 0xD0, 0x65, 0x11, 0x2C, 0x85, 0xC4, 0x6E, 0x6C, 0x35, 0x06, 0x55, 0xD9,
                0x62, 0x50, 0x73, 0xE9, 0x4E, 0x45, 0x28, 0xC0, 0x53, 0xE7, 0x20, 0x20, 0x6C, 0xDB,
                0x78, 0x0A, 0x61, 0xAD, 0x51, 0x20, 0xC4, 0x98, 0xA4, 0xCD, 0x3E, 0x60, 0xC3, 0x34,
                0xD8, 0x70, 0x2F, 0x0F, 0x79, 0xD8, 0x14, 0x18, 0xAF, 0x95, 0x9C, 0xAB, 0x77, 0x22,
                0xEB, 0xDD, 0x30, 0xDF, 0x0A, 0x47, 0x9E, 0x14, 0x03, 0xD3, 0x37, 0xAD, 0x92, 0x3E,
                0x5D, 0xBC, 0xED, 0x4E, 0xE0, 0x48, 0x39, 0x02, 0x0F, 0x77, 0x51, 0xC0, 0x75, 0xA4,
                0x89, 0x2A, 0xAB, 0xBB, 0x0E, 0x00, 0x1A, 0xBA, 0x95, 0x51, 0x5E, 0x1B, 0x65, 0xC6,
                0x3C, 0x50, 0x3D, 0x97, 0xE5, 0x1E, 0xAF, 0xB9, 0xAC, 0x93, 0x0F, 0x5A, 0x1D, 0xBC,
                0xB7, 0xE9, 0x38, 0x8B, 0x5E, 0xF1, 0xE7, 0x5E, 0x89, 0x02, 0x0D, 0x1E, 0x51, 0x81,
                0xB4, 0xD1, 0x20, 0x60, 0xB6, 0xA8, 0xA8, 0x14, 0x3F, 0x27, 0x5E, 0x38, 0xD5, 0x59,
                0xD5, 0xF1, 0x8A, 0x15, 0x0F, 0x49, 0xE0, 0xCB, 0x73, 0xA2, 0x97, 0x2D, 0xB8, 0x4F,
                0x57, 0xDA, 0x46, 0xEF, 0x4D, 0x8A, 0xD1, 0x1F, 0x2B, 0x4F, 0x5A, 0x95, 0x26, 0xA7,
                0x79, 0x01, 0xBA, 0x60, 0x0F, 0xC6, 0xE9, 0xC9, 0xE6, 0xF0, 0x7A, 0xC2, 0xD9, 0xAD,
                0x24, 0x40, 0xBC, 0x2B, 0xB1, 0x71, 0x43, 0x9A, 0xED, 0x32, 0x96, 0x43, 0x5D, 0x3B,
                0x7F, 0x10, 0x2C, 0xE3, 0x13, 0x5D, 0x0D, 0x51, 0xC4, 0xDA, 0x05, 0x2E, 0x91, 0x63,
                0xC6, 0xA1, 0x64, 0x6E, 0xD8, 0x0F, 0xCA, 0x39, 0x09, 0x60, 0xE3, 0xC4, 0x02, 0x24,
                0x1F, 0x5F, 0xC0, 0x53, 0xBC, 0x57, 0x03, 0x55, 0x8A, 0xD9, 0xE9, 0xCC, 0xDB, 0x5A,
                0xC4, 0x28, 0x4B, 0x6A, 0x69, 0xD3, 0x6C, 0x04, 0x2E, 0x29, 0x7C, 0x9C, 0x48, 0xC6,
                0x1A, 0x85, 0x73, 0x5D, 0x8C, 0x02, 0x0D, 0x2F, 0x43, 0x77, 0x43, 0x68, 0xCC, 0x18,
                0x35, 0x38, 0xFF, 0xE6, 0xD7, 0x02, 0x10, 0x29, 0xA3, 0xE5, 0xF1, 0xEC, 0x56, 0xF5,
                0xEE, 0x9E, 0xCB, 0xC0, 0x9F, 0x5F, 0xE0, 0x92, 0xF8, 0x76, 0x57, 0xD3, 0x41, 0x2B,
                0x3F, 0x66, 0xF0, 0x5C, 0x99, 0xCA, 0x6F, 0x27, 0x04, 0xB4, 0x3F, 0xE5, 0x42, 0x2D,
                0x34, 0x2B, 0x48, 0x17, 0xCF, 0x3A, 0x51, 0xE6, 0x2F, 0xAE, 0x28, 0x21, 0xD7, 0xA0,
                0xCB, 0xA8, 0x35, 0x18, 0x79, 0x45, 0x11, 0x20, 0x05, 0x8B, 0xD8, 0x63, 0xD1, 0x89,
                0xEC, 0x80, 0x75, 0x04, 0x89, 0xF8, 0xDE, 0xB4, 0x35, 0xD3, 0x9E, 0xA1, 0x03, 0x47,
                0xF7, 0xD0, 0xD0, 0x41, 0x9D, 0x4A, 0x9A, 0xDD, 0x5B, 0x24, 0x40, 0x2D, 0xAA, 0x99,
                0xB5, 0x5C, 0x94, 0xC5, 0xB7, 0x1C, 0x5B, 0x35, 0xDD, 0x7F, 0xAC, 0x20, 0x4B, 0x98,
                0x4D, 0xE3, 0x39, 0x69, 0x0D, 0x6D, 0xF8, 0x9E, 0x27, 0x6E, 0x85, 0x5A, 0x63, 0x56,
                0x77, 0xBB, 0xEE, 0x38, 0x0A, 0x1C, 0x21, 0x75, 0x1B, 0x6D, 0x0F, 0x13, 0x14, 0x5F,
                0xE4, 0x76, 0x9A, 0x01, 0x03, 0x72, 0x0A, 0xFF, 0x6F, 0x14, 0x18, 0x1A, 0x42, 0x49,
                0x23, 0x03, 0x55, 0x3E, 0xFF, 0x46, 0x90, 0xBC, 0x39, 0x7D, 0x4D, 0x43, 0xBF, 0xBF,
                0x30, 0x9A, 0x58, 0x90, 0x6F, 0x32, 0xFD, 0x54, 0x9A, 0x6B, 0xE5, 0x5A, 0xAD, 0x7E,
                0x12, 0x29, 0xCA, 0xAA, 0xD7, 0x18, 0x21, 0x4C, 0x6F, 0xEE, 0x36, 0x90, 0xA4, 0x55,
                0x32, 0xB2, 0xC4, 0xE8, 0x2B, 0xC4, 0xB5, 0x4F, 0x2D, 0x8C, 0xA4, 0x30, 0x20, 0xA2,
                0xF1, 0x66, 0x30, 0x68, 0x60, 0xD3, 0x01, 0xD5, 0x63, 0xF0, 0x1C, 0xA3, 0xBD, 0xE5,
                0x9A, 0x3F, 0xC7, 0x9D, 0xBC, 0x7A, 0x62, 0x20, 0xFF, 0x77, 0x93, 0x3E, 0x2E, 0x5B,
                0x59, 0xBC, 0x93, 0x8C, 0xB8, 0x6B, 0x31, 0xD7, 0x41, 0x63, 0xB4, 0xF3, 0x7A, 0x0B,
                0xD9, 0xC5, 0x80, 0x82, 0xF1, 0xED, 0xC8, 0x9F, 0xD5, 0x2D, 0xDC, 0xB6, 0x33, 0xC0,
                0x01, 0x0C, 0x01, 0xE0, 0x99, 0x3D, 0xF8, 0x67, 0xFE, 0x63, 0x29, 0x9C, 0x01, 0x0A,
                0xF0, 0x76, 0x78, 0x99, 0xB4, 0x5B, 0x05, 0xB9, 0xBD, 0x8A, 0x76, 0x48, 0x7D, 0x58,
                0x28, 0x39, 0x1D, 0xE2, 0x57, 0x9B, 0x7B, 0x65, 0x3C, 0xC0, 0xCC, 0x4C, 0x1E, 0x69,
                0x4F, 0x71, 0x80, 0xF3, 0xB1, 0x20, 0x9F, 0x27, 0xC6, 0xE7, 0xEB, 0xE3, 0x67, 0xE6,
                0x27, 0x16, 0x67, 0xB2, 0xE1, 0x64, 0xC9, 0x4D, 0x0D, 0xDE, 0x36, 0xE3, 0x25, 0x52,
                0xDD, 0xC0, 0x13, 0x55, 0xB3, 0x6D, 0x86, 0x57, 0x2D, 0x82, 0x08, 0x57, 0x3F, 0x0A,
                0x2C, 0x30, 0x91, 0x21, 0x32, 0x8E, 0x56, 0xA6, 0xA5, 0x0F, 0x0B, 0x6A, 0xCE, 0x20,
                0x2C, 0x37, 0x6C, 0x93, 0x73, 0x76, 0xF6, 0x42, 0xF2, 0x69, 0x27, 0x99, 0x23, 0x37,
                0x36, 0x81, 0xDE, 0x9F, 0x9F, 0xC7, 0x37, 0x5E, 0x75, 0xF3, 0x43, 0x33, 0x33, 0xD5,
                0x8D, 0x72, 0x1F, 0xBB, 0xEC, 0x08, 0xCB, 0x99, 0xDD, 0x82, 0x97, 0x7F, 0x4B, 0x3D,
                0xF9, 0x07, 0xD6, 0x40, 0xEF, 0xDC, 0x19, 0xDD, 0xD5, 0xC2, 0xC6, 0x37, 0x4B, 0x6F,
                0xB2, 0xC6, 0x73, 0x4B, 0x85, 0x08, 0xDD, 0xBC, 0xB8, 0x2E, 0x69, 0x26, 0xAD, 0x12,
                0x6F, 0x34, 0x73, 0x9A, 0x62, 0xE5, 0xB0, 0xF6, 0x6F, 0x3C, 0xA2, 0x0C, 0x42, 0xE8,
                0xDC, 0x0E, 0xDF, 0x4B, 0xB9, 0x72, 0x27, 0x33, 0x4E, 0x5C, 0x3A, 0x90, 0xD1, 0xA5,
                0xA3, 0x92, 0xD7, 0x34, 0x1B, 0x37, 0x07, 0xF0, 0x24, 0x59, 0xD1, 0x2F, 0xFF, 0x12,
                0x08, 0xD0, 0xC8, 0x8A, 0x2B, 0x7C, 0x61, 0x78, 0x88, 0x7E, 0x42, 0xD8, 0x2A, 0xD7,
                0x02, 0x1C, 0x09, 0x30, 0xCF, 0x67, 0x23, 0x33, 0x2D, 0xD2, 0x0B, 0xB4, 0xAB, 0x81,
                0x42, 0xBB, 0xF5, 0xD7, 0x96, 0x9F, 0x2B, 0x96, 0x20, 0x58, 0xE7, 0x8F, 0xC8, 0x71,
                0xA3, 0x3E, 0x53, 0x00, 0x26, 0xDE, 0x24, 0x90, 0x40, 0x5D, 0xD4, 0x40, 0x84, 0xD1,
                0x30, 0x0C, 0xF9, 0x87, 0x02, 0xEB, 0x53, 0x82, 0x40, 0x2F, 0x81, 0xA4, 0xBD, 0xD3,
                0xB8, 0x51, 0x4F, 0x0E, 0x76, 0x38, 0x26, 0x0A, 0xB4, 0x18, 0xA4, 0x67, 0x3F, 0x7D,
                0xF3, 0x59, 0x5B, 0x96, 0x17, 0x0B, 0x41, 0xBB, 0x87, 0x59, 0x5A, 0x41, 0xC5, 0xBE,
                0xF5, 0x6C, 0xDD, 0xA8, 0x31, 0x78, 0x1E, 0x13, 0xCB, 0xEB, 0x3C, 0xAE, 0x75, 0x6C,
                0x6C, 0xDC, 0x73, 0xBC, 0x2A, 0xF8, 0x2F, 0x7E, 0xA1, 0xD1, 0x82, 0x1B, 0x2F, 0x55,
                0x97, 0xA7, 0x56, 0x13, 0x67, 0xE1, 0x4A, 0xA5, 0x13, 0xBA, 0x9F, 0xC7, 0x13, 0x82,
                0x38, 0x10, 0xB8, 0xC7, 0xA5, 0xD7, 0xAA, 0xB8, 0x1E, 0x5F, 0xFF, 0x62, 0x49, 0xF8,
                0x03, 0x42, 0xC7, 0xE5, 0x73, 0xA6, 0xC6, 0x0C, 0x8E, 0x82, 0xCF, 0x39, 0x94, 0x7B,
                0xC8, 0x20, 0xF0, 0xC7, 0x71, 0x0C, 0xB3, 0x20, 0x1C, 0x17, 0x89, 0x1E, 0xEF, 0xCD,
                0x89, 0xE3, 0xB1, 0xCC, 0x50, 0x65, 0x99, 0x3B, 0x4D, 0x48, 0xBE, 0xC3, 0xF2, 0x98,
                0xC4, 0xF3, 0xEB, 0x23, 0x12, 0x5A, 0x4E, 0xEC, 0xF5, 0xD1, 0xA2, 0x9D, 0xCD, 0xFC,
                0x60, 0xC9, 0x6B, 0x40, 0xF5, 0x84, 0x99, 0x6D, 0x0B, 0x91, 0x15, 0x4A, 0xFD, 0xDF,
                0xBA, 0x2C, 0x19, 0xDA, 0x34, 0xD8, 0x23, 0x89, 0xF7, 0x36, 0x1F, 0xCB, 0xBA, 0xC6,
                0x9F, 0x2A, 0x88, 0x33, 0x65, 0x39, 0xB0, 0x20, 0x40, 0x3F, 0x3A, 0x19, 0xB9, 0x2D,
                0x5B, 0xB6, 0x56, 0x10, 0x11, 0x6C, 0xED, 0x92, 0x15, 0x1C, 0xE2, 0xD1, 0x76, 0xC2,
                0x77, 0x78, 0xB0, 0x8A, 0x7D, 0x4A, 0xAF, 0x6D, 0xA4, 0x74, 0xD1, 0xB3, 0x3C, 0xA0,
                0xC1, 0x39, 0x05, 0x48, 0x3B, 0x0A, 0xC8, 0x4C, 0x53, 0x51, 0xFC, 0x3B, 0x7A, 0xBE,
                0xCD, 0x57, 0xDA, 0xDC, 0x12, 0x6A, 0x62, 0x10, 0xC4, 0x52, 0x88, 0xF5, 0x69, 0x57,
                0x7A, 0x52, 0xF2, 0x99, 0x19, 0x2B, 0x1B, 0x6F, 0x41, 0x4B, 0x29, 0x23, 0x71, 0x9E,
                0x78, 0x38, 0x53, 0x2E, 0x4D, 0xA6, 0xC3, 0x5B, 0xAF, 0xDE, 0x56, 0x20, 0xCF, 0xD2,
                0xAB, 0x75, 0x1D, 0x46, 0x52, 0xDB, 0x3D, 0x06, 0x24, 0x77, 0xB5, 0xC9, 0x62, 0x63,
                0x01, 0x23, 0xD6, 0x80, 0x16, 0xD8, 0xD7, 0xAB, 0xE3, 0x25, 0xEF, 0x4D, 0x20, 0x11,
                0x8F, 0x21, 0x7C, 0x74, 0x66, 0x6C, 0xB1, 0x42, 0x1E, 0x74, 0x56, 0x95, 0x8F, 0x25,
                0xD2, 0x34, 0x94, 0x29, 0xB1, 0x24, 0x32, 0x80, 0xCD, 0xB0, 0xD0, 0xF2, 0x2E, 0x8F,
                0xDE, 0xC6, 0xE9, 0x1E, 0x51, 0x45, 0xD4, 0xA0, 0x08, 0x5C, 0x07, 0x7E, 0xAE, 0x0E,
                0xB3, 0xB5, 0xCD, 0x73, 0xDA, 0xF3, 0xD1, 0xF0, 0x5C, 0x29, 0x70, 0xF3, 0xE9, 0x62,
                0xB6, 0x65, 0xA6, 0x04, 0xDC, 0xB5, 0x88, 0x87, 0xA0, 0x83, 0xCC, 0x5A, 0x91, 0x1F,
                0x40, 0xEF, 0x6B, 0x78, 0x00, 0x9B, 0x14, 0x31, 0xC1, 0x5C, 0xCE, 0x02, 0xB8, 0xC0,
                0x84, 0x1D, 0xAF, 0x4D, 0x39, 0x75, 0x0F, 0xD7, 0x3E, 0xB8, 0xA5, 0x00, 0x42, 0xBC,
                0xA6, 0x13, 0x9F, 0x76, 0x44, 0x81, 0xFC, 0x96, 0xCE, 0xB4, 0x68, 0xCD, 0xAC, 0x15,
                0x76, 0xAE,
            ];
            let mut ss1 = [0u8; CRYPTO_BYTES];

            crypto_kem_dec(&mut ss1, &ct, &sk)?;

            let ss_ref = [
                0x49, 0xAC, 0x4D, 0x5D, 0x16, 0x34, 0xC6, 0xAF, 0xFA, 0x5A, 0x08, 0xC2, 0xB2, 0x28,
                0xEC, 0x80, 0x6D, 0x78, 0x70, 0xB1, 0x51, 0x79, 0x90, 0x72, 0x86, 0x63, 0xD2, 0xD8,
                0xBB, 0xC1, 0x84, 0xF2,
            ];
            assert_eq!(ss_ref, ss1);
        }

        #[cfg(feature = "ntruhps4096821")]
        {
            let ct = [
                0xC3, 0x1A, 0x29, 0x03, 0x4A, 0xEE, 0xF0, 0x46, 0x97, 0x18, 0x54, 0x9D, 0xFF, 0x4F,
                0x83, 0x80, 0x23, 0xA1, 0x9A, 0xBA, 0x55, 0xBF, 0xB2, 0x49, 0xB4, 0x05, 0x9D, 0x37,
                0xF8, 0x79, 0x21, 0x9B, 0x0A, 0x22, 0xF1, 0x7D, 0xAB, 0x3F, 0xF1, 0x6B, 0xA0, 0x5B,
                0x3D, 0xFD, 0x44, 0xE9, 0xFD, 0x43, 0x11, 0x8A, 0x41, 0xD4, 0x31, 0x7E, 0xA3, 0x54,
                0x07, 0xAA, 0xE2, 0x47, 0xE3, 0x1E, 0x90, 0xCD, 0x9E, 0x96, 0xBC, 0x11, 0x79, 0xC3,
                0x37, 0xE4, 0x5C, 0x34, 0x3D, 0xC7, 0xC7, 0x83, 0xDF, 0x7C, 0x33, 0x0F, 0x83, 0x29,
                0x88, 0xF0, 0xDF, 0xCA, 0x92, 0x33, 0xE2, 0x84, 0x2A, 0x45, 0xDB, 0xFB, 0xAD, 0xF2,
                0xF6, 0xBE, 0x8B, 0x4C, 0xCF, 0xC2, 0xE5, 0xAF, 0xB1, 0xBA, 0xBC, 0xC2, 0x2A, 0x78,
                0x08, 0xE5, 0xD6, 0x61, 0x67, 0xE6, 0x34, 0x10, 0xC3, 0x12, 0x7C, 0x08, 0x69, 0xA9,
                0x96, 0x9E, 0x5C, 0x10, 0x94, 0xAC, 0xCB, 0xFD, 0x2F, 0x6F, 0xFF, 0xED, 0xE2, 0x51,
                0x71, 0x87, 0xF7, 0x3A, 0x6A, 0x63, 0xB3, 0x9A, 0x9E, 0x09, 0xA1, 0x77, 0x8E, 0x2C,
                0xF1, 0x62, 0x92, 0x46, 0x5D, 0xA1, 0xFF, 0xDE, 0x8E, 0xEA, 0x45, 0xBF, 0x99, 0x61,
                0xC4, 0x8C, 0xEB, 0x3E, 0xF3, 0x70, 0x3C, 0x54, 0x69, 0x60, 0x54, 0x9F, 0xF1, 0x97,
                0xA0, 0x4B, 0x59, 0xC8, 0xA2, 0x52, 0x1C, 0x22, 0xA1, 0x71, 0xE8, 0xB2, 0x14, 0xC6,
                0x97, 0x48, 0x43, 0x83, 0x93, 0x9A, 0x3E, 0x4D, 0xCF, 0x54, 0x52, 0xE7, 0xE4, 0x17,
                0x90, 0x22, 0xDB, 0x52, 0x46, 0x34, 0x4C, 0x19, 0xFF, 0xDD, 0x25, 0x23, 0xF8, 0xB3,
                0xAA, 0xCD, 0xB3, 0x41, 0xA3, 0xD0, 0x8A, 0xEC, 0xB3, 0xAA, 0x42, 0xD4, 0x46, 0x55,
                0xB4, 0xF8, 0xB0, 0xDE, 0xB8, 0x88, 0x39, 0xEE, 0xE3, 0x86, 0xB5, 0x2D, 0xB3, 0xA6,
                0x35, 0x19, 0x14, 0x55, 0xF9, 0x28, 0xEA, 0xE7, 0x53, 0x5B, 0x45, 0xB5, 0xD1, 0x35,
                0x39, 0xB5, 0xFF, 0x03, 0x31, 0xD0, 0x9F, 0xE0, 0xD5, 0xFD, 0x0C, 0x45, 0x53, 0x69,
                0xB3, 0x86, 0x92, 0x12, 0xD5, 0xD5, 0x9E, 0x17, 0x94, 0xF6, 0x80, 0xDE, 0x7D, 0xC8,
                0x04, 0x26, 0xB3, 0x96, 0x57, 0x9E, 0x70, 0x6E, 0x35, 0x50, 0x12, 0x22, 0xAE, 0x1F,
                0x14, 0x1D, 0xD8, 0xEF, 0xE7, 0xC9, 0xF4, 0x79, 0x12, 0x25, 0xA4, 0x38, 0x62, 0xAF,
                0x0C, 0x42, 0x06, 0x7F, 0x69, 0xC5, 0xB6, 0x68, 0xA6, 0x9C, 0xFC, 0xC2, 0x7B, 0xA3,
                0x6A, 0xC2, 0x27, 0xD5, 0xC4, 0xA5, 0xFC, 0x7C, 0x87, 0x3F, 0xBA, 0xBB, 0x60, 0x4B,
                0x27, 0x14, 0x5A, 0x56, 0xD0, 0xFD, 0x07, 0x41, 0xB1, 0x9E, 0x93, 0x04, 0x16, 0xB2,
                0xAB, 0x1F, 0x21, 0x3D, 0x7B, 0x09, 0x44, 0xCA, 0x8B, 0xEA, 0xA8, 0x61, 0x44, 0x5A,
                0xBC, 0xF5, 0x9B, 0x21, 0x7F, 0x30, 0x9F, 0x6D, 0xA8, 0xA2, 0xE3, 0x9C, 0x58, 0x3F,
                0x0E, 0xE2, 0xAE, 0x01, 0x8C, 0x81, 0xB9, 0x37, 0x9E, 0xED, 0x51, 0xAC, 0x0B, 0x9A,
                0xD6, 0x27, 0xB3, 0x87, 0x0E, 0x5F, 0x89, 0x81, 0xA2, 0xA0, 0xF0, 0x6D, 0x15, 0x3D,
                0x23, 0xA0, 0x32, 0x57, 0xF6, 0x9A, 0x42, 0xC3, 0x65, 0x0F, 0x8C, 0xCB, 0xC0, 0xA1,
                0x08, 0x3D, 0x37, 0xA0, 0x24, 0x13, 0xFD, 0xF3, 0x64, 0x35, 0xA1, 0xFA, 0xE0, 0xE4,
                0x8B, 0x6E, 0xD0, 0x03, 0x30, 0xD0, 0x21, 0x06, 0x11, 0x9A, 0x83, 0x0A, 0x9C, 0x0E,
                0x0D, 0xF7, 0x6B, 0x0B, 0x96, 0xC1, 0x87, 0xB1, 0x72, 0x43, 0xBE, 0x11, 0xE3, 0x5A,
                0x2C, 0x84, 0x7F, 0x6F, 0xF3, 0xF6, 0xD8, 0x53, 0x7D, 0x42, 0xCB, 0x63, 0xE6, 0xE6,
                0xF8, 0x9B, 0x26, 0x2B, 0x7E, 0x7B, 0x92, 0x46, 0xA4, 0x9A, 0x33, 0x46, 0x22, 0xE6,
                0x5E, 0xC8, 0xB7, 0x78, 0x70, 0xAA, 0x34, 0x40, 0xEA, 0xC7, 0x70, 0x58, 0xBD, 0x13,
                0x73, 0xD1, 0x2A, 0xEC, 0x34, 0x34, 0x8A, 0x12, 0xE7, 0x4E, 0x65, 0x7F, 0xEB, 0x59,
                0x21, 0xF1, 0x96, 0x7E, 0xB1, 0x55, 0x48, 0xFF, 0x3C, 0xB9, 0x7D, 0xF8, 0xDD, 0xC7,
                0x35, 0xBD, 0x20, 0xD9, 0x9F, 0x93, 0x0D, 0xE9, 0x62, 0x17, 0x3B, 0x4E, 0xD0, 0x5C,
                0xF0, 0x3F, 0xDF, 0x53, 0xBD, 0x3C, 0xE7, 0x2E, 0x7A, 0x1B, 0x2A, 0x0C, 0xE2, 0xF7,
                0x1C, 0x64, 0x50, 0x2C, 0xD5, 0x30, 0xC5, 0x09, 0xCE, 0x16, 0x42, 0x04, 0x4E, 0x1B,
                0x4F, 0x3D, 0xC4, 0xBC, 0x79, 0xE9, 0xCB, 0x81, 0x8C, 0x93, 0xE6, 0x80, 0x35, 0x87,
                0x2E, 0x7C, 0x2A, 0x95, 0xA5, 0xC5, 0x0C, 0xDB, 0x84, 0x39, 0x75, 0x40, 0xCE, 0xB7,
                0x4E, 0xF4, 0x17, 0x1A, 0x1F, 0xB9, 0xC7, 0x75, 0x11, 0x3B, 0x68, 0xF8, 0xE4, 0x1A,
                0xCC, 0x11, 0xB6, 0x3F, 0xCF, 0x9D, 0x5A, 0x95, 0xCF, 0x84, 0x0F, 0xF5, 0x3F, 0x34,
                0x40, 0xAA, 0x4E, 0x7F, 0x06, 0x45, 0x44, 0xDE, 0x45, 0x28, 0xF0, 0x34, 0xA8, 0x28,
                0x87, 0x7B, 0x2A, 0x11, 0xCC, 0x5B, 0xB3, 0xDA, 0x6E, 0xCD, 0x10, 0x0E, 0xAD, 0x0D,
                0x83, 0x9C, 0xF8, 0xB3, 0x79, 0x04, 0xEE, 0xBD, 0x7D, 0xB2, 0xF9, 0xBD, 0xB9, 0x13,
                0xBF, 0x3A, 0x35, 0x1C, 0xAE, 0xA6, 0x01, 0x33, 0x0D, 0xDF, 0x97, 0xF9, 0xED, 0xE4,
                0x98, 0xFA, 0x5C, 0xC2, 0xB1, 0x09, 0x69, 0x9E, 0x43, 0xA9, 0xC9, 0x86, 0x99, 0xA1,
                0x87, 0xDB, 0x24, 0xA5, 0x95, 0x53, 0x7E, 0xB2, 0x2B, 0xA8, 0x9B, 0x58, 0xE0, 0x86,
                0x24, 0xD7, 0x41, 0x9A, 0x06, 0xF2, 0x3C, 0xA5, 0xAF, 0x59, 0x97, 0xEE, 0xE4, 0x95,
                0xC2, 0xBF, 0x3D, 0xEA, 0xDA, 0x61, 0x3F, 0x60, 0xFC, 0x3B, 0x9C, 0x34, 0x22, 0x50,
                0x54, 0x1B, 0xB0, 0x0B, 0x19, 0xB9, 0x6A, 0x6A, 0x65, 0xEC, 0x1A, 0x0D, 0xD4, 0xDD,
                0xCE, 0xD7, 0x34, 0xAE, 0x0E, 0x7F, 0xD1, 0x24, 0x4E, 0xA1, 0x4E, 0xFD, 0xC1, 0x50,
                0xBF, 0x3B, 0x55, 0xC0, 0xD5, 0xF2, 0x59, 0x64, 0x69, 0xC5, 0x75, 0xCC, 0xE3, 0x5B,
                0xF9, 0x87, 0x2B, 0x0D, 0x14, 0xDD, 0x22, 0x96, 0xBA, 0x76, 0x91, 0x5F, 0x6D, 0xA5,
                0xD3, 0x41, 0xD8, 0xA3, 0xB8, 0x08, 0xD3, 0x46, 0xA0, 0x7D, 0x6F, 0xBE, 0xA5, 0xDB,
                0x0C, 0x0C, 0xB0, 0x33, 0x2D, 0xD5, 0x24, 0x0D, 0xC0, 0x10, 0x56, 0x80, 0xEC, 0x3C,
                0x80, 0xA1, 0xE8, 0xE7, 0x27, 0x53, 0x37, 0xBD, 0xFA, 0x32, 0x03, 0xCC, 0xC3, 0x86,
                0xC9, 0x3D, 0x9C, 0x51, 0x27, 0x13, 0x29, 0x33, 0x99, 0x3F, 0xCB, 0x2D, 0x50, 0xF1,
                0x6D, 0xC0, 0x30, 0x2B, 0xC5, 0x5E, 0x8B, 0x72, 0xE0, 0x6C, 0x91, 0x42, 0x64, 0x29,
                0x90, 0x20, 0xC2, 0x53, 0x51, 0xF5, 0xE0, 0x7C, 0xE4, 0xE7, 0x26, 0xE9, 0x4D, 0x96,
                0x34, 0x7A, 0xF8, 0xF0, 0x4A, 0x5E, 0x59, 0xE2, 0x66, 0x24, 0x6E, 0x8D, 0x9C, 0xB2,
                0x2D, 0x7C, 0x6B, 0x8C, 0x42, 0x25, 0x9D, 0xAC, 0xEC, 0xF1, 0x0C, 0xF7, 0x58, 0x51,
                0xFB, 0x59, 0x09, 0xB7, 0xBF, 0xB4, 0x44, 0xC2, 0x6A, 0xC8, 0x01, 0xE9, 0xC1, 0x74,
                0x3D, 0x51, 0x4A, 0xBB, 0xE7, 0x3B, 0x69, 0x29, 0x8D, 0x8F, 0x4D, 0xED, 0x58, 0x76,
                0x27, 0xA4, 0x44, 0x71, 0x27, 0x0A, 0xFC, 0x43, 0xE5, 0x79, 0x1F, 0x71, 0xD6, 0x99,
                0xAE, 0x36, 0xC8, 0x98, 0xFE, 0xF4, 0x68, 0x6F, 0x94, 0x0C, 0x0C, 0x63, 0x36, 0x4A,
                0x04, 0x08, 0x40, 0xD8, 0x02, 0x2D, 0xC0, 0xA5, 0x3E, 0xA4, 0xF2, 0x46, 0x02, 0xCC,
                0x36, 0x0E, 0x72, 0x4A, 0x9F, 0x08, 0xF7, 0xC7, 0x61, 0x4D, 0x1B, 0xD9, 0x12, 0x08,
                0xC7, 0x89, 0xE1, 0x99, 0xB3, 0xEE, 0xFA, 0xBD, 0x1A, 0xC4, 0x07, 0xF2, 0x47, 0x88,
                0x92, 0xF1, 0x20, 0x28, 0x8A, 0xA0, 0x0C, 0x98, 0x25, 0xB1, 0x06, 0xAA, 0x49, 0x40,
                0xC4, 0x46, 0x37, 0x45, 0xFB, 0x0F, 0x73, 0x99, 0xB5, 0x70, 0xDB, 0xFD, 0x75, 0xAA,
                0xFD, 0xA2, 0x21, 0x7C, 0x9B, 0x52, 0x9F, 0x32, 0x7D, 0xBB, 0xC4, 0xF7, 0x92, 0x3B,
                0x17, 0xAC, 0xEE, 0x60, 0xDF, 0xB8, 0x10, 0x5C, 0x46, 0xC9, 0x30, 0x9E, 0xA1, 0x8C,
                0x84, 0x7F, 0x25, 0xD3, 0x25, 0x67, 0x26, 0xC4, 0x11, 0xA0, 0xE1, 0x73, 0xE7, 0x0F,
                0x79, 0xC0, 0x4C, 0x6F, 0xAD, 0x60, 0xF1, 0xE3, 0xC1, 0x52, 0xCE, 0x7D, 0xA9, 0x00,
                0xD9, 0x04, 0x4B, 0xC4, 0xE9, 0x7F, 0xD2, 0xFC, 0x83, 0xEE, 0x2E, 0xB0, 0x59, 0x41,
                0x6F, 0x5B, 0xBD, 0x6B, 0x24, 0x2A, 0xE8, 0x9A, 0xFF, 0x3A, 0x48, 0xD4, 0x9A, 0xCC,
                0x39, 0x7A, 0xD3, 0x08, 0x3C, 0x75, 0x8E, 0x1E, 0xDE, 0xD7, 0x9F, 0xE8, 0xEF, 0x4F,
                0xDE, 0x6F, 0x20, 0x8B, 0x1C, 0x6E, 0x96, 0x14, 0xBD, 0xA6, 0x80, 0x63, 0xD8, 0xC7,
                0xC9, 0x58, 0x42, 0xD8, 0xA1, 0x0E, 0x45, 0x38, 0x87, 0x6E, 0x20, 0x53, 0x03, 0x71,
                0x06, 0xAF, 0x38, 0xFF, 0xB1, 0x2F, 0x0A, 0xC9, 0x66, 0x64, 0x30, 0x63, 0xA0, 0xF1,
                0x67, 0x4C, 0x57, 0x20, 0x99, 0x33, 0xFD, 0x61, 0xFD, 0xD2, 0x5F, 0xA2, 0xCC, 0x5E,
                0x9F, 0x7D, 0x66, 0x24, 0x12, 0xD9, 0x98, 0xE0, 0x78, 0x8C, 0xDF, 0x87, 0xED, 0x76,
                0xF1, 0x15, 0xD8, 0xF5, 0x45, 0x24, 0x7D, 0xC8, 0x27, 0x41, 0xEE, 0xB6,
            ];
            let sk = [
                0xD0, 0x67, 0xD9, 0x8F, 0x00, 0x55, 0xE2, 0xC3, 0xDE, 0xEF, 0x10, 0x76, 0xBB, 0xB7,
                0x55, 0xAF, 0xD0, 0x65, 0x11, 0x2C, 0x85, 0xC4, 0x6E, 0x6C, 0x35, 0x06, 0x55, 0xD9,
                0x62, 0x50, 0x73, 0xE9, 0x4E, 0x45, 0x28, 0xC0, 0x53, 0xE7, 0x20, 0x20, 0x6C, 0xDB,
                0x78, 0x0A, 0x61, 0xAD, 0x51, 0x20, 0xC4, 0x98, 0xA4, 0xCD, 0x3E, 0x60, 0xC3, 0x34,
                0xD8, 0x70, 0x2F, 0x0F, 0x79, 0xD8, 0x14, 0x18, 0xAF, 0x95, 0x9C, 0xAB, 0x77, 0x22,
                0xEB, 0xDD, 0x30, 0xDF, 0x0A, 0x47, 0x9E, 0x14, 0x03, 0xD3, 0x37, 0xAD, 0x92, 0x3E,
                0x5D, 0xBC, 0xED, 0x4E, 0xE0, 0x48, 0x39, 0x02, 0x0F, 0x77, 0x51, 0xC0, 0x75, 0xA4,
                0x89, 0x2A, 0xAB, 0xBB, 0x0E, 0x00, 0x1A, 0xBA, 0x95, 0x51, 0x5E, 0x1B, 0x65, 0xC6,
                0x3C, 0x50, 0x3D, 0x97, 0xE5, 0x1E, 0xAF, 0xB9, 0xAC, 0x93, 0x0F, 0x5A, 0x1D, 0xBC,
                0xB7, 0xE9, 0x38, 0x8B, 0x5E, 0xF1, 0xE7, 0x5E, 0x89, 0x3E, 0x1B, 0x5E, 0x49, 0x95,
                0x5E, 0x7E, 0x3E, 0x4B, 0x38, 0x19, 0xDB, 0xA9, 0xEA, 0x9D, 0x23, 0xB0, 0x98, 0x6F,
                0x0D, 0xA2, 0xAD, 0xDF, 0xF0, 0xA3, 0x10, 0x59, 0x52, 0x67, 0x56, 0x08, 0x29, 0xBC,
                0xAC, 0x32, 0x5D, 0x3E, 0x5C, 0xD1, 0x75, 0x5D, 0x2B, 0xAF, 0xBA, 0x02, 0xD3, 0x5C,
                0x4E, 0x42, 0x6A, 0x93, 0xBD, 0xAF, 0x18, 0x4C, 0xB6, 0x6F, 0x78, 0xE2, 0x8F, 0x5C,
                0x7C, 0x8C, 0xA6, 0x6B, 0x87, 0x7B, 0x5D, 0x06, 0xDB, 0xAF, 0xA4, 0x85, 0x59, 0x73,
                0x66, 0xB2, 0x14, 0xEB, 0xD0, 0x85, 0x81, 0xA8, 0x21, 0x9F, 0xE9, 0x3D, 0xB6, 0xA3,
                0x67, 0x95, 0x98, 0xB5, 0x61, 0x00, 0x7E, 0x58, 0xB0, 0x7F, 0x13, 0xB5, 0x3A, 0xCB,
                0xAC, 0x3F, 0x1A, 0x42, 0xAA, 0xD8, 0x82, 0x87, 0x66, 0xB8, 0xB9, 0xB4, 0x3A, 0xA8,
                0x4E, 0xC6, 0xAD, 0x1D, 0x66, 0x29, 0x3F, 0x88, 0x22, 0x76, 0x7B, 0x2A, 0xC7, 0xD9,
                0xA1, 0xD8, 0xB4, 0x71, 0x59, 0x0B, 0x5B, 0x25, 0x30, 0x45, 0xBB, 0x5C, 0x62, 0x21,
                0x12, 0xA0, 0xF2, 0xC0, 0xC5, 0xDB, 0x11, 0xBF, 0x74, 0x9C, 0x82, 0xBC, 0x86, 0xB9,
                0x3F, 0x92, 0xDF, 0x31, 0x67, 0x06, 0x6C, 0x9D, 0x7B, 0x0F, 0xCC, 0xB9, 0xCD, 0xD4,
                0x53, 0x36, 0xDD, 0x74, 0x66, 0x9A, 0x65, 0x51, 0xD2, 0x05, 0x92, 0x28, 0xE2, 0xB0,
                0xDB, 0x81, 0x74, 0x75, 0x60, 0x99, 0xB2, 0x5F, 0xD4, 0x1D, 0x5B, 0xEB, 0x34, 0x2E,
                0x1D, 0x3A, 0x2E, 0x55, 0x66, 0xAC, 0xED, 0xAF, 0x9F, 0x6C, 0xF9, 0x82, 0x77, 0x89,
                0x43, 0x8F, 0xC8, 0xC2, 0xB9, 0x8C, 0x6C, 0x06, 0xEF, 0x88, 0x8D, 0xBF, 0x01, 0x67,
                0x73, 0xBD, 0x34, 0x0B, 0xB6, 0xCF, 0x89, 0x67, 0x3F, 0x07, 0xF9, 0x3F, 0x89, 0xDF,
                0xC8, 0x53, 0xC2, 0x40, 0x27, 0x39, 0xB7, 0x29, 0x1E, 0x8C, 0xD7, 0x65, 0xD7, 0x63,
                0x24, 0xDD, 0x1A, 0x1B, 0xE1, 0x76, 0x15, 0x61, 0xB2, 0x9E, 0x91, 0xED, 0xA3, 0x9C,
                0x49, 0xE1, 0x8B, 0x38, 0x2C, 0x3A, 0xC0, 0x83, 0x06, 0x18, 0xF5, 0x55, 0x0B, 0x6E,
                0xC7, 0x19, 0xF7, 0x0D, 0x51, 0x0F, 0x87, 0x49, 0xD2, 0x51, 0x8C, 0x27, 0x11, 0xD4,
                0x49, 0x25, 0xAA, 0x5C, 0xEE, 0x7D, 0xAF, 0x64, 0xA3, 0x41, 0x1C, 0x00, 0x48, 0x9E,
                0xD5, 0xA5, 0xDD, 0xC1, 0xE3, 0xF4, 0x69, 0x64, 0xB5, 0xE2, 0x57, 0x41, 0x7F, 0xEA,
                0x5C, 0x68, 0xF1, 0x56, 0x51, 0xD6, 0x66, 0x04, 0xC4, 0xA3, 0xC3, 0x76, 0xB7, 0xD7,
                0x3D, 0xC7, 0xAC, 0x10, 0x10, 0x4C, 0xCA, 0xA6, 0x94, 0x9A, 0xC3, 0x54, 0x6C, 0x14,
                0x0F, 0xCD, 0x39, 0x80, 0x66, 0xC2, 0xEB, 0xDE, 0x9B, 0x46, 0x75, 0x57, 0xCA, 0x00,
                0xC1, 0x25, 0xD3, 0x0F, 0xFC, 0xF2, 0xB3, 0x78, 0x87, 0x2B, 0x0B, 0x8C, 0x06, 0x4B,
                0x19, 0x17, 0xE1, 0x79, 0xDC, 0x13, 0xE0, 0x00, 0x71, 0x4B, 0x3E, 0x80, 0x58, 0xF5,
                0xF3, 0xA2, 0xF5, 0x82, 0xC5, 0x0A, 0x78, 0x3D, 0x3C, 0xF3, 0xF3, 0xA3, 0xF8, 0x56,
                0x4A, 0x19, 0xE7, 0x5C, 0xDA, 0x2A, 0x1F, 0x62, 0xC4, 0xAC, 0xFD, 0xF3, 0x4A, 0xFE,
                0x02, 0x1E, 0xA2, 0xD4, 0x24, 0x3A, 0x07, 0x37, 0xB7, 0x12, 0xA8, 0xC1, 0xED, 0x04,
                0x55, 0x8F, 0xB4, 0x00, 0xA6, 0xBA, 0x1D, 0x02, 0x06, 0xBC, 0x72, 0xEA, 0xD9, 0x52,
                0x59, 0xC6, 0x6C, 0xD2, 0xDC, 0xE5, 0x68, 0x3B, 0xC7, 0xCD, 0xBC, 0xF9, 0x25, 0x88,
                0xBE, 0xF6, 0x44, 0x99, 0xA8, 0xED, 0xE1, 0xE6, 0xF6, 0xC8, 0x6C, 0x97, 0x38, 0x8A,
                0x20, 0xAD, 0x2F, 0xF7, 0x13, 0x6A, 0x13, 0x5A, 0x89, 0x5A, 0xF1, 0x8E, 0x27, 0xD2,
                0xDE, 0x58, 0x23, 0xD3, 0xC5, 0xA1, 0x74, 0x15, 0x28, 0x2C, 0x6B, 0x1B, 0xDD, 0x59,
                0xC4, 0xC4, 0xD9, 0x4D, 0x94, 0xD5, 0x21, 0xAF, 0x15, 0x1A, 0xC7, 0x0C, 0x08, 0x54,
                0xFA, 0x3C, 0x18, 0x07, 0x9F, 0x30, 0xF9, 0x25, 0x96, 0x7E, 0xE0, 0x03, 0x25, 0xFC,
                0xD6, 0x62, 0x9A, 0xD3, 0x5F, 0x0F, 0xC5, 0xE1, 0x5A, 0xE7, 0xB1, 0xE4, 0x3C, 0x38,
                0x07, 0x09, 0x87, 0xEA, 0xD6, 0x28, 0x79, 0x21, 0x99, 0x2E, 0x70, 0x65, 0xA1, 0x70,
                0xB3, 0x0E, 0x1A, 0xD7, 0x79, 0x4E, 0x7E, 0x86, 0x60, 0xE2, 0xF9, 0xC6, 0x10, 0x8A,
                0x53, 0xE1, 0x77, 0x43, 0xEB, 0x27, 0x88, 0x76, 0xC9, 0x20, 0xF2, 0x3D, 0x9B, 0xE6,
                0xFE, 0xA0, 0xCD, 0xA7, 0xDA, 0x05, 0x88, 0x15, 0x65, 0x05, 0x98, 0x29, 0xA8, 0x4F,
                0xC0, 0xF2, 0x33, 0x55, 0x48, 0x61, 0x13, 0x08, 0x88, 0x59, 0x5C, 0xC0, 0x50, 0xFB,
                0xEE, 0xB7, 0xE7, 0x18, 0x9F, 0x02, 0x38, 0x5A, 0x81, 0x45, 0x4F, 0x9D, 0xEF, 0xF0,
                0xA8, 0x9B, 0x13, 0xCC, 0xAB, 0xF2, 0x97, 0x0C, 0xD3, 0x95, 0xDD, 0x0D, 0xED, 0xC8,
                0xC3, 0x33, 0x2A, 0xD1, 0x7B, 0x6C, 0xFF, 0x2C, 0x1E, 0x2B, 0xA8, 0x67, 0xE4, 0x74,
                0x39, 0x47, 0xAE, 0x90, 0xEB, 0x89, 0xB7, 0x2D, 0x9A, 0x3F, 0x5E, 0x1A, 0xB1, 0x2A,
                0x97, 0x55, 0xCD, 0x3B, 0x57, 0x5C, 0x7E, 0x5A, 0x15, 0x24, 0x60, 0x87, 0x62, 0xA7,
                0x88, 0xD9, 0x23, 0xDE, 0x23, 0x3E, 0x81, 0x10, 0xE5, 0x84, 0xC2, 0xD4, 0x50, 0x30,
                0x8A, 0xDA, 0x06, 0xCC, 0xA1, 0xB6, 0xD5, 0x6A, 0x22, 0x2A, 0xE9, 0x3E, 0xD6, 0x01,
                0xDA, 0xA4, 0x29, 0xD1, 0x2C, 0x11, 0xEA, 0xE0, 0xDA, 0xAE, 0xEE, 0x36, 0xB8, 0xC7,
                0xD9, 0x72, 0xC2, 0x29, 0xBD, 0x59, 0x98, 0x87, 0x26, 0xCA, 0x90, 0xBF, 0xF0, 0x9D,
                0x99, 0x5B, 0x42, 0xF0, 0x1A, 0x69, 0xFC, 0xD4, 0xBA, 0x10, 0xFB, 0xEF, 0x43, 0xC1,
                0xC8, 0x74, 0xB4, 0x93, 0xE8, 0x3B, 0x90, 0x4E, 0xCC, 0x50, 0x65, 0x7F, 0x6B, 0xE4,
                0xB5, 0x54, 0xA5, 0x33, 0x1C, 0xEE, 0x26, 0xEA, 0x9F, 0x33, 0xD2, 0xA3, 0x7A, 0x83,
                0x12, 0xD2, 0x29, 0xEA, 0x5C, 0xA5, 0x5C, 0xAF, 0xCD, 0x1B, 0xC6, 0xF4, 0x9E, 0x15,
                0xC1, 0xC0, 0x55, 0x91, 0xAE, 0x74, 0x6B, 0x22, 0xF0, 0x7C, 0x9D, 0x33, 0xA8, 0xB7,
                0x9A, 0x9D, 0x7E, 0x0B, 0x32, 0xBE, 0x80, 0x0B, 0x02, 0xCC, 0xCB, 0x2B, 0xD8, 0x06,
                0x86, 0x56, 0x0A, 0xA3, 0x96, 0xF8, 0x79, 0xB1, 0x50, 0xA5, 0x78, 0x4F, 0x89, 0x20,
                0x52, 0x86, 0x4B, 0xA9, 0xCF, 0xF1, 0x41, 0x4A, 0x03, 0xA0, 0xBE, 0xA2, 0xDC, 0xF1,
                0x12, 0x5A, 0x34, 0x41, 0xF0, 0xA6, 0x25, 0x99, 0x91, 0x10, 0x64, 0xFC, 0x31, 0xEC,
                0x64, 0x49, 0xB1, 0xF1, 0x73, 0x0B, 0xF3, 0x37, 0x80, 0x49, 0xB9, 0xA7, 0x0E, 0xFF,
                0xE4, 0x9D, 0x95, 0x73, 0xB8, 0x9D, 0x71, 0x44, 0xAD, 0xB6, 0xFD, 0x65, 0x86, 0x31,
                0x80, 0x81, 0xBB, 0x6E, 0xAD, 0x9F, 0x20, 0xD6, 0x21, 0x00, 0x26, 0xAE, 0x5F, 0x8C,
                0x1E, 0x4E, 0x6D, 0xFB, 0xF3, 0x34, 0x65, 0x09, 0x64, 0xCD, 0x62, 0xB3, 0xDC, 0x73,
                0xA1, 0x79, 0xD9, 0xFE, 0xEB, 0x39, 0x76, 0x73, 0x77, 0xD7, 0x66, 0x4C, 0x76, 0xB5,
                0x68, 0xC7, 0xF8, 0x5A, 0x0B, 0xBE, 0x65, 0xAA, 0xB6, 0x1F, 0x2C, 0x20, 0x72, 0x6D,
                0x43, 0x32, 0x96, 0x04, 0x8E, 0xBE, 0x64, 0xAB, 0x04, 0x96, 0xCA, 0xDF, 0x38, 0x47,
                0xE1, 0x0B, 0x91, 0x32, 0x95, 0x34, 0x23, 0x4D, 0x8E, 0x7B, 0x8F, 0x28, 0x68, 0x64,
                0x73, 0x69, 0xCD, 0xF0, 0xFF, 0x57, 0x07, 0xBE, 0xD8, 0xED, 0xB5, 0x8F, 0x25, 0xDD,
                0x68, 0x15, 0x9F, 0xC8, 0xA0, 0x16, 0x2A, 0x85, 0xBB, 0xEA, 0x35, 0x31, 0x23, 0x51,
                0xE5, 0x08, 0x63, 0x1F, 0xBD, 0x70, 0xF1, 0x58, 0xAC, 0xE9, 0xDE, 0xC7, 0x3A, 0x5A,
                0xBE, 0x2A, 0x04, 0x72, 0xC8, 0x19, 0x15, 0x2C, 0xF7, 0xCF, 0xA1, 0x6D, 0x7B, 0x7A,
                0x43, 0x48, 0x9A, 0x5E, 0x48, 0xA4, 0xC2, 0xA1, 0x79, 0x3B, 0x61, 0xFA, 0x5C, 0x3C,
                0x9F, 0x02, 0xD0, 0x72, 0xDE, 0x4E, 0xED, 0x0C, 0xB3, 0xFE, 0x78, 0x9B, 0xAE, 0xFA,
                0x40, 0x10, 0x04, 0x64, 0x6B, 0x1B, 0x9C, 0x9D, 0x5C, 0x63, 0xCF, 0x2C, 0xE7, 0xAB,
                0x91, 0xC3, 0xC6, 0x9C, 0x1D, 0x97, 0xE8, 0x1E, 0x6B, 0x2B, 0x14, 0x12, 0x03, 0x35,
                0xDB, 0x22, 0x04, 0x56, 0x97, 0xD7, 0xBB, 0x7F, 0x1E, 0x67, 0x65, 0xAF, 0x8F, 0xD7,
                0xE5, 0xD8, 0xAB, 0xCB, 0xD6, 0x59, 0x8E, 0x7B, 0x1B, 0x3F, 0xEC, 0x79, 0x70, 0xE2,
                0x00, 0xF6, 0x22, 0x67, 0xA1, 0x62, 0xE9, 0xDA, 0x90, 0x9C, 0xAE, 0x1A, 0x93, 0xE8,
                0x8E, 0xBB, 0x82, 0xB7, 0x37, 0x8C, 0x33, 0x06, 0xA9, 0xDD, 0xEF, 0x07, 0xFF, 0x13,
                0x7D, 0xCA, 0xCF, 0x4B, 0x6B, 0x09, 0x12, 0xB6, 0x5B, 0x43, 0x71, 0x1F, 0x27, 0xEE,
                0xBF, 0xE2, 0xC5, 0x60, 0x06, 0x9B, 0xA0, 0x5A, 0x0C, 0x48, 0xD0, 0x6E, 0xFD, 0xAA,
                0xEE, 0xAB, 0x06, 0x30, 0x9C, 0x12, 0x5A, 0x05, 0x1B, 0xD7, 0x96, 0xAA, 0xA0, 0xEC,
                0x75, 0x6C, 0xEC, 0x8A, 0x2E, 0x41, 0xAA, 0x5A, 0x4E, 0xE3, 0x68, 0x56, 0xEE, 0xB4,
                0x83, 0xC1, 0xBD, 0x91, 0xCA, 0x53, 0x04, 0xC6, 0xA1, 0x62, 0x6B, 0x54, 0xC4, 0xAD,
                0x49, 0x42, 0x26, 0x9D, 0xBB, 0xC1, 0xE8, 0x2B, 0xD3, 0xC1, 0x68, 0x31, 0xB8, 0xFB,
                0xB4, 0xE8, 0x44, 0x92, 0x45, 0xC0, 0x76, 0x08, 0x29, 0x04, 0xD7, 0x56, 0xBF, 0x19,
                0x5A, 0xCB, 0xD7, 0xC2, 0x30, 0x97, 0x96, 0xD9, 0x45, 0x86, 0x93, 0xAB, 0x54, 0x1E,
                0x20, 0x45, 0x95, 0xC4, 0xCB, 0x0B, 0x13, 0x92, 0xA8, 0x51, 0xCD, 0x71, 0x83, 0xDF,
                0x5C, 0x02, 0x56, 0xC0, 0x59, 0x9C, 0x18, 0x91, 0x45, 0x3E, 0x3A, 0x95, 0xE7, 0x7B,
                0x8D, 0x9A, 0x5B, 0x2D, 0xFC, 0x0B, 0x3F, 0xF4, 0x38, 0xBA, 0x0C, 0xEF, 0x61, 0xCF,
                0x97, 0xC2, 0x9B, 0x99, 0xE1, 0x6D, 0x6B, 0xCD, 0xDA, 0xE8, 0x1F, 0x4E, 0x4C, 0x05,
                0x55, 0xF8, 0x10, 0x12, 0xE1, 0x82, 0x3F, 0x36, 0xA0, 0xD8, 0x5C, 0x0B, 0x7D, 0x80,
                0x11, 0x2E, 0x94, 0x99, 0x56, 0x16, 0x2F, 0x97, 0x55, 0xEC, 0x80, 0x90, 0x26, 0xCF,
                0x9B, 0xDC, 0x1A, 0xA3, 0x91, 0x10, 0x8D, 0x05, 0xC0, 0x43, 0x17, 0x4B, 0x7A, 0x10,
                0x50, 0xE0, 0x8D, 0x41, 0xBE, 0x4D, 0xD8, 0x6C, 0x22, 0xED, 0x4A, 0x8E, 0x5A, 0xF7,
                0x8D, 0x2C, 0x8B, 0x5F, 0x0C, 0x2A, 0x35, 0xFF, 0x44, 0x2B, 0xF2, 0x2A, 0x91, 0x95,
                0x63, 0x1B, 0x71, 0x3D, 0x31, 0x35, 0x80, 0x0A, 0xF2, 0x3A, 0x4E, 0x29, 0x8F, 0x95,
                0x5D, 0x2D, 0x0A, 0x62, 0x9A, 0x75, 0x80, 0x3A, 0x21, 0xEF, 0xAE, 0x69, 0x9D, 0x29,
                0x19, 0x75, 0x3F, 0x18, 0x98, 0x32, 0xA4, 0x5D, 0x71, 0x66, 0x5F, 0x71, 0xBA, 0xC9,
                0x6F, 0x48, 0x52, 0x8D, 0x97, 0xE7, 0x0E, 0xE5, 0x79, 0x44, 0x93, 0xF0, 0x10, 0xD3,
                0x50, 0x4F, 0x5B, 0xD3, 0x29, 0xDB, 0x16, 0x5A,
            ];
            let mut ss1 = [0u8; CRYPTO_BYTES];

            crypto_kem_dec(&mut ss1, &ct, &sk)?;

            let ss_ref = [
                0x29, 0x39, 0x92, 0x00, 0x0D, 0xC2, 0x88, 0xE8, 0x15, 0x2F, 0x94, 0x51, 0xF0, 0x6D,
                0xD8, 0x35, 0xC7, 0x5E, 0xA0, 0x08, 0x66, 0x2B, 0xAC, 0xE0, 0xFB, 0x97, 0xA9, 0x7B,
                0x3A, 0xFB, 0x54, 0xE4,
            ];
            assert_eq!(ss_ref, ss1);
        }

        #[cfg(feature = "ntruhrss701")]
        {
            let ct = [
                0x4F, 0xB2, 0x9C, 0xE5, 0xCE, 0x75, 0x34, 0x44, 0xA0, 0xB9, 0x41, 0xD8, 0x77, 0x30,
                0x5F, 0x93, 0x38, 0xED, 0x3F, 0x4F, 0x30, 0xF5, 0xE8, 0xA0, 0xBC, 0x1F, 0x41, 0xAD,
                0x98, 0x17, 0xB1, 0x52, 0x8C, 0x9E, 0x86, 0xCC, 0xFF, 0x4E, 0x6F, 0x6B, 0xD3, 0x64,
                0xC1, 0xEB, 0xA2, 0xAF, 0xF7, 0xD4, 0x83, 0x35, 0xE7, 0x1F, 0xAC, 0x9B, 0xFE, 0x5B,
                0x85, 0xE9, 0xFF, 0x19, 0xB3, 0xE6, 0x0E, 0xAA, 0x35, 0x17, 0xE8, 0x89, 0x13, 0xF4,
                0x1C, 0xD0, 0x10, 0x49, 0x2B, 0x9C, 0x06, 0x0D, 0x89, 0xBA, 0xDF, 0xA5, 0x25, 0x7B,
                0x1E, 0xA7, 0x7F, 0x1E, 0x32, 0xAE, 0x2C, 0x2F, 0xAE, 0xAA, 0xB5, 0x87, 0xAD, 0x30,
                0x5F, 0xBB, 0x99, 0x96, 0x1C, 0xE8, 0x8D, 0x39, 0xDB, 0xA4, 0x8D, 0x8B, 0xC5, 0x98,
                0xEF, 0x3D, 0xC8, 0xBF, 0xEC, 0xE9, 0x7E, 0x67, 0x30, 0x1B, 0xCE, 0x08, 0x1E, 0xF8,
                0xEE, 0x52, 0x1E, 0xF2, 0x3F, 0xE7, 0x0A, 0x98, 0x83, 0x4B, 0x63, 0xE0, 0x21, 0xB4,
                0x16, 0x1D, 0x1A, 0x9C, 0xD0, 0x03, 0xC3, 0x00, 0xF2, 0xE9, 0xBE, 0xF0, 0xFD, 0xD8,
                0x41, 0x91, 0x53, 0x19, 0x43, 0x63, 0x13, 0x7F, 0x28, 0x81, 0x04, 0x62, 0xD5, 0x12,
                0x16, 0x1C, 0x82, 0xFE, 0xF9, 0xD5, 0xC1, 0x1E, 0x88, 0x25, 0x0B, 0x00, 0xBF, 0x51,
                0x20, 0x2E, 0x02, 0x34, 0x10, 0x78, 0xB4, 0x0F, 0x89, 0xBB, 0xE5, 0xEA, 0x8B, 0xF0,
                0x13, 0x2C, 0x6E, 0xFB, 0xD7, 0x5A, 0x2C, 0x27, 0x28, 0x6D, 0xF1, 0x46, 0x34, 0x42,
                0x55, 0x20, 0xC7, 0xBF, 0x66, 0x03, 0xA6, 0xB8, 0x05, 0xD1, 0xB3, 0x74, 0x90, 0xE2,
                0xF1, 0x51, 0x64, 0xB4, 0xD2, 0x76, 0xC1, 0xB4, 0xDE, 0x29, 0x7F, 0x39, 0x7B, 0x48,
                0x02, 0x75, 0xB8, 0xE3, 0x99, 0x08, 0xA7, 0x85, 0xE5, 0xB4, 0x9B, 0x29, 0x64, 0x73,
                0xDA, 0x62, 0x27, 0x0C, 0x2D, 0x0A, 0x39, 0x6D, 0xB1, 0x89, 0x8D, 0x19, 0x0A, 0x04,
                0x3D, 0x04, 0x2D, 0x4F, 0xD7, 0xFF, 0x4E, 0xDB, 0x90, 0x46, 0x68, 0x61, 0xB2, 0xC5,
                0x90, 0x09, 0x14, 0x59, 0xD0, 0x22, 0xF9, 0x82, 0x63, 0x16, 0xBC, 0xE8, 0xED, 0x5A,
                0x57, 0x76, 0x9D, 0xB3, 0xEF, 0xB5, 0x1B, 0xB2, 0xFD, 0xDC, 0x3C, 0x78, 0x82, 0xD3,
                0xFC, 0x38, 0x60, 0xC3, 0x38, 0xE1, 0x66, 0x4B, 0x7C, 0xCD, 0x93, 0xAC, 0xCA, 0x5E,
                0xB9, 0x34, 0x2D, 0xBD, 0xF1, 0x0B, 0x1D, 0x9B, 0x53, 0xD3, 0x9E, 0xF8, 0x91, 0x7D,
                0x9C, 0xDB, 0x0F, 0x22, 0x2F, 0x6B, 0xB2, 0xBC, 0xB7, 0xED, 0x50, 0x98, 0x28, 0x46,
                0xF3, 0x9F, 0xA4, 0x86, 0x23, 0x9E, 0x73, 0x8E, 0x48, 0xE5, 0x90, 0x81, 0xBD, 0xC4,
                0x64, 0x85, 0xCF, 0x64, 0xB8, 0xD3, 0xA0, 0xCD, 0x1D, 0x43, 0x0D, 0x26, 0x6B, 0x1E,
                0x89, 0xDA, 0xC0, 0xC0, 0x0B, 0x8D, 0xE1, 0x9D, 0x8A, 0x89, 0x01, 0xBC, 0x65, 0x05,
                0xFC, 0xD7, 0xA3, 0x84, 0xA8, 0xC0, 0x44, 0x64, 0x29, 0x8D, 0x9E, 0x27, 0x42, 0x0C,
                0x6D, 0x97, 0x31, 0x77, 0xB6, 0xC0, 0x0B, 0x8F, 0x4E, 0x28, 0xDF, 0x7B, 0x67, 0x50,
                0xAF, 0x0B, 0x7F, 0x66, 0x97, 0x27, 0x08, 0x06, 0xDA, 0x80, 0x66, 0x47, 0x91, 0x82,
                0xDD, 0xBC, 0x86, 0x65, 0x17, 0xAA, 0x49, 0x45, 0x89, 0xA3, 0x85, 0x7B, 0x63, 0xA4,
                0x00, 0x46, 0x72, 0xBE, 0x2D, 0x52, 0x10, 0x52, 0x82, 0xC7, 0x52, 0xE0, 0xD5, 0x5F,
                0x1C, 0x5A, 0x29, 0xA2, 0x34, 0x5B, 0xE5, 0x1B, 0x39, 0x1A, 0x6D, 0xD9, 0x6F, 0x4B,
                0xFB, 0x34, 0xEF, 0x0A, 0x6E, 0x43, 0x01, 0x4C, 0x6F, 0x71, 0x2E, 0x3B, 0x53, 0xF5,
                0xF6, 0x0A, 0x50, 0x25, 0x02, 0x48, 0x10, 0xCB, 0xDE, 0x01, 0x2C, 0x2F, 0xFB, 0x9B,
                0xF4, 0x86, 0x4B, 0xEC, 0x3A, 0x6B, 0x1E, 0x08, 0xD6, 0xDA, 0xCF, 0xBD, 0xFE, 0x54,
                0xC9, 0xD8, 0x75, 0xCE, 0x88, 0xF3, 0x48, 0x8F, 0x2C, 0xFB, 0xBA, 0xFA, 0x38, 0x33,
                0x40, 0x62, 0x15, 0x16, 0x7A, 0x86, 0xDD, 0x4A, 0x6B, 0xC0, 0xEF, 0xC2, 0xE4, 0xCB,
                0x6F, 0x28, 0x6F, 0xF4, 0x67, 0x0F, 0x23, 0xCA, 0x8D, 0x8A, 0xB7, 0x8F, 0x55, 0x00,
                0x0F, 0x14, 0x38, 0xAA, 0x2C, 0xA1, 0x19, 0x27, 0xB9, 0x7F, 0xD5, 0x9D, 0x77, 0xC4,
                0x4B, 0xBD, 0xA6, 0xB7, 0xAE, 0xD9, 0x6F, 0xB8, 0xF6, 0x6A, 0x25, 0x7A, 0x3D, 0x10,
                0x1F, 0x4C, 0xE0, 0x24, 0xD9, 0x8E, 0x9D, 0xF1, 0x57, 0xC8, 0x97, 0x86, 0x24, 0xC8,
                0x38, 0x76, 0x58, 0x54, 0x3D, 0x9D, 0x96, 0x45, 0xB5, 0x7B, 0x92, 0x3D, 0xE2, 0x70,
                0x2D, 0x4F, 0xB1, 0x00, 0x74, 0x0A, 0x82, 0xBB, 0x21, 0xBD, 0x8B, 0x79, 0xF1, 0x4F,
                0x79, 0x55, 0x68, 0x4E, 0x49, 0xC4, 0xEC, 0x7E, 0x6B, 0x72, 0xA2, 0xF1, 0x73, 0x6F,
                0xAA, 0x1F, 0xEC, 0x3B, 0xB3, 0xB3, 0x1D, 0xDA, 0xFE, 0xD9, 0x5B, 0x1E, 0x50, 0xA7,
                0x13, 0xEC, 0x00, 0x50, 0xD5, 0xA9, 0x5B, 0x94, 0x89, 0xAA, 0xCF, 0x77, 0x84, 0x7B,
                0x87, 0xD1, 0xCB, 0xFF, 0xAE, 0xA9, 0x0F, 0x1A, 0xF0, 0xEE, 0x5E, 0x2A, 0xAC, 0xB5,
                0xD8, 0x0E, 0x9B, 0xF1, 0x42, 0xCA, 0xB9, 0x27, 0x22, 0xB0, 0xED, 0x72, 0xBE, 0x31,
                0x83, 0x5C, 0x9F, 0xD0, 0xCF, 0x16, 0x0C, 0x14, 0x00, 0x51, 0x6B, 0xDC, 0x81, 0xAF,
                0x7C, 0xC7, 0x10, 0x14, 0x68, 0x40, 0xFA, 0xE0, 0x56, 0xA7, 0x89, 0x8F, 0xFE, 0xFF,
                0xD8, 0x19, 0xB6, 0x4C, 0x3E, 0x2E, 0x94, 0x6D, 0x2A, 0xB4, 0x38, 0x06, 0x31, 0xD9,
                0x97, 0x8E, 0xDB, 0x61, 0x0B, 0xE0, 0x91, 0x8E, 0xDB, 0x7C, 0x31, 0x4A, 0x6F, 0x97,
                0x14, 0xE3, 0x94, 0xDF, 0xDD, 0x2D, 0xE7, 0xC7, 0x88, 0xB0, 0xE6, 0xB0, 0x1A, 0x57,
                0xFB, 0x46, 0xD0, 0xCF, 0x27, 0xFB, 0xF6, 0xFD, 0x45, 0x9F, 0xBF, 0xC7, 0xFE, 0xF8,
                0x40, 0xD0, 0xB8, 0x63, 0xED, 0x6E, 0xF1, 0xE0, 0xA1, 0xA9, 0x3E, 0xD0, 0xCD, 0x24,
                0x32, 0xC5, 0x5E, 0x8C, 0xDA, 0xAD, 0xF6, 0xCB, 0x31, 0xB1, 0xF1, 0x87, 0x2B, 0xDE,
                0x39, 0x03, 0x7F, 0x5D, 0xBC, 0x94, 0xCC, 0xDE, 0x7A, 0x43, 0xAB, 0xBF, 0x89, 0x02,
                0x29, 0x1F, 0xCA, 0xF0, 0x63, 0xD6, 0xC4, 0xC1, 0x70, 0x52, 0xF6, 0x3A, 0x06, 0xAD,
                0x88, 0xAE, 0xDA, 0x5B, 0xBE, 0xCE, 0xFB, 0x91, 0x4D, 0xA4, 0xCA, 0xB6, 0x0F, 0x33,
                0xE9, 0xA7, 0xAC, 0x89, 0xEA, 0xD9, 0xA0, 0xDF, 0xB0, 0x3A, 0xFE, 0xEA, 0x45, 0x6A,
                0xF7, 0x55, 0x86, 0x49, 0x13, 0x3E, 0x77, 0x86, 0x2F, 0x28, 0x76, 0xA3, 0xE2, 0x1F,
                0xB5, 0x7A, 0xB2, 0x29, 0x3F, 0xCA, 0xFE, 0xF6, 0xBF, 0x71, 0xE5, 0x53, 0x08, 0xFA,
                0xED, 0x72, 0x02, 0x16, 0xBB, 0x66, 0xD5, 0xA8, 0x91, 0x72, 0xE3, 0x85, 0x87, 0xB1,
                0xB3, 0x5D, 0x07, 0xF8, 0xDB, 0xAA, 0x54, 0x70, 0xF7, 0x26, 0xC3, 0x61, 0xC3, 0x00,
                0xC7, 0xB9, 0x67, 0x2A, 0xE9, 0xB9, 0xF2, 0xDD, 0x8E, 0x7A, 0x17, 0x1E, 0x38, 0x07,
                0xDB, 0x74, 0x5D, 0xDE, 0x43, 0x4A, 0xC3, 0x20, 0x60, 0xCD, 0x27, 0xF4, 0x46, 0xF8,
                0xAA, 0xDB, 0xB2, 0xC2, 0x34, 0x97, 0x5D, 0x42, 0x30, 0x8C, 0xE3, 0x7C, 0x63, 0x72,
                0x27, 0xFC, 0xF7, 0x97, 0x7B, 0x0C, 0x01, 0x42, 0x77, 0x2A, 0xAF, 0xC0, 0x47, 0x67,
                0x85, 0x08, 0x08, 0x6B, 0x5F, 0xEA, 0x05, 0x95, 0x06, 0xDD, 0x66, 0x01, 0xE1, 0x52,
                0xDA, 0xCD, 0x2C, 0xF2, 0x35, 0xB7, 0x2F, 0xFF, 0x07, 0xB5, 0x08, 0x74, 0xF2, 0x53,
                0x12, 0x55, 0x1B, 0x28, 0xF6, 0xDB, 0x1E, 0x92, 0x50, 0x1E, 0x1E, 0x04, 0x59, 0xA8,
                0x2A, 0xD5, 0x4D, 0xF1, 0xCC, 0x90, 0x67, 0xFF, 0xA3, 0x16, 0xD3, 0x1C, 0xCE, 0x67,
                0x47, 0x64, 0x3E, 0xF5, 0x14, 0xAB, 0x6C, 0x04, 0x61, 0x89, 0xD8, 0x1E, 0x04, 0x2E,
                0x7E, 0x8F, 0xC9, 0xF2, 0xD1, 0x90, 0x09, 0x0B, 0x1F, 0x44, 0x7C, 0x8F, 0x86, 0x72,
                0x36, 0x5A, 0x0F, 0x52, 0x83, 0x8A, 0x17, 0x85, 0x37, 0xB0, 0x91, 0x8B, 0x2A, 0x28,
                0xCE, 0x1F, 0x3A, 0x20, 0x71, 0x15, 0x63, 0x1F, 0x5A, 0x9C, 0x53, 0xCE, 0xEC, 0x45,
                0x33, 0x06, 0xDC, 0x03, 0xAB, 0x07, 0x8F, 0xB6, 0x72, 0xBF, 0x9A, 0x87, 0x65, 0x36,
                0x4E, 0x70, 0x90, 0x4F, 0x1E, 0xD2, 0xE5, 0x43, 0x2F, 0x5E, 0x9B, 0x83, 0xC5, 0xC7,
                0xB5, 0xA7, 0xDF, 0x47, 0xD3, 0xB5, 0xE2, 0xB1, 0xC0, 0x15, 0xA4, 0xE6, 0x72, 0x2E,
                0xCF, 0x9C, 0x8A, 0x06,
            ];
            let sk = [
                0xD0, 0x67, 0xD9, 0x8F, 0x00, 0x55, 0xE2, 0xC3, 0xDE, 0xEF, 0x10, 0x76, 0xBB, 0xB7,
                0x55, 0xAF, 0xD0, 0x65, 0x11, 0x2C, 0x85, 0xC4, 0x6E, 0x6C, 0x35, 0x06, 0x55, 0xD9,
                0x62, 0x50, 0x73, 0xE9, 0x4E, 0x45, 0x28, 0xC0, 0x53, 0xE7, 0x20, 0x20, 0x6C, 0xDB,
                0x78, 0x0A, 0x61, 0xAD, 0x51, 0x20, 0xC4, 0x98, 0xA4, 0xCD, 0x3E, 0x60, 0xC3, 0x34,
                0xD8, 0x70, 0x2F, 0x0F, 0x79, 0xD8, 0x14, 0x18, 0xAF, 0x95, 0x9C, 0xAB, 0x77, 0x22,
                0xEB, 0xDD, 0x30, 0xDF, 0x0A, 0x47, 0x9E, 0x14, 0x03, 0xD3, 0x37, 0xAD, 0x92, 0x3E,
                0x5D, 0xBC, 0xED, 0x4E, 0xE0, 0x48, 0x39, 0x02, 0x0F, 0x77, 0x51, 0xC0, 0x75, 0xA4,
                0x89, 0x2A, 0xAB, 0xBB, 0x0E, 0x00, 0x1A, 0xBA, 0x95, 0x51, 0x5E, 0x1B, 0x65, 0xC6,
                0x3C, 0x50, 0x3D, 0x97, 0xE5, 0x1E, 0xAF, 0xB9, 0xAC, 0x93, 0x0F, 0x5A, 0x1D, 0xBC,
                0xB7, 0xE9, 0x38, 0x8B, 0x5E, 0xF1, 0xE7, 0x5E, 0x89, 0x3E, 0x1B, 0x5E, 0x49, 0x95,
                0xE9, 0x0F, 0x71, 0x69, 0x1C, 0x40, 0xB0, 0x03, 0x52, 0xA9, 0x6C, 0xB9, 0xBE, 0x2E,
                0x1D, 0x94, 0x42, 0x3F, 0x89, 0x7D, 0xED, 0x5F, 0xD4, 0x66, 0xE8, 0x4C, 0xB7, 0x22,
                0x90, 0x01, 0x49, 0x81, 0xDE, 0x96, 0x2E, 0xBF, 0x41, 0xB4, 0x2B, 0xE1, 0xE8, 0xB8,
                0xB7, 0xBE, 0x2E, 0xDF, 0xDB, 0x49, 0xA5, 0x58, 0xA6, 0x81, 0x45, 0x81, 0x96, 0xD3,
                0xB3, 0x95, 0x0A, 0xA7, 0x3F, 0x4D, 0xE4, 0xE8, 0x08, 0x9B, 0xD4, 0xE1, 0x61, 0x14,
                0x6F, 0xD1, 0x16, 0x51, 0xC9, 0x55, 0xDB, 0x46, 0x61, 0xDF, 0xE0, 0xC1, 0x81, 0x85,
                0xEE, 0x2D, 0x7A, 0x25, 0xE2, 0x4D, 0xBA, 0xCA, 0x2E, 0xAD, 0x84, 0x54, 0x1D, 0xAA,
                0xAC, 0x37, 0x95, 0xBB, 0x1F, 0xD2, 0x53, 0x6C, 0xCE, 0xBC, 0x57, 0x2E, 0xA4, 0x71,
                0x84, 0x5B, 0x66, 0x9C, 0x33, 0x6F, 0x0F, 0xE7, 0x05, 0x5D, 0xBF, 0x16, 0x19, 0x7F,
                0x51, 0x95, 0x36, 0x94, 0x46, 0x78, 0x9F, 0x0C, 0xAF, 0x8A, 0x0E, 0x38, 0x8F, 0xE6,
                0x84, 0x6E, 0x5F, 0x35, 0xB0, 0xE3, 0x09, 0xF9, 0x6A, 0x93, 0x2F, 0x05, 0x50, 0x2F,
                0xDF, 0xFE, 0xCB, 0x30, 0x71, 0xCF, 0x7F, 0xAD, 0x85, 0x3A, 0x05, 0xB2, 0x4A, 0x71,
                0x5C, 0xEF, 0xCA, 0xD7, 0x2E, 0x4C, 0x56, 0xEA, 0x03, 0xED, 0x49, 0x8C, 0x90, 0x87,
                0x3D, 0x99, 0xC3, 0xA8, 0x0D, 0xA4, 0x8D, 0xA2, 0xAA, 0x70, 0x24, 0xF7, 0xE4, 0xAF,
                0x7D, 0xC9, 0xA6, 0x54, 0x6D, 0xCD, 0x59, 0x6D, 0xBC, 0x8F, 0x5A, 0x65, 0xD5, 0x64,
                0xE0, 0x8A, 0x11, 0x97, 0x43, 0x76, 0x8E, 0xF8, 0x4A, 0x79, 0x9D, 0xCE, 0xB4, 0x9D,
                0xF2, 0x43, 0x21, 0x2B, 0x82, 0x3E, 0x5F, 0x0C, 0x85, 0x39, 0xCA, 0x9F, 0x60, 0x04,
                0xF0, 0x56, 0x12, 0x33, 0x30, 0x45, 0x0E, 0x25, 0xD0, 0xA1, 0xC3, 0xCD, 0x3C, 0x8A,
                0x4B, 0xF3, 0x31, 0xF4, 0xE5, 0xAF, 0xA7, 0x70, 0xB2, 0xF7, 0x95, 0x41, 0x2B, 0x47,
                0x2D, 0x04, 0xFF, 0xAC, 0x26, 0xF1, 0xE8, 0x26, 0xE7, 0x2C, 0xE3, 0x36, 0xA2, 0x95,
                0x85, 0x7A, 0x7E, 0x65, 0xEB, 0xB1, 0x07, 0x7C, 0x74, 0xF3, 0x61, 0x02, 0xFC, 0xEE,
                0xF3, 0x45, 0x52, 0x95, 0x4A, 0xA8, 0x31, 0xA3, 0xCA, 0x6E, 0xA4, 0xED, 0xEE, 0x22,
                0x64, 0xDB, 0x71, 0x04, 0x7F, 0xBC, 0xD4, 0x22, 0x00, 0xD2, 0xE9, 0x73, 0xE0, 0x1A,
                0x92, 0xCD, 0xD7, 0x1D, 0xFC, 0xD2, 0x90, 0xD4, 0xE5, 0x50, 0x79, 0xA9, 0x52, 0x05,
                0x1D, 0x6E, 0x79, 0x0A, 0xA0, 0xA8, 0x56, 0x61, 0xD5, 0x04, 0x11, 0x3F, 0xD2, 0x94,
                0xC9, 0x67, 0xB8, 0x48, 0xD7, 0x2F, 0x58, 0xDB, 0xBE, 0xE6, 0x7C, 0x27, 0x93, 0x47,
                0xBF, 0x03, 0xDB, 0x5D, 0xEC, 0x7B, 0x0C, 0xD1, 0xD4, 0x44, 0xA8, 0x52, 0xEA, 0x83,
                0x1B, 0x18, 0x55, 0xE9, 0x45, 0x85, 0x57, 0xE6, 0x25, 0x0A, 0x9B, 0xD4, 0xCE, 0x5F,
                0x29, 0x7E, 0xA4, 0x69, 0x73, 0x2A, 0x0C, 0xDC, 0x01, 0x84, 0x2A, 0x38, 0xA8, 0xF8,
                0x9C, 0xBB, 0x9B, 0xAD, 0x23, 0x19, 0xA6, 0xEF, 0xA9, 0x5F, 0xE4, 0x72, 0xA7, 0xD8,
                0x7F, 0xAD, 0xBD, 0x95, 0x38, 0x83, 0x7B, 0x6D, 0x00, 0xCF, 0x12, 0x3A, 0xAB, 0x07,
                0x58, 0x5E, 0x2A, 0x0A, 0x8E, 0xEC, 0x91, 0xAC, 0x7A, 0xE0, 0x87, 0x59, 0x2A, 0xC7,
                0x37, 0x69, 0x5A, 0x7C, 0xC6, 0x6A, 0x1C, 0xA1, 0x99, 0x62, 0x2D, 0x2C, 0xCF, 0x3C,
                0x46, 0xD3, 0xB5, 0x3D, 0x1B, 0x4D, 0x84, 0x0A, 0x5B, 0x22, 0x31, 0xB4, 0xE7, 0x68,
                0x0F, 0xAE, 0xF7, 0xDB, 0x87, 0x76, 0xE0, 0x2E, 0x5B, 0x49, 0x6C, 0x22, 0x09, 0x1D,
                0x03, 0xEC, 0x07, 0xA4, 0x4D, 0x9E, 0x91, 0x0F, 0x59, 0x1C, 0xA4, 0x98, 0xBF, 0xCF,
                0x02, 0xF7, 0x92, 0xD0, 0x3E, 0xEB, 0x55, 0x64, 0x3A, 0x91, 0x76, 0x89, 0x10, 0x11,
                0x61, 0x94, 0x0D, 0x6B, 0x59, 0xF9, 0x94, 0x3B, 0x72, 0x03, 0x66, 0xE3, 0xB0, 0x55,
                0x49, 0xCE, 0x7D, 0x89, 0x72, 0xDF, 0x1A, 0x4F, 0x3C, 0x63, 0x25, 0x1D, 0xB6, 0x10,
                0x84, 0xFE, 0x93, 0x65, 0xFF, 0xED, 0xCE, 0xB7, 0xCE, 0xDE, 0x9E, 0x27, 0x7E, 0xBA,
                0xA4, 0x3A, 0xBE, 0x56, 0x7F, 0x92, 0xC9, 0x7A, 0x53, 0x45, 0x25, 0x62, 0xF0, 0x5F,
                0x32, 0xA7, 0xEE, 0x34, 0x7E, 0xA1, 0x60, 0x0E, 0x0A, 0xB9, 0x7E, 0x68, 0x6D, 0x0E,
                0xBA, 0xF0, 0xC2, 0x58, 0xEF, 0x8C, 0x76, 0x32, 0xD6, 0xCC, 0xAA, 0xA5, 0x32, 0x3D,
                0xAC, 0x2E, 0xD8, 0xFD, 0xD5, 0x45, 0x78, 0x90, 0x49, 0x31, 0xA6, 0x14, 0xCD, 0x4E,
                0x8D, 0x33, 0x86, 0xC9, 0x3A, 0x41, 0x60, 0xC5, 0x7C, 0x2F, 0x72, 0xB4, 0x2E, 0xB2,
                0xD3, 0x5A, 0xA2, 0x87, 0xF1, 0x82, 0x4D, 0xA8, 0x21, 0xF9, 0x0C, 0x20, 0x42, 0x25,
                0x5F, 0xFF, 0xD3, 0x8D, 0xB3, 0x99, 0x83, 0x55, 0x79, 0x96, 0xF8, 0x67, 0x73, 0x96,
                0x13, 0x79, 0x63, 0xFD, 0x03, 0x06, 0xB5, 0xA8, 0x70, 0x27, 0x14, 0x47, 0x7B, 0xB3,
                0x50, 0x1F, 0x75, 0x58, 0xE6, 0x6D, 0xF1, 0x09, 0x07, 0xCA, 0x94, 0x10, 0xC5, 0x37,
                0x0C, 0x2A, 0x99, 0x07, 0x5A, 0x7C, 0xB0, 0x66, 0xD1, 0xAF, 0x78, 0xEE, 0x91, 0xD9,
                0x1D, 0xDA, 0x37, 0x7B, 0x36, 0xAE, 0x2D, 0xFA, 0x26, 0x5A, 0xF6, 0x20, 0x43, 0x6F,
                0xA1, 0x13, 0x32, 0xBB, 0xD6, 0x1D, 0xAC, 0x03, 0x24, 0xFB, 0x90, 0xBA, 0x58, 0xD7,
                0x4C, 0xE6, 0x29, 0x9E, 0x29, 0xC0, 0x4E, 0xBF, 0x2F, 0xE4, 0x44, 0x15, 0x54, 0x53,
                0x70, 0x47, 0xE3, 0x5B, 0x3C, 0xA4, 0x36, 0x11, 0x8E, 0xF2, 0x86, 0x38, 0xA3, 0x2A,
                0x30, 0xEC, 0x1A, 0xCE, 0xB2, 0x56, 0xFC, 0x1B, 0xB8, 0x4C, 0xEE, 0x81, 0x27, 0xC8,
                0x2A, 0x72, 0x5D, 0x1E, 0x80, 0x3B, 0x61, 0x95, 0x6E, 0x50, 0xC4, 0xD6, 0xAE, 0x0D,
                0xC5, 0x3F, 0xCB, 0x5C, 0xD0, 0x68, 0x5D, 0x66, 0xE1, 0x50, 0x7B, 0xFF, 0x52, 0xFF,
                0xDC, 0x8E, 0xE6, 0xD0, 0x1B, 0xD2, 0x49, 0x2C, 0xB4, 0x42, 0x00, 0x19, 0x5B, 0x26,
                0x94, 0x86, 0x08, 0xF1, 0xEE, 0xA7, 0x27, 0x1C, 0xAF, 0x89, 0x70, 0xD4, 0x97, 0x44,
                0x4E, 0xA5, 0x46, 0x37, 0x72, 0x70, 0x17, 0x1A, 0x53, 0x4A, 0xFE, 0xE0, 0x5D, 0x01,
                0xE0, 0x5D, 0xA2, 0xDE, 0xE2, 0x50, 0x49, 0xC1, 0x44, 0x92, 0xB9, 0x2E, 0x72, 0xB8,
                0xF8, 0x9E, 0xBB, 0x1E, 0x43, 0x56, 0x30, 0xB7, 0x44, 0x02, 0x26, 0xEB, 0x3E, 0x85,
                0x66, 0x28, 0x40, 0x65, 0x12, 0x3D, 0xC0, 0x25, 0xA5, 0x3B, 0x8E, 0x98, 0x99, 0xE4,
                0x1B, 0x63, 0x65, 0x54, 0x82, 0x8F, 0xF8, 0xA7, 0x9B, 0x41, 0x04, 0x33, 0x14, 0xC5,
                0x45, 0x79, 0x81, 0xA8, 0x88, 0x6C, 0xE2, 0xF5, 0xB5, 0x5D, 0xE2, 0x05, 0x82, 0xF3,
                0x6C, 0xCF, 0xD5, 0xA8, 0x5E, 0x43, 0x92, 0x28, 0xD6, 0x9B, 0x1A, 0x92, 0x75, 0x16,
                0xBB, 0x05, 0x4F, 0x37, 0x95, 0xBA, 0x25, 0x30, 0x94, 0x1E, 0x49, 0x73, 0xF2, 0xD0,
                0x9D, 0xD3, 0xC7, 0x32, 0x74, 0xF3, 0xAD, 0x04, 0x6A, 0x1D, 0xCE, 0x4A, 0xF7, 0x8E,
                0xA5, 0xAA, 0x18, 0xB7, 0x68, 0x36, 0x8A, 0xA0, 0x97, 0xE7, 0xD6, 0xDA, 0x65, 0x47,
                0x57, 0x54, 0xF9, 0x75, 0x03, 0x88, 0x03, 0xF9, 0x83, 0x2E, 0x5A, 0x9A, 0x0F, 0xC7,
                0xA7, 0xF7, 0x4F, 0xA3, 0x0E, 0x41, 0x0A, 0x89, 0x93, 0x3B, 0xC1, 0x24, 0xC9, 0x5B,
                0x31, 0x65, 0x73, 0x71, 0xE3, 0xBD, 0x6E, 0x53, 0x78, 0xB6, 0xE8, 0x86, 0x6A, 0x47,
                0xD3, 0xF1, 0x41, 0xED, 0x57, 0xEE, 0xF1, 0xFF, 0x5A, 0xAA, 0x64, 0xDD, 0xB7, 0xD9,
                0x9A, 0xE3, 0x66, 0x3A, 0x54, 0x48, 0xE9, 0xA5, 0xE3, 0xF1, 0xEC, 0x06, 0xBC, 0xD3,
                0x24, 0x58, 0x92, 0xD6, 0xC5, 0x6C, 0xC0, 0xE7, 0x6B, 0x0E, 0x7B, 0x36, 0x03, 0x06,
                0xEE, 0x9C, 0x12, 0x7A, 0xAB, 0xD1, 0x43, 0xAD, 0xAA, 0x38, 0x60, 0xC4, 0x2E, 0xF1,
                0xAF, 0x59, 0x2C, 0x96, 0x86, 0x42, 0xC3, 0x9C, 0x1D, 0xB8, 0xD3, 0xD0, 0xA0, 0x30,
                0xE8, 0x82, 0xEF, 0xC5, 0xC3, 0x03, 0x74, 0xF5, 0xCA, 0x68, 0xF1, 0x9B, 0x32, 0x5F,
                0x88, 0x20, 0x1E, 0xD3, 0x33, 0x8F, 0xD8, 0xE0, 0xA0, 0xF6, 0x1D, 0x49, 0x13, 0xC0,
                0xB3, 0xCB, 0x68, 0x5D, 0x75, 0xF9, 0x6D, 0x55, 0xE2, 0x4B, 0x97, 0x69, 0xB0, 0x6E,
                0xB0, 0x5D, 0x17, 0xEF, 0xBB, 0x84, 0xE3, 0xF5, 0x50, 0xA9, 0x2F, 0xB3, 0x98, 0x12,
                0x72, 0x14, 0xD1, 0x67, 0xE7, 0x66, 0x39, 0x84, 0x4D, 0x02, 0xE6, 0xCF, 0x83, 0x84,
                0x00, 0x7C, 0xD1, 0xB2, 0x1B, 0x54, 0x0D, 0x0F, 0xC3, 0x7F, 0xC8, 0x0D, 0x84, 0x88,
                0xB3, 0xC4, 0xF2, 0x8F, 0x7F, 0x77, 0x49, 0x8D, 0x9E, 0x60, 0x69, 0x0B, 0x73, 0x3E,
                0xE4, 0x7E, 0x0D, 0x54, 0x28, 0xAF, 0x4F, 0xCE, 0xFD, 0x47, 0x39, 0x33, 0x3D, 0xA1,
                0xE9, 0x4E, 0x3B, 0x7B, 0x94, 0xF6, 0xF1, 0xE5, 0xEF, 0x8D, 0x90, 0xA7, 0xBE, 0x0F,
                0x91, 0xC3, 0x79, 0xE9, 0xDF, 0x15, 0x61, 0xA3, 0x46, 0x84, 0xF0, 0xF1, 0xB2, 0xEA,
                0x6B, 0x15, 0x98, 0xCE, 0xB1, 0xB6, 0x0C, 0x24, 0x9C, 0xC8, 0x8C, 0x4A, 0x11, 0x5A,
                0xFC, 0x06, 0xB1, 0xA3, 0xA7, 0x77, 0x9E, 0x2F, 0x6A, 0xB7, 0x09, 0x30, 0x81, 0x7C,
                0x7A, 0xC3, 0x36, 0x72, 0x84, 0x23, 0xF1, 0x3F, 0x91, 0x91, 0x72, 0xA1, 0xD6, 0xF8,
                0x2D, 0x2B, 0x09, 0xA2, 0x64, 0x61, 0x0F, 0x3D, 0xE0, 0x4F, 0x05, 0x66, 0x02, 0xBC,
                0x1C, 0x37, 0x7B, 0x0E, 0xB1, 0x41, 0x93, 0xA1, 0xC6, 0xE6, 0xFC, 0x6D, 0xA8, 0xE7,
                0x09, 0xC0, 0xB3, 0xB4, 0x56, 0xF9, 0x96, 0x33, 0x75, 0x36, 0xDF, 0x40, 0xEE, 0x5A,
                0xC8, 0x57, 0x1F, 0xE7, 0xC2, 0xBC, 0x7A, 0x19,
            ];
            let mut ss1 = [0u8; CRYPTO_BYTES];

            crypto_kem_dec(&mut ss1, &ct, &sk)?;

            let ss_ref = [
                0x10, 0xAF, 0x7B, 0xA1, 0xD6, 0x25, 0xB1, 0x61, 0x72, 0xC5, 0xB8, 0x0E, 0x2E, 0xE5,
                0x3A, 0xE9, 0xB7, 0xF3, 0xED, 0xBE, 0x2E, 0x22, 0x6F, 0x11, 0x3E, 0xDE, 0x5A, 0x0E,
                0xA8, 0xD1, 0xA9, 0x78,
            ];
            assert_eq!(ss_ref, ss1);
        }

        Ok(())
    }
}
